#!/usr/bin/env python
try:
    import os, sys, platform, threading
except ImportError:
    print('Unable to import system modules. Exiting.')
    raise

try:
    import re, subprocess, time, signal
    from subprocess import Popen
    from subprocess import PIPE
except ImportError:
    print('Unable to import regex and subprocess modules. Exiting.')
    raise

import logging
import logging.handlers


class TimeOut(Exception):
    pass


def sigAlarmHandler(signum, frame):
    raise Timeout


LOGGER_INFO = 1
LOGGER_DEBUG = 2
DEFAULT_LOG_FILE_PATH = "/tmp/CVGreenplumBkpRstUtility.log"


# def getLoggerObject(_fn="/tmp/CVGreenplumBkpRstUtility.log"):
#     logging_config = dict(
#         version = 1,
#         formatters = {
#             'f': {'format':'%(asctime)s %(name)-3s: %(levelname)-2s : %(message)s',
#                   'datefmt': '%m-%d-%Y %I:%M:%S %p'
#                   }
#         },
#         handlers = {
#             'default' : {'level': logging.DEBUG,
#                          'formatter': 'f',
#                          'class': 'logging.StreamHandler',},
#             'rotate_file': {
#                 'level': logging.DEBUG,
#                 'formatter': 'f',
#                 'class': 'logging.handlers.RotatingFileHandler',
#                 'filename': _fn,
#                 'maxBytes': 100000,
#                 'backupCount': 1,}
#         },
#         root = {
#             'handlers': ['rotate_file'],
#             'level': logging.DEBUG,
#         },
#     )
#     dictConfig(logging_config)
#     return logging.getLogger()

def getLoggerObject(logfilepath=DEFAULT_LOG_FILE_PATH):
    log = logging.getLogger(logfilepath)
    # default is INFO logging.. DEBUG mode has to be enabled with command line input
    log.setLevel(logging.INFO)
    fileformatter = logging.Formatter("%(process)d %(thread)x %(asctime) -4s %(levelname) -4s %(message)s",
                                      "%m/%d %H:%M:%S")
    filehandler = logging.handlers.RotatingFileHandler(logfilepath, maxBytes=5242880, backupCount=5)
    filehandler.setFormatter(fileformatter)
    log.addHandler(filehandler)

    return log


def info(_str):
    logging.info(_str)


def debug(_str):
    logging.debug(_str)


def warning(_str):
    logging.warning(_str)


def error(_str):
    logging.error(_str)


class ExecuteCommand:
    def __init__(self, logger, command_timeout):
        self.logger = logger
        self.commandTimeout = command_timeout
        signal.signal(signal.SIGALRM, sigAlarmHandler)
        self.monitorPidsList_lock = threading.Lock()
        self.pidsListToMonitor = []

    def runCommand(self, cmd, monitor=False):
        try:

            # -- timer for command. default is 60 secs
            # signal.alarm(self.commandTimeout)
            p = Popen(cmd, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                      close_fds=True)
            self.logger.info('Launched process %d -- > %s' % (p.pid, cmd))
            if monitor == True:
                self.monitorPidsList_lock.acquire()
                self.pidsListToMonitor.append(p.pid)
                self.monitorPidsList_lock.release()
            (out, err) = p.communicate()
            if p.returncode is 0:
                self.logger.debug('Command pid=%d successfully executed' % p.pid)
            else:
                self.logger.debug('Command pid=%d failed' % p.pid)


        except TimeOut:
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tb_str = ''.join(traceback.format_exception(exc_type, exc_value, exc_traceback))
            self.logger.error('Timeout exception for command process pid=%d ex=%s' % (p.pid, tb_str))
            try:
                p.kill()
                self.logger.error('Killed process pid=%d' % (p.pid))
            except OSError:
                exc_type, exc_value, exc_traceback = sys.exc_info()
                tb_str = ''.join(traceback.format_exception(exc_type, exc_value, exc_traceback))
                self.logger.error('Exception encountered while killing process pid=%d ex=%s' % (p.pid, tb_str))
        except:
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tb_str = ''.join(traceback.format_exception(exc_type, exc_value, exc_traceback))
            self.logger.error('Exception for command process pid=%d ex=%s' % (p.pid, tb_str))

        if out:
            self.logger.info('Command pid=%d STDOUT:\n%s' % (p.pid, str(out)))
        if err:
            self.logger.info('Command pid=%d STDERR:\n%s' % (p.pid, str(err)))

        # -- reset the timer
        # signal.alarm(0)

        return (p.returncode, out.strip(), err.strip())
