/*!
 * jQuery plugin
 * Component for search results view
 */

(function(factory) {
	var getDependencyMissingMsg = function(dependency) {
		return "cvDataGrid depends on " + dependency + " and seems to be missing. Please include necessary files.";
	};
	if (!jQuery) {
		throw getDependencyMissingMsg('jQuery');
	} else {factory(jQuery);}

}(function($) {
	"use strict";

	/**
	 * @function
	 * @name cvDataGrid
	 * @Description Creates a new cvDataGrid with given configuration
	 * @param {object}
	 *            config The initialization parameters supplied by the user
	 * @description The bare minimum arguments that you need to pass through config are listed below:
	 * 
	 * <pre><code>
	 * $('your_selector').cvDataGrid({
	 * 	dom : &quot;t&quot;,
	 * 	data : myData
	 * })
	 * </code></pre>
	 * 
	 * Where "t" means the final output contains the grid only. 'myData' is the javascript array or object
	 * that contains your data. You can read more about 'config' in the appropriate documentation page.
	 */
	$.fn.cvDataGrid = function(config) {
		return this.each(function() {
			var $this = $(this), data = $this.data('cvDataGrid');
			if (!data){$this.data('cvDataGrid', new DataGridView(this, config));}
		});
	};
	$.fn.cvDataGrid.id = 0;
}));
function DataGridView(element, config) {
	this.$element = $(element);
	this._origConfig = config;
	this._config = $.extend(true, {}, this.defaults, config);

	this._id = $.fn.cvDataGrid.id++;
	this.columnConfig = cvSearchUtil.sortArrayByKey(this._config.columnConfig, "dispOrder");
	// if static data
	if (!$.isEmptyObject(config.data)){this.data = config.data;}
	this.initParams();
	this.init();
	this.reloadGridView();
	this.initPoll();
};

//Represents a group
function ValueBucketObject() {
	this.value = null;
	this.count = 0;
}

//Used with bread crumbs
function State() {
	this.level = null;
	this.filterQuery = null;
	this.sortingOption = null;
	this.groupingCategory = null;
	this.data = [];
}

DataGridView.prototype.Constants = {
	ID_PREFIX : "cvDataGrid_",

	GRID_HOLDER_CLASS : "cvGridViewHolder",
	GRID_EMPTY_MSG_CLASS : "cvGridEmptyMsg",

	GRID_CONTAINER_CLASS : "cvDataGridContainer",
	GRID_PAGINATION_CLASS : "cvDataGridPagination",
	GRID_FILTER_CLASS : "cvDataGridFilter",
	GRID_LENGTH_CLASS : "cvDataGridLength",
	GRID_INFO_CLASS : "cvDataGridInfo",
	GRID_SWITCH_TOTABLE_CLASS : "cvDataGridTableSwitch",
	GRID_SWITCH_TOGRID_CLASS : "cvDataGridtoGridSwitch",
	GRID_TABLE_CLASS : "cvDataGridTable",
	GRID_SORTING_CLASS : "cvDataGridSorting",
	GRID_SORTING_OPTIONS : "cvDataGridSortingOptions",
	GRID_GROUPING_CLASS : "cvDataGridGrouping",
	GRID_GROUPING_OPTIONS : "cvDataGridGroupingOptions",
	GRID_BREADCRUMB : "cvBreadCrumb",

	SORTING_DESC : "_d",
	SORTING_ASC : "_a",

	PAGINATE_BUTTON_CLASS : "paginate_button",
	PAGINATE_MORE_BUTTON_CLASS : "btn-show-more",
	PAGINATE_LESS_BUTTON_CLASS : "btn-show-less",
	PAGINATE_SHOW_ALL_BUTTON_CLASS : "btn-show-all",
	PAGINATE_SCROLL_CLASS : "paginate-scroll",
	PAGINATE_SCROLL_THRESHOLD : 15,

	PAGINATION_TYPE_DEFAULT : 'page',
	PAGINATION_TYPE_MORE : 'more',
	PAGINATION_TYPE_SCROLL : 'scroll',

	RENDER_MODE_DATA : "renderData",
	RENDER_MODE_GROUP : "renderGroup",

	//Doing an OR because localMsg is not available when running jasmine test cases
	groupSortConstants : {
		VALUE : "Group Name",
		COUNT : "Documents"
	},
	EVENTS : {
		ORDER : "cvDataGrid.order"
	}
};

DataGridView.prototype.templates = {

	gridContainerTmpl : function(id, className) {
		var tmpl = "<div id='" + id + "' class='" + className + " form-inline'></div>";
		return tmpl;
	},
	emptyMsgHtml : function(msg) {
		//handle showing error msg
		return "<div class='itemDiv col-xs-12 col-sm-6 col-md-4 col-lg-4 " +
				DataGridView.prototype.Constants.GRID_EMPTY_MSG_CLASS + "'>" + msg + "</div>";
	},
	eachGroupHolderTmpl : function(className) {
		return "<div class='" + className + "'></div>";
	}
};

/**
 * @namespace
 * @name config
 * @description The initialization parameters that can be supplied to cvDataGrid()
 * @example $("#selector").cvDataGrid(config)
 * @property {boolean} autoRefresh sets autoRefresh true or false
 * @property {boolean} serverSide whether to fetch data from a server or not
 * @property {boolean} pagination to use pagination or not (requires 'p' in the dom string)
 * @property {string} paginationType What kind of pagination to use. Supports 'default', 'more', and 'scroll'.
 *           'default' is pagination with page numbers. 'more' introduces a single 'load more' button to load
 *           more data, and optionally a 'show less' button to hide data. 'scroll' implements a simple
 *           infinite scroll.
 * @property {number} pageLength The number of data elements that constitute a page
 * @property {number} displayStart The page to be loaded initially. Defaults to page 1
 * @property {function} fnItemTmpl A user defined function that receives each data element as an argument and
 *           returns the html content that should be displayed.
 * @property {function|string} ajax Specifies the URL from which server side data should be fetched. If ajax
 *           is a string, cvDataGrid automatically adds parameters to the supplied string and makes an ajax
 *           call, so the user need to provide only the base string. If ajax is a function, it is the
 *           responsibility of the user to construct the URL and make an ajax request.
 * @property {object} filterOptions Parameters for filter
 * @property {object} pagingOptions Parameters for pagination
 */

DataGridView.prototype.defaults = {
	autoRefresh : false,
	refreshInterval : 60000, //60 secs
	maxRefresh : 5,
	pageLength : 5,
	serverSide : false,
	paging : true,
	displayStart : 1,
	paginationType : 'page',

	/**
	 * @namespace
	 * @name pagingOptions
	 * @description Additional options to tweak pagination
	 * @property {boolean} lessButton If set to true, 'more' pagination will display a 'less' button that the
	 *           user can press to hide some content
	 */
	pagingOptions : {
		moreButtonText : "More",
		lessButtonText : "Less",
		lessButton : false,
		showAllButtonText : "Show All",
		showAllButton : false,
		loadDelay : 0,
	},

	/**
	 * @namespace
	 * @name filterOptions
	 * @description Additional options to tweak how filtering behaves
	 * @property {Array} notSearchable Content of any keys that appear in this list is exempted from the
	 *           search. Yes, this assumes that your data is an array of JSON objects.
	 */
	filterOptions : {
		placeholder : "",
	},
	//In breadcrumb, if clicking a level should delete the crumb and its children, or should delete just its children
	deleteClickedLevel : false,
	fnItemTmpl : function(model) {
		var html = "";
		cvSearchUtil.traverse(model, null, function(value, key) {

			if (key && value == null){html += "<div style='padding-left: 10px;'>" + key + "</div>";}else if (key && value != null){html += "<div>" + key + ": " + value + "</div>";} else {html += "<div>" + value + "</div>";}
		});

		return html;
	},
	language : {
		info : "Showing _START_ to _END_ of _TOTAL_ entries",
		infoFiltered : "filtered from _MAX_ total entries",
		filter : "Search: _INPUT_",
		search : "Search"
	}
};

DataGridView.prototype.initParams = function() {

	/**
	 * @namespace
	 * @name params
	 * @description Internal parameters for pagination. You won't usually need to touch this.
	 * @property {number} pageIndex Used internally to set the start index in the 'data' array from where data
	 *           is displayed
	 * @property {number} length Used internally to store the length of a page
	 * 
	 */
	var params = {};

	params.length = this._config.pageLength;
	if (params.length == null){params.length = 5;}

	if (this._config.displayStart == null || this._config.displayStart <= 1) {
		params.pageIndex = 0;
	} else {params.pageIndex = (this._config.displayStart - 1) * params.length;}

	params.columns = this._config.columns;
	this.params = params;
};

DataGridView.prototype.init = function() {

	this.filterCache = [];

	//renderCache is used for searching
	this.populateRenderCacheFlag = 0;

	this.renderCache = [];
	this.renderData = [];
	this.sortedCache = [];
	this.processDataIndices = [];
	//Stores states
	this.groupChain = [];
	this.breadCrumb = null;
	var timer = undefined;
	var initState = new State();
	this.groupChain.push(initState);

	this.renderMode = this.Constants.RENDER_MODE_DATA;
	try {
		var gridHolderId = this.Constants.ID_PREFIX + this._id;
		var gridHolderElm = $(this.templates.gridContainerTmpl(gridHolderId, this.Constants.GRID_HOLDER_CLASS));
		this.$element.append(gridHolderElm);
		var domArr = this._config.dom.split('');
		var parent = gridHolderElm, featureNode, newNode, ch, chNext, customAttr;
		for (var i = j = 0; i < domArr.length; i++) {
			featureNode = null;
			ch = domArr[i];
			if (ch == '<') {
				/* New container div */
				newNode = $('<div/>')[0];

				/* Check to see if we should append an id and/or a class name to the container */
				chNext = domArr[i + 1];
				if (chNext == "'" || chNext == '"') {
					customAttr = "";
					j = 2;//1 char for < and 1 char for quotes, so j=2
					while (domArr[i + j] != chNext) {
						customAttr += domArr[i + j];
						j++;
					}

					/*
					 * The attribute can be in the format of "#id.class", "#id" or "class" This logic breaks
					 * the string into parts and applies them as needed
					 */
					if (customAttr.indexOf('.') != -1) {
						var aSplit = customAttr.split('.');
						newNode.id = aSplit[0].substr(1, aSplit[0].length - 1);
						newNode.className = aSplit[1];
					} else if (customAttr.charAt(0) == "#") {
						newNode.id = customAttr.substr(1, customAttr.length - 1);
					} else {
						newNode.className = customAttr;
					}

					i += j; /* Move along the position array */
				}

				parent.append(newNode);
				parent = $(newNode);
			} else if (ch == '>') {
				/* End container div */
				parent = parent.parent();
			} else if (ch == 'l') {
				/* Length */
				featureNode = this.getHtmlForLength();
			} else if (ch == 'f') {
				/* Filter */
				featureNode = this.getHtmlForFilter();
			} else if (ch == 't') {
				/* Table */
				featureNode = this.getHtmlForGrid();
			} else if (ch == 'i') {
				/* Info */
				featureNode = this.getHtmlForInfo();
			} else if (ch == 'p') {
				/* Pagination */
				featureNode = this.getHtmlForPagination();
			} else if (ch == 's') {
				featureNode = this.getHtmlForSwitch();
			} else if (ch == 'o') {
				featureNode = this.getHtmlForSorting();
			} else if (ch == 'g') {
				featureNode = this.getHtmlForGrouping();
			} else if (ch == 'b') {
				featureNode = this.getHtmlForBreadCrumb();
			}
			parent.append(featureNode)
		}
		this.attachListeners();
	} catch (e) {
		console.error(e);
	}
};
DataGridView.prototype.getHtmlForGrid = function() {
	return $("<div class='" + this.Constants.GRID_CONTAINER_CLASS + "'></div>");
};
DataGridView.prototype.getHtmlForPagination = function() {
	if (!this._config.paging) {
		return null;
	}
	return $("<div class='" + this.Constants.GRID_PAGINATION_CLASS + "'></div>");
};
DataGridView.prototype.getHtmlForInfo = function() {
	return $("<div class='" + this.Constants.GRID_INFO_CLASS + "'></div>");
};
DataGridView.prototype.getHtmlForFilter = function() {

	var placeholder = null;
	if (this._config.filterOptions.placeholder){placeholder = this._config.filterOptions.placeholder;} else {placeholder = " ";}

	var filterBox = this._config.language.search + "<input type='text' placeholder='" + placeholder +
			"' class='form-control'>";
	var filterHtml = "<div class='" + this.Constants.GRID_FILTER_CLASS + "'><label>" + filterBox + "</label></div>";
	return filterHtml;
};

DataGridView.prototype.updateSortingOptions = function() {
	var optionDOM = this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS);
	optionDOM.empty();
	optionDOM.append("<option value='default'>Default</option>");

	if (this.renderMode == this.Constants.RENDER_MODE_DATA) {
		for (var i = 0; i < this._config.columns.length; i++) {
			if (this._config.columns[i].orderable) {
				var title = null;
				if (this._config.columns[i].title){title = this._config.columns[i].title;} else {title = this._config.columns[i].data;}

				optionDOM.append("<option value='" + this._config.columns[i].data + "_a'>" + title +
						" - ascending </option>");
				optionDOM.append("<option value='" + this._config.columns[i].data + "_d'>" + title +
						" - descending </option>");
			}
		}
	} else if (this.renderMode == this.Constants.RENDER_MODE_GROUP) {
		var dummy = new ValueBucketObject();

		for ( var member in dummy) {
			if (dummy.hasOwnProperty(member)) {
				if (typeof dummy[member] == "function") {
					continue;
				}

				var title = this.mapMemberToTitle(member);
				optionDOM.append("<option value='" + member + "_a'>" + title + " - ascending</option>");
				optionDOM.append("<option value='" + member + "_d'>" + title + " - descending</option>")
			}
		}
	}

	//Not really necessary to return since we are appending to the DOM
	//Improves readablity I guess.
	return optionDOM;
};

//When displaying groups, you should be able to sort groups based on their members. Mapping object properties to names
DataGridView.prototype.mapMemberToTitle = function(member) {
	switch (member) {
	case "value":
		return this.Constants.groupSortConstants.VALUE;
	case "count":
		return this.Constants.groupSortConstants.COUNT;
	}
}

DataGridView.prototype.getHtmlForSorting = function() {
	var sortingDiv = $("<div class='" + this.Constants.GRID_SORTING_CLASS + "'><label>Sort By: </label></div>");
	var options = $("<select class='" + this.Constants.GRID_SORTING_OPTIONS + " form-control'></select>");

	options.append("<option value='default'>Default</option>");

	for (var i = 0; i < this._config.columns.length; i++) {
		if (this._config.columns[i].orderable) {
			var title = null;
			if (this._config.columns[i].title){title = this._config.columns[i].title;} else {title = this._config.columns[i].data;}

			options
					.append("<option value='" + this._config.columns[i].data + "_a'>" + title +
							" - ascending </option>");
			options.append("<option value='" + this._config.columns[i].data + "_d'>" + title +
					" - descending </option>");
		}
	}
	sortingDiv.append(options);
	return sortingDiv;
};

DataGridView.prototype.getHtmlForGrouping = function() {
	var groupingDiv = $("<div class='" + this.Constants.GRID_GROUPING_CLASS + "''><label>" + localMsg.GroupBy +
			": </label></div>");
	var options = $("<select class='" + this.Constants.GRID_GROUPING_OPTIONS + " form-control'></select>");
	options.append("<option value='default'>None</option");
	for (var i = 0; i < this._config.columns.length; i++) {
		if (this._config.columns[i].groupable) {
			var title = null;
			if (this._config.columns[i].title){title = this._config.columns[i].title;} else {title = this._config.columns[i].data;}
			options.append("<option value='" + this._config.columns[i].data + "'>" + title + "</option>");
		}
	}

	groupingDiv.append(options);
	return groupingDiv;
};

DataGridView.prototype.getTitleForGroup = function(category) {
	for (var i = 0; i < this._config.columns.length; i++) {
		if (this._config.columns[i].groupable && this._config.columns[i].data == category) {
			if (this._config.columns[i].title) {
				return this._config.columns[i].title;
			} else {
				return this._config.columns[i].data;
			}
		}
	}
}

DataGridView.prototype.getHtmlForLength = function() {
	return $("<div class='" + this.Constants.GRID_LENGTH_CLASS + "'></div>");
};

DataGridView.prototype.getHtmlForSwitch = function() {
	return $("<div class='cvDataGridSwitch'><button class='btn btn-default " + this.Constants.GRID_SWITCH_TOGRID_CLASS +
			"' title='Grid' data-toggle='button' disabled><span class='glyphicon glyphicon-th-list'></span></button>" +
			"<button class='btn-default btn " + this.Constants.GRID_SWITCH_TOTABLE_CLASS +
			"' title='Table' data-toggle='button'><span class='glyphicon glyphicon-list-alt'></span></button></div>");
};

DataGridView.prototype.getHtmlForBreadCrumb = function() {

	var self = this;
	var crumbContainer = $("<div class='" + this.Constants.GRID_BREADCRUMB + "'></div>");
	var crumb = crumbContainer.cvBreadCrumb({
		levelZeroName : "All",
		renderFromLevel : 1,
		levelZeroResets : this._config.levelZeroResets,
		breadCrumbRenderFromLevel : this._config.breadCrumbRenderFromLevel
	});
	this.breadCrumb = crumb.data('cvBreadCrumb');
	crumbContainer.css("min-height", "17px");
	return crumbContainer;
}

DataGridView.prototype.render = function(startIndex, pageLength, recordsTotal, tempData) {

	if (!$.isArray(tempData)) {
		tempData = this.convertDataObjectToArray(tempData);
		recordsTotal = tempData.length;
	}

	if (recordsTotal === undefined) {
		console
				.error("recordsTotal undefined. If your data is a JSON object rather than an array, make sure you set the recordsTotal property");
		return;
	}

	if (this._config.paging === false) {
		startIndex = 0;
		//We are essentially displaying a single page with all the data in it - and no pagination buttons if control reaches here
		//setting length of page as total number of records
		pageLength = recordsTotal;
	} else {this.renderPagination(startIndex, pageLength, recordsTotal);}

	this.renderContent(startIndex, pageLength, recordsTotal, tempData);
	this.renderInfo(startIndex, pageLength, recordsTotal);
	
};

DataGridView.prototype.defaultPagination = function(recordsTotal, pageLength, paginationDisplayStartIndex, pageNo,
		lastPageNo) {
	var NO_OF_PAGES = 5;
	var isPrev = true;
	var isNext = true;
	var pageUl = $("<ul class='pagination'></ul>");
	var liClass = this.Constants.PAGINATE_BUTTON_CLASS;
	var prevPageLink = $("<li class='" + liClass + " previous' data-from='" + ((pageNo - 1) * pageLength - pageLength) +
			"'><a href='#'> Previous </a></li>");

	if (pageNo == 1){isPrev = false;}
	if (pageNo == lastPageNo){isNext = false;}

	if (!isPrev){prevPageLink.addClass("disabled");}
	pageUl.prepend(prevPageLink);
	if (recordsTotal > 0) {
		var i = 0;
		while (i < NO_OF_PAGES) {
			var pageLi = $("<li class='" + liClass + "' data-from=" + ((paginationDisplayStartIndex - 1) * pageLength) +
					"></li>");
			if (pageNo == paginationDisplayStartIndex){pageLi.addClass("active");}
			var eachPageLink = $("<a href='#'>" + paginationDisplayStartIndex + "</a>");
			pageLi.append(eachPageLink);
			pageUl.append(pageLi);
			paginationDisplayStartIndex++;
			i++;
			if (paginationDisplayStartIndex == lastPageNo + 1) {
				break;
			}
		}
	}
	var nextPageLink = $("<li class='" + liClass + " next' data-from=" + ((pageNo - 1) * pageLength + pageLength) +
			"><a href='#'> Next </a></li>");
	if (!isNext){nextPageLink.addClass("disabled");}
	pageUl.append(nextPageLink);

	return pageUl;
};

DataGridView.prototype.moreButtonPagination = function(pageLength, pageNo, lastPageNo) {
	var pageUl = $("<ul class=''></ul>");
	var isPrev = true;
	var isNext = true;
	var liClass = this.Constants.PAGINATE_MORE_BUTTON_CLASS;

	if (pageNo == this._config.displayStart){isPrev = false;}
	if (pageNo == lastPageNo){isNext = false;}

	var moreButton = $("<li class='" + liClass + " btn btn-primary' data-from=" + ((pageNo) * pageLength) +
			"  style='display:inline;  margin-left: 10px;'><a>" + this._config.pagingOptions.moreButtonText +
			"</a></li>");
	var lessButton = $("<li class='" + this.Constants.PAGINATE_LESS_BUTTON_CLASS + " btn btn-primary' data-from=" +
			((pageNo - 1) * pageLength) + " style='display:inline margin-left: 10px;'><a>" +
			this._config.pagingOptions.lessButtonText + "</a></li>");
	var showAllButton = $("<li class='" + this.Constants.PAGINATE_SHOW_ALL_BUTTON_CLASS +
			" btn btn-primary' data-from=" + ((pageNo) * pageLength) +
			" style='display:inline; margin-left: 10px;'><a>" + this._config.pagingOptions.showAllButtonText +
			"</a></li>");

	if (isNext === false) {
		moreButton.css("display", "none");
		showAllButton.css("display", "none");
	}
	if (isPrev === false) {
		lessButton.css("display", "none");
	} else if (lessButton.is(":hidden")) {
		lessButton.css("display", "inline");
	}

	pageUl.prepend(moreButton);
	if (this._config.pagingOptions.showAllButton){pageUl.prepend(showAllButton);}
	if (this._config.pagingOptions.lessButton){pageUl.prepend(lessButton);}

	return pageUl;
};

DataGridView.prototype.scrollPagination = function(pageLength, pageNo) {
	var pageUl = $("<ul class='pagination'></ul>");
	var liClass = this.Constants.PAGINATE_SCROLL_CLASS;

	var scrollDiv = $("<div class='" + liClass + " btn btn-default' style='display: block;' data-from=" +
			((pageNo) * pageLength) + "></div>");

	pageUl.prepend(scrollDiv);
	return pageUl;
};

DataGridView.prototype.renderPagination = function(startIndex, pageLength, recordsTotal) {
	/*
	 * startIndex = index in this.data[] from which we render content this.params.pageIndex is: (current Page
	 * number - 1) * pageLength
	 * 
	 * params.pageIndex is what determines the current page number. startIndex determines the position in the
	 * data array from where we start rendering data. Sometimes startIndex = params.pageIndex
	 * 
	 * Just don't confuse the two ;)
	 */

	var container = this.$element.find("." + this.Constants.GRID_PAGINATION_CLASS);
	if (container.length === 0) {
		return;
	}

	var total, len, start, isPrev = true, isNext = true;
	var NO_OF_PAGES = 5;

	//we have a function indexToPage() to do this but vars total and len are passed to defaultPagination() below.
	var pageCount = Math.ceil(recordsTotal / pageLength);

	if (recordsTotal === 0 || pageCount <= 1){isPrev = isNext = false;}

	var paginationDisplayStartIndex = 1;

	var pageNo = Math.floor((this.params.pageIndex / pageLength) + 1);

	this.setCurrentPage(pageNo);

	var lastPageNo;
	if (recordsTotal > 0){lastPageNo = Math.ceil(recordsTotal / pageLength);}else if (recordsTotal === 0){lastPageNo = pageNo; //Both will be 1
}

	if (pageNo > lastPageNo) {
		console.error("ERROR: start page number greater than the last page number");
		return;
	}

	if (this.params.pageIndex >= pageLength && pageCount > NO_OF_PAGES) {
		var mid = Math.ceil(NO_OF_PAGES / 2);
		var sub = Math.floor(NO_OF_PAGES / 2);
		var diff = lastPageNo - pageNo;
		if (pageNo > mid) {
			paginationDisplayStartIndex = pageNo - mid;
		}
	}

	var paginationElm = null;

	switch (this._config.paginationType) {

	case this.Constants.PAGINATION_TYPE_DEFAULT: {
		paginationElm = this.defaultPagination(recordsTotal,
				pageLength,
				paginationDisplayStartIndex,
				pageNo,
				lastPageNo);
		break;
	}

	case this.Constants.PAGINATION_TYPE_MORE: {
		paginationElm = this.moreButtonPagination(pageLength, pageNo, lastPageNo);
		break;
	}

	case this.Constants.PAGINATION_TYPE_SCROLL: {
		paginationElm = this.scrollPagination(pageLength, pageNo);
		break;
	}
	}

	container.empty().append(paginationElm);

	this.attachPaginationListeners(paginationElm, this._config.paginationType);
};

DataGridView.prototype.renderInfo = function(startIndex, pageLength, recordsTotal) {
	var container = this.$element.find("." + this.Constants.GRID_INFO_CLASS);
	if (container.length === 0) {
		return;
	}
	var msg = "";
	if (recordsTotal > 0) {
		var start = null;

		if (this._config.serverSide){start = this.params.pageIndex;}else if (!this.shouldClearGridContainer()){start = 0;} else {start = startIndex;}

		var end = null;

		if (startIndex + pageLength > recordsTotal || this.params.pageIndex + pageLength > recordsTotal){end = recordsTotal;} else {end = this.params.pageIndex + pageLength;}

		msg = cvFormatters.formatNamedArguments(this._config.language.info, {
			START : cvFormatters.formatNumber(start + 1),
			END : cvFormatters.formatNumber(end),
			TOTAL : cvFormatters.formatNumber(recordsTotal)
		}, null, "_", "_");

		//tip represents the data that we are rendering
		//if recordsTotal parameter passed to this function is less that length of tip.data,
		//then that means we have some sort of filter applied
		var tip = this.groupChain[this.groupChain.length - 1];
		if (recordsTotal < tip.data.length) {
			msg += " (" + cvFormatters.formatNamedArguments(this._config.language.infoFiltered, {
				MAX : cvFormatters.formatNumber(tip.data.length)
			}, null, "_", "_") + ")";
		}
	}
	container.empty().append("<span>" + msg + "</span>");
};

/* startIndex - the index of the element in the array */
DataGridView.prototype.renderContent = function(startIndex, pageLength, recordsTotal, tempData) {

	var self = this;
	var container = this.$element.find("." + this.Constants.GRID_CONTAINER_CLASS);

	if (this.shouldClearGridContainer()){container.empty();}

	if (this.renderMode == this.Constants.RENDER_MODE_DATA) {
		if ($.isEmptyObject(tempData)) {
			container.append(this.templates.emptyMsgHtml(cvSearchMessages.EmptyMsg));
		} else {
			var fn = self._config.fnItemTmpl;
			if (fn) {
				if (typeof fn === "string"){fn = cvSearchUtil.getFunctionByName(fn);}
				if (typeof fn === "function") {

					var pageNo = this.indexToPage(this.params.pageIndex);
					var pageElm = $("<div class = 'dataGrid-page' data-page = '" + pageNo + "'></div>");

					this.addContentToDom(pageElm, startIndex, pageLength, recordsTotal, tempData, fn);
					container.append(pageElm);
				}
			} else {
				container.append(this.templates.emptyMsgHtml(cvSearchMessages.ParsingErrorMsg));
			}
		}
	}

	else if (this.renderMode == this.Constants.RENDER_MODE_GROUP) {

		if ($.isEmptyObject(tempData)) {
			container.append(this.templates.emptyMsgHtml(cvSearchMessages.EmptyMsg));
		} else {
			var fn = self._config.fnGroupTmpl;
			if (fn) {
				if (typeof fn === "string"){fn = cvSearchUil.getFunctionByName(fn);}
				if (typeof fn === "function") {
					var pageNo = this.indexToPage(this.params.pageIndex);
					var pageElm = $("<div class = 'dataGrid-page' data-page='" + pageNo + "'></div");
					this.addContentToDom(pageElm, startIndex, pageLength, recordsTotal, tempData, fn);
					container.append(pageElm);
				}
			}
		}
	}
};

DataGridView.prototype.addContentToDom = function(targetDom, startIndex, pageLength, recordsTotal, tempData, fn) {
	var self = this;
	for (var i = startIndex; this.renderContentCondition(startIndex, pageLength, recordsTotal, i); i++) {
		var model = tempData[i];
		if (model === undefined) {
			continue;
		}
	    if(this.populateRenderCacheFlag != 1)
	    {
			var eachElm = fn.call(self, model);
			if (typeof (eachElm) === "string") {
				if ($(eachElm).length != 0){eachElm = $(eachElm);}
			}
		}
     	else
     		eachElm = this.renderCache[this.processDataIndices[i]];
		targetDom.append(eachElm);
	}
};

DataGridView.prototype.convertDataObjectToArray = function(data) {

	if (data === undefined){data = this.data;}

	if (!(data instanceof Object)) {
		return [ data ];
	}

	var arr = $.map(data, function(val, key) {
		var obj = {};
		obj[key] = val;
		return obj;
	});

	return arr;
};

/*
 * Refresh - whether the call is due to the autorefresh or not If the call is not due to autorefresh, we do
 * not want to show the load mask.
 */
DataGridView.prototype.reloadGridView = function(reset, mask) {
	var self = this;
	var url = null;

	if (reset !== undefined && reset == true){self.params.pageIndex = 0;}

	if (self._config.serverSide && self._config.ajax) {
		if (typeof self._config.ajax == "string") {
			url = self._config.ajax + "/?pageIndex=" + self.params.pageIndex + "&length=" + self.params.length;
			$.ajax({
				url : url,
				success : function(data) {

					if (data == null || $.isEmptyObject(data)){self.render(0, self.params.length, 1, []);}else {
						self.setData(data);

						//Because renderContent renders from startIndex to end of page (or end of data).\
						//startIndex earlier was, say, 12. Pagelength is 3. The array 'data' is now maybe [data13, data14, data15].
						//If the below line is not there, renderContent would try to access data[12].
						//Whereas what we really need is data[0] to data[pageLength-1]
						self.render(0, self.params.length, self.recordsTotal, self.data);
					}
				}
			});
		} else if (typeof self._config.ajax == "function") {
			self._config.ajax.call(self, self.params, function(data) {
				if (data == null || $.isEmptyObject(data)){self.render(0, self.params.length, 1, []);}else {
					self.setData(data);
					self.render(0, self.params.length, self.recordsTotal, self.data);
				}
			}, null);
		}
	}

	/* TODO: Test autorefresh with serverSide set to true. Then test with server side false */
	/* TODO: What happens when we introduce groups? */
	else if (self._config.autoRefresh && self._config.ajax) {
		var numberOfPages = (self.params.pageIndex) / self.params.length;
		var tempParams = {};

		if (typeof self._config.ajax == "string") {
			if (self._config.paginationType == self.Constants.PAGINATION_TYPE_MORE) {
				//In more button pagination, autoRefresh should fetch data from page 1 till current page
				var start = 0;
				var end = self.params.pageIndex + self.params.length;
				url = self._config.ajax + "/?pageIndex=" + start + "&length=" + end;
				$.ajax({
					url : url,
					success : function(data) {
						if (data == null || $.isEmptyObject(data)){self.render(0, self.params.length, 1, []);}else {
							self.setData(data);
							self.$element.find("." + self.Constants.GRID_CONTAINER_CLASS).empty();
							self.params.pageIndex = 0;
							//Render the first page
							self.render(0, self.params.length, self.recordsTotal, self.data);

							//We now have refreshed data. Click the more button till we reach our current page
							for (var i = 0; i < numberOfPages; i++) {self.$element.find("." + self.Constants.PAGINATE_MORE_BUTTON_CLASS).trigger("click");}

							tempParams.pageIndex = start;
							tempParams.length = self._config.pageLength;
							self.handleSearchAndSortingAfterRefresh(tempParams);
						}
					}
				});
			} else if (self._config.paginationType == self.Constants.PAGINATION_TYPE_DEFAULT) {
				url = self._config.ajax + "/?pageIndex=" + self.params.pageIndex + "&length=" + self.params.length;
				$.ajax({
					url : url,
					success : function(data) {
						if (data == null || $.isEmptyObject(data)){self.render(0, self.params.length, 1, []);}else {
							self.setData(data);
							self.render(self.params.pageIndex, self.params.length, self.recordsTotal, self.data);
							self.handleSearchAndSortingAfterRefresh(self.params);
						}
					}
				});
			}
		} else if (typeof self._config.ajax == "function") {
			var tempParams = $.extend(true, {}, self.params);

			if (self._config.paginationType == self.Constants.PAGINATION_TYPE_MORE) {
				tempParams.pageIndex = 0;
				//tempParams.length = self.params.pageIndex + self.params.length;
				tempParams.length = self._config.pageLength;
				tempParams.mask = mask;
				self._config.ajax.call(self, tempParams, function(data) {
					if (data == null || $.isEmptyObject(data)){self.render(0, self.params.length, 1, []);}else {
						self.setData(data);
						self.params.pageIndex = 0;

						//Doing sorting and searching will set renderData
						self.handleSearchAndSortingAfterRefresh(tempParams);
						// self.$element.find("." + self.Constants.GRID_CONTAINER_CLASS).empty();
						//
						// /*since renderData is set by search and sort function called bt handleSearchAndSortingAfterRefresh(),
						// all we need to do is display the renderData. If no search or sort was applied, this.renderData = this.data
						//
						// Why don't we render this.data here rather than this.renderData?
						// 	Because if we do that, search and sort results that are stored only in this.renderData won't be displayed.
						// */
						// self.render(0, self.params.length, self.renderData.length, self.renderData);
						//
						// for(var i=0; i<numberOfPages; i++)
						// 	self.$element.find("." + self.Constants.PAGINATE_MORE_BUTTON_CLASS).trigger("click");

					}
				}, null);
			} else if (self._config.paginationType == self.Constants.PAGINATION_TYPE_DEFAULT) {
				self._config.ajax.call(self, self.params, function(data) {
					self.setData(data);
					self.render(self.params.pageIndex, self.params.length, self.recordsTotal, self.data);
					self.handleSearchAndSortingAfterRefresh(self.params);
				})
			}
		}
	}
	//adding support for simple client side data
	else if (!self._config.serverSide && !self._config.ajax && self._config.data) {
		self.setData(self._config.data);
		self.render(self.params.pageIndex, self.params.length, self.recordsTotal, self.data);
	}
};

DataGridView.prototype.handleSearchAndSortingAfterRefresh = function(tempParams) {
	var query = this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").val();
	var sortingOption = this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS).val();

	var tip = this.groupChain[this.groupChain.length - 1];
	tip.filterQuery = query;
	tip.sortingOption = sortingOption;

	this.processFilterAndSorting(tempParams);
}

DataGridView.prototype.clear = function() {
	$("#" + this.getIdStr()).remove();
};
DataGridView.prototype.getIdStr = function() {
	return this.Constants.ID_PREFIX + this._id;
};
DataGridView.prototype.mask = function() {
	$("#" + this.getIdStr()).mask("");
};
DataGridView.prototype.unmask = function() {
	$("#" + this.getIdStr()).unmask();
};
DataGridView.prototype.attachListeners = function() {
	var self = this;
	this.$element.off('remove').on('remove', function(e, settings) {
		if (self.pollObj) {
			clearInterval(self.pollObj);
			self.pollObj = undefined;
		}
	});

	this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").off("input").on("input", function() {
		if(self.timer != undefined)
		{
			 clearTimeout(self.timer);
		}
		var query = $(this).val();
		var tip = self.groupChain[self.groupChain.length - 1];
		self.params.pageIndex = 0;

		tip.filterQuery = query;
		self.timer = setTimeout(function(){
		
		self.processFilterAndSorting.call(self);},500);
		
	});

	this.$element.find("." + this.Constants.GRID_SORTING_CLASS + " ." + this.Constants.GRID_SORTING_OPTIONS)
			.off("change").on("change", function() {
				var sortingOption = $(this).val();
				var tip = self.groupChain[self.groupChain.length - 1];
				self.params.pageIndex = 0;

				tip.sortingOption = sortingOption;
				self.processFilterAndSorting();
				/*
				 * Triggering an order event so that external code can do something when cvDataGrid is sorted
				 */
				self.$element.trigger(self.Constants.EVENTS.ORDER, sortingOption);
			});

	this.$element.find("." + this.Constants.GRID_GROUPING_CLASS + " ." + this.Constants.GRID_GROUPING_OPTIONS)
			.off("change").on("change", function() {
				var category = $(this).val();

				//Already grouped. But the user changes the drop down grouping option again
				if (self.renderMode == self.Constants.RENDER_MODE_GROUP && self.breadCrumb) {
					var level = self.breadCrumb.getTipLevel();
					var tempParams = self.breadCrumb.getParamsByLevel(level - 1);
					self.breadCrumb.removeLevel(tempParams.level);
					//removing breadrumb level will set the value of grouping combo box to default. Restoring it.
					$(this).val(category);
					//self.goBackInGroupChain(tipParams);
				}
				//If there is no breadcrumb. Then we are restricted to only one level of grouping
				else if (self.renderMode == self.Constants.RENDER_MODE_GROUP) {
					var dummyParams = {
						level : 0
					};
					self.goBackInGroupChain(dummyParams);
					$(this).val(category);
				}

				self.params.pageIndex = 0;
				self.processGrouping(category);
			});

	this.$element.off("cvDataGridGroupExpand").on("cvDataGridGroupExpand", function(e, val) {
		$("." + self.Constants.GRID_CONTAINER_CLASS).empty();
		self.setRenderMode(self.Constants.RENDER_MODE_DATA);
		self.expandGroup(val);
	});

	this.$element.off("cvBreadCrumbRemoveLevel", $("." + this.Constants.BREADCRUMB_HOLDER).parent())
			.on("cvBreadCrumbRemoveLevel",
					$("." + this.Constants.BREADCRUMB_HOLDER).parent(),
					function(e, crumbParams) {
						self.params.pageIndex = 0;
						self.goBackInGroupChain(crumbParams);
					});
};

DataGridView.prototype.attachPaginationListeners = function(pageUl, type) {
	var self = this;

	switch (type) {
	case this.Constants.PAGINATION_TYPE_DEFAULT: {
		pageUl.on("click", "li." + this.Constants.PAGINATE_BUTTON_CLASS, function(e) {
			if ($(this).hasClass('disabled')) {
				return;
			}

			self.params.pageIndex = parseInt($(this).data('from'));

			if (self._config.serverSide){self.reloadGridView();} else {self.render(self.params.pageIndex, self.params.length, self.renderData.length, self.renderData);}

		});
		break;
	}

	case this.Constants.PAGINATION_TYPE_MORE: {
		pageUl.on("click", "li." + this.Constants.PAGINATE_MORE_BUTTON_CLASS, function(e) {
			self.params.pageIndex = $(this).data('from');
			if (self._config.serverSide){self.reloadGridView();} else {self.render(self.params.pageIndex, self.params.length, self.renderData.length, self.renderData);}
		});

		pageUl.on("click", "li." + this.Constants.PAGINATE_LESS_BUTTON_CLASS, function(e) {
			var lastPage = $("." + self.Constants.GRID_CONTAINER_CLASS).find(".dataGrid-page").last();
			self.setPageIndexFromPage(lastPage.data("page") - 1);

			lastPage.remove();
			self.renderPagination(self.params.pageIndex, self.params.length, self.recordsTotal);
			self.renderInfo(self.params.pageIndex, self.params.length, self.recordsTotal);
		});

		pageUl.on("click", "li." + this.Constants.PAGINATE_SHOW_ALL_BUTTON_CLASS, function() {
			self.params.pageIndex = $(this).data('from');
			var startIndex = self.params.pageIndex;
			self.renderTillPage(startIndex, self.params.length, self.renderData.length, self.renderData, self
					.indexToPage(self.recordsTotal));
		});
		break;
	}

	case this.Constants.PAGINATION_TYPE_SCROLL: {
		$(window).off("scroll").on("scroll", function(e) {
			var scrollBottom = $(document).height() - $(window).height() - $(window).scrollTop();
			var cur = self.params.currentPage - 1;
			if (scrollBottom < self.Constants.PAGINATE_SCROLL_THRESHOLD) {
				if (self._config.paginationType == 'scroll') {
					var lastPageNo = Math.ceil(self.recordsTotal / self.params.length);

					if (self.params.currentPage >= lastPageNo) {
						self.hideSpinner();
						return;
					}
					self.turnNextPage();
					self.reloadGridView();

				}
			}
		});
		break;
	}
	}
};

DataGridView.prototype.setData = function(resp) {
	if (!$.isEmptyObject(resp)) {
		if (!$.isArray(resp)) {
			//Do we even need this? FORCE the user send his JSON data encapsulated in an array!
			this.recordsTotal = resp.recordsTotal;
			this.data = resp.data;
		} else {
			this.recordsTotal = resp.length;
			this.data = resp;
		}
		this.setRenderData(this.data);

		var tip = this.groupChain[this.groupChain.length - 1];
		tip.data = this.data;
	} else {
		this.recordsTotal = 0;
		this.data = [];
	}
};

DataGridView.prototype.populateRenderCache = function(data) {

	this.renderCache = [];

	if (this.renderMode === this.Constants.RENDER_MODE_DATA) {
		var fn = this._config.fnItemTmpl;
	} else if (this.renderMode === this.Constants.RENDER_MODE_GROUP) {
		fn = this._config.fnGroupTmpl;
	}

	if (typeof fn === "string"){fn = cvSearchUtil.getFunctionByName(fn);}

	if ($.isArray(data)) {
		for (var i = 0; i < data.length; i++) {
			this.renderCache.push($(fn.call(this, data[i])));
		}
	} else {this.renderCache = $(fn.call(this, data));}
};

DataGridView.prototype.getCellDisplayValue = function(data, config) {
	if (data) {
		var node = $("<span/>").attr("title", data);
		if (config && config.renderer && config.renderer.dispType == "html"){node.html(data);} else {node.text(data);}
		return node[0].outerHTML;
	}
	return cvSearchMessages.NotAvailable;
};

DataGridView.prototype.initPoll = function() {
	var self = this;
	if (self._config.autoRefresh) {
		var noOfRefreshes = 0;
		self.pollObj = setInterval(function() {
			if (noOfRefreshes < self._config.maxRefresh) {
				noOfRefreshes++;
				self.reloadGridView(false, false);
			} else {
				console.log("Clearing the timer as the no of refreshes reached the max refresh count:" +
						self._config.maxRefresh);
				clearInterval(self.pollObj);
				self.pollObj = undefined;
			}
		}, self._config.refreshInterval);
	}
};

DataGridView.prototype.shouldClearGridContainer = function() {

	if (this._config.paginationType === 'page') {
		return true;
	} else {
		return false;
	}
};

DataGridView.prototype.turnNextPage = function() {
	this.params.currentPage += 1;
	this.setPageIndexFromPage(this.params.currentPage);
};

DataGridView.prototype.indexToPage = function(index) {
	var pageLength = this.params.length;

	return Math.floor(index / pageLength) + 1;
};

DataGridView.prototype.setPageIndexFromPage = function(page) {
	page = parseInt(page);
	this.params.pageIndex = (page - 1) * this.params.length;
};

DataGridView.prototype.setCurrentPage = function(pageNo) {
	this.params.currentPage = pageNo;
};

DataGridView.prototype.renderContentCondition = function(startIndex, pageLength, recordsTotal, i) {

	if (this._config.paging === false) {

		if (i < recordsTotal) {
			return true;
		} else {
			return false;
		}
	} else {
		if (i < startIndex + pageLength && i < recordsTotal) {
			return true;
		} else {
			return false;
		}
	}
};

DataGridView.prototype.serverSideFilter = function(query) {

	if (typeof this._config.search === "function"){this._config.search(query);} else {console.error("No search function provided");}

	return;
};

DataGridView.prototype.clientSideFilter = function(query, data,indices) {

	var result = [];
	var re = new RegExp(query, 'i')
	var textNodes = null;

	for (var i = 0; i < this.renderCache.length; i++) {
		textNodes = this.getTextNodesIn(this.renderCache[i].find("." + this._config.searchableClass));
		for (var j = 0; j < textNodes.length; j++) {
			if (textNodes[j].textContent.match(re)) {
				result.push(data[i]);
				indices.push(i);
				break;
			}
		}
	}

	this.setFilterCache(result);
	delete textNodes;
	return result;
};

DataGridView.prototype.setFilterCache = function(result) {
	this.filterCache = result;
};

DataGridView.prototype.getTextNodesIn = function($searchables) {
	var textNodes = [];
	var nonWhiteSpaceMatcher = /\S/;
	$.each($searchables, function(i, node) {
		if (node.textContent.length != 0) {
			if (nonWhiteSpaceMatcher.test(node.nodeValue)){textNodes.push(node);}
		}
	});
	return textNodes;
};

DataGridView.prototype.sort = function(origData, key, desc) {

	//making a copy
	var data = origData.slice();

	if (key === "default") {
		return data;
	}

	if (data.length > 0) {

		if (typeof data[0][key] === "number") {
			if (desc) {
				data.sort(function(a, b) {
					return b[key] - a[key];
				});
			} else {
				data.sort(function(a, b) {
					return a[key] - b[key];
				});
			}
		} else if (typeof data[0][key] === "string") {
			if (desc) {
				data.sort(function(a, b) {
					if (a[key].toLowerCase() > b[key].toLowerCase()) {
						return -1;
					} else {
						return 1;
					// return b[key].localeCompare([a[key]]);
					}
				});
			} else {
				data.sort(function(a, b) {
					if (a[key].toLowerCase() < b[key].toLowerCase()) {
						return -1;
					} else {
						return 1;
					}
				});
			}
		}
	}

	return data;
};

DataGridView.prototype.group = function(data, params) {

	var tip = this.groupChain[this.groupChain.length - 1];
	var sourceData = tip.data;
	var category = params;
	var groupedData = [];

	for (var i = 0; i < data.length; i++) {
		var currentValue = data[i][category]

		if (!this.bucketAlreadyAdded(groupedData, currentValue, category)) {
			var bucket = new ValueBucketObject();
			bucket.value = currentValue;
			bucket.count = 1;
			groupedData.push(bucket);
		} else {
			this.updateBucketCount(groupedData, currentValue, category);
		}
	}

	return groupedData;
};

DataGridView.prototype.bucketAlreadyAdded = function(bucketList, value, category) {
	for (var i = 0; i < bucketList.length; i++) {
		if (bucketList[i].value == value) {
			return true;
		}
	}
	return false;
};

DataGridView.prototype.updateBucketCount = function(bucketList, value, category) {
	for (var i = 0; i < bucketList.length; i++) {
		if (bucketList[i].value == value){bucketList[i].count = bucketList[i].count + 1;}
	}
};

DataGridView.prototype.setRenderMode = function(mode) {
	this.renderMode = mode;
};

//This is the callback when cvBreadCrumb removes a level of chaining
DataGridView.prototype.goBackInGroupChain = function(crumbParams) {
	var tip = this.groupChain[this.groupChain.length - 1];
	var level = crumbParams.level;
	var state = this.groupChain[level];
	var processData = null;
	var renderData = null;
	this.$element.find("." + this.Constants.GRID_CONTAINER_CLASS).empty();

	//If the crumb we are going to represents a grouping
	if (state.groupingCategory) {
		var prevState = this.groupChain[level - 1];
		processData = prevState.data;
		if (prevState.filterQuery) {
			//prevState was NOT a grouping state. So renderCache should contain data items, not group items. Hence changing render mode
			var prevRenderMode = this.renderMode;
			this.setRenderMode(this.Constants.RENDER_MODE_DATA);
			this.populateRenderCache(processData);

			var result = this.clientSideFilter(prevState.filterQuery, processData);
			processData = result;

			//Restoring original render mode
			this.setRenderMode(prevRenderMode);
		}

		var newState = new State();
		newState.level = level;
		newState.filterQuery = state.filterQuery;
		newState.sortingOption = state.sortingOption;
		newState.groupingCategory = state.groupingCategory;

		this.groupChain.splice(level, 0, newState);
		var groupedData = this.group(processData, state.groupingCategory);
		processData = groupedData;
		newState.data = groupedData;

		this.setRenderMode(this.Constants.RENDER_MODE_GROUP);
		this.updateSortingOptions();

		if (state.filterQuery) {
			this.setRenderMode(this.Constants.RENDER_MODE_GROUP);
			this.populateRenderCache(processData);
			var result = this.clientSideFilter(state.filterQuery, processData);
			var processData = result;
			this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").val(state.filterQuery);
		}

		if (state.sortingOption) {
			var keyString = state.sortingOption;
			var key = keyString.substring(0, keyString.length - 2);
			var desc = false;
			var re = new RegExp("(_d)$");

			if (keyString.match(re)) {
				desc = true;
			}

			var sortedData = this.sort(processData, key, desc);
			processData = sortedData;
			this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS).val(state.sortingOption);
		}

		this.$element.find("." + this.Constants.GRID_GROUPING_OPTIONS).val(state.groupingCategory);
		toRenderdata = processData;
	}
	//The crumb contains normal data
	else {
		this.setRenderMode(this.Constants.RENDER_MODE_DATA);
		this.updateSortingOptions();
		processData = state.data;

		if (state.filterQuery) {
			this.populateRenderCache(processData)
			var result = this.clientSideFilter(state.filterQuery, processData);
			processData = result;
			this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").val(state.filterQuery);
		} else {this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").val("");}

		this.$element.find("." + this.Constants.GRID_GROUPING_OPTIONS).val("default");

		if (state.sortingOption) {
			var keyString = state.sortingOption;
			var key = keyString.substring(0, keyString.length - 2);
			var desc = false;
			var re = new RegExp("(_d)$");

			if (keyString.match(re)) {
				desc = true;
			}

			var sortedData = this.sort(processData, key, desc);
			processData = sortedData;

			this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS).val(state.sortingOption);
		} else {this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS).val("default");}

	}

	toRenderData = processData;
	this.groupChain.splice(level + 1, this.groupChain.length - level - 1);

	this.setRenderData(toRenderData);
	this.render(this.params.pageIndex, this.params.length, toRenderData.length, toRenderData);
}

DataGridView.prototype.processFilterAndSorting = function(params) {
	var processData = null;
	var toRenderData = null;
	this.timer = undefined;
	this.$element.find("." + this.Constants.GRID_CONTAINER_CLASS).empty();

	var tip = this.groupChain[this.groupChain.length - 1];
	processData = tip.data;

	if (tip.filterQuery) {
		this.populateRenderCache(processData);
		
		this.populateRenderCacheFlag = 1;
		this.processDataIndices = [];
		var result = this.clientSideFilter(tip.filterQuery, processData,this.processDataIndices);
		
		processData = result;
	}
	else
		this.populateRenderCacheFlag = 0;

	if (tip.sortingOption && tip.sortingOption != 'default') {
		this.populateRenderCacheFlag = 0;
		var keyString = tip.sortingOption;
		var key = keyString.substring(0, keyString.length - 2);
		var desc = false;
		var re = new RegExp("(_d)$");

		if (keyString.match(re)) {
			desc = true;
		}

		var sortedData = this.sort(processData, key, desc);
		processData = sortedData;
	}

	toRenderdata = processData;
	this.setRenderData(toRenderdata);

	//Param argument passed can overwrite default behavior
	if (!params) {
		this.renderTillPage(this.params.pageIndex,
				this.params.length,
				toRenderdata.length,
				toRenderdata,
				this.params.currentPage);
	} else {
		this
				.renderTillPage(params.pageIndex,
						params.length,
						toRenderdata.length,
						toRenderdata,
						this.params.currentPage);
	}

};

DataGridView.prototype.processGrouping = function(category) {
	var processData = null;
	var toRenderData = null;
	this.$element.find("." + this.Constants.GRID_CONTAINER_CLASS).empty();

	var tip = this.groupChain[this.groupChain.length - 1];
	processData = tip.data;

	if (tip.filterQuery) {
		this.populateRenderCache(processData);
		var result = this.clientSideFilter(tip.filterQuery, processData);
		processData = result;
	}

	toRenderData = processData;

	if (category != 'default') {
		var newState = new State();
		newState.level = tip.level + 1;
		newState.groupingCategory = category;

		var groupedData = this.group(processData, category);
		newState.data = groupedData;
		this.groupChain.push(newState);
		toRenderData = groupedData;
		if (this.breadCrumb){this.breadCrumb.addLevel("Group By: " + this.getTitleForGroup(category));}

		this.setRenderMode(this.Constants.RENDER_MODE_GROUP);
	}
	this.updateSortingOptions();
	this.setRenderData(toRenderData);
	this.render(this.params.pageIndex, this.params.length, toRenderData.length, toRenderData);
}

/*
 * So what exactly is the difference between this.renderData and this.data?
 * 
 * renderData is data that would be rendered. In most cases this would be the sames as this.data. However, if
 * the user does a search or sort, renderData is the result of the search while this.data is the original
 * data. We always display the renderData. All calls to render() that pass this.data to be displayed does so
 * because in those cases renderData = this.data. We might as well pass renderData there.
 */
DataGridView.prototype.setRenderData = function(data) {
	this.renderData = data;
}

DataGridView.prototype.expandGroup = function(val) {
	var tip = this.groupChain[this.groupChain.length - 1];
	var sourceData = this.groupChain[this.groupChain.length - 2].data;
	var category = tip.groupingCategory;

	var toRenderData = this.getGroupedItemsByVal(sourceData, val, category)

	var newState = new State();
	newState.level = tip.level + 1;
	newState.data = toRenderData;
	this.groupChain.push(newState);

	this.$element.find("." + this.Constants.GRID_GROUPING_OPTIONS).val("default");
	this.$element.find("." + this.Constants.GRID_SORTING_OPTIONS).val("default");
	this.$element.find("." + this.Constants.GRID_FILTER_CLASS + " input").val("");

	//expanding from group to data. Change sorting options
	this.updateSortingOptions();

	this.setRenderData(toRenderData);
	this.render(0, this.params.length, toRenderData.length, toRenderData);
	if (this.breadCrumb){this.breadCrumb.addLevel(val);}
};

DataGridView.prototype.getGroupedItemsByVal = function(data, val, category) {
	var result = [];
	for (var i = 0; i < data.length; i++) {
		if (data[i][category] == val){result.push(data[i]);}
	}
	return result;
}

/* Has the capability to render UP TO a certain page number */
DataGridView.prototype.renderTillPage = function(pageIndex, pageLength, recordsTotal, toRenderdata, targetPage) {
	var currentPage = this.indexToPage(pageIndex);
	var paginationType = this._config.paginationType;
	var nextBtn = null;
	
	this.render(pageIndex, pageLength, recordsTotal, toRenderdata);

	while (currentPage < targetPage) {
		switch (paginationType) {
		case this.Constants.PAGINATION_TYPE_DEFAULT:
			nextBtn = $("." + this.Constants.PAGINATE_BUTTON_CLASS + ".next");
			break;
		case this.Constants.PAGINATION_TYPE_MORE:
			nextBtn = $("." + this.Constants.PAGINATE_MORE_BUTTON_CLASS);
			break;
		}
		if (!nextBtn.is(":hidden")){nextBtn.trigger("click");}

		currentPage++;
	}
}

DataGridView.prototype.getTip = function() {
	return this.groupChain[this.groupChain.length - 1];
};
