var qb = qb || {}; // qb is the namespace for SOLR Query Builder and processor
/*
 * qb.base is the base object for other sub objects (qb.rv, qb.sv, qb.tv, etc.)
 * Base functionalities should be defined here. Prefix with _ for all private
 * properties and member functions
 */

qb.base = (function () {
	// private properties, start
	var _state = {};
	var _commonState = {};
	var _facetRequests = [];
	var _constants = {
		dateConstants: {
			nowDay: "NOW/DAY",
			nowYear: "NOW/YEAR"
		},
		allowedFieldTypeForFaceting: [
			'string',
			'int',
			'tinit',
			'pint',
			'long',
			'tlong',
			'plong',
			'double',
			'tdouble',
			'pfloat',
			'pdouble',
			'date',
			'tdate',
			'pdate',
			'epoch',
			'roundSecondsDate',
			'secondsDate',
			'lowercase',
			'boolean',
			'string_ci',
			'utcdatetime'
		],
		allowedFieldTypeForDateRange: ['date', 'roundSecondsDate', 'secondsDate', 'tdate', 'pdate', 'epoch', 'utcdatetime'],
		allowedFieldTypeForNumberRange: ['int', 'tinit', 'pint', 'long', 'tlong', 'plong'],
		allowedFieldTypeForDecimalRange: ['double', 'tdouble', 'pdouble', 'pfloat'],
		allowedFieldTypeForFacetSuggestion: ['string', 'lowercase', 'string_ci'],
		allowedFieldTypeForStringOp: ['string', 'lowercase', 'cvlowercase', 'string_ci', 'text_general', 'text_general_rev', 'text_rev'],
		defaultFieldsToBeExcluded: ['sea_push_counter', '_version_'],
		statsOperations: ["sum", "avg", "min", "max"],
		specialCharsRegExp: "\\+|\\-|\\&|\\||\\!|\\(|\\)|\\{|\\}|\\[|\\]|\\^|\\\"|\\~|\\?|\\:|\\\\|\\/",
		allowedSystemFields: ['data_source', 'data_source_type', 'data_source_name', 'timestamp']
	};
	var _solrParams = {
		QUERY: "q",
		PARSER: "defType",
		RETURN_TYPE: "wt",
		FIELDS: "fl",
		FILTER_QUERY: "fq",
		SORT: "sort",
		START: "start",
		ROWS: "rows",
		FACET: "facet",
		FACET_QUERY: "facet.query",
		FACET_FIELD: "facet.field",
		FACET_SIZE: "facet.size",
		FACET_F: "f.",
		FACET_LIMIT: ".facet.limit",
		FACET_MINCOUNT: ".facet.mincount",
		FACET_SORT: ".facet.sort",
		FACET_PREFIX: ".facet.prefix",
		HIGHLIGHT_QUERY: "hl",
		HIGHLIGHT_LIST: "hl.fl",
		HIGHLIGHT_FRAGMENT_SIZE: "hl.fragsize",
		FQ_OR: " OR ",
		FQ_AND: " AND ",
		RANGE_OP: " TO ",
		TAG_PREFIX: "tag_",
		JSON_FACET: "json.facet",
		JSON_FACET_TERMS: "terms",
		JSON_FACET_RANGE: "range",
		JSON_FACET_QUERY: "query",
		JSON_FACET_EXCLUDE_TAGS: "excludeTags",
		OPEN_PLACE_HOLDER: "QB_OPEN_BRACKET",
		CLOSE_PLACE_HOLDER: "QB_CLOSE_BRACKET"
	};
	var _solrProps = {
		defaultQuery: "*:*",
		defaultReturnType: "json",
		defaultParser: "edismax",
		defaultSelectHandler: "select",
		defaultSuggestHandler: "autocomplete",
		dtFormat: "YYYY-MM-DDTHH:mm:ss.SSS",
		defaultHighlightFragSize: 800,
		defaultNoOfColumnsToFetch: 15,
		// val - in seconds
		dtUnitMap: {
			"s": {
				unit: "SECOND",
				val: 1
			},
			"m": {
				unit: "MINUTE",
				val: 60
			},
			"h": {
				unit: "HOUR",
				val: 3600
			},
			"d": {
				unit: "DAY",
				val: 86400
			},
			"M": {
				unit: "MONTH",
				val: 2592000
			},
			"y": {
				unit: "YEAR",
				val: 31536000
			}
		}
	};

	// priate properties, end
	// private member functions, start
	/* initialize the state and other properties */
	function initialize(id, config) {
		_facetRequests = [];
		// currently config can be multiple; if needed can add filterMap here
		_state[id] = {
			config: $.extend(true, {}, config)
		};
		_commonState = {
			sReq: _getDefaultSearchReq(),
			filterMap: {
				'override': {},
				'normal': {}
			},
			customFilters: []
		};
		var fMap = _state[id].config.fieldMapping;
		var schema = _state[id].config.schema;

		if (!$.isEmptyObject(fMap)) {
			$.each(fMap, function (key, val) {
				for (var i = 0; i < schema.length; i++) {
					if (schema[i].fieldName === key) {
						// old field name itself found in schema, no need of
						// alternate field mapping
						// so delete it from fMap
						delete fMap[key];
						break;
					}
				}
			});
		} else {
			_state[id].config.fieldMapping = {};
		}
	}
	/* to get default search request */
	function _getDefaultSearchReq() {
		var req = {};
		req[_solrParams.QUERY] = _solrProps.defaultQuery;
		req[_solrParams.RETURN_TYPE] = _solrProps.defaultReturnType;
		req[_solrParams.ROWS] = 0;
		req[_solrParams.PARSER] = _solrProps.defaultParser;
		return req;
	}

	function _getFieldTypeForReport(fType) {
		if (_isDateRangeAllowed(fType))
			return "TimeStamp";
		if (_isNumberRangeAllowed(fType))
			return "Long";
		if (_isDecimalRangeAllowed(fType))
			return "Double";
		if (_isTextField(fType))
			return "String";
		return fType;
	}

	function _getFieldTypeForSolr(fType) {
		switch(fType) {
			case "Date" :
				return "date";
			case "TimeStamp" :
				return "utcdatetime";
			case "Integer" :
				return "int";
			default:
				return "string";
		}
	}

	function _isNumberRangeAllowed(fType) {
		return $.inArray(fType, _constants.allowedFieldTypeForNumberRange) > -1;
	}

	function _isDecimalRangeAllowed(fType) {
		return $.inArray(fType, _constants.allowedFieldTypeForDecimalRange) > -1;
	}

	function _isDateRangeAllowed(fType) {
		return $.inArray(fType, _constants.allowedFieldTypeForDateRange) > -1 ||
			fType.startsWith(qb.base.constants.schemaCustomDateFormat);
	}

	function _isTextField(fType) {
		return $.inArray(fType, _constants.allowedFieldTypeForStringOp) > -1;
	}

	function _getIndexField(id, fName) {
		var schemaObj = _getSchemaObj(id,fName+"_idx");
		if(schemaObj.fieldName)
			return schemaObj.fieldName;
		var idxField = _getFieldByTypeMatch(id, fName, "_idx");
		return (idxField === null) ? fName : idxField;
	}

	function _getHierarchicalField(id, fName) {
		return _getFieldByTypeMatch(id, fName, "_path");
	}

	function _getSLevelField(id, fName) {
		return (fName.toLowerCase().indexOf("slevel_") > -1 ? fName : null);
	}

	function _getFieldName(id, fName) {
		return (_state[id].config.fieldMapping.hasOwnProperty(fName) ? _state[id].config.fieldMapping[fName] : fName);
	}
	// Look up the schema response for the given field name and returns the
	// object if found
	function _getSchemaObj(id, fName) {
		var schemaObj = $.grep(_state[id].config.schema, function (o) {
			return o.fieldName === fName;
		})[0];
		if ($.isEmptyObject(schemaObj)) {
			// if not found in schema, default to String type
			schemaObj = {
				origType: "string"
			}
		}
		return schemaObj;
	}
	// Returns the copy field if its type contains matchStr
	// Else Returns the field name if its type contains matchStr
	// Else NULL
	function _getFieldByTypeMatch(id, fName, matchStr) {
		var fNameToReturn = null;
		var schemaObj = _getSchemaObj(id, fName);
		if (schemaObj) {
			var fType = schemaObj.origType || schemaObj.type;
			var isNumeric = _isNumberRangeAllowed(fType) || _isDecimalRangeAllowed(fType);
			if (public.isDateRangeAllowed(fType) || isNumeric) {
				return fName;
			}
			if (schemaObj.copyFields) {
				$.each(schemaObj.copyFields, function (i, eachField) {
					if (eachField.toLowerCase().indexOf(matchStr) > -1) {
						fNameToReturn = eachField;
						return true;
					}
				});
			} else if (fType.toLowerCase().indexOf(matchStr) > -1) {
				fNameToReturn = fName;
			}
		}
		return fNameToReturn;
	}

	function _isGET(searchReq) {
		// if (!$.isEmptyObject(searchReq[_solrParams.FIELDS]) &&
		// searchReq[_solrParams.FIELDS].length >
		// _solrProps.defaultNoOfColumnsToFetch)
		// return false;
		// if (!$.isEmptyObject(searchReq[_solrParams.JSON_FACET]))
		// return false;
		// Using POST for all request as there is an issue with + in GET API
		if(!$.isEmptyObject(searchReq))
			return searchReq.reqType === "GET";
		return false;
	}

	function _getPOSTSearchRequest(searchReq) {
		// temp code
		if (searchReq.hasOwnProperty("useDCubeReq") && searchReq.useDCubeReq === true) {
			return _getDCubeSearchRequest(searchReq);
		} else {
			return _getSimplifiedSearchRequest(searchReq);
		}
	}

	function _getDCubeSearchRequest(searchReq) {
		var dCubeReq = {};
		var searchParams = [];
		$.each(searchReq, function (key, obj) {
			if (obj instanceof Array) {
				if (!$.isEmptyObject(obj)) {
					for (var i = 0; i < obj.length; i++) {
						if (obj[i] !== undefined) {
							searchParams.push({
								"key": key,
								"value": obj[i]
							});
						}
					}
				}
			} else {
				if (!(obj instanceof Object && $.isEmptyObject(obj)) && obj !== undefined && obj !== null &&
					obj !== "null") {
					searchParams.push({
						"key": key,
						"value": typeof (obj) === "string" ? obj : JSON.stringify(obj)
					});
				}
			}
		});
		dCubeReq.searchParams = searchParams;
		return JSON.stringify(dCubeReq);
	}
	/* submit search to the server */
	// commenting for now to avoid dcubeSearchReq
	/*
	 * function _executeQuery(id, searchReq, fnToGetRequest, callback) { var
	 * reqObj = {}; if (_isGET(searchReq)) { reqObj = { reqType : "GET", reqStr :
	 * _state[id].config.searchUrl + "?" +
	 * $.param(_getSimplifiedSearchRequest(searchReq), true), params : null }; }
	 * else { reqObj = { reqType : "POST", reqStr : _state[id].config.searchUrl,
	 * params : _getDCubeSearchRequest(searchReq) }; }
	 * console.time(fnToGetRequest); cvUtil.loadPage(reqObj.reqStr,
	 * reqObj.params, function(resp) { console.timeEnd(fnToGetRequest);
	 * callback(resp, reqObj); }, function(xhr, ajaxOptions, thrownError) {
	 * console.log(thrownError); callback({}, reqObj); }, { "callType" :
	 * reqObj.reqType, "contentType" : "text/plain; charset=utf-8", "addCsrf" :
	 * false }); }
	 */
	function _getQueryString(url, queryParams) {
		return url + (url.indexOf("?") === -1 ? "?" : "&") + queryParams;
	}

	function _executeQuery(id, searchReq, fnToGetRequest, callback) {
		var reqObj = {};
		var config = _state[id].config;
		var originalReqParams = _.cloneDeep(searchReq);
		if (config.hasOwnProperty("additionalParams") && !$.isEmptyObject(config.additionalParams)) {
			searchReq = $.extend({}, searchReq, config.additionalParams);
		}
		// Adding all the executed facet requests to a map
		if(searchReq['json.facet']){
			if(!_facetRequests){
				_facetRequests = [];
			}
			_facetRequests.push(searchReq);
		}
		if (_isGET(searchReq)) {
			//delete request type parameter as this should get added to API request URL
			delete originalReqParams.reqType;
			reqObj = {
				reqType: "GET",
				reqStr: _getQueryString(_state[id].config.searchUrl, $.param(originalReqParams, true)),
				params: null
			};
		} else {
			reqObj = {
				reqType: "POST",
				reqStr: _state[id].config.searchUrl,
				params: _getPOSTSearchRequest(searchReq)
			};
		}
		if (searchReq.invalidSearchUrl) {
			window.setTimeout(function () {
				callback({}, reqObj);
			}, 100);
			return;
		}
		var reqOptions = {
			callType: reqObj.reqType,
			addCsrf: false,
			traditional: true
		};
		// temp code
		if (searchReq.hasOwnProperty("useDCubeReq") && searchReq.useDCubeReq === true) {
			reqOptions.contentType = "text/plain;";
			reqOptions.headers = {
				Accept : "application/json, text/plain, */*"
			}
		}

		console.time(fnToGetRequest);
		cvUtil.loadPage(reqObj.reqStr, reqObj.params, function (resp) {
			console.timeEnd(fnToGetRequest);
			if(typeof resp === "string")
				resp = JSON.parse(resp);
			callback(resp, reqObj);
		}, function (xhr, ajaxOptions, thrownError) {
			console.log(thrownError);
			callback({}, reqObj);
		}, reqOptions);
	}

	/* to reset all facet values in the search request */
	function _resetFacetParamsInSearchReq(searchReq) {
		delete searchReq[_solrParams.FACET_PARAM];
		delete searchReq[_solrParams.FACET_SIZE];
		delete searchReq[_solrParams.FACET_FIELD];
		delete searchReq[_solrParams.FILTER_QUERY];
		delete searchReq[_solrParams.FACET_QUERY];
		$.each(searchReq, function (key) {
			if (key.indexOf(_solrParams.FACET_F) > -1) {
				delete searchReq[key];
			}
		});
		return searchReq;
	}
	/* to append any custom filter query to search request */
	function _appendFilterQuery(searchReq, fq) {
		var fqArr = searchReq[_solrParams.FILTER_QUERY];
		if (!$.isEmptyObject(fqArr)) {
			fqArr.push(fq);
		} else {
			fqArr = [fq];
		}
		searchReq[_solrParams.FILTER_QUERY] = fqArr;
		return searchReq;
	}
	/* to append any highlight query to the search request */
	function _appendHighlightQuery(searchReq, hlParams, hlKey) {
		var hlFieldsArr = searchReq[_solrParams.HIGHLIGHT_LIST];
		var hlArr = [];
		if (!$.isEmptyObject(hlParams)) {
			$
				.each(hlParams,
					function (i, hlObj) {
						if (hlObj.fragmentSize) {
							searchReq[_solrParams.FACET_F + hlObj.fieldName + "." +
								_solrParams.HIGHLIGHT_FRAGMENT_SIZE] = hlObj.fragmentSize;
						}
						hlArr.push(hlObj.fieldName);
					});
		}
		if (!$.isEmptyObject(hlFieldsArr)) {
			hlFieldsArr = hlFieldsArr.concat(hlArr);
		} else {
			hlFieldsArr = hlArr;
		}
		if (!$.isEmptyObject(hlFieldsArr)) {
			searchReq[_solrParams.HIGHLIGHT_QUERY] = true;
			searchReq[_solrParams.HIGHLIGHT_LIST] = hlFieldsArr;
			searchReq[_solrParams.HIGHLIGHT_FRAGMENT_SIZE] = _solrProps.defaultHighlightFragSize;
		}
	}

	function _mapHightlightResp(docs, hlMap, hlKey) {
		var docsMap = {};
		var docsToReturn = [];
		if (!hlKey) {
			hlKey = "contentid";
		}
		$.each(hlMap, function (key, hlObj) {
			if (!$.isEmptyObject(hlObj)) {
				var hlProcessedObj = {};
				// if highlight response contains array, get the value at index
				// 0
				$.each(hlObj, function (k, v) {
					if ($.isArray(v)) {
						hlProcessedObj[k] = v[0];
					} else {
						hlProcessedObj[k] = v;
					}
				});
				hlMap[key] = hlProcessedObj;
			}
		});
		$.each(docs, function (i, docsObj) {
			docsMap[docsObj[hlKey]] = docsObj;
		});
		var resultsMap = $.extend(true, {}, docsMap, hlMap);
		$.each(resultsMap, function (key, docsObj) {
			docsToReturn.push(docsObj);
		});
		// console.log(docsToReturn);
		return docsToReturn;
	}
	/* to append any custom raw query */
	function _appendRawQuery(reqObj, rawQuery) {
		var rawQueryObj = rawQuery;
		if (typeof rawQuery === 'string') {
			rawQueryObj = $.parseParams(rawQuery);
		}
		if (!$.isEmptyObject(rawQueryObj)) {
			$.each(rawQueryObj, function (key, val) {
				// for now lets not overwrite the value in the reqObj
				if (!reqObj.hasOwnProperty(key)) {
					reqObj[key] = val;
				}
			});
		}
	}
	/* facet.query */
	function _getFacetQueryStr(fName, eachValueObj) {
		var str = _getFqStr(fName, eachValueObj);
		var keyPrefix = "";
		if (eachValueObj.key) {
			keyPrefix = "{!key=" + eachValueObj.key + "}";
		}
		return keyPrefix + str;
	}
	/* filter query fq=<something> */
	function _getFqStr(fName, eachValueObj) {
		var str = "";
		if ($.isEmptyObject(eachValueObj)) {
			return;
		}
		if (eachValueObj.isRange) {
			str = fName + ":[" + _getRangeValue(eachValueObj.start) + " TO " + _getRangeValue(eachValueObj.end) + "]";
		} else if ($.isArray(eachValueObj.value)) {
			var i = 0;
			while (i < eachValueObj.value.length) {
				str += fName + ":\"" + _escapeSpecialChars(eachValueObj.value[i]) + "\"";
				i++;
				if (i < eachValueObj.value.length) {
					str += (eachValueObj.operator ? eachValueObj.operator : _solrParams.FQ_OR);
				}
			}
		} else {
			str = fName + ":\"" + _escapeSpecialChars(eachValueObj.value) + "\"";
		}
		return str;
	}

	function _escapeSpecialChars(str, isSortField) {
		// escape
		// http://lucene.apache.org/core/2_9_4/queryparsersyntax.html#Escaping
		// Special Characters
		var solrSpecialChars = _constants.specialCharsRegExp;
		solrSpecialChars = isSortField ? solrSpecialChars + "|\\ " : solrSpecialChars;
		return str.replace(new RegExp(solrSpecialChars, "g"), "\\$&");
	}

	function _unEscapeSpecialChars(str) {
		// replace \\+ as +
		return str.replace(new RegExp("(\\\\(" + _constants.specialCharsRegExp + "))", "g"), "$2");
	}

	function _isUnaryOp(char) {
		return $.inArray(char, ['!', '=', '<', '>']) > -1;
	}

	// TODO: for now introducing processColon so that we process the colon only
	// in case of custom groups
	function _getSOLRExpr(id, fName, exp, fType, isExact, processColon, dontEscapeSpecialChars) {
		try {
			fName = _getFieldName(id, fName);
			var firstChar = "",
				secondChar = "",
				prefix = "";
			var val = exp;
			var isNumeric = _isNumberRangeAllowed(fType) || _isDecimalRangeAllowed(fType);
			if (!isExact || isNumeric) {
				firstChar = exp.charAt(0);
			}
			// (minus) should not be allowed in the second unaryop check, so not
			// including in the array
			if (_isUnaryOp(firstChar) || firstChar === "-") {
				val = exp.substring(1);
				if (exp.length > 1) {
					secondChar = exp.charAt(1);
					if (_isUnaryOp(secondChar)) {
						val = val.substring(1);
					}
				}
			}
			if (processColon && val.indexOf(":") > -1) {
				var splitAt = val.indexOf(":");
				// \: will be used for escaping :
				if (val.charAt(splitAt - 1) !== "\\") {
					fName = val.substring(0, splitAt);
					val = val.substring(splitAt + 1);
				}
			}
			if (!processColon) {
				// if criteria doesn't have \" and \" (someone searching for
				// double quotes)
				// AND the criteria has space AND enclosed with double quotes
				if (val.search(/^\\\"[^\\]+\\\"$/g) === -1 && ((val.split(' ') || []).length > 1) && val.search(/^\"[^"]+\"$/g) !== -1) {
					val = val.substring(1, val.length - 1);
					isExact = true;
				} else if (val.search(/^\\\"[^\\]+\\\"$/g) !== -1) {
					val = val.substring(2, val.length - 2);
					val = '"' + val + '"';
					isExact = true;
				} else {
					if (!dontEscapeSpecialChars) {
						val = _escapeSpecialChars(val);
					}
				}
			}

			var isDateRange = public.isDateRangeAllowed(fType);
			if (isDateRange) {
				var dtObj;
				if (firstChar && (firstChar === '>' || firstChar === '<')) {
					dtObj = _getDateRangeObj({
						'relative': firstChar + val
					});
				} else {
					dtObj = _getDateRangeObj(val);
				}
				if (!$.isEmptyObject(dtObj) && dtObj.start) {
					val = dtObj.start;
					if (firstChar === "<") {
						val = dtObj.end;
					}
				}
			}
			switch (firstChar) {
				case "-":
					if (isExact && !isNumeric && !isDateRange) {
						val = "\"" + val + "\"";
					}
					prefix = "-";
					break;
				case "!":
					if (isExact && !isNumeric && !isDateRange) {
						val = "\"" + val + "\"";
					}
					prefix = "!";
					break;
				case "=":
					if (isExact && !isNumeric && !isDateRange) {
						val = "\"" + val + "\"";
					}
					break;
				case "<":
					if (secondChar === "=") {
						val = "[*" + " TO " + val + "]";
					} else {
						val = "{*" + " TO " + val + "}";
					}
					break;
				case ">":
					if (secondChar === "=") {
						val = "[" + val + " TO *]";
					} else {
						val = "{" + val + " TO *}";
					}
					break;
				default:
					//No need to escape/enclose value in quotes("") for empty value
					if (isExact && !isNumeric && !isDateRange && _getQueryForEmptyVal() !== val) {
						val = "\"" + val + "\"";
					}
					break;
			}
			if (!isExact) {
				fName = _getIndexField(id, fName);
			}
			return prefix + fName + ":" + val;
		} catch (e) {
			console.error(e);
			return exp;
		}
	}

	function _processAndOrQuery(id, fName, exp, fType, isExact, processColon) {
		//&& takes higher precedence
		var andParts = exp.split(_solrParams.FQ_AND),
			orParts, andArr = [],
			orArr = [];
		for (var i = 0; i < andParts.length; i++) {
			//|| takes next precedence
			orParts = andParts[i].split(_solrParams.FQ_OR);
			orArr = [];
			for (var j = 0; j < orParts.length; j++) {
				orArr.push(_getSOLRExpr(id, fName, orParts[j], fType, isExact, processColon));
			}
			andArr.push(orArr.join(_solrParams.FQ_OR));
		}
		return andArr.join(_solrParams.FQ_AND);
	}

	function _processFilterQuery(id, fName, fValue, fType, isExact, isExclude, processColon) {
		var arr, exp, q, prefix = "";
		isExact = processColon ? false : isExact;
		if (typeof fValue === 'undefined' || fValue === "") {
			return "";
		}

		if(_getQueryForEmptyVal() === fValue) {
			return _getSOLRExpr(id, fName, fValue, fType, true, true)
		}
		// replace && by <space>AND<space>, replace || by <space>OR<space>,
		// replace != by !
		fValue = fValue.split(",").join("||");
		fValue = fValue.replace(/\s*&&\s*/g, _solrParams.FQ_AND).replace(/\s*\|\|\s*/g, _solrParams.FQ_OR)
			.replace(/\!\=/g, "!");

		//parentheses takes higher precedence
		var pattern = /\([^()]*\)/g;
		var arr = pattern.exec(fValue);
		//no parentheses found
		if (arr === null || isExact === true)
			return _processAndOrQuery(id, fName, fValue, fType, isExact, processColon);

		//parentheses found
		do
		{
			exp = arr[0], q = "";
			//remove parentheses
			exp = exp.substring(1, exp.length-1);
			//inside the loop, use PLACE_HOLDER instead of parantheses
			//so that SOLR expression will not get processed again by the reg exp
			q = _solrParams.OPEN_PLACE_HOLDER + _processAndOrQuery(id, fName, exp, fType, isExact, processColon) + _solrParams.CLOSE_PLACE_HOLDER;
			if (arr.index > 0 && (fValue[arr.index - 1] === "!" || fValue[arr.index - 1] === "-")) {
				prefix = fValue[arr.index - 1];
				q = "-" + q;
			}
			fValue = fValue.replace(prefix + arr[0], q);
		} while ((arr = pattern.exec(fValue)) != null);
		fValue = fValue.replace(new RegExp(_solrParams.OPEN_PLACE_HOLDER,"g"), "(").replace(new RegExp(_solrParams.CLOSE_PLACE_HOLDER,"g"), ")");
		return fValue;
	}

	function _handleEscapingForQueryVal(value, fieldName) {
		//remove starting and ending braces ()
		var enclosedInParenthesis  = value.startsWith("(") && value.endsWith(")");
		var startToken = "";
		var endToken = "";
		if(enclosedInParenthesis) {
			startToken = "(";
			endToken = ")";
			value = value.slice(1,value.length - 1);
		}
		var isSortField = fieldName.endsWith("_sort");
		return startToken + _escapeSpecialChars(value, isSortField) + endToken;
	}

	function _parseSearchObject(id,searchObject,searchVal) {
		var query;
		var fName = searchObject.fName;
		if (!$.isEmptyObject(searchObject)) {
			if(searchObject.value)
				searchVal = searchObject.value;

			if (fName === '*')
				return searchVal;

			var isUrlField = fName === "Url" || fName === "filePath";
			var schemaObj = _getSchemaObj(id, fName + (isUrlField ? '_sort' : '_idx'));
			if(schemaObj.fieldName)
				fName = schemaObj.fieldName;
			var logicalRegex = /(\) OR \(|\) AND \()/gi; // ") OR (" or ") AND ("
			var complexLogicalRegex = /(\(\(|\)\))/g; // ignore complex logical operations like "((this) OR (this)) AND (that)" i.e. parenthesis hierarchy
			if (logicalRegex.test(searchVal) && !complexLogicalRegex.test(searchVal)) {
				logicalRegex = /( OR | AND )/gi; // " OR " or " AND "
				var searchTokens = _.split(searchVal, logicalRegex); // => ['(test*)', ' OR ', '(test)']
				query = '';
				searchTokens.forEach(function(token) {
					if (!logicalRegex.test(token)) {
						query += fName + ':' + _handleEscapingForQueryVal(token, fName);
					} else {
						query += token;
					}
				});
			} else {
				query = fName + ':' + _handleEscapingForQueryVal(searchVal.trim(), fName);
			}
		}
		return query || searchVal;
	};

	/*
	 * filter query fq=<something>; advanced version TODO: support logical
	 * expressions in facets view
	 */
	function _getFilterQuery(id, fName, fValue, fType, isExact, isExclude, processColon) {
		var str = "",
			rValue = "";

		if (!fType) {
			fName = _getFieldName(id, fName);
			var schemaObj = _getSchemaObj(id, fName);
			// CR2 origType has the SOLR datatype where as search view has it in
			// type and origType will be undefined
			fType = schemaObj.origType || schemaObj.type;
		}
		if ($.isEmptyObject(fValue)) {
			return;
		}
		if ($.isArray(fValue) || typeof fValue !== 'object') {
			var filterQuery = "";
			if ($.isArray(fValue)) {
				fValue = $.extend(true, [], fValue);
				var combinedQuery = "";
				$.each(fValue, function (index, value) {
					var processColon = false,
						val = value + "";
					if (typeof value === 'object' && value.groupName) {
						val = value.groupValues;
						isExact = false;
						processColon = true;
					}
					if (val !== "") {
						combinedQuery += (isExclude ? "-" : "") + "(" + _processFilterQuery(id, fName, val, fType, isExact, isExclude, processColon) + ")" + _solrParams.FQ_OR;
					}
				});
				// to exclude last ' OR '
				filterQuery += combinedQuery.slice(0, -(_solrParams.FQ_OR.length));
				//for single expression, SOLR doesn't need query to be surrounded by (), so remove it
				if(fValue.length === 1 && filterQuery.charAt(0) === "(" && filterQuery.charAt(filterQuery.length-1) === ")")
					filterQuery = filterQuery.substring(1, filterQuery.length - 1);
			} else {
				// in case of date range component fvalue will be string
				// this is getting used for date range and searchbar
				filterQuery = (isExclude ? "-" : "") + "(" + _processFilterQuery(id, fName, fValue, fType, isExact, isExclude, false) + ")";
			}
			return filterQuery;
		} else if (typeof fValue === "object" && public.isDateRangeAllowed(fType)) {
			var dtObj = _getDateRangeObj(fValue);
			if (!$.isEmptyObject(dtObj) && dtObj.start && dtObj.end) {
				var endQuery = fValue.excludeEnd ? "}" : "]";
				return ((isExclude ? "-" : "") + fName + ":[" + dtObj.start + " TO " + dtObj.end + endQuery);
			}
		}
	}

	/* to process SOLR date */
	function _getDateFromSEDate(dateStr) {
		return cvSearchUtil.rtrim(dateStr, "Z");
	}
	/* get date range in SOLR format */
	function _getRangeValue(v) {
		if (v === "*") {
			return "*";
		}
		if (typeof (v) === "object") {
			return moment(v).utc().format(_solrProps.dtFormat) + "Z";
		}
		return v;
	}
	/* get size range */
	function _getSizeRangeObj(gapObj) {
		var obj = {
			isRange: true,
			start: 0,
			end: cvFormatters.Constants.MAX_SIZE
		};
		if (gapObj.start) {
			obj.start = cvFormatters.getSizeInBytes(gapObj.start.value, gapObj.start.unit);
		}

		if (gapObj.end) {
			obj.end = cvFormatters.getSizeInBytes(gapObj.end.value, gapObj.end.unit);
		}
		return obj;
	}

	function _getDateRangeInterval(interval, numPoints) {
		var gap = interval / numPoints,
			unitVal;
		var order = ["y", "M", "d", "h", "m", "s"];
		// as dtUnitMap is a associative array which will be sorted based on the
		// key
		// instead of sorting it maintain the array for order
		for (var i = 0; i < order.length; i++) {
			unitVal = _solrProps.dtUnitMap[order[i]].val;
			if (gap > unitVal) {
				gap = Math.floor(gap / unitVal);
				break;
			}
		}
		gap = (gap < 1 ? 1 : gap) + "" + _solrProps.dtUnitMap[order[i]].unit;
		return gap;
	}
	/* get date range query for abs|rel */
	function _getDateRangeObj(rangeObj, numPoints) {
		var obj = {};
		var interval = 0;
		try {
			if (rangeObj.hasOwnProperty("relative")) {
				// relative
				rangeObj = rangeObj.relative;
				var suffix = rangeObj.substring(rangeObj.length - 1);
				var rangeOp = rangeObj.charAt(0);
				var secondChar = rangeObj.charAt(1);
				// if (_isUnaryOp(firstChar)) {
				// rangeObj = substring(1);
				// }

				var val;
				if (secondChar !== '-' && secondChar !== '+') {
					secondChar = "+";
					val = parseInt(rangeObj.substring(1, (rangeObj.length - suffix.length)));
				} else {
					val = parseInt(rangeObj.substring(2, (rangeObj.length - suffix.length)));
				}

				//converting weeks into days
				if(suffix === 'w') {
					val *= 7;
					suffix = 'd';
				}
				var unitObj = _solrProps.dtUnitMap[suffix];
				var roundUnitObj = unitObj;
				if (suffix === 'M' || suffix === 'y') {
					roundUnitObj = _solrProps.dtUnitMap['d'];
				}
				if (rangeOp === '<') {
					if (secondChar === '-') {
						obj.start = '*';
					} else {
						obj.start = "NOW/" + roundUnitObj.unit;
					}
					obj.end = "NOW/" + roundUnitObj.unit + secondChar + val + unitObj.unit;
				} else if (rangeOp === '>') {
					if (secondChar === '+') {
						obj.end = '*';
					} else {
						obj.end = "NOW";
					}
					obj.start = "NOW/" + roundUnitObj.unit + secondChar + val + unitObj.unit;
				}
				interval = val * unitObj.val; // usecase?
			} else if (rangeObj.hasOwnProperty("from")) {
				// absolute
				obj.start = _getRangeValue(rangeObj.from);
				obj.end = _getRangeValue(rangeObj.to);
				interval = moment(rangeObj.to).diff(moment(rangeObj.from), 'seconds');
			}
			if (numPoints) {
				obj.gap = _getDateRangeInterval(interval, numPoints);
			}
		} catch (e) {
			console.error(e);
		}
		return obj;
	}
	/* get base facet request */
	function _getBaseFacetReqObj(facetConfig) {
		var req = {};
		var fName = facetConfig.fieldName;
		if (facetConfig.noOfValuesToShow &&
			!(typeof facetConfig.listType !== "undefined" && facetConfig.listType === "finite")) {
			req[_solrParams.FACET_F + fName + _solrParams.FACET_LIMIT] = facetConfig.noOfValuesToShow;
		}
		if (facetConfig.sortValuesBy) {
			req[_solrParams.FACET_F + fName + _solrParams.FACET_SORT] = facetConfig.sortValuesBy;
		}
		req[_solrParams.FACET_F + fName + _solrParams.FACET_MINCOUNT] = 1;
		if (facetConfig.prefix) {
			req[_solrParams.FACET_F + fName + _solrParams.FACET_PREFIX] = facetConfig.prefix;
		}
		return req;
	}
	/* facet.query for predefined values */
	function _getFacetQueryForPredefinedValues(facetConfig) {
		var facetQueryArr = [];
		var fName = facetConfig.fieldName;
		var pvObj = facetConfig.predefinedValues;
		if (pvObj.type === "fixed") {
			if (public.isDateRangeAllowed(facetConfig.schema.type)) {
				var now = new Date();
				for (var i = 0; i < pvObj.gaps; i++) {
					var obj = {
						end: moment(now).subtract(i, pvObj.gapUnit).endOf("day"),
						start: moment(now).subtract(i + 1, pvObj.gapUnit).startOf("day"),
						isRange: true
					};
					facetQueryArr.push(_getFacetQueryStr(fName, obj));
				}
			}
		} else if (pvObj.type === "fixedList") {
			for (var facet in pvObj.facets)
				facetQueryArr.push(_getFacetQueryStr(fName, pvObj.facets[facet]));
		} else if (!$.isEmptyObject(pvObj.gaps)) {
			if (facetConfig.renderer && facetConfig.renderer.type === "size") {
				$.each(pvObj.gaps, function (i, gapObj) {
					facetQueryArr.push(_getFacetQueryStr(fName, _getSizeRangeObj(gapObj)));
				});
			}
		}
		return facetQueryArr;
	}
	/* parse response for facet suggestion */
	function _getFacetRespFromSuggestionResp(seResp, fieldName) {
		var seFacetResp = [];
		var facetValues = [];

		if (!$.isEmptyObject(seResp) && !$.isEmptyObject(seResp["facet_counts"]) &&
			!$.isEmptyObject(seResp["facet_counts"]["facet_fields"])) {
			seFacetResp = seResp["facet_counts"]["facet_fields"];
		}
		if (!$.isEmptyObject(seFacetResp) && !$.isEmptyObject(seFacetResp[fieldName])) {
			var respArr = seFacetResp[fieldName];

			for (var i = 0; i < respArr.length;) {
				var v = respArr[i++];
				var c = respArr[i++];
				facetValues.push({
					value: v,
					count: c
				});
			}

		}
		return facetValues;
	}
	/* request for keyword suggestions */
	function _getKeywordSuggestReqStr(id) {
		var req = $.extend({}, _getDefaultSearchReq(), _commonState.sReq);
		req[_solrParams.QUERY] = "%QUERY";
		return _state[id].config.suggestUrl + "?" + decodeURIComponent($.param(req, true));
	}
	/* request for facet suggestions */
	function _getFacetSuggestReqStr(id, facetConfig) {
		var req = $.extend({}, _getDefaultSearchReq(), _getBaseFacetReqObj(facetConfig), _commonState.sReq);
		var fName = facetConfig.fieldName;
		req[_solrParams.FACET] = true;
		req[_solrParams.FACET_FIELD] = [fName];
		req[_solrParams.QUERY] = _getIndexField(id, fName) + ":%QUERY*";
		return _state[id].config.searchUrl + "?" + decodeURIComponent($.param(req, true));
	}
	/* parse response for keyword suggestion */
	function _getSuggestionFromSearchResp(seResp) {
		var suggestions = [];
		if (!$.isEmptyObject(seResp) && !$.isEmptyObject(seResp.spellcheck) &&
			!$.isEmptyObject(seResp.spellcheck.suggestions)) {
			// check with someone reg. the logic
			$.each(seResp.spellcheck.suggestions, function (k, obj) {
				if (!$.isEmptyObject(obj.suggestion)) {
					$.each(obj.suggestion, function (i, v) {
						suggestions.push({
							value: v
						});
					});
				}
			});
		}
		return suggestions;
	}
	/* default renderer, predefined values for date and size */
	function _setDefaultRenderer(config) {
		var fieldName = config.fieldName;
		if (config.renderer && config.renderer.type == "size" && !config.renderer.unitObj) {
			config.renderer.unitObj = {
				unit: cvFormatters.Constants.sizeUnits[1],
				value: cvFormatters.Constants.oneKb
			};
			config.predefinedValues = {
				type: "fixedList",
				facets: {
					below1MB: {
						start: 0,
						end: cvFormatters.getSizeInBytes(1, "MB"),
						key: fieldName + ":below1MB",
						dispName: "Below 1 MB",
						isRange: true
					},
					oneMBto10MB: {
						start: cvFormatters.getSizeInBytes(1, "MB"),
						end: cvFormatters.getSizeInBytes(10, "MB"),
						key: fieldName + ":oneMBto10MB",
						dispName: "1 MB to 10 MB",
						isRange: true
					},
					tenMBto50MB: {
						start: cvFormatters.getSizeInBytes(10, "MB"),
						end: cvFormatters.getSizeInBytes(50, "MB"),
						key: fieldName + ":tenMBto50MB",
						dispName: "10 MB to 50 MB",
						isRange: true
					},
					fiftyMBto1GB: {
						start: cvFormatters.getSizeInBytes(50, "MB"),
						end: cvFormatters.getSizeInBytes(1, "GB"),
						key: fieldName + ":fiftyMBto1GB",
						dispName: "50 MB to 1 GB",
						isRange: true
					},
					morethan1GB: {
						start: cvFormatters.getSizeInBytes(1, "GB"),
						end: "*",
						key: fieldName + ":morethan1GB",
						dispName: "Above 1 GB",
						isRange: true
					}
				}
			};
		}

		if (public.isDateRangeAllowed(config.schema.type)) {
			if (!config.renderer) {
				config.renderer = {
					type: "date"
				};
			}
			if (config.renderer.type == "date" && !config.renderer.fmt) {
				config.renderer.fmt = cvFormatters.Constants.defaultDateFormat;
			}
			var nowDay = _constants.dateConstants.nowDay;
			var nowYear = _constants.dateConstants.nowYear;

			config.predefinedValues = {
				type: "fixedList",
				facets: {
					today: {
						start: nowDay,
						end: "*",
						key: fieldName + ":today",
						dispName: "Today",
						isRange: true
					},
					last7days: {
						start: nowDay + "-7DAYS",
						end: nowDay,
						key: fieldName + ":last7days",
						dispName: "Last 7 Days",
						isRange: true
					},
					last30days: {
						start: nowDay + "-30DAYS",
						end: nowDay,
						key: fieldName + ":last30days",
						dispName: "Last 30 Days",
						isRange: true
					},
					lastyear: {
						start: nowDay + "-1YEAR",
						end: nowDay,
						key: fieldName + ":lastyear",
						dispName: "Last Year",
						isRange: true
					},
					beyond1year: {
						start: "*",
						end: nowDay + "-1YEAR",
						key: fieldName + ":beyond1year",
						dispName: "Beyond One Year",
						isRange: true
					}
				}
			};
		}
		config.isSortable = public.isSortableField(config.schema);
	}
	/* TODO: check if really needed */
	function _getSimplifiedSearchRequest(obj) {
		if ($.isEmptyObject(obj)) {
			return obj;
		}
		var objToReturn = $.extend({}, obj);
		// convert fl array to comma separated list
		if (objToReturn.hasOwnProperty("fl")) {
			var flArr = objToReturn.fl;
			if (!$.isEmptyObject(flArr) && $.isArray(flArr)) {
				objToReturn.fl = flArr.join(",");
			}
		}
		return objToReturn;
	}

	function _executeMethod(self, argsObj, fnToGetRequest, fnToProcessResp) {
		// People are saying var args = Array.prototype.slice.call(arguments);
		// is slow; so using below stmt
		var args = (argsObj.length === 1 ? [argsObj[0]] : Array.apply(null, argsObj));
		var len = argsObj.length;
		// no of arguments depends on the sub classes; so pass all the arguments
		// as such
		// first argument should be the ID.
		// last argument should always be a call back
		var id = args[0],
			params = args.slice(0, len - 1),
			callback = args[len - 1];
		var sReq = self[fnToGetRequest].apply(self, params);
		// _saveCommonCriteria(sReq);
		_executeQuery(id, sReq, fnToGetRequest, function (SEResp, SEReq) {
			params.push(SEResp);
			// send the resp, searchReq
			callback(self[fnToProcessResp].apply(self, params), SEReq);
		});
	}

	/*
	 * remember the common criteria which affects widgets in a single
	 * report/dashboard currently q and fq this logic has to be revisited
	 * because filter criteria has to be formed out of all components first and
	 * then trigger refresh
	 */
	function _saveCommonCriteria(sReq) {
		var q = sReq[_solrParams.QUERY],
			fq = sReq[_solrParams.FILTER_QUERY];
		_commonState.sReq[_solrParams.QUERY] = q;
		_commonState.sReq[_solrParams.FILTER_QUERY] = fq;
	}

	function _getQueryForEmptyVal () {
		return "\"\"";
	}
	// private member functions, end
	var public = {
		// public properties, start
		constants: {
			schemaCustomDateFormat: "DateFormat :",
		},
		// public properties, end
		// public member functions, start
		getSolrParams: function () {
			return $.extend({}, _solrParams);
		},
		getSolrProps: function () {
			return $.extend({}, _solrProps);
		},
		/* getter to get a read only copy of current search request */
		getCurrentSearchReq: function () {
			return $.extend(true, {}, _commonState.sReq);
		},
		/* getter to get a read only copy of configuration */
		getConfig: function (id) {
			if(_state[id] === undefined)
				return {};
			return $.extend(true, {}, _state[id].config);
		},
		/* getter to get a map of all the facet search requests */
		getAllFacetSearchRequests : function(id) {
			return $.extend(true, [], _facetRequests);
		},
		isFieldToBeExcluded: function (fName) {
			return $.inArray(fName, _constants.defaultFieldsToBeExcluded) > -1;
		},
		isFacetingAllowed: function (fType) {
			return $.inArray(fType, _constants.allowedFieldTypeForFaceting) > -1 ||
				fType.startsWith(qb.base.constants.schemaCustomDateFormat);
		},
		isFacetSuggestionAllowed: function (fType) {
			return $.inArray(fType, _constants.allowedFieldTypeForFacetSuggestion) > -1;
		},
		isSortableField: function (schema) {
			// https://lucene.apache.org/solr/guide/6_6/common-query-parameters.html#CommonQueryParameters-ThesortParameter
			return !schema.multiValued && (schema.indexed || schema.docValues);
		},
		isAllowedSystemField: function (fName) {
			return $.inArray(fName, _constants.allowedSystemFields) > -1;
		},
		doSearch: function () {
			_executeMethod(this, arguments, "getSearchRequest", "processSearchResponse");
		},
		doFacet: function () {
			_executeMethod(this, arguments, "getFacetRequest", "processFacetResponse");
		},
		doQuery: function () {
			_executeMethod(this, arguments, "getCustomQueryRequest", "processCustomQueryResponse");
		},
		doFacetFunction: function () {
			_executeMethod(this, arguments, "getFacetFnRequest", "processFacetFnResponse");
		},
		getChartData: function() {
			_executeMethod(this, arguments, "getRequestParams" , "processFacetResponse");
		},
		setKeyword: function (id, keyword, fieldObj) {
			if (typeof keyword === "undefined" || keyword === null || keyword === "") {
				_commonState.sReq[_solrParams.QUERY] = _solrProps.defaultQuery;
			} else if ($.isEmptyObject(fieldObj)) {
				// not escaping the keyword as there could be
				// fielded search
				_commonState.sReq[_solrParams.QUERY] = keyword;
			} else {
				if(!Array.isArray(fieldObj)) fieldObj = [fieldObj];
				var combinedQuery = "";
				fieldObj.forEach(function(searchQueryObject) {
					combinedQuery +=  "(" + _parseSearchObject(id,searchQueryObject,keyword) + ")" + _solrParams.FQ_OR;
				});
				combinedQuery = combinedQuery.slice(0, -(_solrParams.FQ_OR.length));
				_commonState.sReq[_solrParams.QUERY] = combinedQuery;
			}
		},
		/**
		 * Append 'fq' to the current list of fqs
		 * @param {*} fq Final formatted fq as string
		 */
		addFilterQuery: function(fq) {
			if (typeof fq !== 'undefined' && fq !== null && fq !== '' && !_commonState.customFilters.find(function(filter) { return filter === fq; })) {
				_commonState.customFilters.push(fq);
				if (!Array.isArray(_commonState.sReq[_solrParams.FILTER_QUERY])) {
					_commonState.sReq[_solrParams.FILTER_QUERY] = [];
				}
				_commonState.sReq[_solrParams.FILTER_QUERY].push(fq);
			}
		},
		/**
		 * Remove the 'fq' from currently applied fqs
		 * @param {*} fq Final formatted fq as string, that'll be matched with the current fqs
		 */
		removeFilterQuery: function(fq) {
			if (typeof fq !== 'undefined' && fq !== null && fq !== '') {
				_.remove(_commonState.customFilters, function(filter ) { return filter === fq; });
				if (Array.isArray(_commonState.sReq[_solrParams.FILTER_QUERY])) {
					_.remove(_commonState.sReq[_solrParams.FILTER_QUERY], function(filter) { return filter === fq; });
				}
			}
		},
		/**
		 * Return the 'customFilters' value from common state. (fq added through 'addFilterQuery' method)
		 * @param {*} isAdvFq return the first value vs return the whole array
		 */
		getCustomQuery: function(isAdvFq) {
			if (isAdvFq) {
				return _commonState.customFilters.length > 0 ? _commonState.customFilters[0] : '';
			}
			return _commonState.customFilters;
		},

		/**
		 * get corresponding solr operator for advanced filters
		 *
		 * @param {string}
		 *            operator
		 * @return {string} returns solr operator
		 */
		getSolrOperator: function (operator, value, dataType) {
			var isNumeric = _isNumberRangeAllowed(dataType) || _isDecimalRangeAllowed(dataType);
			switch (operator) {
				case "eq":
					return value;
				case "neq":
					return "!" + value;
				case "doesnotcontain":
					return "!*" + value + "*";
				case "startswith":
					return value + "*";
				case "doesnotstartswith":
					return "!" + value + "*";
				case "contains":
					return "*" + value + "*";
				case "endswith":
					return "*" + value;
				case "less":
					return "<" + value;
				case "leq":
					return "<=" + value;
				case "lt":
					return "<" + value;
				case "gt":
					return ">" + value;
				case "greater":
					return ">" + value;
				case "less":
					return "<" + value;
				case "geq":
					return ">=" + value;
				case "isempty":
					if(isNumeric)
					return '!["" TO *]';
					else
						return '!*';
				case "isnotempty":
					if(isNumeric)
					return '["" TO *]';
					else
						return '*';
				default:
					return value;
			}
		},
		/**
		 * get advanced filter query
		 *
		 * @param {string}
		 *            filterParam dimension
		 * @param {Object}
		 *            advancedFilter {filters: [{field, operator, value}], logic
		 * @return {string} returns advanced query
		 */
		getAdvancedFilterQuery: function (id, filterParam, advancedFilter) {
			var fqArr = [];
			var self = this;
			var schemaObj = _getSchemaObj(id, filterParam);
			if (Array.isArray(advancedFilter) && advancedFilter.length && schemaObj) {
				$.each(advancedFilter[0].filters, function (index, filter) {
					filter.value = filter.value.toString();
					if (filter.value.toString().trim() != "" ||
						(filter.operator === 'isempty' || filter.operator === 'isnotempty')) {
						var fType = schemaObj.origType || schemaObj.type;
						fqArr.push(_getSOLRExpr(id,
							filter.field,
							self.getSolrOperator(filter.operator, filter.value, fType),
							fType,
							(filter.operator === "eq" || filter.operator === "neq") ? true : false,
							(filter.operator === "isempty" || filter.operator === "isnotempty") ? true : false));
					}
				});
			}
			if (fqArr.length > 0) {
				return fqArr.join(" " + advancedFilter[0].logic.toUpperCase() + " ");
			} else {
				return "";
			}
		},
		/**
		 * get filter query from either "d1:v1 && d2:v2" , "d1:(>10 && <15) &&
		 * d2:(>50 && <70)"
		 *
		 * @param {string}
		 *            filterString
		 * @return {string} returns filter query for solr
		 */
		getFilterQueryFromString: function (id, filterString) {
			var self = this;
			var filterQuery = "";
			var filters = filterString.match(/([\s\d\w_-]*:\([\w&<>\s_-]*\))/g);
			if ($.isEmptyObject(filters)) { // in case of d1:v1 && d2:v2
				filters = filterString.split("&&");
			}
			$.each(filters, function (index, filter) {
				var filterSplit = filter.split(":");
				filterQuery += self.getFilterQuery(id,
						(filterSplit[0] || "").trim(),
						(filterSplit[1] || "").trim(),
						undefined,
						true) +
					_solrParams.FQ_AND;
			});
			filterQuery = filterQuery.slice(0, filterQuery.lastIndexOf(_solrParams.FQ_AND));
			return "(" + filterQuery + ")";
		},
		/**
		 * get filter query for tuple d1:v1 && d2:v2
		 *
		 * @param {number}
		 *            id state id
		 * @param {object}
		 *            tupleObject contains dimensions (array of dimensions) and
		 *            tupleFilters (array of query strings)
		 * @return {string} returns filter query for tuple with all the related
		 *         dimensions with a or in between.
		 */
		getTupleFilterQuery: function (id, tupleObject) {
			var self = this;
			var tupleFilterQuery = "";
			var tagPrefix = "";
			$.each(tupleObject.tupleFilters, function (index, filterString) {
				tupleFilterQuery += self.getFilterQueryFromString(id, filterString) + _solrParams.FQ_OR;
			});
			// append other dimension include as OR condition to current filter
			// query
			$.each(tupleObject.dimensions, function (index, dimension) {
				if (typeof _commonState.filterMap.normal[dimension] != 'undefined' &&
					_commonState.filterMap.normal[dimension].include.length > 0) {
					tupleFilterQuery += self.getFilterQuery(id,
							dimension,
							_commonState.filterMap.normal[dimension].include,
							undefined,
							true) +
						_solrParams.FQ_OR;
				}
				tagPrefix += dimension + ",";
			});
			tupleFilterQuery = tupleFilterQuery.slice(0, tupleFilterQuery.lastIndexOf(_solrParams.FQ_OR));
			tagPrefix = tagPrefix.slice(0, tagPrefix.lastIndexOf(","));
			tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + tagPrefix + "}";
			if ((tupleFilterQuery || "").trim() != "") {
				tupleFilterQuery = tagPrefix + tupleFilterQuery;
			}
			return tupleFilterQuery;
		},
		// TODO: this has to be deprecated
		setFilter: function (id, facetConfig, facetReq) {
			_commonState.sReq[_solrParams.FILTER_QUERY] = this.getFacetRequest(id, facetConfig, facetReq, true)[_solrParams.FILTER_QUERY];
		},
		setFilterMap: function (id, filtersMap, fqAlias, isExact, filterType) {
			var self = this;
			var tuplesDimensions = [];
			var eachFilter;
			var filterMapType = "normal";
			if (filterType === "page_override") {
				filterMapType = "override";
			}
			$.each(filtersMap, function (filterName, filterObj) {
				eachFilter = {};
				if (filterName.indexOf('tuples_') != -1) {
					/*
					 * for now we can only include values using tuple selection
					 * include array will store objects [{ dimension1 : value1,
					 * dimension2 : value2 }] make query in the from of
					 * dimension1 : value1 && dimension2 : value2 push above
					 * filter queries in tupleFilters array and the dimensions
					 * involved in dimensions array for exclude tags
					 */
					var tupleFilters = [];
					if (!$.isEmptyObject(filterObj.include)) {
						var dimensions = [];
						$.each(filterObj.include, function (index, filter) {
							var tupleFilter = "";
							$.each(filter, function (dimension, value) {
								tupleFilter += dimension + ":" + value + " && ";
								if (tuplesDimensions.indexOf(dimension) === -1) {
									tuplesDimensions.push(dimension);
									dimensions.push(dimension);
								}
							});
							tupleFilter = tupleFilter.slice(0, tupleFilter.lastIndexOf(' && '));
							tupleFilters.push(tupleFilter);
						});
						if ($.isEmptyObject(tupleFilters)) {
							return;
						}
						eachFilter.tupleFilters = tupleFilters;
						eachFilter.dimensions = dimensions;
					}
				} else if (filterName === "globalSearch") {
					eachFilter = filterObj;
				} else {
					/*
					 * isExact eachFilter is going to store, include, exclude -
					 * values selected for filtering the data advancedFilters -
					 * advancedFilters object searchQuery for search bar (with
					 * recent changes in schema searchQuery is case sensitive)
					 */
					// in case of clear filters we have
					// isExact as undefined so
					// setting it to true by default
					eachFilter.isExact = isExact || true;
					eachFilter.include = filterObj.include;
					if (typeof filterObj.exclude != 'undefined') {
						eachFilter.exclude = filterObj.exclude;
					}
					if (Array.isArray(filterObj.advancedFilters) && filterObj.advancedFilters.length) {
						eachFilter.advancedFilters = filterObj.advancedFilters;
					}
					if (!$.isEmptyObject(filterObj.tableFilters)) {
						eachFilter.tableFilters = filterObj.tableFilters;
					}
					if (!$.isEmptyObject(filterObj.searchQuery)) {
						eachFilter.searchQuery = filterObj.searchQuery;
					}
					if (!$.isEmptyObject(filterObj.daterange)) {
						eachFilter.daterange = filterObj.daterange;
					}
				}
				/*
				 * store global filters in fitlerMap TODO: need to support for
				 * multiple datasets.
				 */
				_commonState.filterMap[filterMapType][filterName] = eachFilter;
			});
			var fqArrNormal = self.getFilterQueryFromFiltersMap(id,
				_commonState.filterMap.normal,
				isExact,
				"normal",
				false,
				tuplesDimensions);
			var fqArrOverride = self.getFilterQueryFromFiltersMap(id,
				_commonState.filterMap.override,
				isExact,
				"override",
				false,
				tuplesDimensions);
			var fqArr = (fqArrNormal || []).concat(fqArrOverride || []);
			_commonState.sReq[fqAlias ? fqAlias : _solrParams.FILTER_QUERY] = fqArr;
			// As above fq overrides the previously applied fq. Reapply any 'custom' fq that were present
			if (!Array.isArray(_commonState.sReq[_solrParams.FILTER_QUERY])) {
				_commonState.sReq[_solrParams.FILTER_QUERY] = [];
			}
			_commonState.customFilters.forEach(function(fq) {
				_commonState.sReq[_solrParams.FILTER_QUERY].push(fq);
			});
		},
		/**
		 * get solr filter queries in an array from filtersMap, this function
		 * will get called for both global filters and component level filters
		 *
		 * @param {number}
		 *            id handlerId
		 * @param {Object}
		 *            filtersMap filtersMap will have ({fieldName: { 'include' :
		 *            [], 'exclude' : [], 'advancedFilters': [] } })
		 * @param {Boolean}
		 *            isExact if we need to use filterVal as phrase query
		 * @param {string}
		 *            filterType filters applied from page properties or global
		 *            filters
		 * @param {Boolean}
		 *            isDefaultFilters in case filtersMap contains component
		 *            level filters in that case we don't need exclude tags
		 * @param {Array}
		 *            tuplesDimensions all the dimensions related to tuple
		 *            filters
		 * @return {Array} it will return an array of solr filter queries
		 */
		getFilterQueryFromFiltersMap: function (id, filtersMap, isExact, filterType, isDefaultFilters, tuplesDimensions, excludeTableRowFilters) {
			if (!filtersMap) {
				return [];
			}
			var self = this;
			var fqArr = [];

			function _appendFq(currFq, filterQuery) {
				if ((currFq || "").trim() !== "") {
					filterQuery += currFq + _solrParams.FQ_OR;
				}
				return filterQuery;
			}
			$.each(filtersMap, function (fName, filterObj) {
				var schemaObj = _getSchemaObj(id, fName);
				var fType = schemaObj.origType || schemaObj.type;
				var filterQuery = "";
				/*
				 * !isDefaultFilters - component level filters do not support
				 * tuple selection tuplesDimensions - for dimensions with tuple
				 * selections rest of the filters will be added in a single
				 * query in getTupleFilterQuery
				 */
				if (!isDefaultFilters && fName.indexOf('tuples_') !== -1) {
					if ($.isEmptyObject(filterObj)) {
						return;
					}
					filterQuery = self.getTupleFilterQuery(id, filterObj);
					fqArr.push(filterQuery);
				} else if ((tuplesDimensions || []).indexOf(fName) === -1) {
					var tagPrefix = "";
					if ((!$.isEmptyObject(filterObj.include) && filterObj.include !== "") ||
						!$.isEmptyObject(filterObj.exclude) || !$.isEmptyObject(filterObj.daterange)) {
						// page level default filters (from properties)
						var tagPrefixSuffix = "";
						if (filterType === 'override') {
							tagPrefixSuffix = 'page_';
						}
						var fVal = filterObj.include;
						var customGroups =[];
						var normalFilterValues = [];
						if ($.isArray(fVal)) {
							$.each(fVal, function (index, value) {
								if (typeof value === 'object' && value.groupName)
									customGroups.push(value);
								else
									normalFilterValues.push(value);
							});
						}
						if (!Array.isArray(filterObj.exclude) || filterObj.exclude.length === 0) {
							/*
						    * append (include)filter queries with an OR between
						    * them to filterQuery. pushing separate fq for custom group and normal values
							* if 2 component are using same field then custom group from one component
							should get treated as different from normal values of other component
						    */

							if(normalFilterValues.length) {
								tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + tagPrefixSuffix + fName +
									",tag_page_override}";
								filterQuery = tagPrefix + self.getFilterQuery(id, fName, normalFilterValues, fType, true);
								//pushing custom group query separately
								fqArr.push(filterQuery);
								filterQuery = "";
							}

							if(customGroups.length) {
								tagPrefixSuffix = 'group_';
								tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + tagPrefixSuffix + fName + "}";
								filterQuery = tagPrefix + self.getFilterQuery(id, fName, customGroups, fType, true);
								//pushing normal filter query separately
								fqArr.push(filterQuery);
								filterQuery = "";
							}
						} else {
							/*
						    * append (exclude)filter queries with an OR between
						    * them to filterQuery
						    */
						   tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + "exclude_" + tagPrefixSuffix + fName +
								",tag_page_override}";

						   // isExact will be true always for exclude filters
							filterQuery = _appendFq(self
								.getFilterQuery(id, fName, filterObj.exclude, fType, true, true), filterQuery);

						}
					}
					/*
					 * append (advanced)filter queries with an OR between them
					 * to filterQuery
					 */
					if (Array.isArray(filterObj.advancedFilters) && filterObj.advancedFilters.length) {
						filterQuery = _appendFq(self.getAdvancedFilterQuery(id, fName, filterObj.advancedFilters),
							filterQuery);
					}
					/*
					 * append (table)filter queries with an OR between them to
					 * filterQuery
					 */
					if (!$.isEmptyObject(filterObj.tableFilters)) {
						//for treegrid component
						if($.isArray(filterObj.tableFilters)) {
							filterQuery = _appendFq(self.getAdvancedFilterQuery(id, fName, filterObj.tableFilters),
	                           filterQuery);
						} else {
							//for table component
							if (!$.isEmptyObject(filterObj.tableFilters.include) && filterObj.tableFilters.include.length > 0 && !excludeTableRowFilters) {
								var filterVal  = filterObj.tableFilters.include;
								if(fType === "string") {
									tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + fName + "}";
								} else if(fType !== "string" && filterVal.length === 1)
									filterVal = filterVal[0];
								filterQuery = _appendFq(self.getFilterQuery(id, fName, filterVal, fType, true), filterQuery);
							}

							if (!$.isEmptyObject(filterObj.tableFilters.exclude) && filterObj.tableFilters.exclude.length > 0 && !excludeTableRowFilters) {
								var filterVal  = filterObj.tableFilters.exclude;
								if(fType === "string") {
									tagPrefix = "{!tag=" + _solrParams.TAG_PREFIX + fName + "}";
								} else if(fType !== "string" && filterVal.length === 1)
									filterVal = filterVal[0];
								filterQuery = _appendFq(self.getFilterQuery(id, fName, filterVal, fType, true, true), filterQuery);
							}

							if(Array.isArray(filterObj.tableFilters.advancedFilters) && filterObj.tableFilters.advancedFilters.length)
								filterQuery = _appendFq(self.getAdvancedFilterQuery(id, fName, filterObj.tableFilters.advancedFilters),
									filterQuery);

							if(filterObj.tableFilters.daterange) {
								if (typeof filterObj.tableFilters.daterange === 'string') {
									filterObj.tableFilters.daterange = {
										'relative': filterObj.tableFilters.daterange
									};
								}
								filterQuery = _appendFq(self.getFilterQuery(id, fName, filterObj.tableFilters.daterange, fType, false),
								filterQuery);
							}
						}
					}
					/*
					 * append (search query)filter queries with an OR between
					 * them to filterQuery
					 */
					if (filterObj.searchQuery) {
						filterQuery = _appendFq(self.getFilterQuery(id, fName, filterObj.searchQuery, fType, false),
							filterQuery);
					}
					/*
					 * append (date range query)filter queries with an OR
					 * between them to filterQuery
					 */
					if (filterObj.daterange) {
						if (typeof filterObj.daterange === 'string') {
							filterObj.daterange = {
								'relative': filterObj.daterange
							};
						}
						filterQuery = _appendFq(self.getFilterQuery(id, fName, filterObj.daterange, fType, false),
							filterQuery);
					}

					/*global search filters*/

					if(fName === "globalSearch" && !$.isEmptyObject(filterObj.columns)) {
						$.each(filterObj.columns, function (index, filterCol) {
							var filterVal = "(" + filterObj.filterValue + "*) OR (" +filterObj.filterValue+")";
							filterQuery = _appendFq(self.getFilterQuery(id, filterCol.column, filterVal, filterCol.type, isExact), filterQuery);
						});
					}

					if (fName === "customFilterQuery") {
						filterQuery = _appendFq(filterObj,filterQuery);
					}

					/*
					 * remove the last trailing or from the filterQuery add
					 * tagPrefix(exclude tags) in case of global filters
					 * (!isDefaultFilters) push the filterQuery in fqArr
					 */
					if (filterQuery.trim() !== "") {
						filterQuery = filterQuery.slice(0, filterQuery.lastIndexOf(_solrParams.FQ_OR));
						if (!isDefaultFilters) {
							if ((tagPrefix || "").trim() !== "") {
								filterQuery = tagPrefix + filterQuery;
							}
						}
						fqArr.push(filterQuery);
					}
				}
			});
			return fqArr;
		},
		getDateFilterApplied: function (fieldName, numPoints) {
			var fObj = (!$.isEmptyObject(_commonState.filterMap) && _commonState.filterMap.hasOwnProperty(fieldName) && !$
					.isEmptyObject(_commonState.filterMap[fieldName].values)) ? _commonState.filterMap[fieldName] :
				undefined;
			if (fObj && fObj.values !== "") {
				fObj = _getDateRangeObj(fObj.values, numPoints);
			}
			return fObj;
		},
		isStatsKey: function (key) {
			key = key.toLowerCase();
			return $.inArray(key, _constants.statsOperations) > -1;
		},
		getDefaultSearchHandler: function () {
			return _solrProps.defaultSelectHandler;
		},
		getDefaultSuggestHandler: function () {
			return _solrProps.defaultSuggestHandler;
		},
		getDefaultNoOfColumns: function () {
			return _solrProps.defaultNoOfColumnsToFetch;
		},
		getFilterMap: function () {
			return _commonState.filterMap;
		},
		init: initialize,
		isNumberRangeAllowed: _isNumberRangeAllowed,
		isDecimalRangeAllowed: _isDecimalRangeAllowed,
		isDateRangeAllowed: _isDateRangeAllowed,
		getHierarchicalField: _getHierarchicalField,
		getIndexField: _getIndexField,
		getSLevelField: _getSLevelField,
		getFieldName: _getFieldName,
		setDefaultRenderer: _setDefaultRenderer,

		getDefaultSearchReq: _getDefaultSearchReq,
		getKeywordSuggestReqStr: _getKeywordSuggestReqStr,
		getFacetSuggestReqStr: _getFacetSuggestReqStr,
		getSuggestionFromSearchResp: _getSuggestionFromSearchResp,
		getFacetRespFromSuggestionResp: _getFacetRespFromSuggestionResp,
		getFqStr: _getFqStr,
		getFilterQuery: _getFilterQuery,
		getFacetQueryStr: _getFacetQueryStr,
		getBaseFacetReqObj: _getBaseFacetReqObj,
		getFacetQueryForPredefinedValues: _getFacetQueryForPredefinedValues,
		appendHighlightQuery: _appendHighlightQuery,
		mapHightlightResp: _mapHightlightResp,
		appendRawQuery: _appendRawQuery,
		getDateFromSEDate: _getDateFromSEDate,
		// TODO:review resetFacetParamsInSearchReq
		// getting used in web analytics
		resetFacetParamsInSearchReq: _resetFacetParamsInSearchReq,
		appendFilterQuery: _appendFilterQuery,
		unEscapeSpecialChars: _unEscapeSpecialChars,
		escapeSpecialChars: _escapeSpecialChars,
		getFieldTypeForReport: _getFieldTypeForReport,
		getSchemaObj: _getSchemaObj,
		getFieldTypeForSolr : _getFieldTypeForSolr,
		getQueryForEmptyVal : _getQueryForEmptyVal,
		getSOLRExpr : _getSOLRExpr,
		parseSearchObject : _parseSearchObject
		// public member functions, end
	};
	return public;
})();
