(function(factory) {
	var getDependencyMissingMsg = function(dependency) {
		return "cvDataViewer depends on " + dependency + " and seems to be missing. Please include necessary files.";
	};
	if (!jQuery)
		throw getDependencyMissingMsg('jQuery');
	else
		factory(jQuery);

}(function($) {
	"use strict";

	$.fn.cvDataViewer = function(config) {
		return this.each(function() {
			var $this = $(this), data = $this.data('cvDataViewer');
			if (!data)
				$this.data('cvDataViewer', new DataViewer(this, config));
		});
	};
	$.fn.cvDataViewer.id = 0;
}));

function DataViewer(element, config) {
	this.$element = $(element);
	this._origConfig = config;
	this._config = $.extend(true, {}, this.defaults, config);

	this._id = $.fn.cvDataViewer.id++;
	this.init();
}

DataViewer.prototype.defaults = {
	refreshInterval : 60000,
	maxRefresh : 5
};

DataViewer.prototype.Constants = {
	SWITCH_BUTTONS : "cvDataViewerSwitch",
	GRID_SWITCH_TOGRID_CLASS : "cvDataViewerGrid",
	GRID_SWITCH_TOTABLE_CLASS : "cvDataViewerTable",
	GRID_HOLDER_CLASS : "cvGridViewHolder",
	GRID_SORT_CLASS : "cvDataGridSortingOptions",
	DATATABLE : "cvDataGridTable",
	TABLE_HOLDER_CLASS : "dataTables_wrapper",
	DEFAULT_SORTING : "default"
};

DataViewer.prototype.init = function() {
	//Polling for datatable if autorefresh is set
	this.tablePoll = null;
	if (window.sessionStorage.getItem("datasourcesViewType") === "table")
		this.initToTable();
	else {
		this.initToGrid();
		this.pollObj = this.dataGrid.pollObj;
		// this.addSwitchButtons();
	}
};

/**
 * @namespace
 * @name cvDataViewerSwitch
 * @description Don't forget to add docs for this
 */
DataViewer.prototype.addSwitchButtons = function() {
	var buttons = this.createSwitchButtons();
	this.$element.find("." + this.Constants.SWITCH_BUTTONS).append(buttons);
	this.attachListeners();
};

DataViewer.prototype.createSwitchButtons = function() {
	return $("<button class='btn btn-default " + this.Constants.GRID_SWITCH_TOGRID_CLASS +
			"' title='Grid' data-toggle='button' disabled><span class='glyphicon glyphicon-th-list'></span></button>" +
			"<button class='btn-default btn " + this.Constants.GRID_SWITCH_TOTABLE_CLASS +
			"' title='Table' data-toggle='button'><span class='glyphicon glyphicon-list-alt'></span></button>" +
			"<div class='clearfix'></div>");
};

DataViewer.prototype.attachListeners = function() {
	var self = this;
	this.$element.find("." + this.Constants.SWITCH_BUTTONS + " ." + this.Constants.GRID_SWITCH_TOTABLE_CLASS)
			.off("click").on("click", function() {
				self.initToTable();
				window.sessionStorage.setItem("datasourcesViewType", "table");
			});

	this.$element.find("." + this.Constants.SWITCH_BUTTONS + " ." + this.Constants.GRID_SWITCH_TOGRID_CLASS)
			.off("click").on("click", function() {
				self.initToGrid();
				window.sessionStorage.setItem("datasourcesViewType", "grid");
			});

	this.$element.off("cvDataGrid.order").on("cvDataGrid.order", function(e, sortingOption) {
		window.sessionStorage.setItem("datasourcesOrdering", sortingOption);
	});

	this.$element.on('remove', function(e, settings) {
		if (self.pollObj)
			clearInterval(self.pollObj);
	});
};

DataViewer.prototype.initToTable = function() {
	var self = this;
	var sortingOption = null;
	this.$element.find("." + this.Constants.GRID_HOLDER_CLASS).remove();
	var configCopy = $.extend(true, {}, this._config);
	delete configCopy.datatableOptions;

	var dtConfig = $.extend(true, {}, configCopy, this._config.dataTableOptions);
	var table = $("<table class='dataTable table table-bordered cvDataGridTable'></table>");

	//Preserving the sort between grid and table view
	// if(!$.isEmptyObject(this.dataGrid))
	// 	sortingOption = this.dataGrid.getTip().sortingOption;
	sortingOption = window.sessionStorage.getItem("datasourcesOrdering");

	if (sortingOption != null && sortingOption != this.Constants.DEFAULT_SORTING)
		dtConfig.order = this.convertGridSortingParamToDT(sortingOption, dtConfig);

	//If the datatable is already there, do nothing
	if ($.fn.dataTable.isDataTable(this.$element.find("." + this.Constants.DATATABLE)))
		return;
	this.$element.append(table);
	this.$element.find("." + this.Constants.DATATABLE).DataTable(dtConfig).on('preXhr.dt', function(e, settings, data) {
		data.mask = false;
	});
	this.addSwitchButtons();
	this.$element.find("." + this.Constants.SWITCH_BUTTONS + " ." + this.Constants.GRID_SWITCH_TOTABLE_CLASS)
			.prop("disabled", true);
	this.$element.find("." + this.Constants.SWITCH_BUTTONS + " ." + this.Constants.GRID_SWITCH_TOGRID_CLASS)
			.prop("disabled", false);

	//Handling autoRefresh for datatables
	if (this._config.autoRefresh) {
		var noOfRefreshs = 0;
		var tableObject = table.DataTable();
		//For some reason, if we don't do a clear() once before starting setInterval, table will end up
		//having twice as many rows until next refresh.
		tableObject.clear();
		clearInterval(this.pollObj);
		this.pollObj = setInterval(function() {
			if (noOfRefreshs < self._config.maxRefresh) {
				noOfRefreshs += 1;
				tableObject.ajax.reload();
			}
		}, this._config.refreshInterval);
	}

	this.datatable = table.DataTable();
	this.dataGrid = undefined;

	table.off('order.dt').on('order.dt', function() {
		var order = self.datatable.order();
		sessionStorage.setItem('datasourcesOrdering', self.convertDTSortingParamToGrid(order));
	});
};

DataViewer.prototype.initToGrid = function() {
	var self = this;

	self.$element.find("." + self.Constants.TABLE_HOLDER_CLASS).remove();
	self.$element.removeData("cvDataGrid");

	var sortingOption = window.sessionStorage.getItem("datasourcesOrdering");
	// var sortingOption = self.convertDTSortingParamToGrid(prevSorting);

	self.$element.cvDataGrid(self._origConfig);
	self.dataGrid = self.$element.data('cvDataGrid');
	if (sortingOption != null && sortingOption.length)
		self.$element.find("." + self.Constants.GRID_SORT_CLASS).val(sortingOption).trigger("change");

	self.addSwitchButtons();
	self.$element.find("." + self.Constants.SWITCH_BUTTONS + " ." + self.Constants.GRID_SWITCH_TOTABLE_CLASS)
			.prop("disabled", false);
	self.$element.find("." + self.Constants.SWITCH_BUTTONS + " ." + self.Constants.GRID_SWITCH_TOGRID_CLASS)
			.prop("disabled", true);

	if (self._config.autoRefresh) {
		clearInterval(self.pollObj);
		self.pollObj = self.dataGrid.pollObj;
	}

	self.datatable = undefined;
};

DataViewer.prototype.convertGridSortingParamToDT = function(sortingOption, dtConfig) {
	if (sortingOption === this.Constants.DEFAULT_SORTING)
		return;

	//datagrid sorting options are of the format '<data>_<direction>'. eg: name_a, age_d, age_a e.t.c
	var splitIndex = sortingOption.indexOf("_");
	var sortData = sortingOption.slice(0, splitIndex);
	var sortDirection = sortingOption.slice(splitIndex + 1, sortingOption.length);
	var convertedSort = null;
	var columnNumber = null;
	if (sortData != undefined)
		columnNumber = this.getDataColumnNumber(sortData, dtConfig);
	if (columnNumber === null) {
		console.error("could not find the sorting parameter in the columns list");
		return;
	}

	if (sortDirection === "a")
		convertedSort = [ [ columnNumber, 'asc' ] ];
	else if (sortDirection === "d")
		convertedSort = [ [ columnNumber, 'desc' ] ];
	else
		console.error("unknown sort direction");

	return convertedSort;
};

DataViewer.prototype.getDataColumnNumber = function(data, dtConfig) {
	var columns = dtConfig.columns;
	for (var i = 0; i < columns.length; i++) {
		if (columns[i].data === data)
			return i;
	}
	return null;
};

DataViewer.prototype.convertDTSortingParamToGrid = function(sortingOption) {
	var columnNumber = sortingOption[0][0];
	var sortDirection = sortingOption[0][1];

	if (!this._origConfig.columns[columnNumber].orderable)
		return "default";

	var data = this._origConfig.columns[columnNumber].data;
	var convertedSort = "";

	if (sortDirection === "asc")
		convertedSort = data + "_a";
	else if (sortDirection === "desc")
		convertedSort = data + "_d";

	return convertedSort;
};