(function () {
    "use strict";

    angular
        .module("reports")
        .controller("mapCtrl", mapCtrl);

    mapCtrl.$inject = ['$scope', 'customReportSvc', 'dataSource', '$timeout', 'reportService','$interval'];

    function mapCtrl($scope, customReportSvc, $dataSource, $timeout, $reportService,$interval) {

        // TODO: check below block of code and move to common place
        var exportType = customReportSvc.getParameterByName("exportType");
        var readme = customReportSvc.getParameterByName("readme");
        $scope.setChildScope($scope);
        $scope.columnDefinitionById = {};
        var isExport = false;
        if (exportType || readme === "true")
            isExport = true;

        if ($scope.component.name && !$scope.component.id) {
            $scope.component.id = angular.copy($scope.component.name);
            $scope.component.name = undefined;
        }
        //TODO: check above block of code and move to common place
        //Web GL is not supported in PhantomJS browser simulator
        //show an error message instead of showing angular JS code
        if (typeof (mapboxgl) === "undefined") {
            $scope.showErrorDiv = true;
            $scope.component.isComponentLoading = false;
            return;
        }

        //defaults and declarations, section start
        mapboxgl.accessToken = 'pk.eyJ1IjoiY29tbXZhdWx0LW1hcHMiLCJhIjoiNk81djMzYyJ9.1fAPKqwczcfXkTEBZevStA';
        var numericDataTypes = ['Double', 'Float', 'Integer', 'Long', 'Short', 'Decimal'];
        if (!$scope.component.inputType) {
            $scope.component.inputType = 'latlon';
        }

        var limit = 999999,
            srcPrefix = 'geoJsonSource_',
            worldBounds = [-180.0000, -90.0000, 180.0000, 90.0000];
        var MAX_ATTEMPTS_FOR_STYLE_LOAD = 5;
        var DEFAULT_ZOOM = 1.7,
            MAX_ZOOM = 14,
            DEFAULT_COLOR = "#00cee6",
            MARKER_ID = "cv-marker";
        var additionalDimensionProperties = {
            'numPointsToDisplay': {
                'includeAll': true,
                'maxPoints': 15
            }
        };
        var cache = {},
        worldGeoJson = undefined;

        const DEFAULT_CIRCLE_COLOR = "#ffffff";
        
        //defaults and declarations, section end

        //register common events, section start
        customReportSvc.registerCallback("redrawAllComponents", function (data) {
            $scope.loadComponent();
        }, $scope.component.id);

        customReportSvc.registerCallback("refreshComponent", function (
            componentId) {
            if ($scope.component.id === componentId) {
                $scope.loadComponent();
            }
        }, $scope.component.id);

        customReportSvc.registerCallback("redrawComponent", function (
            componentId) {
            if ($scope.component.id === componentId) {
                loadMap(0);
            }
        }, $scope.component.id);
        //register common events, section end

        //local methods, section start
        function convertArrayToObject(records, columns) {
            var arrOfObj = records,
                eachRecord, obj, incBy;
            $scope.categoryMap = undefined;
            if (records && records.length > 0) {
                if (Array.isArray(records[0])) {
                    arrOfObj = [];
                    for (var i = 0; i < records.length; i++) {
                        eachRecord = records[i];
                        obj = {};
                        if ($scope.component.inputType === 'cc') {
                            obj[$scope.component.countrycode.dataField] = eachRecord[0];
                            if ($scope.component.categoryColumn) {
                                obj[$scope.component.categoryColumn.dataField] = eachRecord[1];
                                incBy = 2;
                            } else
                                incBy = 1;
                        } else {
                            obj[$scope.component.latitude.dataField] = eachRecord[0];
                            obj[$scope.component.longitude.dataField] = eachRecord[1];
                            incBy = 2;
                        }
                        for (var j = 0; j < columns.length; j++) {
                            obj[columns[j].id] = eachRecord[j + incBy];
                            obj[columns[j].id + "_formatted"] = customReportSvc.getFormattedValue(eachRecord[j + incBy], columns[j].cellExpression);
                        }
                        arrOfObj.push(obj);
                    }
                } else if ($scope.component.inputType === 'cc' && $scope.component.categoryColumn) {
                    $scope.categoryMap = new Map();
                    arrOfObj = [];
                    for (var i = 0; i < records.length; i++) {
                        eachRecord = records[i];
                        obj = {};
                        obj[eachRecord.field] = eachRecord.value;
                        for (var j = 0; j < columns.length; j++) {
                            var col = columns[j];
                            obj[col.id] = eachRecord.aggregates[col.id][col.aggrType.toLowerCase()];
                            obj[col.id + "_formatted"] = customReportSvc.getFormattedValue(obj[col.id], col.cellExpression);
                        }
                        if (eachRecord.items) {
                            let customValuesFromData = [];
                            var categories = [];
                            for (var j = 0; j < eachRecord.items.length; j++) {
                                var eachItem = eachRecord.items[j];
                                var cObj = {};
                                cObj[eachItem.field] = eachItem.value;
                                $scope.categoryMap.set(eachItem.value, "");
                                //as categories are sorted by the measure, the first category decides the color
                                if (!$scope.component.showBubbleColorByCustomColorOrder && j === 0) {
                                    obj['categoryToBeColored'] = eachItem.value;
                                }

                                if($scope.component.showBubbleColorByCustomColorOrder && eachItem.count > 0)
                                      customValuesFromData.push(eachItem.value);

                                for (var k = 0; k < columns.length; k++) {
                                    var col = columns[k];
                                    cObj[col.id] = eachItem.aggregates[col.id][col.aggrType.toLowerCase()];
                                    cObj[col.id + "_formatted"] = customReportSvc.getFormattedValue(cObj[col.id], col.cellExpression);
                                }
                                categories.push(cObj);
                            }
                             if($scope.component.showBubbleColorByCustomColorOrder && angular.isDefined($scope.component.seriesFormatting)
                                 && $scope.component.seriesFormatting.length > 0) {
                                //Show first custom color by default if all custom color values has 0 count
                                if(customValuesFromData.length === 0)
                                    obj['categoryToBeColored'] = $scope.component.seriesFormatting[0].value;
                                else {
                                    //show bubble color as topmost custom color
                                    const customColorMap = $scope.component.seriesFormatting.filter(function(color) {
                                        return $.inArray(color.value,customValuesFromData) !== -1;
                                    });
                                    obj['categoryToBeColored'] = customColorMap[0].value;
                                } 
                             }
                             

                            obj['categories'] = categories;
                        }
                        arrOfObj.push(obj);
                    }
                }
            }
            if ($scope.categoryMap && $scope.categoryMap && $scope.categoryMap.size > 0) {
                $scope.component.currDataSeries = [...$scope.categoryMap.keys()].map(function (eachVal) {
                    return {
                        value: eachVal
                    }
                });
            } else
                $scope.component.currDataSeries = [];
            return arrOfObj;
        }

        function getDefaultTooltipForDataPoint(row) {
            var tooltip = "";
            if ($scope.component.inputType === "cc" && $scope.component.countrycode) {
                tooltip += "<span class='tooltip-country-name'>" + row['__countryname__'] + "</span><br/>";
            }
            angular.forEach($scope.component.columns, function (eachColumn, index) {
                tooltip += "<b>" + eachColumn.displayName + "</b>: " + customReportSvc.getFormattedValue(row[eachColumn.id], eachColumn.cellExpression) + "<br/>";
            });
            if ($scope.component.inputType === 'cc' && $scope.component.categoryColumn && row.categories) {
                var cArr = JSON.parse(row.categories);
                if (!$.isEmptyObject(cArr)) {
                    angular.forEach(cArr, function (cObj) {
                        tooltip += "<b>" + cObj[$scope.component.categoryColumn.dataField] + "</b>: " + customReportSvc.getFormattedValue(cObj[$scope.bubbleSizeByCol.id], $scope.bubbleSizeByCol.cellExpression) + "<br/>";
                    });
                }
            }
            return tooltip;
        }

        function getDefaultTooltipForCluster(row) {
            var tooltip = "";
            angular.forEach($scope.component.columns, function (eachColumn, index) {
                tooltip += "<b>" + eachColumn.displayName + ": </b>" + row[eachColumn.id + "_" + eachColumn.aggrType + "_formatted"] + "<br/>";
            });
            return tooltip;
        }

        function getGeoMetry(eachRecord) {
            var gObj = {};
            if ($scope.component.inputType === 'cc') {
                var ccVal = eachRecord[$scope.component.countrycode.dataField];
                if (cache.hasOwnProperty(ccVal)) {
                    gObj = cache[ccVal];
                } else {
                    let country = worldGeoJson.find(function (countryInfo) {
                        return countryInfo.country_code === ccVal;
                    }); 
                    if(country) {
                        gObj.geometry = {
                            'type': 'Point',
                            'coordinates': country.latlng.reverse()
                        };
                         gObj['__countryname__'] = country.name;
                    } else {
                        console.log("Couldn't form geometry object for ", ccVal);
                    }
                    cache[ccVal] = gObj;
                }
            } else {
                gObj.geometry = {
                    'type': 'Point',
                    'coordinates': [eachRecord[$scope.component.longitude.dataField],
                        eachRecord[$scope.component.latitude.dataField]
                    ]
                };

            }
            return gObj;
        }

        function getGeoJsonData(records) {
            var featuresArr = [],
                obj, args, geo;
            angular.forEach(records, function (eachRecord, index) {
                geo = getGeoMetry(eachRecord);
                if (!$.isEmptyObject(geo)) {
                    obj = {
                        'type': 'Feature',
                        'properties': eachRecord,
                        'geometry': geo.geometry
                    };
                    obj.properties['__countryname__'] = geo['__countryname__'];
                    featuresArr.push(obj);
                }
            });
            return {
                type: 'FeatureCollection',
                features: featuresArr
            };
        }

        function getSuperCluster() {
            var columns = $scope.component.columns;
            return new supercluster({
                radius: 50,
                maxZoom: MAX_ZOOM,
                initial: function () {
                    var col, obj = {
                        count: 0
                    };
                    for (var i = 0; i < columns.length; i++) {
                        col = columns[i];
                        obj[col.id + "_" + col.aggrType] = 0;
                    }
                    return obj;
                },
                map: function (properties) {
                    var col, obj = {
                        count: 1
                    };
                    for (var i = 0; i < columns.length; i++) {
                        col = columns[i];
                        obj[col.id + "_" + col.aggrType] = Number(properties[col.id]);
                    }
                    return obj;
                },
                reduce: function (accumulated, properties) {
                    var col, key;
                    accumulated.count += properties.count;
                    for (var i = 0; i < columns.length; i++) {
                        col = columns[i];
                        key = col.id + "_" + col.aggrType;
                        if (col.aggrType === "Sum" || col.aggrType === "Count" || col.aggrType === "CountDistinct")
                            accumulated[key] += properties[key];
                        else if (col.aggrType === "Min")
                            accumulated[key] = Math.min(accumulated[key], properties[key]);
                        else if (col.aggrType === "Max")
                            accumulated[key] = Math.max(accumulated[key], properties[key]);
                        else if (col.aggrType === "Avg")
                            accumulated[key] = (accumulated[key] + properties[key]) / accumulated.count;
                        accumulated[key + "_formatted"] = customReportSvc.getFormattedValue(accumulated[key],
                            col.cellExpression);
                    }
                }
            });
        }

        function getNormalizedValues(arr) {
            var nArr = $.extend([], arr);
            if (arr && arr.length > 0) {
                var c = Math.log(arr.length);
                for (var i = 0; i < nArr.length; i++) {
                    nArr[i] = Math.log(nArr[i]);
                    if (nArr[i] === -Infinity)
                        nArr[i] = 0;
                    nArr[i] *= c;
                }
                var min = nArr[0],
                    max = nArr[nArr.length - 1],
                    diff = max - min;
                for (var i = 0; i < nArr.length; i++) {
                    nArr[i] = diff === 0 ? 1 : (nArr[i] - min) / diff;
                }
            }
            return nArr;
        }

        function createRadiusStops(arr, nArr, minRadius, maxRadius) {
            var stops = [],
                diff = maxRadius - minRadius;
            for (var i = 0; i < arr.length; i++)
                stops.push([arr[i], (minRadius + (nArr[i] * diff))]);
            return stops;
        }

        function getRadiusStops(geoData, isCluster) {
            var aggValues, nAggValues, radiusStops;
            var propKey = $scope.bubbleSizeByCol.id;
            if (isCluster) {
                propKey = $scope.bubbleSizeByCol.id + "_" + $scope.bubbleSizeByCol.aggrType;
            }
            aggValues = geoData.features.filter(function (obj) {
                return !isCluster || obj.properties.cluster;
            }).map(function (fObj) {
                return fObj.properties[propKey];
            });
            if (aggValues && aggValues.length > 0) {
                aggValues = aggValues.sort(function (v1, v2) {
                    return v1 - v2;
                });
                nAggValues = getNormalizedValues(aggValues);
                radiusStops = createRadiusStops(aggValues, nAggValues, 10, 40);
            }
            return radiusStops;
        }

        function getCategoryColor(val, index) {
            var colorToReturn;
            if (!$.isEmptyObject($scope.customColorMap)) {
                colorToReturn = $scope.customColorMap[val];
            }
            if (!colorToReturn) {
                //if not found in custom color, pick it from default color list
                var colorCount = customReportSvc.defaultMultiChartColors.length;
                colorToReturn = customReportSvc.defaultMultiChartColors[index %
                    colorCount];
            }
            return colorToReturn ? colorToReturn : DEFAULT_COLOR;
        }


        function getPaintProps(radiusStops, isCluster) {
            var ccDefn, keys = [],
                color;
            if ($scope.component.inputType === 'cc' && $scope.component.categoryColumn && $scope.categoryMap && $scope.categoryMap.size > 0) {
                $scope.colorMapForToolTip = {};
                ccDefn = ['match', ['get', 'categoryToBeColored']];
                keys = [...$scope.categoryMap.keys()];
                for (var i = 0; i < keys.length; i++) {
                    color = getCategoryColor(keys[i], i);
                    ccDefn.push(keys[i]);
                    ccDefn.push(color);
                    $scope.colorMapForToolTip[keys[i]] = color;
                }
                //push color for others; mandatory in mapbox syntax
                ccDefn.push(DEFAULT_COLOR);
            }
            var circleColor = ccDefn ? ccDefn : $scope.component.mapColor,
            paintProps = {
                "circle-color": circleColor,
                "circle-radius": 4,
                "circle-opacity" : 0.5,
                "circle-stroke-width": 1,
                "circle-stroke-color": circleColor,
            };
            if (radiusStops && radiusStops.length > 0) {
                var propKey = $scope.bubbleSizeByCol.id;
                if (isCluster) {
                    propKey = $scope.bubbleSizeByCol.id + "_" + $scope.bubbleSizeByCol.aggrType;
                }
                paintProps['circle-radius'] = {
                    property: propKey,
                    type: "interval",
                    stops: radiusStops
                };
            }
            return paintProps;
        }

        function addDataLayer(paintProps) {
            var dataLayer = {
                id: "data-layer",
                type: "symbol",
                source: srcPrefix + $scope.component.id,
                filter: ["!has", "point_count"],
                layout: {
                    "icon-image": MARKER_ID,
                    "icon-allow-overlap": true
                }
            };
            if($scope.component.enableHtmlClusters && !$.isEmptyObject($scope.clusterProps)) {
                let circleColor =  [];
                Object.keys($scope.clusterProps).forEach(val => {
                    circleColor = [...circleColor,$scope.clusterProps[val][1][1],$scope.clusterColorMap[val]]
                });
                dataLayer["paint"] = {
                    "icon-color" :  ['case',...circleColor, '#6aa5e7']
                }
            } else if ($scope.bubbleSizeByCol) {
                if (!$scope.component.enableClustering) {
                    dataLayer.type = "circle";
                    dataLayer.paint = paintProps;
                    dataLayer.layout = {};
                } else if ($scope.component.showLabels) {
                    dataLayer.layout["text-field"] = "{" + $scope.bubbleSizeByCol.id + "_formatted}";
                    dataLayer.layout["text-font"] = ["Open Sans Regular"];
                    dataLayer.layout["text-size"] = 12;
                    dataLayer.layout["text-allow-overlap"] = true;
                }
            }

            $scope.map.addLayer(dataLayer);
            
        }

        function addLabelLayer() {
            var labelLayer = {
                id: "label-layer",
                type: "symbol",
                source: srcPrefix + $scope.component.id,
                layout: {
                    "text-field": $scope.propKeyForTextField,
                    "text-font": ["Open Sans Regular"],
                    "text-size": 12
                }
            };
            if ($scope.component.enableClustering)
                labelLayer.filter = ["has", "point_count"];
            $scope.map.addLayer(labelLayer);
        }

        function removeAllLayers() {
            if ($scope.map.getLayer('label-layer')) {
                $scope.map.removeLayer('label-layer');
            }
            if ($scope.map.getLayer('data-layer')) {
                $scope.map.removeLayer('data-layer');
            }
            if ($scope.map.getLayer("cluster-layer")) {
                $scope.map.removeLayer("cluster-layer");
            }
        }

        function initMapListeners() {
            $scope.isStyleLoaded = false;
            $scope.popup = new mapboxgl.Popup({
                closeButton: false,
                closeOnClick: false
            });
            $scope.map.on('style.load', function () {
                $scope.isStyleLoaded = true;
            });
            $scope.map.on('zoom', function () {

                var newZoom = $scope.map.getZoom();

                if (Math.floor($scope.component.currentZoom) == 0) {
                    $scope.component.currentZoom = 1;
                };

                if (Math.floor(newZoom) != Math.floor($scope.component.currentZoom)) {
                    $scope.component.currentZoom = newZoom;
                    if ($scope.component.enableClustering && $scope.component.bubbleSizeBy) {
                        loadClusters(true);
                    }
                }
            });

            //add listener for data points popup and cursor 
            $scope.map.on('mouseenter', 'data-layer', function (e) {
                $scope.map.getCanvas().style.cursor = 'pointer';

                var coordinates = e.features[0].geometry.coordinates.slice();
                var prop = e.features[0].properties;
                var tooltip = "";
                if ($scope.component.tooltipExpForDataPoint) {
                    prop.colorMap = $scope.colorMapForToolTip;
                    if ($scope.bubbleSizeByCol) {
                        prop.aggrColumn = $scope.bubbleSizeByCol.id;
                    }
                    if ($scope.component.categoryColumn) {
                        prop.categoryColumn = $scope.component.categoryColumn.id;
                    }
                    tooltip = customReportSvc.evalExpression($scope.component.tooltipExpForDataPoint, prop);
                } else {
                    tooltip = getDefaultTooltipForDataPoint(prop);
                    tooltip = '<div class="mapbox-popup">' + tooltip + '</div>'
                }

                $scope.popup.setLngLat(coordinates)
                    .setHTML(tooltip)
                    .addTo($scope.map);
            });
            $scope.map.on('click', 'data-layer', function (e) {
                if ($scope.component.inputType === 'latlon' && $scope.component.onClickMarkerExp) {
                    const props = e.features[0].properties;
                    let exp = $scope.component.onClickMarkerExp.trim();
                    if(exp.startsWith(":="))
                        exp = exp.substr(2);
                    exp = customReportSvc.evalExpression(exp,props);
                    window.open(exp, '_blank');
                }

                if ($scope.component.disableInteractivity)
                    return;
                if ($scope.component.inputType === 'cc') {
                    $scope.addFilters($scope.component.countrycode.dataField,
                        e.features[0].properties[$scope.component.countrycode.dataField],
                        $scope.component,
                        true,
                        true,
                        undefined,
                        undefined,
                        "viewer",
                        "include",
                        true);
                }
            });

            //add listener for cluster popup
            $scope.map.on('mouseenter', 'cluster-layer', function (e) {
                $scope.map.getCanvas().style.cursor = 'pointer';

                var coordinates = e.features[0].geometry.coordinates.slice();
                var prop = e.features[0].properties;
                var tooltip = "";

                if ($scope.component.tooltipExpForClustered)
                    tooltip = customReportSvc.evalExpression($scope.component.tooltipExpForClustered, prop);
                else
                    tooltip = getDefaultTooltipForCluster(prop);

                $scope.popup.setLngLat(coordinates)
                    .setHTML('<div class="mapbox-popup">' + tooltip + '</div>')
                    .addTo($scope.map);
            });

            function onMouseLeave(e) {
                if (e && e.originalEvent && e.originalEvent.toElement &&
                    !($(e.originalEvent.toElement).parents().hasClass("mapboxgl-popup"))) {
                    $scope.map.getCanvas().style.cursor = '';
                    $scope.popup.remove();
                }
            }
            $scope.map.on('mouseleave', 'data-layer', onMouseLeave);
            $scope.map.on('mouseleave', 'cluster-layer', onMouseLeave);
            $("#" + $scope.component.id).off("mouseleave", ".mapboxgl-popup-content")
                .on("mouseleave", ".mapboxgl-popup", function (e) {
                    $scope.map.getCanvas().style.cursor = '';
                    $scope.popup.remove();
                });

            $scope.map.on('render', () => {
                if (!$scope.map.loaded() || $.isEmptyObject($scope.clusterProps)) return;           
                updateMarkers();
            });
        }

        function loadClusters(repaint) {
            var clusterData, radiusStops;
            if (!repaint) {
                $scope.cluster = getSuperCluster();
                $scope.cluster.load($scope.geoJsonData.features);
            }
            clusterData = {
                type: "FeatureCollection",
                features: $scope.cluster.getClusters(worldBounds, Math.floor($scope.component.currentZoom))
            };
            radiusStops = getRadiusStops(clusterData, true);

            if (!repaint) {
                if ($scope.map.getLayer("cluster-layer"))
                    $scope.map.removeLayer("cluster-layer");
                if (radiusStops && radiusStops.length > 0) {
                    $scope.map.addLayer({
                        id: "cluster-layer",
                        type: "circle",
                        source: srcPrefix + $scope.component.id,
                        filter: ["has", "point_count"],
                        paint: getPaintProps(radiusStops, true)
                    });
                }
            } else {
                if (radiusStops && radiusStops.length > 0) {
                    $scope.map.setPaintProperty('cluster-layer', 'circle-radius', {
                        property: $scope.bubbleSizeByCol.id + "_" + $scope.bubbleSizeByCol.aggrType,
                        stops: radiusStops
                    });
                }
            }
            $scope.map.getSource(srcPrefix + $scope.component.id).setData(clusterData);
        }

       
        function updateMarkers() {
            let newHtmlMarkers = {};
            //get features for current source
            const features = $scope.map.querySourceFeatures(srcPrefix + $scope.component.id);
            
            // for every cluster on the screen, create an HTML marker for it (if we didn't yet),
            // and add it to the map if it's not there already
            features.forEach(feature => {
                const props = feature.properties;
                //if it's not a cluster then do nothing
                if(!props.cluster) return;

                const clusterId = props.cluster_id;
                let htmlMarker = $scope.htmlMarkers[clusterId];
                let popup;
                //If html marker for clusterId is not already present then create it
                if(!htmlMarker) {
                    const donutElement = createDonutChart(props);
                    const coords = feature.geometry.coordinates;
                    let tooltip = "";
                    if($scope.component.htmlClusterTooltipExp) {
                        var args = {
                            expression : $scope.component.htmlClusterTooltipExp,
                            props
                        };
                        tooltip = rpt.evalExpression(args);
                    } else {
                        Object.keys(props).forEach(prop => {
                            tooltip+= Object.keys($scope.clusterProps).includes(prop) && props[prop] !==0  ? `<div>${prop}:${props[prop]}</div>` : "";
                        });
                    }
                    popup = new mapboxgl.Popup({offset: 25,closeButton: false}).setHTML(tooltip);
                   
                    htmlMarker = $scope.htmlMarkers[clusterId] = new mapboxgl.Marker({
                        element: donutElement
                    }).setLngLat(coords).setPopup(popup); 
                }
                newHtmlMarkers[clusterId] = htmlMarker;
                
                //if newly created marker not already present on screen then add it to Map
                if (!$scope.htmlMarkersOnScreen[clusterId]) htmlMarker.addTo($scope.map);
            });

            // for every marker we've added previously, remove those that are no longer visible
            for (var id in $scope.htmlMarkersOnScreen) {
                if (!newHtmlMarkers[id]) $scope.htmlMarkersOnScreen[id].remove();
            }
            $scope.htmlMarkersOnScreen = newHtmlMarkers;
        };

        //function to create the donut segment based on start,end,radius and fill it with given color
        function donutSegment(start, end, radius, r0, color) {
            if (end - start === 1) end -= 0.00001;
            const a0 = 2 * Math.PI * (start - 0.25);
            const a1 = 2 * Math.PI * (end - 0.25);
            const x0 = Math.cos(a0);
            const y0 = Math.sin(a0);
            const x1 = Math.cos(a1);
            const y1 = Math.sin(a1);
            const largeArc = end - start > 0.5 ? 1 : 0;

            return `<path d="M ${radius + r0 * x0} ${radius + r0 * y0} L ${radius + radius * x0} ${radius + radius * y0} A 
             ${radius} ${radius} ${0} ${largeArc} ${1} ${radius + radius * x1} ${radius + radius * y1} L 
             ${radius + r0 * x1} ${radius + r0 * y1} A ${r0} ${r0}  ${0} ${largeArc} ${0} ${radius + r0 * x0} 
             ${radius + r0 * y0}" fill="${color}"/>`;
            
        }

        function getFontSizeForSVG(totalCount) {
            switch(true) {
                case totalCount >= 1000 :
                    return 22;
                case totalCount >= 100 :
                   return 20;
                case totalCount >= 10:
                    return 18;
                default:
                    return 16;
            }
        }

        function getCircleRadius(totalCount) {
            switch(true) {
                case totalCount >= 1000 :
                    return 50;
                case totalCount >= 100 :
                   return 32;
                case totalCount >= 10:
                    return 24;
                default:
                    return 18;
            }
        }
        // function for creating an SVG donut chart from feature properties
        function createDonutChart (props) {
            const colors = ['#fed976', '#feb24c', '#fd8d3c', '#fc4e2a', '#e31a1c'];
            let offsets = [];
            let propNames = Object.keys($scope.clusterProps);
            //get count for every property/filter
            let counts = propNames.map(propName => props[propName]);
            let total = 0;
            //calculate the total of all property count and form offsets Array
            counts.forEach(count => {
                offsets.push(total);
                total += count;
            });
            
            const fontSize = getFontSizeForSVG(total);
            const radius = getCircleRadius(total);
            var r0 = Math.round(radius * 0.6);
            var size = radius * 2;

            let html = `<div style="cursor:pointer"><svg width="${size}" height="${size}" viewbox="0 0 ${size} ${size}" 
            text-anchor="middle" style="font:${fontSize}px sans-serif; display: block">`;
            
            counts.forEach((count, indx) => {
                const segmentstart = offsets[indx] / total;
                const segmentEnd = (offsets[indx] + count) / total;
                const propName = propNames[indx];//get corresponding property
                html += donutSegment(segmentstart, segmentEnd, radius , r0, $scope.clusterColorMap[propName]);
            });
           
            html += `<circle cx="${radius}" cy="${radius}" r="${r0}" fill="${$scope.component.innerCircleColor}"/>
                    <text dominant-baseline="central" transform="translate(${radius},${radius})"> 
                    ${total.toLocaleString()}</text></svg></div>`;
            
            const divElement = document.createElement('div');
            divElement.innerHTML = html;
            return divElement.firstChild;
        }

		function getClusterExpression(operator, value) {
            const categoryCol = $scope.component.categoryColumn;
			let columnValue = ["get", categoryCol.column];
            if(categoryCol.type.toLowerCase() === "string") {
                //converting expression value and column value to lowercase to make comparison case insensitive
                columnValue = ["downcase", columnValue];
                value = ["downcase", value];
            } else {
                //for integer/float  covert string to number. For number operators if used with string value throws an error
                value = Number(value);
            }
			
            switch(operator) {
                case "empty" :
                    return ["==",columnValue, ""];
                case "!empty" :
                    return ["!=", columnValue, ""];
                case "==":
                    return ["==", columnValue, value];
                case "!=":
                    return ["!=", columnValue, value];
                case ">" : 
                return [">", columnValue, value];
                case ">=":
                    return [">=", columnValue, value];
                case "<":
                    return ["<", columnValue, value];
                case "<=":
                    return ["<=", columnValue, value];
                default :
                    return [];
            }
		}
		
        function getclusterFilter(filterExp) {
            const andGroups = filterExp.split('&&');
			let clusterFilter = andGroups.length === 1 ? [] : ['all'];
			const type = $scope.component.categoryColumn.type;
            const operators = customReportSvc.getFilterOpsForHtmlCluster(type);
            andGroups.map(andGroup => {
                const orGroups = andGroup.split('||');
                let orFilter = orGroups.length === 1 ? [] : ['any'];
                orGroups.forEach(orGroup => {
                    const exp = orGroup.trim();
                    const operator = operators.find(op => exp.startsWith(op));
                    //if opeartor is supported then only form an expression
                    if(operator) {
                        // Once we get the operator then remaining string will be value
                        let val = exp.substr(operator.length).trim();
                        //if value is enclosed in quotes then discard quotes 
                        if((val.startsWith("'") || val.startsWith('"')) &&
                        (val.endsWith("'") || val.endsWith('"'))) {
                            val = val.slice(1,-1);
                        }
                        const filterExp = getClusterExpression(operator, val);
                        orFilter = orFilter.length === 0 ? filterExp : [...orFilter, filterExp];
                    }                    
                });
                //if there is any expression then only add it. Wrong expressions will get discarded.
                if(orFilter.length > 1) 
                    clusterFilter = clusterFilter.length === 0 ? orFilter : [...clusterFilter, orFilter];
            });
            return clusterFilter;
        }
        function setClusterProperties() {
            $scope.clusterProps = {};
            $scope.clusterColorMap = {};
            const categoryCol = $scope.component.categoryColumn;
           if(categoryCol && categoryCol.mapFilters && categoryCol.mapFilters.length)
            { 
				const colors = customReportSvc.defaultMultiChartColors;
				const isUngrouped = !$.isEmptyObject($scope.component.categoryColumn) 
				                    && $scope.component.categoryColumn.includeUngrouped;
                let ungroupedFilter = isUngrouped ? ['any'] : [];
                $scope.component.categoryColumn.mapFilters.forEach((filter,indx) => {
					const clusterFilter = getclusterFilter(filter.value);
					if(isUngrouped)
					    ungroupedFilter.push(clusterFilter);
                    $scope.clusterProps[filter.name]=['+',['case',clusterFilter, 1, 0]];
                    $scope.clusterColorMap[filter.name] = filter.color || colors[indx];
				});
				
				if(ungroupedFilter.length > 1) {
					//add cluster property for ungrouped values if include ungroup is selected
					ungroupedFilter = ["!",ungroupedFilter];
					const label = $scope.component.categoryColumn.ungroupedLabel || 'Other'; 
					const color = $scope.component.categoryColumn.ungroupedColor || colors[0];
                    $scope.clusterProps[label] = ['+',['case', ungroupedFilter, 1, 0]];
                    $scope.clusterColorMap[label] = color;
                }
            } 
        }

        function loadMap(attempt) {
            removeAllLayers();
            $scope.customColorMap = {};
            if (!$.isEmptyObject($scope.component.seriesFormatting)) {
                angular.forEach($scope.component.seriesFormatting, function (
                    eachSeries) {
                    $scope.customColorMap[eachSeries.value] = eachSeries.color;
                });
            }
            if ($scope.component.bubbleSizeBy) {
                //aggregate of bubbleSizeBy column need not get saved in XML, so put it in scope
                $scope.bubbleSizeByCol = $.extend({}, $scope.component.columns.filter(function (eachCol) {
                    return eachCol.id === $scope.component.bubbleSizeBy;
                })[0]);
                $scope.propKeyForTextField = "{" + $scope.bubbleSizeByCol.id + "_formatted}";
                if ($scope.component.enableClustering) {
                    $scope.propKeyForTextField = "{" + $scope.bubbleSizeByCol.id + "_" + $scope.bubbleSizeByCol.aggrType + "_formatted}";
                }
            } else
                $scope.bubbleSizeByCol = undefined;
            if (!$scope.isStyleLoaded) {
                if (attempt < MAX_ATTEMPTS_FOR_STYLE_LOAD) {
                    //check after a second
                    $timeout(function () {
                        console.log("Waiting for map style to be loaded. Attempt: " + attempt);
                        loadMap(++attempt);
                    }, 2000);
                } else {
                    customReportSvc.errorToast("Failed to load map.");
                    $scope.component.isComponentLoading = false;
                }
            } else {
                var radiusStops = (!$scope.component.enableClustering && $scope.bubbleSizeByCol) ?
                    getRadiusStops($scope.geoJsonData) : [];
                var paintProps = getPaintProps(radiusStops);
                var src = $scope.map.getSource(srcPrefix + $scope.component.id);
                if (src) {
                    src.setData($scope.geoJsonData);
                    addDataLayer(paintProps);
                } else {
                    let props = {
                        type: "geojson",
                        buffer: 1,
                        maxzoom: MAX_ZOOM,
                        data: $scope.geoJsonData
                    }
                   if($scope.component.enableHtmlClusters && $scope.component.categoryColumn) {
                        setClusterProperties();
                        const clusterProps = {
                            cluster: true,
                            clusterRadius: 80,
                            clusterProperties:  $scope.clusterProps
                        };
                        props = {...props,...clusterProps};
                    }
                    $scope.map.addSource(srcPrefix + $scope.component.id, props);
                    addDataLayer(paintProps);
                }
                if ($scope.component.enableClustering && $scope.bubbleSizeByCol) {
                    loadClusters(false);
                }
                //label layer needs to be added after adding cluster-layer
                if ($scope.component.showLabels && $scope.bubbleSizeByCol) {
                    addLabelLayer();
                }
                //below code is repaint data layer and label layer; just adding them doesn't repaint
                //so need to set paint and layout property
                if (src && $scope.bubbleSizeByCol) {
                    //when clustering is enabled, data points will be symbol(pin)
                    if (!$scope.component.enableClustering) {
                        $scope.map.setPaintProperty('data-layer', 'circle-radius', paintProps['circle-radius']);
                    }
                    if ($scope.component.showLabels) {
                        $scope.map.setLayoutProperty('label-layer', 'text-field', $scope.propKeyForTextField);
                    }
                }
                $scope.component.isComponentLoading = false;
            }
        }

        function initMap() {
            $scope.map = new mapboxgl.Map({
                container: 'cvmap_' + $scope.component.id,
                //possible values for style: basic, streets, bright, light, dark, satellite
                style: $scope.component.mapStyle,
                //center: [0, 25.2],
                zoom: $scope.component.currentZoom,
                renderWorldCopies: false
            });
            $scope.map.addControl(new mapboxgl.NavigationControl());
            //$scope.map.scrollZoom.disable();
            initMapListeners();
        }

        function onSuccessGetMapData(resultData) {
            $scope.geoJsonData = {};
            $scope.htmlMarkers = {};
            $scope.htmlMarkersOnScreen= {};
            var response = resultData.data || resultData;
            //When category column is applied, for data cube, records will be in groups            
            var records = (($scope.component.inputType === 'cc' && 
            $scope.component.categoryColumn && $scope.dataSet.endpoint === 'DATACUBE') ? response.groups :
                response.records) || [];
            if (records.length === 0) {
                $scope.showNoDataDiv = true;
                $scope.component.isComponentLoading = false;
                return;
            }

            //convert array of array values(if exist) to array of objects
            console.time('ArrayToObject');
            records = convertArrayToObject(records, $scope.component.columns);
            console.timeEnd('ArrayToObject');
            console.time('GeoJsonConversion');
            $scope.geoJsonData = getGeoJsonData(records);
            console.timeEnd('GeoJsonConversion');
            loadMap(0);
        }

        function getMapData() {
            var dataSetName = $scope.dataSet.dataSet.dataSetName;
            var dimensionDataField = [],
                measureDataField = [];

            if ($scope.component.inputType === 'latlon') {
                if ($
                    .isEmptyObject($scope.component.latitude.numPointsToDisplay)) {
                    $scope.component.latitude = $.extend({},
                        $scope.component.latitude,
                        additionalDimensionProperties);
                    $scope.component.longitude = $.extend({},
                        $scope.component.longitude,
                        additionalDimensionProperties);
                }
                var latitudeColumn = {
                    'column': $scope.component.latitude.dataField,
                    'dataField': $scope.component.latitude.dataField,
                    'numPointsToDisplay': $scope.component.latitude.numPointsToDisplay,
                    'sortOrder': "NONE"
                };
                var longitudeColumn = {
                    'column': $scope.component.longitude.dataField,
                    'dataField': $scope.component.longitude.dataField,
                    'numPointsToDisplay': $scope.component.longitude.numPointsToDisplay,
                    'sortOrder': "NONE"
                };
                dimensionDataField.push(latitudeColumn);
                dimensionDataField.push(longitudeColumn);
                limit = latitudeColumn.numPointsToDisplay.includeAll ? limit :
                    latitudeColumn.numPointsToDisplay.maxPoints;
                if($scope.component.enableHtmlClusters && $scope.component.categoryColumn) {
                    const column = $scope.component.categoryColumn.column;
                    const dataCol = $scope.component.columns.find(col => col.column === column);
                    //if category column not present in data columns then only add it
                    if(!dataCol) {
                        const columnInfo = {
                            column,
                            dataField: column,
                            aggrType: 'None'
                        };
                        measureDataField.push(columnInfo);
                    }
                    
                }
            } else if ($scope.component.inputType === 'cc') {
                if ($
                    .isEmptyObject($scope.component.countrycode.numPointsToDisplay)) {
                    $scope.component.countrycode = $.extend({},
                        $scope.component.countrycode,
                        additionalDimensionProperties);
                }
                var ccColumn = {
                    'column': $scope.component.countrycode.dataField,
                    'dataField': $scope.component.countrycode.dataField,
                    'numPointsToDisplay': $scope.component.countrycode.numPointsToDisplay,
                    'sortOrder': "NONE"
                };
                dimensionDataField.push(ccColumn);
                if ($scope.component.categoryColumn) {
                    if ($
                        .isEmptyObject($scope.component.categoryColumn.numPointsToDisplay)) {
                        $scope.component.categoryColumn = $.extend({},
                            $scope.component.categoryColumn,
                            additionalDimensionProperties);
                    }
                    var categoryColumn = $scope.applyInputsToCustomGroups([$scope.component.categoryColumn])[0];
                    categoryColumn.column = $scope.component.categoryColumn.dataField;
                    dimensionDataField.push(categoryColumn);
                }
                limit = ccColumn.numPointsToDisplay.includeAll ? limit :
                    ccColumn.numPointsToDisplay.maxPoints;
            }
            angular.forEach($scope.component.columns, function (column,index) {
                var columnInfo = {
                    'column': column.dataField,
                    'dataField': column.dataField,
                    'aggrType': column.aggrType || 'None'
                };
				if(index === 0 && $scope.component.isGroupBy && columnInfo.aggrType === 'None' ){
                	$scope.component.isGroupBy = undefined; // removing the isGroupBy property so that the map is drawn with out any aggregation.
                }
                measureDataField.push(columnInfo);
            });
            var columns = dimensionDataField;
            columns = columns.concat(measureDataField);
            var tableParams = {
                columns: customReportSvc
                    .getSelectedColumnsDefn($scope.dataSet.fields,
                        columns),
                offset: 0,
                limit: limit
            };
            if ($scope.dataSet.endpoint === 'DATACUBE') {
                const isCountryCodeCategory = $scope.component.inputType === 'cc' && $scope.component.categoryColumn;
                $dataSource
                    .getDataSource($scope.dataSet.endpoint)
                    .getMapData({
                            dataSet: $scope.dataSet,
                            componentType: $scope.component.type,
                            measureDataField: measureDataField,
                            dimensionDataField: dimensionDataField,
                            sortOptions: (!$scope.component.showBubbleColorByCustomColorOrder && isCountryCodeCategory) ? {
                                sortAxis: "YAxis",
                                direction: "desc"
                            } : undefined,
                            tableParams: tableParams,
                            isObjFormat: isCountryCodeCategory ? true : false,
                            constructGroupStructure: isCountryCodeCategory ? true : false,
                            filters: $scope.component.filters &&
                                $scope.component.filters['builder'] ? $scope.component.filters['builder'][dataSetName] : {},
                            doNotExclude: true, // as of now, multi selection in map component is not supported
                            inputParams: customReportSvc
                                .applyInputsToDataSet($scope.dataSet,
                                    $scope.page.inputs)
                        },
                        function (response) {
                            onSuccessGetMapData(response);
                        });
            } else {
                //for CRE data sources to work, we need dimension and measure data fields in the scope
                //these need not be exported in XML; need to fix
                $scope.component.dimensionDataField = dimensionDataField;
                $scope.component.measureDataField = measureDataField;
                $scope.getData().then(function (resultData) {
                    onSuccessGetMapData(resultData);
                }, function (error) {
                    console.log(error);
                    $scope.component.isComponentLoading = false;
                });
            }
        }
        //local methods, section end

        //scope methods, section start        
        //should check if below three methods can move to a common place
        $scope.getBasicColumnDef = function (col) {
            var column = {
                column: col.name,
                dataField: col.name,
                displayName: col.name,
                id: $scope.generateColumnId(col.dataField),
                type: col.type,
                origType: col.hasOwnProperty("origType") ? col.origType : "",
                visible: true,
                hidden: false,
                aggrType: 'None',
                componentId: $scope.component.id,
                minCount: 1
            };
            if ($scope.reportMode === "preview" && $scope.component.columns.length > 9) {
                column.visible = false;
            }
            $scope.columnDefinitionById[column.id] = column;
            return column;
        };

        $scope.generateColumnId = function (dataField) {
            var id = dataField.replace(/\s/g, "");
            var orginalId = dataField.replace(/\s/g, "");
            var i = 1;
            while (id in $scope.columnDefinitionById) {
                id = orginalId + i;
                i++;
            }
            return id;
        };

        $scope.getColumnIndexByID = function (colId) {
            for (var i = 0; i < $scope.component.columns.length; i++) {
                if (colId.toLowerCase() == $scope.component.columns[i].id.toLowerCase()) {
                    return i;
                }
            }
        };
        //should check if above three methods can move to a common place

        $scope.dropped = function (dragEl, dropEl) {
            var source = document.getElementById(dragEl);
            if ($scope.reportMode == "viewer") {
                return;
            }

            if (!$scope.component.isSelected) {
                customReportSvc
                    .errorToast('Select the component to add a column.');
                return;
            }

            var dest = document.getElementById(dropEl);
            var src = document.getElementById(dragEl);
            var drag = angular.element(src);
            var drop = angular.element(dest);
            var dropType = drop.data('droptype');

            //TODO: check if this is needed
            //check is needed because rearranging columns also triggered drop
            if (drag.data("componenttype") != "COLUMN") {
                $scope.columnRearranged = true;
                return;
            }

            var columnName = drag.attr("data-name");
            var columnType = drag.attr("data-type");
            var origType = drag.attr("data-origtype");
            var dataField = drag.attr("data-datafield");
            var dataSetEntity = drag.data("datasetentity");
            var dataSetName = dataSetEntity.dataSetName;

            if (!$scope.component.dataSet ||
                !$scope.component.dataSet.dataSetName) {
                $scope.associateDataSetToComponent(dataSetEntity);
            } else if ($scope.component.dataSet.dataSetName != dataSetName) {
                alert("Mismatched data Sets");
                return;
            }

            var column = $scope.getBasicColumnDef({
                name: columnName,
                dataField: dataField,
                type: columnType,
                origType: origType
            });
            if (numericDataTypes.indexOf(columnType) > 0) {
                column.showNumberOps = true;
            }

            if (drop.data('droptype') == "latitude") {
                $scope.component.latitude = column;
            } else if (drop.data('droptype') == "longitude") {
                $scope.component.longitude = column;
            } else if (drop.data('droptype') == "cc") {
                $scope.component.countrycode = column;
            } else if (drop.data('droptype') == "category") {
                $scope.component.categoryColumn = column;
                if($scope.component.inputType === "cc") {
                    if ($scope.component.columns.length > 0 && $scope.component.columns[0].aggrType === 'None') {
                        //when category is applied, only faceting is allowed; so change from None to Count
                        $scope.component.columns[0].aggrType = 'Count';
                    }
                } else if($scope.component.enableHtmlClusters) {
                    //For html clusters no need to do faceting. Grouping will be taken care by cluster filters
                    if ($scope.component.columns.length > 0)  {
                        if($scope.component.columns[0].aggrType !== 'None')
                            $scope.component.columns[0].aggrType = 'None';
                    } else {
                        //if there is no data column then add category column as data column
                        $scope.component.columns = [column];
                    }
                         
                }
                
            } else {
                if ($scope.component.columns.length > 0) {
                    if ($scope.component.columns[0].aggrType === 'None') {
                        column.aggrType = 'None';
                    } else {
                        column.aggrType = 'Count';
                    }
                } else if($scope.component.enableHtmlClusters) {
                    column.aggrType = 'None';
                } else {
                    //by default let it be Count as None takes time to load
                    column.aggrType = 'Count';
                }
                $scope.component.columns.push(column);
            }

            $scope.loadComponent();
            $scope.$apply();
        };

        $scope.deleteColumn = function (index) {
            if (index === 'longitude') {
                $scope.component.longitude = undefined;
            } else if (index === 'latitude') {
                $scope.component.latitude = undefined;
            } else if (index === 'cc') {
                $scope.component.countrycode = undefined;
            } else if (index === 'category') {
                $scope.component.categoryColumn = undefined;
            } else {
                if ($scope.component.bubbleSizeBy === $scope.component.columns[index].id)
                    $scope.component.bubbleSizeBy = undefined;
                $scope.component.columns.splice(index, 1);
            }
            $scope.loadComponent();
        }

        $scope.setColumnAggregation = function(aggr) { 
            if (aggr !== 'None') {
                //set all other columns aggr type to Count if it is None
                if ($scope.component.columns.length > 1) {
                    for (var i = 1; i < $scope.component.columns.length; i++) {
                        if ($scope.component.columns[i].aggrType === 'None')
                            $scope.component.columns[i].aggrType = 'Count';
                    }
                }
                $scope.component.isAggrColPresent = true;
            } else {
                //set all other columns aggr type to None
                for (var i = 0; i < $scope.component.columns.length; i++) {
                    $scope.component.columns[i].aggrType = 'None';
                }
                $scope.component.isAggrColPresent = false;
                $scope.component.enableClustering = false;
                $scope.component.bubbleSizeBy = undefined;
            }
        }

        $scope.getMappingForOldMapProps = function () {            
            $scope.component.inputType = "latlon";
            $scope.component.tooltipExpForDataPoint = $scope.component.tooltipFormatter;
            delete $scope.component.tooltipFormatter;

            $scope.component.type = "MAP_v2";

            if(!$scope.component.columns)
                return;
            
            if($scope.component.columns.length > 0) {
                if($scope.component.columns[0].aggrType) 
                    $scope.setColumnAggregation($scope.component.columns[0].aggrType);

                angular.forEach($scope.component.columns, function(col) {
                    if(!col.displayName)
                        col.displayName = col.column;
                });
            }     
        }

        $scope.loadComponent = function () {
            if ($scope.map) {
                //remove the map so that the resources are released and recreated using initMap
                //without removing the map, page was getting sluggish
                $scope.map.remove();
            }
            if ($scope.component.inputType === 'cc' && !$scope.component.mapType) {
                $scope.component.mapType = 'bubble';
            }
            if ($scope.component.mapColor === undefined)
                $scope.component.mapColor = DEFAULT_COLOR;
            if ($scope.component.currentZoom === undefined)
                $scope.component.currentZoom = DEFAULT_ZOOM;
            if (!((($scope.component.inputType === 'latlon' && $scope.component.latitude && $scope.component.longitude) ||
                        ($scope.component.inputType === 'cc' && $scope.component.countrycode)) &&
                    ($scope.component.columns && $scope.component.columns.length > 0))) {
                $scope.component.isComponentLoading = false;
                return;
            }
            if(!$scope.component.mapStyle)
                $scope.component.mapStyle = 'mapbox://styles/mapbox/basic-v9?optimize=true';
            
            //add white color by default
            if(!$scope.component.innerCircleColor)
                $scope.component.innerCircleColor = DEFAULT_CIRCLE_COLOR;
            
            $scope.showNoDataDiv = false;
            $scope.component.isComponentLoading = true;

            $scope.loadingButton();
             $("#reportArea #cvmap_" + $scope.component.id).removeClass("ng-hide");           
            initMap();

            function afterLoadImage() {
                if ($scope.component.inputType === 'cc' && worldGeoJson === undefined) {
                    //get world country codes
                    $reportService.getCountriesGeoJson().then(function (resp) {
                        worldGeoJson = resp.data;
                        getMapData();
                    });
                } else
                    getMapData();
            }

            if (!$scope.map.hasImage(MARKER_ID)) {
               $scope.map.loadImage(cvUtil.getContextPath() + '/common/thirdParty/mapbox/v2/marker-pin.png',
                function (error, image) {
                    if (error) {
                        console.log("Error while loading marker image", error);
                    } else {
                        $scope.map.addImage(MARKER_ID, image, {sdf: true});
                    }
                    afterLoadImage();
                });                
            } else
                afterLoadImage();
        }

        $scope.reloadComponent = function () {
            $scope.loadComponent();
        };

        $scope.$on('resize', function (sizes, gridster) {
            if (gridster && gridster.length > 1) {
                var componentResizedScope = angular.element(gridster[1])
                    .scope();
                var component = componentResizedScope.$parent.component;
                var currentCompId = component.id;
                if ($scope.component.id == currentCompId) {
                    if ($scope.map) {
                        $timeout(function () {
                            $scope.map.resize();
                        }, 50);
                    }
                }
            }
        });
        $scope.$on('gridsterItemWidthChanged', function (event,
            isLeftPanelToggle, componentId) {
            if ($scope.map) {
                $timeout(function () {
                    $scope.map.resize();
                }, 50);
            }
        });
        //handle data set edit
        //TODO : not working; check
        $scope.$on('updateComponentFields', function (e, dataSet) {
            if ($scope.component.dataSet &&
                dataSet.dataSet.originalDataSetName === $scope.component.dataSet.dataSetName) {
                $scope.component.dataSet.dataSetName = dataSet.dataSet.dataSetName;
                $scope.dataSet = dataSet;
                $scope.loadComponent();
            }
        });
        //scope methods, section end

        //watch methods, section start
        $scope.$watch('component.inputType', function (newValue, oldValue) {
            if ($scope.component.isComponentLoading) {
                return;
            }
            $scope.component.isComponentLoading = true;
            if (newValue !== undefined && oldValue !== undefined && newValue !== oldValue) {
                $scope.component.latitude = undefined;
                $scope.component.longitude = undefined;
                $scope.component.countrycode = undefined;
                $scope.loadComponent();
            }
        });

        $scope.$watch('component.enableHtmlClusters', (newValue, oldValue) => {
            if (oldValue !== newValue && newValue !== undefined) {
                if(newValue === true) {
                    $scope.setColumnAggregation("None");
                    if (!$scope.component.isComponentLoading) {
                        $scope.loadComponent();
                    }
                }
                
            }
        });

        $scope.$watch('component.columns[0].aggrType', function (newValue,
            oldValue) {
            if (oldValue !== newValue && newValue !== undefined) {
                $scope.setColumnAggregation(newValue);
                if (!$scope.component.isComponentLoading) {
                    $scope.loadComponent();
                }
            }
        });
        $scope.$watch('component.enableClustering', function (newValue,
            oldValue) {
            if (oldValue !== newValue && newValue !== undefined) {
                if (!newValue) {;
                    $scope.component.bubbleSizeBy = undefined;
                }
                if (!$scope.component.isComponentLoading) {
                    $scope.loadComponent();
                }
            }
        });
        $scope.$watch('component.seriesFormatting', function (newValue, oldValue) {
            if ($scope.isComponentLoading || $scope.processing || newValue === undefined || oldValue === undefined || newValue === oldValue) {
                return;
            }
            if (!$.isEmptyObject($scope.component.seriesFormatting) && $scope.categoryMap && $scope.categoryMap.size > 0) {
                var cMap = new Map();
                $scope.categoryMap.forEach(function (obj, key) {
                    cMap.set(key, {
                        selected: false
                    });
                });
                angular.forEach($scope.component.seriesFormatting, function (
                    eachSeries) {
                    if (cMap.has(eachSeries.value)) {
                        cMap.set(eachSeries.value, {
                            selected: true
                        });
                    }
                });
                $scope.component.currDataSeries = [];
                cMap.forEach(function (obj, key) {
                    $scope.component.currDataSeries.push({
                        value: key,
                        selected: obj.selected
                    });
                });
            }
            loadMap(0);
        }, true);

        
        $scope.$watch('component.mapStyle', function (newValue,oldValue) {
            if (oldValue !== newValue && newValue !== undefined && $scope.map) {
               $scope.map.setStyle(newValue); 
               //wait for style to get loaded 
               $scope.loadMapInterval = $interval(function() {
                    if($scope.map.isStyleLoaded()) {
                        if($scope.loadMapInterval) {
                            $interval.cancel($scope.loadMapInterval);
                            $scope.loadMapInterval = undefined;
                        }
                        loadMap(0);  
                    }                    
                  
                },500);
            }
                
         });

        //watch methods, section end

        // init
        if($scope.component.type === "MAP" &&  $scope.component.mapType === 'mapbox' 
            && $scope.component.inputType === "latitudelongitude") {
            $scope.getMappingForOldMapProps();
        }
           

        $timeout(function () {
            $scope.loadComponent();
        });
    }
})();