/**
 * @author kharkut
 */
import { formsModule } from 'common/js/modules';
import 'modules/forms/js/services/forms.svc.js';
import * as workflowScheduleColumnsTemplate from './workflowSchedule.columns.template.js';

export class WorkflowSchedulesListingController {
	constructor(
		$scope,
		cvUtil,
		$stateParams,
		$location,
		cvLoc,
		formsModuleService,
		ENTITY_TYPES,
		cvPermissionFactory,
		$compile,
		cvToaster,
		securityFactory,
		scheduleUIFactory,
		$rootScope,
		cvBreadcrumbsTabsFactory,
		schedulePolicyService,
		schedulePatternFactory,
		schedulePolicyFactory,
		$dialogs,
		scheduleService
	) {
		cvBreadcrumbsTabsFactory.addBreadCrumbs([
			{
				title: cvLoc('label.nav.webconsole.forms'),
				link: '#workflows'
			}
		]);

		this.schedulePolicyFactory = schedulePolicyFactory;
		this.schedulePatternFactory = schedulePatternFactory;
		$scope.workflowId = $stateParams.workflowId;
		this.formsModuleService = formsModuleService;
		this.$scope = $scope;
		this.cvLoc = cvLoc;
		this.ENTITY_TYPES = ENTITY_TYPES;
		this.cvPermissionFactory = cvPermissionFactory;
		this.$compile = $compile;
		this.cvToaster = cvToaster;
		this._init();
		this.securityFactory = securityFactory;
		this.scheduleUIFactory = scheduleUIFactory;
		this.$rootScope = $rootScope;
		this.schedulePolicyService = schedulePolicyService;
		this.$dialogs = $dialogs;
		this.scheduleService = scheduleService;
		this.cvUtil = cvUtil;
		this.initializeEditScheduleListener();
	}

	getWorkflowSchedulesSpecificActions = () => {
		return {
			RUN_SCHEDULE: {
				label: 'Run',
				value: 'RUN_SCHEDULE'
			},
			DELETE_SCHEDULE: {
				label: 'Delete',
				value: 'DELETE_SCHEDULE'
			},
			VIEW_JOBS: {
				label: 'View jobs',
				value: 'VIEW_JOBS'
			}
		};
	};

	deleteSchedule = function(taskId) {
		let self = this;
		let $scope = this.$scope;
		let $dialogs = this.$dialogs;
		let cvLoc = this.cvLoc;

		$dialogs.confirm(cvLoc('label.confirmDelete'), 'Are you sure you want to delete this schedule?', {
			noFunction: function() {},
			yesFunction: function() {
				self.schedulePolicyFactory
					.deleteSchedulePolicy(taskId)
					.success(function() {
						self.grid.refreshData();
					})
					.error(function(err) {
						self.onErrorSave(err);
					});
			}
		});
	};

	runSchedule(entity) {
		let self = this;
		let cvLoc = self.cvLoc;
		var subTaskName = entity.subTasks[0].subTask.subTaskName,
			subTaskId = entity.subTasks[0].subTask.subTaskId,
			taskId = entity.task.task.taskId,
			taskName = '';

		this.schedulePolicyService
			.runSchedule(subTaskId, subTaskName, taskId, taskName)
			.success(function(jobIdList) {
				if (jobIdList) {
					var responseContent;
					if (jobIdList.length == 1) {
						var jobId = jobIdList[0];
						responseContent =
							cvLoc('label.jobStarted', jobId) +
							'  <a target="_blank" href="' +
							'#/jobs/' +
							jobId +
							'">' +
							cvLoc('notification.viewJobs') +
							'</a>';
					} else if (jobIdList.length > 1) {
						responseContent =
							cvLoc('label.jobsStarted') +
							'. ' +
							'<a target="_blank" href="#/activeJobs" >' +
							cvLoc('notification.viewJobs') +
							'</a>';
					}
					cvToaster.showSuccessMessage({
						ttl: '10000', //10 sec
						message: responseContent
					});
				}
			})
			.error(function(err) {
				self.onErrorSave(err);
			});
	}

	editSchedule(entity) {
		let self = this;
		let $scope = self.$scope;
		let cvUtil = this.cvUtil;
		let schedule = {
			taskId: entity.task.task.taskId,
			subtaskId: entity.subTasks[0].subTask.subTaskId
		};
		$scope.wfInfo = entity.associations[0];

		this.scheduleService.getScheduleDetails(schedule, true).success(function(data) {
			//For the database plan we have to allow deletion of the SLA schedules
			// Our time picker needs a javascript Date object here; create one:
			var time = 0;
			var repeatTime = 0;
			data.taskId = schedule.taskId;
			data.subtaskId = schedule.subtaskId;

			if (data.activeStartTime || data.activeStartTime == 0) {
				time = data.activeStartTime;
				data.scheduleTime = cvUtil.convertCvTimeToDate(time);
				data.dailyBackupHours = data.scheduleTime.getHours();
				data.dailyBackupMinutes = data.scheduleTime.getMinutes();
			}

			if (data.repeat.time || data.repeat.time == 0) {
				repeatTime = data.repeat.time;
				data.repeat.time = cvUtil.convertCvTimeToDate(repeatTime);
			}

			$scope.scheduleModel = data;

			let inputs = {};
			if (data.workflowJobOptions) {
				inputs = JSON.parse(data.workflowJobOptions);
			}

			openForm({
				workflowId: $scope.wfInfo.workflowId,
				doNotStartWorkflow: true,
				success: self.scheduleCallBack.bind(self),
				inputs: inputs.inputs,
				showAsModal: true
			});
		});
	}

	scheduleCallBack(inputs) {
		let self = this;
		let $scope = this.$scope;
		$scope.workflowInput = inputs;
		console.log(inputs);
		self.scheduleUIFactory.editWorkflowSchedule($scope.scheduleModel);
	}

	initializeEditScheduleListener() {
		let self = this;
		let $rootScope = self.$rootScope;
		let $scope = self.$scope;

		$rootScope.$on(
			'scheduleEdited',
			function(evt, addedSchedule, planId) {
				let self = this;
				let $scope = this.$scope;

				addedSchedule.repeat = {
					enabled: addedSchedule.repeatEnabled,
					duration: addedSchedule.repeatDuration,
					hrs: addedSchedule.repeatHrs,
					mins: addedSchedule.repeatMins,
					time: addedSchedule.repeatTime
				};

				self.formsModuleService
					.modifyWorkflowSchedule({
						schedule: JSON.stringify(addedSchedule),
						taskId: addedSchedule.taskId,
						subTaskId: addedSchedule.subtaskId,
						workflowInput: JSON.stringify($scope.workflowInput)
					})
					.success(function(data) {
						self.grid.refreshData();
						self.onSuccessSave('Schedule modified successfully');
					})
					.error(function(e) {
						let errMsg = 'Failed to modify schedule';
						if (e.data && e.data.errorMessage) {
							errMsg += ' [ ' + e.data.errorMessage + ']';
						}
						self.onErrorSave(errMsg);
					});
			}.bind(this)
		);
	}

	onErrorSave(err) {
		if (err) {
			this.cvToaster.showErrorMessage({
				message: err
			});
		}
	}

	onSuccessSave(msg) {
		this.grid.refreshData();
		this.cvToaster.showSuccessMessage({
			message: msg
		});
	}

	getSchedules(options) {
		let cvUtil = this.cvUtil;
		let PLAN_PERMISSIONS = this.PLAN_PERMISSIONS;
		let cvLoc = this.cvLoc;
		let ENTITY_TYPES = this.ENTITY_TYPES;
		let $filter = this.$filter;
		let self = this;
		let $scope = this.$scope;
		let _computePermittedActions = function(entity) {
			let arrayOfProfileSpecificActionsToOverride = [
				{
					RUN_SCHEDULE: {
						show: true,
						onClick: self.runSchedule.bind(self, entity)
					}
				},
				{
					DELETE_SCHEDULE: {
						show: true,
						onClick: self.deleteSchedule.bind(self, entity.task.taskId)
					}
				},
				{
					VIEW_JOBS: {
						show: true,
						href: '#jobs?schedulePolicyId=' + entity.task.taskId + '&activeJobs=0&view=showAdminJobs'
					}
				}
			];

			let profilePermittedList = [];
			profilePermittedList = self.cvPermissionFactory.updateAndGetCommonActionsList(
				arrayOfProfileSpecificActionsToOverride,
				false,
				self.getWorkflowSchedulesSpecificActions()
			);
			return profilePermittedList;
		};

		this.formsModuleService
			.getSchedules($scope.workflowId)
			.success(function(data) {
				if (data.taskDetail) {
					_.each(data.taskDetail, schedule => {
						schedule.permittedOptions = {
							//entityType: ENTITY_TYPES.SUBTASK_ENTITY,
							entity: angular.copy(schedule.subTasks[0]),
							entityId: schedule.subTasks[0].subTask.subTaskId,
							entityName: schedule.subTasks[0].subTask.subTaskName,
							appendToBody: true,
							permittedActionList: _computePermittedActions(schedule)
						};
					});
					options.success(data.taskDetail);
				} else {
					options.success([]);
				}
			})
			.error(function(e) {
				self.$scope.serverMessage = {
					message: e,
					type: 'error'
				};
				options.error(self.$scope.serverMessage);
			});
	}

	_initColumnDefs() {
		return workflowScheduleColumnsTemplate.getColumns({
			cvLoc: this.cvLoc,
			cvUtil: this.cvUtil,
			enableSchedule: this.enableSchedule.bind(this),
			schedulePatternFactory: this.schedulePatternFactory
		});
	}

	_onGridDataBound(dataItem, row) {
		let self = this;
		this.dataItem = dataItem;

		const permittedOptions = dataItem.permittedOptions;
		const id = permittedOptions.entityId;
		const template = `<cv-permitted-actions cv-permitted-options="permittedOptions${id}"></cv-permitted-actions>`;
		this.$scope[`permittedOptions${id}`] = permittedOptions;
		row.find('.permittedActions').append(this.$compile(template)(this.$scope));

		const scheduleLink = row.find('.k-grid-schedule-name');
		scheduleLink.click(function() {
			self.editSchedule(dataItem);
		});

		const statusInfo = row.find('.status-checkbox');
		const checked = !dataItem.task.taskFlags.disabled ? true : false;
		statusInfo.prop('checked', checked);
	}

	enableSchedule(e) {
		let self = this;
		e.preventDefault();
		// e.target is the DOM element representing the button
		let tr = $(e.target).closest('tr'); // get the current table row (tr)
		// get the data bound to the current table row
		let schedule = this.grid.grid.dataItem(tr);
		var isDisabled = _.get(schedule, 'task.taskFlags.disabled');
		self.schedulePolicyService
			.toggleStatus({
				taskId: _.get(schedule, 'task.task.taskId'),
				opType: isDisabled ? 'ENABLE' : 'DISABLE'
			})
			.success(function() {
				schedule.task.taskFlags.disabled = !schedule.task.taskFlags.disabled;
				const statusInfo = tr.find('.status-checkbox');
				const checked = !schedule.task.taskFlags.disabled ? true : false;
				statusInfo.prop('checked', checked);
			});
	}

	_init() {
		let $scope = this.$scope;
		this._setupGridOptions();
	}

	_setupGridOptions() {
		const self = this;
		this.gridOptions = {};
		this.$scope.gridOptions = this.gridOptions;
		this.gridOptions.url = this.getSchedules.bind(this);
		this.gridOptions.hasViews = false;
		this.gridOptions.columns = this._initColumnDefs();
		this.gridOptions.gridTitle = 'Schedules';
		this.gridOptions.idField = 'entity.task.taskId';
		this.gridOptions.beforeGridInitialize = ({ grid }) => {
			self.grid = grid;
		};
		this.gridOptions.afterGridInitialize = ({ grid }) => {
			self.grid = grid;
		};
		this.gridOptions.onGridDataBound = this._onGridDataBound.bind(this);
		this.gridOptions.actionMenu = [];
	}
}

WorkflowSchedulesListingController.$inject = [
	'$scope',
	'cvUtil',
	'$stateParams',
	'$location',
	'cvLoc',
	'formsModuleService',
	'ENTITY_TYPES',
	'cvPermissionFactory',
	'$compile',
	'cvToaster',
	'securityFactory',
	'scheduleUIFactory',
	'$rootScope',
	'cvBreadcrumbsTabsFactory',
	'schedulePolicyService',
	'schedulePatternFactory',
	'schedulePolicyFactory',
	'$dialogs',
	'scheduleService'
];

formsModule.controller('WorkflowSchedulesListingController', WorkflowSchedulesListingController);
