import { ediscoveryAppModule } from 'common/js/modules';
import {
	getTagsetGridColumns,
	getTagsetSubgridColumns
} from 'modules/tags/js/columnTemplates/tagset.column.template.js';
import 'modules/tags/js/services/tagManager.svc.js';
import 'modules/tags/js/controllers/addTagset.ctrl.js';
import { DM2EntityType } from 'modules/tags/js/tags.constants.js';
import 'modules/tags/js/controllers/addTag.ctrl.js';
import TagManagerUtility from 'modules/tags/js/services/tagManager.util.js';
import { Url } from 'ediscovery/common/js/constants';

class TagManagerController {
	constructor(
		$state,
		$uibModal,
		cvLoc,
		cvToaster,
		$dialogs,
		tagManagerService,
		tabService,
		cvBreadcrumbsTabsFactory,
		settingsService,
		PERMISSIONS
	) {
		this.$state = $state;
		this.$uibModal = $uibModal;
		this.cvLoc = cvLoc;
		this.cvToaster = cvToaster;
		this.$dialogs = $dialogs;
		this.tagManagerService = tagManagerService;
		this.tabService = tabService;
		this.cvBreadcrumbsTabsFactory = cvBreadcrumbsTabsFactory;
		this.settingsService = settingsService;
		this.PERMISSIONS = PERMISSIONS;

		this.init();
	}

	async init() {
		this.initState();
		await this.initPermissions();
		this.setupTagsetGrid();
	}

	initState() {
		if (this.$state.is('activateTagManager')) {
			this.isActivateState = true;
			this.cvBreadcrumbsTabsFactory.addBreadCrumbs([
				{
					title: this.cvLoc('label.ediscovery'),
					link: Url.Activate
				}
			]);

			this.tabService.init({
				tabs: [
					{
						title: this.cvLoc('label.entitymanager.entities'),
						state: 'entityManager'
					},
					{
						title: this.cvLoc('label.classifiermanager.classifiers'),
						state: 'classifierManager'
					},
					{
						title: this.cvLoc('label.tags'),
						state: 'activateTagManager'
					}
				]
			});
		}
	}

	async initPermissions() {
		await this.settingsService.getUserHasCapability(this.PERMISSIONS.TAG_MANAGEMENT).then(resp => {
			const { data } = resp;
			this.tagManagementPermission = !!data;
		});
	}

	setupTagsetGrid() {
		this.tagsetGridOptions = {};
		this.tagsetGridOptions.gridTitle = this.cvLoc('label.tagManager');
		this.tagsetGridOptions.tableName = 'tagsetTable';
		this.tagsetGridOptions.pageSize = 20;
		this.tagsetGridOptions.hasViews = true;
		this.tagsetGridOptions.showSearchOnFewItems = true;
		this.tagsetGridOptions.gridEmptyMessage = this.cvLoc('info.noTags');
		this.tagsetGridOptions.gridEmptySvg = `<svg class="add-new-placeholder" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 50 50"><path d="M37.6 37.5v10.8H1.7V1.7h24.2v11.6h11.7V19a8.5 8.5 0 0 1 1.6-.3v-6.5L27.1 0H0v50h39.2V37.9a8.5 8.5 0 0 1-1.6-.4zM27.6 3l8.8 8.8h-8.8z"/><circle cx="17.4" cy="10.7" r="3.2"/><circle cx="20" cy="27.7" r="3.2"/><circle cx="15.1" cy="40.3" r="3.2"/><path d="M12.5 11.5h-8a.8.8 0 0 1 0-1.6h8a.8.8 0 1 1 0 1.6zM9 20.3H4.7a.8.8 0 0 1 0-1.6H9a.8.8 0 1 1 0 1.6zM21.3 20.3h-4a.8.8 0 1 1 0-1.6h4a.8.8 0 0 1 0 1.6zM14.4 28.5H8.8a.8.8 0 0 1 0-1.6h4.6a.8.8 0 0 1 0 1.6zM19 35.1H4.7a.8.8 0 0 1 0-1.6H19a.8.8 0 0 1 0 1.6zM10 41.8H8.3a.8.8 0 0 1 0-1.6H10a.8.8 0 1 1 0 1.6zM27.5 41.8h-6a.8.8 0 1 1 0-1.6h6a.8.8 0 0 1 0 1.6z"/><circle cx="13.1" cy="19.5" r=".8"/><circle cx="4.8" cy="41" r=".8"/><circle cx="22.1" cy="34.3" r=".8"/><circle cx="4.8" cy="27.7" r=".8"/><path d="M49.6 21l-6.8-2.5h-.3L35.6 21a.4.4 0 0 0-.3.4v4.8c0 6.3 1 8.6 7.1 11.7h.4C49 34.7 50 32.5 50 26.2v-4.9a.4.4 0 0 0-.4-.3zM42 23.5h.1a2 2 0 0 1 .4 0 2 2 0 0 1 2 2 2 2 0 0 1-1.6 2 1.8 1.8 0 0 1-.4 0 2 2 0 0 1-.5-4zm4.2 7a1 1 0 0 1-1 1H40a1 1 0 0 1-.9-1v-.2a1.8 1.8 0 0 1 .9-1.6 5 5 0 0 1 5.5 0 1.8 1.8 0 0 1 .9 1.6zM19 39.1l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.4.4.8zm1.6-.8L25 35l.8-.5.4.8zm1.7-.9l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.4.4.8zm1.6-.8l-.4-.8.8-.5.4.8zm1.6-.9L33 31l.8-.4.4.8zM35 31l-.4-.8h.1l.4.7zM34.2 28.3h-.7v-1h.7zm-1.6 0h-.9v-1h.9zm-1.8 0h-.9v-1h.9zm-1.8 0h-.9v-1h.9zm-1.8 0h-.9v-1h.9zm-1.8 0h-.9v-1h.9zM34 26.5l.5-.7zm-.8-.6l-.6-.6.6-.7.7.6zM32 24.6l-.6-.6.6-.7.7.7zm-1.3-1.2l-.6-.6.6-.7.7.6zm-1.3-1.2l-.6-.7.6-.6.6.6zM28 20.9l-.7-.6.7-.6.6.6zm-1.3-1.2l-.7-.6.6-.7.7.7zm-1.3-1.2l-.7-.7.6-.6.7.6zM24 17.2l-.7-.6.6-.6.7.6zM22.8 16l-.7-.6.6-.7.7.7zm-1.3-1.2l-.7-.7.6-.6.7.6z"/></svg>`;
		if (this.tagManagementPermission) {
			this.tagsetGridOptions.gridEmptyMenu = [
				{
					label: this.cvLoc('label.tagset.add'),
					onSelect: this.onAddTagset.bind(this)
				}
			];
		}
		this.tagsetGridOptions.columns = getTagsetGridColumns(this.cvLoc, this.isActivateState);
		this.tagsetGridOptions.url = this.getTagsets.bind(this);
		this.tagsetGridOptions.detailInit = this.setupTagsetSubgrid.bind(this);
		this.tagsetGridOptions.beforeGridInitialize = ({ grid }) => {
			this.tagsetGrid = grid;
		};
		this.tagsetGridOptions.onGridDataBound = (e, row) => {
			if (_.isUndefined(e.tags) || _.get(e, 'tags.length') === 0) {
				// if there are no tags, hide the expand marker for this row
				row.find('.k-hierarchy-cell').html('');
			}
		};

		// Toolbar menu
		this.tagsetGridOptions.gridToolbarMenu = [
			{
				id: 'ADD_TAGSET',
				label: this.cvLoc('label.tagset.add'),
				onSelect: () => {
					this.onAddTagset();
				},
				disableOnDeselect: false,
				hidden: !this.tagManagementPermission
			}
		];

		// Action menu
		this.tagsetGridOptions.actionMenu = [
			{
				id: 'EDIT_TAGSET',
				label: this.cvLoc('label.edit'),
				contextMenuOnly: true,
				onSelect: ({ selectedRowValues }) => {
					if (selectedRowValues.length === 1) {
						this.onEditTagset(selectedRowValues[0]);
					}
				},
				hidden: !this.tagManagementPermission
			},
			{
				id: 'DELETE_TAGSET',
				label: this.cvLoc('label.delete'),
				contextMenuOnly: true,
				onSelect: ({ selectedRowValues }) => {
					if (selectedRowValues.length === 1) {
						this.onDeleteTagset(selectedRowValues[0]);
					}
				},
				hidden: !this.tagManagementPermission
			},
			{
				id: 'ADD_TAG',
				label: this.cvLoc('label.tag.add'),
				contextMenuOnly: true,
				onSelect: ({ selectedRowValues }) => {
					if (selectedRowValues.length === 1) {
						this.onAddTag(selectedRowValues[0]);
					}
				},
				hidden: !this.tagManagementPermission
			}
		];
		this.tagsetGridOptions.onActionMenuOpen = this.actionMenuOpen;
	}

	actionMenuOpen(event) {
		['EDIT_TAGSET', 'DELETE_TAGSET', 'ADD_TAG'].forEach(option => {
			if (!_.get(event, 'selectedRowValues[0].allowEdit')) {
				event.grid.disableActionMenuOption(option);
			} else if (_.get(event, 'selectedRowValues[0].isSystem') && option !== 'ADD_TAG') {
				event.grid.disableActionMenuOption(option);
			} else if (_.get(event, 'selectedRowValues[0].isMine') == 'false' && option === 'DELETE_TAGSET') {
				event.grid.disableActionMenuOption(option);
			} else {
				event.grid.enableActionMenuOption(option);
			}
		});
	}

	setupTagsetSubgrid(row) {
		if (row.data.tags.length === 0) {
			return;
		}
		const tagsetSubgridOptions = {};
		tagsetSubgridOptions.columns = getTagsetSubgridColumns(this.cvLoc);
		// Tagset grid has a hidden csv of tags to help search both tagset and tags at the same time. In case there's a search entry, filter out the tags that don't match
		tagsetSubgridOptions.dataSource = {
			data: row.data.tags,
			pageSize: 10,
			filter: { field: 'name', operator: 'contains', value: this.tagsetGrid.gridDataSrc.currentSearch }
		};
		tagsetSubgridOptions.searchable = false;
		tagsetSubgridOptions.hasViews = false;
		tagsetSubgridOptions.hideToolbar = true;
		tagsetSubgridOptions.scrollable = false;
		tagsetSubgridOptions.sortable = true;
		tagsetSubgridOptions.pageable = true;
		angular
			.element('<div/>')
			.appendTo(row.detailCell)
			.kendoGrid(tagsetSubgridOptions);
	}

	getTagsets(options) {
		this.tagManagerService
			.getTags()
			.then(
				resp => {
					const tagsets = TagManagerUtility.formatTagsetResponse(resp.data);
					options.success(tagsets);
				},
				err => {
					options.success([]);
				}
			)
			.catch(err => {
				console.error(err);
			});
	}

	onAddTagset() {
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTagset.jsp`,
			controller: 'addTagsetController',
			controllerAs: 'addTagsetCtrl',
			backdrop: 'static',
			windowClass: 'small-size',
			resolve: {
				isEdit() {
					return false;
				},
				tagset() {
					return undefined;
				}
			}
		});

		modalInstance.result.then(res => {
			this.tagsetGrid.refreshData();
		});
	}

	onEditTagset(row) {
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTagset.jsp`,
			controller: 'addTagsetController',
			controllerAs: 'addTagsetCtrl',
			backdrop: 'static',
			windowClass: 'small-size',
			resolve: {
				isEdit() {
					return true;
				},
				tagset() {
					return row;
				}
			}
		});

		modalInstance.result.then(res => {
			this.tagsetGrid.refreshData();
		});
	}

	onAddTag(row) {
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTag.jsp`,
			controller: 'addTagController',
			controllerAs: 'addTagCtrl',
			resolve: {
				isEdit() {
					return false;
				},
				tagset() {
					return row;
				},
				tag() {
					return {};
				}
			},
			backdrop: 'static',
			windowClass: 'small-size'
		});

		modalInstance.result.then(res => {
			this.tagsetGrid.refreshData();
		});
	}

	onDeleteTagset(row) {
		const self = this;
		const callBackFunctions = {
			noFunction() {},
			yesFunction() {
				self.deleteTagset(row);
			}
		};
		this.$dialogs.confirm(
			this.cvLoc('label.tagset.delete'),
			this.cvLoc('info.tagset.delete', row.name),
			callBackFunctions
		);
	}

	deleteTagset(row) {
		const containers = [
			{
				containerType: DM2EntityType.Tags,
				containerId: row.containerId
			}
		];
		const containerDeleteRequest = {
			entityType: DM2EntityType.Tags,
			containers
		};
		this.tagManagerService
			.deleteTagset(containerDeleteRequest)
			.then(
				resp => {
					if (!_.isUndefined(_.get(resp, 'data.errList[0].errLogMessage'))) {
						this.showErrorNotification(row.name, resp.data.errList[0].errLogMessage);
					} else {
						this.cvToaster.showSuccessMessage({
							ttl: '5000',
							message: this.cvLoc('info.tagset.deleted', row.name)
						});
						this.tagsetGrid.refreshData();
					}
				},
				err => {
					this.showErrorNotification(row.name, '');
				}
			)
			.catch(err => {
				console.error(err);
				this.showErrorNotification(row.name, '');
			});
	}

	showErrorNotification(tagsetName, errorMessage) {
		this.cvToaster.showErrorMessage({
			ttl: '10000',
			message: this.cvLoc('error.tagset.delete', tagsetName, errorMessage)
		});
	}
}

TagManagerController.$inject = [
	'$state',
	'$uibModal',
	'cvLoc',
	'cvToaster',
	'$dialogs',
	'tagManagerService',
	'tabService',
	'cvBreadcrumbsTabsFactory',
	'settingsService',
	'PERMISSIONS'
];

ediscoveryAppModule.controller('tagManagerController', TagManagerController);
export default ediscoveryAppModule;
