import { commonSetupModule } from 'common/js/modules';

/**
 * A service for interacting with the different Setup related API calls.
 */
commonSetupModule.service('setupService', [
	'$http',
	function($http) {
		/**
		 * Checks to see if there is at least 1 gateway configured in the CS.
		 *
		 * @return {HttpPromise}
		 */
		this.isGatewayConfigured = function() {
			return $http.get('isGatewayConfigured.do');
		};

		/**
		 * Mark a setup, given the setup's name, as complete.
		 *
		 * @return {HttpPromise}
		 */
		this.markSetupAsComplete = function(appName) {
			return $http.post('markSetupAsComplete.do', {
				appName: appName
			});
		};

		/**
		 * Mark a setup, given the setup's name, as incomplete.
		 *
		 * @return {HttpPromise}
		 */
		this.markSetupAsIncomplete = function(appName) {
			return $http.post('markSetupAsIncomplete.do', {
				appName: appName
			});
		};

		/**
		 * Referesh the user's session context so that if the user refreshes the page, the session context
		 * will be at least updated.
		 *
		 * @return {HttpPromise}
		 */
		this.refreshSession = function() {
			return $http.get('refreshSession.do', {
				ignoreLoadingBar: true
			});
		};

		/**
		 * Get all of the setup steps for all apps.
		 *
		 * @return {HttpPromise}
		 */
		this.getAllSetupStepsForCurrentUser = function(app) {
			return $http.get('getAllSteps.do');
		};

		/**
		 * Get the setup steps for a given app.
		 *
		 * @param {string}
		 *            The app to get the setup steps for.
		 * @return {HttpPromise}
		 */
		this.getSetupStepsForApp = function(app) {
			return $http.get('getStepsForApp.do?appName=' + app);
		};

		/**
		 * Get the Core setup steps completed for the current user.
		 *
		 * @return {HttpPromise}
		 */
		this.getCoreStepForCurrentUser = function() {
			return $http.get('getCoreStep.do');
		};

		/**
		 * Get the VSA steps completed for the current user.
		 *
		 * @return {HttpPromise}
		 */
		this.getVsaStepForCurrentUser = function() {
			return $http.get('getVsaStep.do');
		};

		/**
		 * Get the Commvault One steps completed for the current user.
		 *
		 * @return {HttpPromise}
		 */
		this.getCVOneStepForCurrentUser = function() {
			return $http.get('getCVOneStep.do');
		};

		/**
		 * Get the DLO steps completed for the current user.
		 *
		 * @return {HttpPromise}
		 */
		this.getDloStepForCurrentUser = function() {
			return $http.get('getDloStep.do');
		};

		/**
		 * Get the DBS steps completed for the current user.
		 *
		 * @return {HttpPromise}
		 */
		this.getDbsStepForCurrentUser = function() {
			return $http.get('getDbsStep.do');
		};
	}
]);

export default commonSetupModule;
