(function() {
	'use strict';

	/**
	 * A factory to assist with anything related to setups.
	 */
	angular.module('common.setup').factory('setupFactory', [
		'setupService',
		'$log',
		function(setupService, $log) {
			var factory = {};

			/**
			 * Tests to make sure that every setup given is complete.
			 *
			 * @param {array|string}
			 *            setups - If an array is given, it will check every setup in the array for
			 *            completion. If a string is given, it will only check that single setup for
			 *            completion.
			 * @return {boolean} True if all given setups are complete; False otherwise.
			 */
			factory.areSetupsComplete = function(setups) {
				if (angular.isArray(setups)) {
					return setups.every(function(setup) {
						return factory.isSetupComplete(setup);
					});
				} else if (angular.isString(setups)) {
					return factory.isSetupComplete(setups);
				}
				return false;
			};

			/**
			 * Returns back whether or not at least 1 setup in the given argument is complete.
			 *
			 * @param {array|string}
			 *            setups - If an array is given, it will check every setup in the array to see if
			 *            at least 1 is complete. If a string is given, it will only check that single
			 *            setup for completion.
			 * @return {boolean} True if at least 1 given setup is complete; False otherwise.
			 */
			factory.isAtLeastOneSetupComplete = function(setups) {
				if (angular.isArray(setups)) {
					for (var i in setups) {
						if (factory.isSetupComplete(setups[i])) {
							return true;
						}
					}
					return false;
				} else if (angular.isString(setups)) {
					return factory.isSetupComplete(setups);
				}
				return false;
			};

			/**
			 * Returns back whether or not the name of the given setup is complete.
			 *
			 * @return {boolean} True if the setup exists and all steps in the setup are completed; False
			 *         otherwise.
			 */
			factory.isSetupComplete = function(setup) {
				if (angular.isString(setup)) {
					setup = setup.trim();
					return (
						angular.isDefined(cv.sessionContext.completedSetups[setup]) && cv.sessionContext.completedSetups[setup]
					);
				}
				return false;
			};

			factory.markSetupComplete = function(setup) {
				if (_.isString(setup)) {
					setupService.markSetupAsComplete(setup).then(
						function(successData) {
							cv.sessionContext.completedSetups[setup] = true;
						},
						function(err) {
							$log.log('Cannot Mark setup' + setup);
						}
					);
				}
			};

			factory.checkAndUpdateCompletedSetups = function(entityCount, solutionName) {
				if (
					_.isNumber(entityCount) &&
					entityCount > 0 &&
					_.isString(solutionName) &&
					!factory.isSetupComplete(solutionName)
				) {
					factory.markSetupComplete(solutionName);
				}
			};

			return factory;
		}
	]);
})();
