import 'storage/js/services/storage.svc.js';
import 'vsa/js/constants/servers.constants.js';
import { cvModuleServerModule } from 'common/js/modules';

var serverMod = cvModuleServerModule;
serverMod.service('serverService', [
	'$http',
	'$q',
	'storageService',
	'$cacheFactory',
	'cvUtil',
	'vendorTypes',
	'VSA_VENDOR',
	'ENTITY_TYPES',
	'VMBrowseReqType',
	function($http, $q, storageService, $cacheFactory, util, vendorTypes, VSA_VENDOR, ENTITY_TYPES, VMBrowseReqType) {
		this.getServers = function(appId) {
			return $http.get('servers.do', {
				params: {
					instanceName: appId ? appId : ''
				}
			});
		};

		this.getVibUrl = function() {
			return $http.get('vibUrl.do');
		};

		this.isAppSupportedVendor = vendorName => {
			const appSupportedVendors = [VSA_VENDOR.KUBERNETES];
			return appSupportedVendors.includes(vendorName);
		};

		this.serversFilter = (serversList, supportedVendorList) => {
			if (
				_.isArray(serversList) &&
				_.isArray(supportedVendorList) &&
				serversList.length &&
				supportedVendorList.length
			) {
				serversList.splice(
					0,
					serversList.length,
					...serversList.filter(hypervisor => supportedVendorList.includes(hypervisor.type))
				);
			}
			return serversList;
		};

		this.getServersForRestoreWithClientId = function(clientId) {
			return $http.get('getServersForRestore.do', {
				params: {
					clientId: clientId
				}
			});
		};
		this.refreshCache = function(serverId, subclientId) {
			return $http.post('refreshCache.do', {
				serverId: serverId,
				subclientId: subclientId
			});
		};
		this.getServersByVendor = function(vendor) {
			return $http.get('servers.do', {
				params: {
					vendor: vendor
				}
			});
		};
		this.installVAIO = function(serverId, serverName, instanceId, instanceName, rows) {
			return $http.post('installVAIO.do', {
				clientId: serverId,
				clientName: serverName,
				instanceId: instanceId,
				instanceName: instanceName,
				clusters: this.getClusterInfo(serverId, serverName, instanceId, instanceName, rows),
				vibUrl: rows.vibUrl
			});
		};
		this.getClusterInfo = function(serverId, serverName, instanceId, instanceName, rows) {
			var clusters = [];
			for (var i = 0; i < rows.length; i++) {
				var blrClusterInfo = {};
				blrClusterInfo.vClientId = serverId;
				blrClusterInfo.instanceId = instanceId;
				blrClusterInfo.name = rows[i].entity.name;
				blrClusterInfo.moRef = rows[i].entity.strGUID;
				clusters.push(blrClusterInfo);
			}
			return JSON.stringify(clusters);
		};
		this.uninstallVAIO = function(serverId, serverName, instanceId, instanceName, rows, detachVMs) {
			return $http.post('uninstallVAIO.do', {
				clientId: serverId,
				clientName: serverName,
				instanceId: instanceId,
				instanceName: instanceName,
				clusters: this.getClusterInfo(serverId, serverName, instanceId, instanceName, rows),
				detachVMs: detachVMs
			});
		};
		// do not use this, it will be removed. use getServers() instead
		this.getServersForRestore = function() {
			return $http.get('servers.do');
		};

		this.createServer = function(model) {
			// TODO clean this code to pass the array of objects directly
			// create a proxy map from json object
			var proxy = '';
			for (var i = 0; i < model.selectedProxy.length; i++) {
				var proxyStr = model.selectedProxy[i].clientId + ':' + model.selectedProxy[i].clientName;
				proxy = proxy.length == 0 ? proxyStr : proxy + ',' + proxyStr;
			}

			var password = model.vsPassword;
			if (model.vendorType === 'AZURE') {
				password = model.thumbprint;
			} else if (model.vendorType === 'AZURE_V2' || model.vendorType === 'AZURE_STACK') {
				password = model.applicationPassword;
				//For ADFS in Azure stack we take thumbprint
				if (model.vendorType === 'AZURE_STACK' && model.thumbprint) {
					password = model.thumbprint;
				}
			}
			var serverObj = {
				vendorType: model.vendorType,
				serverName: model.serverName,
				hostname: model.hostname,
				vsUserName: model.vsUserName,
				vsPassword: password,
				accessKey: model.accessKey,
				secretKey: model.secretKey,
				useIamRole: model.useIamRole,
				subscriptionId: model.subscriptionId,
				tenantId: model.tenantId,
				vOrgName: model.organization == undefined ? '' : model.organization.vOrgName,
				vOrgGuid: model.organization == undefined ? '' : model.organization.vOrgGuid,
				company: model.company,
				companyId: model.companyId,
				companyMode: model.companyMode,
				applicationId: model.applicationId,
				proxy: proxy,
				nodeList: model.nodeStr == undefined ? '' : model.nodeStr.toString(),
				domainName: model.domainName,
				userId: model.userId,
				fingerPrint: model.fingerPrint,
				regionName: model.regionName,
				appTypes: model.appTypes,
				resourceMgrUrl: encodeURIComponent(model.resourceMgrUrl),
				glusterHostCredentials: model.glusterHostCredentials,
				adminInstanceId: model.adminInstanceId,
				serviceToken: model.serviceToken,
				credEntity: model.credEntity,
				secretType: model.secretType
			};

			if (model.isSqlCloud) {
				(serverObj.azureStorageConnectionString = model.azureStorageConnectionString),
					(serverObj.s3BucketName = model.s3BucketName),
					(serverObj.saUserName = model.SAUser ? model.SAUser.userName : ''),
					(serverObj.saPassword = model.SAUser ? model.SAUser.password : '');
			}

			if (cvConfig.enableMetallicCustomizationForK8s && model.orgNamePrefix) {
				serverObj.serverName = model.orgNamePrefix + model.serverName;
			}

			return $http.post('createServer.do', { hypvInfo: JSON.stringify(serverObj) });
		};

		this.deployProxy = function(configurationInfo) {
			let deployObject = {
				clusterEndpoint: configurationInfo.endpointURL,
				serverGroup: { clientGroupName: _.get(configurationInfo, 'clientGroup[0].name') },
				nameSpace: 'cvautoscale',
				serviceToken: 'bearer ' + configurationInfo.credentialsToken.trim(),
				authcode: configurationInfo.authCode,
				clientInfo: configurationInfo.clientEntity,
				subscriberId: configurationInfo.subID.toString(),
				accessNode: { clientName: _.get(configurationInfo, 'clientName') },
				cvImageTag: configurationInfo.imageTag,
				gatewayProxy: _.get(configurationInfo, 'gatewayProxy.length') > 0 ? configurationInfo.gatewayProxy[0] : ''
			};
			return $http({
				method: 'POST',
				url: cv.contextPath + '/proxy/kubernetes/accessnode/deploy',
				data: JSON.stringify(deployObject),
				headers: {
					'Content-Type': 'application/json',
					Accept: 'application/json'
				}
			});
		};

		this.refreshCollection = collectionId => {
			return $http.post('refreshCollection.do', {
				collectionId: collectionId
			});
		};

		this.updateCollectionTimeZone = function(collectionId, timeZone) {
			let tz = {
				TimeZoneID: 0
			};
			if (timeZone[0].tzId !== -1) {
				tz = {
					TimeZoneID: timeZone[0].tzId,
					TimeZoneName: timeZone[0].timezoneStdName
				};
			}
			const UpdateSubClientPropertiesRequest = {
				subClientProperties: {
					commonProperties: {
						TimeZone: tz
					}
				}
			};
			return $http({
				method: 'POST',
				url: cv.contextPath + '/proxy/Subclient/' + collectionId,
				data: JSON.stringify(UpdateSubClientPropertiesRequest),
				headers: {
					'Content-Type': 'application/json',
					Accept: 'application/json'
				}
			});
		};

		this.updateClientTimeZone = function(clientId, timeZone) {
			let SetClientPropertiesRequest = null;
			if (timeZone[0].tzId === -1) {
				SetClientPropertiesRequest = {
					clientProperties: {
						client: {
							timezoneSetByUser: false
						}
					}
				};
			} else {
				SetClientPropertiesRequest = {
					clientProperties: {
						client: {
							TimeZone: {
								TimeZoneID: timeZone[0].tzId,
								TimeZoneName: timeZone[0].timezoneStdName
							},
							timezoneSetByUser: true
						}
					}
				};
			}
			return $http({
				method: 'POST',
				url: cv.contextPath + '/proxy/Client/' + clientId,
				data: JSON.stringify(SetClientPropertiesRequest),
				headers: {
					'Content-Type': 'application/json',
					Accept: 'application/json'
				}
			});
		};

		//To get CS Hostname and client details
		this.getClientDetails = function() {
			return $http.get('clientEntityDetails.do');
		};

		this.pushInstallVsa = function(hostInfo) {
			return $http.post('pushInstallSoftware.do', {
				clientId: hostInfo.clientId,
				hostname: hostInfo.hostname,
				vsUserName: hostInfo.vsUserName,
				vsPassword: hostInfo.vsPassword,
				softwareInstallCode: 'VSA'
			});
		};

		this.editServerVMWare = function(hostInfo) {
			return $http.post('editServerVmWare.do', {
				vsUserName: hostInfo.vsUserName,
				vsPassword: hostInfo.vsPassword,
				vsCurrPassword: hostInfo.vsCurrPassword,
				serverId: hostInfo.serverId,
				oldHostName: hostInfo.oldHostName,
				newHostName: hostInfo.newHostName,
				instanceId: hostInfo.instanceId
			});
		};

		this.editServer = function(appProperties, password, proxyList) {
			return $http.post('editServer.do', {
				appProperties: appProperties,
				password: password,
				proxyList: proxyList
			});
		};

		this.getServerDetails = function(serverId) {
			return $http.get('serverDetails.do', {
				params: {
					serverId: serverId
				}
			});
		};

		this.resubmitPushInstallJob = function(proxyCreds) {
			return $http.post('resubmitPushInstallJob.do', {
				serverId: proxyCreds.serverId,
				vsUserName: proxyCreds.vsUserName,
				vsPassword: proxyCreds.vsPassword,
				node: proxyCreds.node
			});
		};

		this.getRemoteCache = function() {
			return $http.get('api/CommServ/AddRemoveSoftware/RemoteSoftwareCache');
		};

		this.changeServerName = function(serverId, serverName) {
			return $http.post('changeServerName.do', {
				id: serverId,
				name: serverName
			});
		};

		this.getServerVendor = function(serverId) {
			/*
			 * !!! getServerDetails does NOT return the vendor type, so instead we get the server list and
			 * extract the one we want. Consider making a dedicated java-side lookup for this later, in case
			 * the list of servers is very large.
			 */
			var deferred = $q.defer();

			$http
				.get('servers.do')
				.success(function(data) {
					for (var i = 0; i < data.length; i++) {
						if (data[i].clientId == serverId) {
							deferred.resolve(data[i].type);
							return;
						}
					}
					deferred.reject('Failed to retrieve vendor type - server not found for serverId=' + serverId);
				})
				.error(function(e) {
					deferred.reject('Failed to retrieve vendor type for serverId=' + serverId);
				});

			return deferred.promise;
		};

		this.getCompanyList = function() {
			return $http.get('organizations.do');
		};

		this.getSubclientList = function(serverId) {
			return $http.get('subclients.do', {
				params: {
					appId: 106,
					serverId: serverId
				},
				headers: jsonHeader
			});
		};

		this.getCollectionDetails = function(collectionId, hardRefresh) {
			if (!hardRefresh) {
				hardRefresh = false;
			}

			return $http.get('collectionDetails.do', {
				params: {
					collectionId: collectionId,
					hardRefresh: hardRefresh
				}
			});
		};

		this.deleteCollection = function(collectionId) {
			return $http['delete']('collection.do', {
				params: {
					collectionId: collectionId
				}
			});
		};

		this.getVms = function(serverId) {
			return $http.get('vms.do', {
				params: {
					serverId: serverId
				}
			});
		};

		this.getDataStores = function(serverId, hostname, serverDetails, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 0;

			return $http.post('dataStores.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getNetworkInfo = function(serverDetails, hostname, vendor, projectId) {
			var vmDestination = {};
			vmDestination.serverId = serverDetails.clientId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 1;
			vmDestination.projectId = projectId;

			return $http.post('networkInfo.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getAMISelections = (serverDetails, availabilityZone) => {
			const vmDestination = {
				serverId: serverDetails.clientId,
				hostname: availabilityZone,
				requestor: 11
			};

			return $http.post('amiSelections.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getSecurityGroups = function(serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverDetails.clientId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 4;
			return $http.post('securityGroups.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getAzureStorageOptions = function(serverId, hostname, serverName, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.vendor = vendor;
			vmDestination.destinationServerName = serverName;
			vmDestination.requestor = 6;

			return $http.post('getAzureStorageOption.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getOCIImages = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 2;

			return $http.post('getOCIImages.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getVolumeTypes = function(serverId, hostname, browseReqType) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.browseReqType = browseReqType;
			vmDestination.destinationServerName = '';
			vmDestination.serverUser = '';
			vmDestination.serverPassword = '';
			vmDestination.vendor = '';
			vmDestination.requestor = 2;
			return $http.post('volumeTypes.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getEncryptionKeys = function(serverId, hostname, browseReqType) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.browseReqType = browseReqType;
			vmDestination.destinationServerName = '';
			vmDestination.serverUser = '';
			vmDestination.serverPassword = '';
			vmDestination.vendor = '';
			vmDestination.requestor = 2;
			return $http.post('encryptionKeys.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getInstanceTypes = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 2;

			return $http.post('instanceTypes.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getKeyPairs = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 2;

			return $http.post('keypairs.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getSSHKeys = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 10;

			return $http.post('sshKeys.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};
		this.getStagingBuckets = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 2;

			return $http.post('stagingBuckets.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getProtectionDomains = function(serverDetails) {
			return $http.post('protectionDomains.do', {
				serverId: serverDetails.clientId,
				inventoryPath: '\\NONE:Protection Domains'
			});
		};

		this.getVMDestinations = function(serverId, serverDetails, browseReqType, hostName, vsMetaData, vendor, proxyId) {
			var vmDestination = {};
			vmDestination.requestor = 8;
			vmDestination.serverId = serverId;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.hostname = hostName;
			vmDestination.browseReqType = browseReqType;
			vmDestination.vendor = vendor;
			vmDestination.proxyId = parseInt(proxyId) || 0;

			if (
				browseReqType == VMBrowseReqType.VCLOUD_VDC_IN_ORG ||
				browseReqType == VMBrowseReqType.VCLOUD_NETWORK_IN_VAPP ||
				browseReqType == VMBrowseReqType.VCLOUD_USER_IN_ORG ||
				browseReqType == VMBrowseReqType.VCLOUD_STORAGE_PROFILES_IN_VDC
			) {
				vmDestination.orgName = vsMetaData.vcloudOrgName;
				vmDestination.orgvDCName = vsMetaData.vcloudOrgVdcName;
				vmDestination.vCloudAppName = vsMetaData.vcloudVappName;
			}

			return $http.post('vsaDestinationsBrowse.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getAmazonBuckets = function(serverId, serverDetails, hostname, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.destinationServerName = serverDetails.serverName;
			vmDestination.serverUser = serverDetails.username;
			vmDestination.serverPassword = serverDetails.password;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 5;

			return $http.post('amazonBuckets.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getAmazonIAMs = function(instanceId) {
			return $http.get('amazonIAM.do?instanceId=' + instanceId);
		};

		this.getResourcePools = function(serverId, hostname, serverName, serverUser, serverPassword, vendor) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostname;
			vmDestination.requestor = 3;

			if (serverUser) {
				vmDestination.destinationServerName = serverName;
				vmDestination.serverUser = serverUser;
				vmDestination.serverPassword = serverPassword;
				vmDestination.vendor = vendor;
			}
			if (!serverUser) {
				vmDestination.serverUser = 'root'; //default value for serverUser
			}
			return $http.post('resourcepools.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getHosts = function(serverId) {
			return $http.get('hosts.do', {
				params: {
					serverId: serverId
				}
			});
		};

		this.vsaInventoryBrowse = function(
			serverId,
			inventoryPath,
			serverName,
			username,
			password,
			elementSeperator,
			vendor
		) {
			var params = {
				serverId: serverId,
				inventoryPath: inventoryPath,
				vendor: vendor
			};
			if (serverName) {
				params['serverName'] = serverName;
				params['username'] = username;
				params['password'] = password;
			}
			if (elementSeperator) {
				params['elementSeperator'] = elementSeperator;
			}
			return $http({
				method: 'GET',
				url: 'vsaInventoryBrowse.do',
				params: {
					vsaInvBrowseInfo: JSON.stringify(params)
				},

				cache: true
				//need to set some expiration
			});
		};

		/* vendorType must be one of the enum values in VSVendor */
		this.getHyperVDestinationServerList = function(vendorType) {
			return $http.get('vsaClientList.do', {
				params: {
					vendorType: vendorType
				}
			});
		};

		this.getVsaTenantClientsForAdmin = function(instanceId) {
			return $http.get('vsaTenantClientList.do', {
				params: {
					instanceId: instanceId
				}
			});
		};

		this.createCollection = function(collectionInfo) {
			return $http.post('createCollection.do', collectionInfo);
		};

		this.modifyCollectionContent = function(collectionInfo) {
			return $http.post('modifyCollectionContent.do', collectionInfo);
		};

		this.previewCollectionContent = function(collectionInfo) {
			return $http.post('previewCollectionContent.do', collectionInfo);
		};

		this.updateStorageForCollection = function(collectionId, libraryId, retentionDays) {
			return $http.post('updateStorageForCollection.do', {
				collectionId: collectionId,
				libId: libraryId,
				retention: retentionDays
			});
		};

		this.getCollectionContent = function(collectionInfo) {
			return $http.get('getCollectionContent.do', {
				params: {
					collectionId: collectionInfo.collectionId
				}
			});
		};

		this.cloneVMGroup = function(cloneVMGroupInfo) {
			return $http.post('cloneVMGroup.do', cloneVMGroupInfo);
		};

		this.getServerAlerts = function(clientId) {
			return $http.get('serverAlerts.do', {
				params: {
					clientId: clientId,
					subClientId: 0
				}
			});
		};

		this.getCollectionAlerts = function(subClientId) {
			return $http.get('serverAlerts.do', {
				params: {
					clientId: 0,
					subClientId: subClientId
				}
			});
		};

		/**
		 * Backup the most recently setup server.
		 *
		 * @param {function}
		 *            The function to execute on a successful backup.
		 * @param {function}
		 *            The function to execute on a failed backup.
		 */
		this.backupRecentServer = function(success, error) {
			var _this = this;

			// Handle receiving the sub client list for a server.
			var handleSubClientList = function(data) {
				_this.backupSubClient(data.subclients[0].sc.subClientEntity.subclientId, 'FULL').then(function(wr) {
					if (wr.taskId != -1 && angular.isFunction(success)) {
						success(wr);
					} else if (angular.isFunction(error)) {
						error();
					}
				}, error);
			};

			// Handle receiving the server list for the user.
			var handleServers = function(data) {
				if (data.length > 0) {
					_this
						.getSubClientList(data[data.length - 1].clientId)
						.success(handleSubClientList)
						.error(error);
				} else if (angular.isFunction(error)) {
					error();
				}
			};

			_this.getServers().then(handleServers, error);
		};

		this.backupSubClient = function(subClientId, backupType, vmGuidList) {
			return $http.post('backup.do', {
				subClientId: JSON.stringify(subClientId),
				backupType: backupType,
				//'collectMetadata': collectMetadata,
				vmGuidList: JSON.stringify(vmGuidList)
			});
		};

		//								var vmStatusCache = $cacheFactory('vmStatusCache'); //need to set some expiration
		this.getVMList = function(vmInfo) {
			return $http({
				method: 'GET',
				url: 'vmList.do',
				params: {
					vmInfo: JSON.stringify(vmInfo)
				},

				cache: true
				//need to set some expiration
			});
		};

		this.getVmDetails = function(vmInfo) {
			return $http.get('vmDetails.do', {
				params: {
					vmGuid: vmInfo.vmGuid,
					serverName: vmInfo.serverName,
					serverId: vmInfo.serverId,
					collectionId: vmInfo.collectionId
				}
			});
		};
		this.getUserVMList = function(options) {
			if (!options) {
				options = {};
			}
			options.additionalQuery = 'forUser=true&status=0';

			const KUBERNETES_ENUM = 20;
			if (options.vendorType == KUBERNETES_ENUM) {
				options.additionalQuery = options.additionalQuery.concat('&filterVendorId=' + options.vendorType); // To filter data for Kubernetes
			} else {
				options.additionalQuery = options.additionalQuery.concat('&excludeVendorId=' + KUBERNETES_ENUM);
			}

			let url = util.getCachedEntityUrl(options);
			return $http.get('uservms.do', {
				params: {
					url: url
				},
				headers: jsonHeader
			});
		};

		this.getCustomTabs = function() {
			return $http.get('customTabs.do');
		};

		this.getCustomTab = function(tabName) {
			return $http.get('customTab.do', {
				params: {
					tabName: tabName
				}
			});
		};

		this.getWidgets = function(pageName) {
			return $http.get('widgets.do', {
				params: {
					pageName: pageName
				}
			});
		};

		this.getWidget = function(pageName, widgetName) {
			return $http.get('widget.do', {
				params: {
					pageName: pageName,
					widgetName: widgetName
				}
			});
		};

		this.getMonitoringDetails = function(serverId) {
			return $http.get('getHypervMonitoring.do', {
				params: {
					serverId: serverId
				}
			});
		};

		this.manageMonitoring = function(monitorInfo) {
			return $http.post('manageHypervMonitoring.do', {
				action: monitorInfo.action,
				serverId: monitorInfo.serverId
			});
		};

		this.manageActivity = function(activityInfo) {
			return $http.post('manageActivityForServer.do', {
				type: activityInfo.type,
				action: activityInfo.action,
				serverId: activityInfo.serverId
			});
		};

		this.manageActivityForCollection = function(activityInfo) {
			return $http.post('manageActivityForCollection.do', {
				type: activityInfo.type,
				action: activityInfo.action,
				serverId: activityInfo.serverId,
				collectionId: activityInfo.collectionId
			});
		};

		this.manageIndexingActivity = function(activityInfo) {
			return $http.post('manageIndexingActivity.do', {
				action: activityInfo.action,
				serverId: activityInfo.serverId,
				collectionId: activityInfo.collectionId
			});
		};

		this.deleteServer = function(serverId, serverName) {
			return $http.post('deleteServer.do?serverId=' + serverId + '&serverName=' + serverName);
		};

		this.editProxy = function(instanceId, proxyList, subclientId) {
			if (subclientId) {
				return $http.post('editSubClientProxy.do', {
					subclientId: subclientId,
					proxyList: JSON.stringify(proxyList)
				});
			} else {
				return $http.post('editProxy.do', {
					instanceId: instanceId,
					proxyList: JSON.stringify(proxyList)
				});
			}
		};

		this.editEsxProxy = function(clientId, subclientId, esxServer) {
			return $http.post('editSubClientEsxProxy.do', {
				clientId: clientId,
				subclientId: subclientId,
				esxServer: esxServer
			});
		};

		this.editProxyForHyperv = function(info) {
			if (info.subclientId) {
				return $http.post('editSubClientProxy.do', {
					subclientId: info.subclientId,
					proxyList: JSON.stringify(info.changedProxy)
				});
			} else {
				return $http.post('editProxyForHyperv.do', {
					instanceId: info.instanceId,
					clientId: info.clientId,
					vsUserName: info.vsUserName,
					vsPassword: info.vsPassword,
					proxyList: JSON.stringify(info.changedProxy),
					nodes: JSON.stringify(info.changedNode)
				});
			}
		};

		var serverSvc = this;

		this.getProxy = function(instanceId) {
			var deferred = $q.defer();
			$http
				.get('proxy.do?instanceId=' + instanceId)
				.success(function(data) {
					serverSvc.cachedProxy = data.associatedClients.memberServers;
					deferred.resolve(data);
					return;
				})
				.error(function(e) {
					deferred.reject(e);
				});
			return deferred.promise;
		};

		//subclientId is sent when you want proxy from subclient,also setting includeClientProxy to true will retreive client proxy in case  subclient proxy is empty
		//send proxyList if there is chance that subclient proxy is not present in proxy list and you have to fallback to client proxy
		this.getProxyByClient = function(clientId, subclientId, includeClientProxy, proxyList) {
			var _this = this;
			var deferred = $q.defer();
			if (subclientId) {
				$http
					.get('subClientProxy.do?subclientId=' + subclientId + '&includeClientProxy=' + includeClientProxy)
					.success(function(data) {
						var memberServers = data.virtualServerInstance.associatedClients.memberServers;
						var found = true;

						if (proxyList && proxyList.length && memberServers && memberServers.length) {
							found = proxyList.some(function(element) {
								element.clientId == memberServers[0].client.clientId;
							});
						}
						//this is the case where we get proxy from subclient but that is not present in proxyList so we fallback to client
						if (!found && includeClientProxy) {
							//get from client
							_this.getClientProxy(deferred, clientId);
						} else {
							serverSvc.cachedProxy = data.virtualServerInstance.associatedClients.memberServers;
							deferred.resolve(data);
							return;
						}
					})
					.error(function(e) {
						deferred.reject(e);
					});
			} else {
				_this.getClientProxy(deferred, clientId);
			}
			return deferred.promise;
		};

		this.getClientProxy = function(deferred, clientId) {
			$http
				.get('proxyByClient.do?clientId=' + clientId)
				.success(function(data) {
					serverSvc.cachedProxy = data.virtualServerInstance.associatedClients.memberServers;
					deferred.resolve(data);
					return;
				})
				.error(function(e) {
					deferred.reject(e);
				});
		};

		this.getAppProperties = function(clientId) {
			return $http.get('appProperties.do', {
				params: {
					clientId: clientId
				}
			});
		};

		//for child jobs need to pass backupsetId so that parent instance is retreived
		this.getAppPropertiesByInstance = function(instanceId, backupSetId) {
			return $http.get('appPropertiesForInstance.do', {
				params: {
					instanceId: instanceId,
					backupSetId: backupSetId
				}
			});
		};

		this.setAppProperties = function(appProperties, vCloudPassword) {
			return $http.post('appProperties.do', {
				appProperties: appProperties,
				vCloudPassword: vCloudPassword
			});
		};

		this.getHostName = function(clientId) {
			return $http.post('getHostName.do', {
				clientId: clientId
			});
		};

		this.getVcenterCollection = function(clientId, hostName, userName, password, requestType) {
			return $http.post('getVcenterCollection.do', {
				clientId: clientId,
				hostName: hostName,
				userName: userName,
				password: password,
				requestType: requestType
			});
		};

		this.createVcenterAssociation = function(serverId, vcenterList) {
			return $http.post('vcenterAssociation.do', {
				serverId: serverId,
				vcenterList: JSON.stringify(vcenterList)
			});
		};

		this.getProxyServersVendor = function(vendor, platform, packageList) {
			if (!vendor) {
				vendor = '';
			}
			if (_.isNil(platform)) {
				platform = '';
			}
			if (!packageList) {
				packageList = '';
			} else if (Array.isArray(packageList)) {
				packageList = encodeURIComponent(packageList.join('|'));
			}

			return $http.get('proxyServers.do?vendor=' + vendor + '&platform=' + platform + '&packageList=' + packageList);
		};

		//common method to get proxies by client or vendor
		this.getProxyServers = function(vendor, platform, clientId, packageList) {
			if (!clientId) {
				return this.getProxyServersVendor(vendor, platform, packageList);
			} else {
				return $http.get('proxyByClient.do?clientId=' + clientId);
			}
		};

		//common method to get proxies by instanceId
		this.getProxyServersByInstance = function(instanceId) {
			return $http.get('proxyByInstance.do?instanceId=' + instanceId);
		};

		// do not use this, it will be removed, use getProxyServers() instead
		this.getProxyServersForRestore = function() {
			return this.getProxyServers();
		};
		this.getProxyServersAndGroups = function(vendor, platform) {
			// Returns a list of ProxyEntity objects that contains both proxy clients and nodes
			if (!vendor) {
				vendor = '';
			}
			return $http.get('proxyServersAndGroups.do?vendor=' + vendor + '&platform=' + platform);
		};

		this.getCloudAppsAccessNodes = function(instanceType) {
			let url = 'cloudAppsAccessNodes.do';
			if (instanceType !== undefined) {
				url = `${url}?instanceType=${instanceType}`;
			}
			return $http.get(url);
		};

		this.editCollection = function(collectionInfo) {
			return $http.post('editCollection.do', collectionInfo); // !!! COLLECTIONCONTENT - ADDCOLLECTION - NEED TO SEND FILTERS TOO
		};
		this.filterVmStatuses = function(statuses) {
			return statuses;
		};

		this.hasCapability = function(capId) {
			return $http.get('checkCapability.do', {
				params: {
					capId: capId
				}
			});
		};

		this.getCloudAppsInstanceType = function(entityType, entityId) {
			return $http.get('cloudAppsInstanceType.do', {
				params: {
					entityType: entityType,
					entityId: entityId
				}
			});
		};

		this.isMspAdmin = function(capId) {
			return $http.get('isMspAdmin.do');
		};

		this.getNodesForHyperVCluster = function(clientId, hostName) {
			return $http.get('discoverNodesByClientId.do', {
				params: {
					clientId: clientId,
					hostName: hostName || '' // do not send undefined in the hostname
				}
			});
		};
		this.getNodesAndGroupsForHyperVCluster = function(clientId, hostName, proxyId) {
			// Returns a list of ProxyEntity objects that contains both groups and nodes
			return $http.get('getNodesAndGroupsByClientId.do', {
				params: {
					clientId: clientId,
					hostName: hostName || '', // do not send undefined in the hostname
					proxyId: proxyId
				}
			});
		};

		this.getDiscoveredNodes = function(hostname, vsUserName, vsPassword, proxyId, credentialId) {
			return $http.post('discoverNodes.do', {
				hostName: hostname,
				vsUserName: vsUserName,
				vsPassword: vsPassword,
				proxyId: proxyId,
				credentialId: credentialId
			});
		};

		this.manageVMPlugin = function(model) {
			switch (model.action) {
				case 'ADD':
					return $http.post('vmplugin.do?', { model: JSON.stringify(model) });
				case 'FIND':
					return $http.get('vmplugin.do?' + $.param({ model: JSON.stringify(model) }));
				case 'EDIT':
					return $http.put('vmplugin.do', { model: JSON.stringify(model) });
				case 'DELETE':
					return $http['delete']('vmplugin.do?' + $.param({ model: JSON.stringify(model) }));
				default:
					return null;
			}
		};

		this.getWebConsoleHost = function() {
			return $http.get('webconsoleurl.do');
		};

		this.modifyOptions = function(obj) {
			return $http.post('modifyCollectionOptions.do', obj);
		};

		this.modifyAppCredentials = function(obj) {
			return $http.post('modifyAppCredentials.do', obj);
		};

		this.getGlobalParam = function(key, readFromDb) {
			if (!angular.isDefined(readFromDb)) {
				readFromDb = false;
			}
			return $http.get('globalParam.do?key=' + key + '&readFromDb=' + readFromDb);
		};

		/**
		 * Set whether or not the CommServe is an MSP.
		 *
		 * @param {boolean}
		 *            value - True if the CommServe should be an MSP; False otherwise.
		 * @return {HttpPromise}
		 */
		this.setIsMSPCommcell = function(value) {
			return $http
				.post('setIsMSPCommcell.do', {
					isMSPCommcell: value ? true : false
				})
				.success(function() {
					cvApp.showMSP = value;
				});
		};

		/**
		 * Set's a global parameter in the system.
		 *
		 * @param {string}
		 *            key - The key to set.
		 * @param {string}
		 *            value - The value to set the key to.
		 * @return {HttpPromise}
		 */

		this.setGlobalParam = function(key, value) {
			return $http
				.post('setGlobalParam.do', {
					key: key,
					value: value
				})
				.success(function() {
					cvApp.globalParams[key] = value;
				});
		};

		this.saveWebDomainsList = function(value) {
			return $http.post('saveWebDomainsList.do', {
				list: value
			});
		};

		/**
		 * Specifies to not ask any user about setting the CommServe as an MSP.
		 *
		 * @return {HttpPromise}
		 */

		this.dontAskAboutMSP = function() {
			return $http.post('dontAskAboutMSP.do');
		};

		this.getFBRMAList = function(osType) {
			return $http.get('fbrMAList.do?osType=' + osType);
		};

		this.getAzureVmSizes = function(serverId, destinationServerName, vendor, hostName) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.hostname = hostName;
			vmDestination.destinationServerName = destinationServerName;
			vmDestination.vendor = vendor;
			vmDestination.requestor = 7;

			var params = {
				vmDestination: JSON.stringify(vmDestination)
			};
			return $http.post('getAzureVmSizeOptions.do', params);
		};

		this.removeVM = function(vmGuid, vmName, subclientId) {
			return $http.post('removeVM.do', {
				vmGuid: vmGuid,
				vmName: vmName,
				subclientId: subclientId
			});
		};

		this.getSystemState = function(backupsetId, platform, recoveryTime) {
			return $http.post('systemState.do', {
				backupSetId: backupsetId,
				platform: platform,
				recoveryTime: recoveryTime
			});
		};

		this.getIsoPaths = function(serverId, browseDataStore, destinationServer, vendor, destinationHostName) {
			var vmDestination = {};
			vmDestination.serverId = serverId;
			vmDestination.destinationServerName = destinationServer.name;
			vmDestination.serverUser = destinationServer.username;
			vmDestination.serverPassword = destinationServer.password;
			vmDestination.vendor = vendor;
			vmDestination.hostname = destinationHostName;
			vmDestination.browseDataStore = browseDataStore;
			vmDestination.requestor = 9;
			return $http.post('getIsoPaths.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getGatewayTemplates = function(dataCenter, clientId, instanceId) {
			var params = 'dataCenter=' + dataCenter + '&clientId=' + clientId + '&instanceId=' + instanceId;
			return $http.get('getGatewayTemplates.do?' + params);
		};

		this.getVMSizes = function(serverId, hostName, vendor, projectId) {
			var vmDestination = {};
			vmDestination.requestor = 8;
			vmDestination.serverId = serverId;
			vmDestination.destinationServerName = '';
			vmDestination.serverUser = '';
			vmDestination.serverPassword = '';
			vmDestination.hostname = hostName;
			vmDestination.browseReqType = 8;
			vmDestination.vendor = vendor;
			vmDestination.projectId = projectId;

			return $http.post('vsaVMSizes.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		//this is used to manage intellisnap operations for Amazon.
		//vmware currently uses plans. This call can be augmented in case
		//support for other vendors is added
		this.manageIntellisnapOperation = function(intellisnapInfo) {
			return $http.post('manageIntellisnapOperation.do', intellisnapInfo);
		};

		this.getOSTypesAvailable = function(serverId, serverDetails, browseReqType, hostName, vsMetaData, vendor) {
			var vmDestination = {};
			vmDestination.requestor = 8;
			vmDestination.serverId = serverId;
			vmDestination.browseReqType = browseReqType || 14;
			vmDestination.destinationServerName = serverDetails.serverName || '';
			vmDestination.serverUser = serverDetails.username || '';
			vmDestination.serverPassword = serverDetails.password || '';
			vmDestination.hostname = hostName;
			vmDestination.vendor = vendor || '';

			return $http.post('vsaOSTypeBrowse.do', {
				vmDestination: JSON.stringify(vmDestination)
			});
		};

		this.getSnapEngines = function(clientId) {
			return $http.get('snapEngines.do?clientId=' + clientId);
		};

		this.getSLAEntityForAgent = function(params) {
			return $http.get('getSLAEntityForAgent.do', {
				params: params
			});
		};

		this.getAzureVmTemplate = function(regionName, item, clientId, instanceId) {
			var payload = {
				EVGui_GetBrowseInventoryReq: {
					browsePath: '\\Images',
					instanceEntity: {
						clientId: clientId,
						instanceId: instanceId
					},
					appId: {
						clientId: clientId,
						instanceId: instanceId
					},
					InventoryInfo: {
						name: regionName
					}
				}
			};

			var body = { name: item.ItemName };
			while (item.parentNode) {
				body = { name: item.parentNode.ItemName, inventoryInfo: body };
				item = item.parentNode;
			}
			payload.EVGui_GetBrowseInventoryReq.InventoryInfo.inventoryInfo = body;
			return $http.post(cv.contextPath + '/api/qcommand/qoperation execute', JSON.stringify(payload), {
				headers: { 'Content-Type': 'application/json' }
			});
		};

		this.processSelectedVMTemplates = function(result) {
			return cvUtil.loadSynchronously(cv.contextPath + '/processSelectedVMTemplates.do', {
				vmTemplateResult: angular.toJson(result)
			});
		};

		this.getApplicationGroups = vmGroupsList => {
			return vmGroupsList.length
				? vmGroupsList.filter(function(appGrp) {
						return appGrp.sc.vsaSubclientProp.vendorType === VSA_VENDOR.KUBERNETES;
				  })
				: [];
		};

		/**
		 * Gets VSA clients from MongoDB cache
		 */
		this.getServerCachedVSAClients = function(options) {
			options.additionalQuery = `PseudoClientType=VSPseudo`;
			if (options.vendorType === vendorTypes.getVendorEnumValue(VSA_VENDOR.KUBERNETES)) {
				// Kubernetes clusters
				options.additionalQuery = options.additionalQuery.concat('&filterVendorId=' + options.vendorType);
			} else {
				// Hypervisors
				options.additionalQuery = options.additionalQuery.concat(
					'&excludeVendorId=' + vendorTypes.getVendorEnumValue(VSA_VENDOR.KUBERNETES)
				);
			}
			const url = util.getCachedEntityUrl(options);
			return $http.get('serverCachedVSAClients.do', {
				params: { url },
				headers: jsonHeader
			});
		};

		this.getServerCachedSubclientList = function(serverId, options) {
			let cancel = $q.defer();
			options.additionalQuery =
				'clientId=' + serverId + '&applicationId=106&PropertyLevel=5&includeVMPseudoSubclients=false';
			if (options.vendorType === vendorTypes.getVendorEnumValue(VSA_VENDOR.KUBERNETES)) {
				// Kubernetes application groups
				options.additionalQuery = options.additionalQuery.concat('&filterVendorId=' + options.vendorType);
			} else {
				// VM Groups
				options.additionalQuery = options.additionalQuery.concat(
					'&excludeVendorId=' + vendorTypes.getVendorEnumValue(VSA_VENDOR.KUBERNETES)
				);
			}
			let url = util.getCachedEntityUrl(options);
			return $http.get('subclients.do', {
				params: {
					appId: 106,
					serverId: serverId,
					url: url
				},
				timeout: cancel.promise,
				cancel: cancel,
				overrideTimeout: true,
				headers: jsonHeader
			});
		};
		/**
		 * this returns VSA clients with a particular permission, which can be filtered by an optional vendor
		 * param
		 *
		 * @permissionId = {number}
		 * @vendor = {number}
		 *
		 * @return = {promise}
		 */
		this.getVSAClientsWithPermission = (permissionId, vendor, filterAdminClients) => {
			let url = 'proxy/VSPseudoClientsForPermission?permissionId=' + permissionId;
			if (vendor) {
				url += '&vendor=' + vendor;
			}
			if (filterAdminClients) {
				url += '&filterAdminClient=' + filterAdminClients;
			}
			return $http.get(url);
		};

		this.getEligibleEntitiesForMigration = (entities, companyId) => {
			entities.forEach(entity => {
				if (entity.clientId > 0) {
					entity._type_ = ENTITY_TYPES.CLIENT_ENTITY;
				} else if (entity.clientGroupId > 0) {
					entity.clientId = undefined;
					entity._type_ = ENTITY_TYPES.CLIENT_GROUP_ENTITY;
				}
			});

			const request = {
				entities: entities
			};

			const resp = $http.put(
				'proxy/company/' + companyId + '/migration-entities',
				JSON.stringify(request),
				httpHeader.jsonDefaultHeader
			);
			return resp;
		};

		this.changeCompanyForServers = (entities, companyId) => {
			return $http.post('changeCompanyForServers.do', { entities: JSON.stringify(entities), companyId: companyId });
		};

		this.getSourceNetwork = (vmGuid, clientId) => {
			return $http.get('sourceNetworkInfo.do', {
				params: {
					vmGuid: vmGuid,
					clientId: clientId
				}
			});
		};

		this.getBackupCopyStatus = function(object) {
			if (!_.isNil(object)) {
				let url = 'proxy/v2/vsa/' + object.url;
				return $http.get(url);
			}
		};
	}
]);

export default serverMod;
