import 'modules/reports/js/services/reports.svc';

import { reportsDashboardModule } from 'common/js/modules';

var app = reportsDashboardModule;

app.directive('cvReportsHyperscaleHardware', function() {
	return {
		restrict: 'E',
		scope: {},
		templateUrl: appUtil.appRoot + 'modules/reports/partials/directives/cvReportsHyperscaleHardware.jsp',
		require: [],
		controller: 'HyperscaleHardwareController',
		controllerAs: 'ctrl'
	};
});
app.controller('HyperscaleHardwareController', [
	'$location',
	'cvLoc',
	'reportsDashboardService',
	'reportsDashboardFactory',
	'cvToaster',
	'ChartColors',
	'AnomalyTypes',
	function HyperscaleHardwareController(
		$location,
		cvLoc,
		reportsDashboardService,
		reportsDashboardFactory,
		cvToaster,
		ChartColors,
		AnomalyTypes
	) {
		var self = this;
		var path = $location.path().substring(1);
		var imgRoot = appUtil.appRoot + 'common/img/';
		var criticalImg = appUtil.appRoot + 'common/img/critical-status.svg';
		var COLORCODES = {
			GOOD: ChartColors.SUCCESS,
			WARNING: ChartColors.WARNING,
			CRITICAL: ChartColors.FAILURE
		};

		self.$onInit = function() {
			self.tileTitle = cvLoc('appliance.appliance.hardware.title');

			self.systemData = {
				title: cvLoc('appliance.system'),
				detailsLink: appUtil.appRoot + '#/reportDetails?name=Infrastructure%20Load&app=ADMIN_CONSOLE&nav=' + path
			};
			self.hwData = {
				title: cvLoc('appliance.hardware.title'),
				detailsLink: appUtil.appRoot + '#/reportDetails?reportId=Hardware%20Details&app=ADMIN_CONSOLE&nav=' + path
			};

			self.hwData.criticalIconSrc = appUtil.appRoot + 'common/img/critical-status.svg';

			self.entityMetaInfo = {
				disk: _getHardwareEntityObj(cvLoc('appliance.hardware.disk'), imgRoot + 'disk.svg', criticalImg),
				'cache-disk': _getHardwareEntityObj(
					cvLoc('appliance.hardware.cacheDisk'),
					imgRoot + 'cache-disk.svg',
					criticalImg
				),
				'power-supply': _getHardwareEntityObj(
					cvLoc('appliance.hardware.powerSupply'),
					imgRoot + 'power-supply.svg',
					criticalImg
				),
				'fan-state': _getHardwareEntityObj(
					cvLoc('appliance.hardware.fanState'),
					imgRoot + 'fan-state.svg',
					criticalImg
				),
				'nic-state': _getHardwareEntityObj(cvLoc('appliance.hardware.nicState'), imgRoot + 'nic-state.svg', criticalImg)
			};

			function _getHardwareEntityObj(name, imgSrc, criticalImgSrc) {
				return {
					name: name,
					imgSrc: imgSrc,
					criticalImgSrc: criticalImgSrc
				};
			}

			processSystemData();
			processHardwareData();
		};

		function processSystemData() {
			var anomalyType = 0;
			anomalyType = anomalyType | AnomalyTypes.HighCPUUsage;
			anomalyType = anomalyType | AnomalyTypes.HighMemoryUsage;
			anomalyType = anomalyType | AnomalyTypes.MemoryLeak;

			self.systemData.showLoader = true;
			reportsDashboardService
				.getCPUandMemoryUsage(anomalyType)
				.then(function(data) {
					if (data) {
						data = data.data;
					}

					if (!data || !data.HighMemoryUsage || data.HighMemoryUsage.Total <= 0) {
						self.systemData.showLoader = false;
						self.systemData.noData = { text: cvLoc('dashboard.msg.noDataAvailable') };
						return;
					}
					if (data.HighMemoryUsage && data.HighMemoryUsage.Total > 0 && data.HighMemoryUsage.TotalDetected <= 0) {
						self.systemData.showLoader = false;
						self.systemData.noData = { text: cvLoc('appliance.system.nodes.offiline.msg') };
						return;
					}

					var memoryUsage = { total: { value: 0, unit: 'GB' }, used: { value: 0, unit: 'GB' } };
					if (data.MemoryConsumption) {
						if (data.MemoryConsumption.Total) {
							memoryUsage.total = reportsDashboardFactory.getSizeAndUnits(
								reportsDashboardFactory.autoConvertFromMB(data.MemoryConsumption.Total / 1024)
							);
						}
						if (data.MemoryConsumption.TotalDetected) {
							memoryUsage.used = reportsDashboardFactory.getSizeAndUnits(
								reportsDashboardFactory.autoConvertFromMB(data.MemoryConsumption.TotalDetected / 1024)
							);
						}
					}
					self.memoryUsage = memoryUsage;
					var formattedData = formatSystemData(data);
					drawChart(formattedData.memory, cvLoc('appliance.system.memory'), 'memory.svg', 'memory');
					drawChart(formattedData.cpu, cvLoc('appliance.system.cpu'), 'cpu.svg', 'cpu');
					if (data.HighMemoryUsage && data.HighMemoryUsage.Total > data.HighMemoryUsage.TotalDetected) {
						var offlineCnt = data.HighMemoryUsage.Total - data.HighMemoryUsage.TotalDetected;
						self.offlineNotes = cvLoc('appliance.system.nodes.offiline.note', offlineCnt, data.HighMemoryUsage.Total);
					}
					self.systemData.showLoader = false;
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						self.systemData.showLoader = false;
						self.systemData.noData = { text: cvLoc('dashboard.error.server') };
						reportsDashboardFactory.showErrorMessage(err, 'HyperScale System');
						console.log(err);
					}
				});
		}

		function processHardwareData() {
			self.hwData.showLoader = true;

			reportsDashboardService
				.getHardwareData()
				.then(function(data) {
					if (data) {
						data = data.data;
					}
					if (!data['records']) {
						console.log('No records found for hardware.');
						self.hwData.showLoader = false;
						self.hwData.noData = { text: cvLoc('dashboard.msg.noDataAvailable') };
						return;
					}

					if (data.totalRecordCount <= 0) {
						console.log('Records has no rows. Returning false.');
						self.hwData.showLoader = false;
						self.hwData.noData = { text: cvLoc('dashboard.msg.noDataAvailable') };
						return;
					}

					self.hwData.showLoader = false;
					formatHardwareData(data);
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						self.hwData.showLoader = false;
						self.hwData.noData = { text: cvLoc('dashboard.error.server') };
						reportsDashboardFactory.showErrorMessage(err, 'HyperScale Hardware');
						console.log(err);
					}
				});
		}

		function formatSystemData(rawData) {
			var data = {};
			data.memory = {
				critical: 0,
				warning: 0,
				good: 0,
				total: 0
			};

			data.cpu = {
				critical: 0,
				warning: 0,
				good: 0,
				total: 0
			};
			if (rawData.HighMemoryUsage) {
				data.memory.total = rawData.HighMemoryUsage.Total;
				if (rawData.HighMemoryUsage.Critical) {
					data.memory.critical = rawData.HighMemoryUsage.Critical;
				}
				if (rawData.HighMemoryUsage.Warning) {
					data.memory.warning = rawData.HighMemoryUsage.Warning;
				}
				data.memory.good = rawData.HighMemoryUsage.TotalDetected - (data.memory.critical + data.memory.warning);
			}
			if (rawData.HighCPUUsage) {
				rawData.HighCPUUsage.TotalDetected = rawData.HighMemoryUsage.TotalDetected;
				data.cpu.total = rawData.HighCPUUsage.Total;
				if (rawData.HighCPUUsage.Critical) {
					data.cpu.critical = rawData.HighCPUUsage.Critical;
				}
				if (rawData.HighCPUUsage.Warning) {
					data.cpu.warning = rawData.HighCPUUsage.Warning;
				}
				data.cpu.good = rawData.HighCPUUsage.TotalDetected - (data.cpu.critical + data.cpu.warning);
			}
			return data;
		}

		function formatHardwareData(data) {
			var hardwareTypeMap = {
				1: 'disk',
				2: 'cache-disk',
				4: 'fan-state',
				8: 'power-supply',
				16: 'nic-state'
			};
			var reportCompMap = {
				1: { table: 'Table1538692662762', filter: 'EntityStatusStr=Offline' }, //Disks
				2: { table: 'Table1538692743393', filter: 'EntityStatusStr=Offline' }, //Cache
				4: { table: 'Table1538692884743', filter: 'EntityStatusStr=Off' }, //Cooling Devices/Fans
				8: { table: 'Table1538693005924', filter: 'EntityStatusStr=Disconnected' }, //Power supply
				16: { table: 'Table1538693098987', filter: 'EntityStatusStr=Disconnected' } //NICs
			};

			var ID_IDX = 1;
			var COUNT_IDX = 3;
			var OFFLINE_IDX = 4;
			var baseReportLink =
				appUtil.appRoot + '#/reportDetails?reportId=Hardware%20Details&app=ADMIN_CONSOLE&nav=' + path;

			if (data.records) {
				angular.forEach(data.records, function(rowData) {
					var id = rowData[ID_IDX];
					var entity = self.entityMetaInfo[hardwareTypeMap[id]];
					if (!entity) {
						return;
					}

					entity.entityCount = rowData[COUNT_IDX];
					entity.offlineCount = rowData[OFFLINE_IDX];
					if (entity.offlineCount > 0) {
						var count = '<span class="entity-count">' + entity.entityCount + '</span>';
						var offline = '<span class="entity-offline">' + entity.offlineCount + '</span>';
						if (id == 1 || id == 2) {
							entity.errorMsg = cvLoc('appliance.hardware.offline.msg', offline, count);
						} else if (id == 4 || id == 16) {
							entity.errorMsg = cvLoc('appliance.hardware.failed.msg', offline, count);
						} else if (id == 8) {
							entity.errorMsg = cvLoc('appliance.hardware.failed.msg', offline, count);
						}
						entity.detailsLink =
							baseReportLink +
							'&showComponents=' +
							reportCompMap[id].table +
							'&' +
							reportCompMap[id].table +
							'.filter.' +
							reportCompMap[id].filter;
					}
				});
			}

			var entities = [];
			entities.push(self.entityMetaInfo['disk']);
			entities.push(self.entityMetaInfo['cache-disk']);
			entities.push(self.entityMetaInfo['power-supply']);
			entities.push(self.entityMetaInfo['fan-state']);
			entities.push(self.entityMetaInfo['nic-state']);

			self.hwData.entities = entities;
		}
		function drawChart(data, title, image, type) {
			var chartData = [
				{
					name: cvLoc('appliance.system.good'),
					y: data.good
				},
				{
					name: cvLoc('appliance.system.warning'),
					y: data.warning
				},
				{
					name: cvLoc('appliance.system.critical'),
					y: data.critical
				}
			];

			var dialColors = [COLORCODES['GOOD'], COLORCODES['WARNING'], COLORCODES['CRITICAL']];
			var total = data.good + data.warning + data.critical;

			var chartConfig = {
				options: {
					chart: {
						type: 'pie',
						margin: [0, 0, 0, 0],
						spacingTop: 0,
						spacingBottom: 30,
						spacingLeft: 0,
						spacingRight: 0,
						outlineColor: ''
					},
					tooltip: {
						formatter: function() {
							return this.point.name + ': <b>' + this.point.y + '</b>';
						}
					},
					plotOptions: {
						pie: {
							startAngle: 0,
							allowPointSelect: false,
							cursor: 'pointer',
							dataLabels: {
								enabled: false
							},
							showInLegend: true,
							point: {
								events: {
									click: function() {
										if (this.options.details) {
										}
									}
								}
							}
						},
						series: {
							states: {
								hover: {
									enabled: false
								}
							}
						}
					},
					legend: {
						enabled: false
					},
					exporting: {
						enabled: false
					},

					colors: dialColors
				},
				title: {
					text: ''
				},
				subtitle: {
					text:
						'<div class="dial-center-wrapper"><div class="dial-center-value">' +
						total +
						'</div><div class="dial-center-label">' +
						cvLoc('appliance.system.nodes') +
						'</div></div>',
					x: 0,
					y: 55,
					floating: true,
					useHTML: true
				},
				credits: {
					enabled: false
				},

				series: [
					{
						colorByPoint: true,
						animation: true,
						innerSize: '85%',
						data: chartData
					}
				]
			};

			var customLegend = [];
			if (data.warning > 0) {
				customLegend.push({
					value: data.warning,
					label: cvLoc('appliance.system.warning')
				});
			}

			if (data.critical > 0) {
				customLegend.push({
					value: data.critical,
					label: cvLoc('appliance.system.critical')
				});
			}

			self.systemData.entities = self.systemData.entities || [];
			self.systemData.entities.push({
				type: type,
				title: title,
				imgSrc: appUtil.appRoot + 'common/img/' + image,
				chartConfig: chartConfig,
				customLegend: customLegend
			});

			self.systemData.customLegend = [];
		}
	}
]);

export default app;
