import { reportsDashboardModule } from 'common/js/modules';
import 'modules/reports/js/services/reports.svc';
import 'modules/reports/js/factory/reports.factory.js';
import 'modules/reports/js/constants/reports.constants.js';

var app = reportsDashboardModule;

app.directive('cvReportsCurrentCapacity', function() {
	return {
		restrict: 'E',
		scope: {},
		templateUrl: appUtil.appRoot + 'modules/reports/partials/directives/cvReportsCurrentCapacity.jsp',
		bindToController: {
			capacityData: '='
		},
		controller: 'DashboardCapacityController',
		controllerAs: 'capacity'
	};
});
app.controller('DashboardCapacityController', [
	'$location',
	'cvLoc',
	'reportsDashboardService',
	'reportsDashboardFactory',
	'ChartColors',
	'cvToaster',
	'$log',
	'multiCommcellFactory',
	function DashboardCapacityController(
		$location,
		cvLoc,
		reportsDashboardService,
		reportsDashboardFactory,
		ChartColors,
		cvToaster,
		$log,
		multiCommcellFactory
	) {
		var self = this;
		this.$onInit = function() {
			self.data = {
				title: cvLoc('capacity.title'),
				titletooltip: cvLoc('currentcapacity.title.tooltip')
			};
			const DEFAULT_PRIORITY = 9999;
			var path = $location.path().substring(1);
			self.data.showLoader = true;
			self.calloutTemplate = appUtil.appRoot + 'modules/reports/partials/perDashboardCallout.jsp';
			reportsDashboardService
				.getMetricsCommUniqId()
				.then(function(data) {
					if (data) {
						data = data.data;
					}
					var commUniId = 0;
					if (!cv.isCometApp) {
						self.data.showLoader = false;
						if (!data['records']) {
							$log.error('API format is incorrect.');
							return 0;
						}
						if (data.totalRecordCount <= 0) {
							$log.error('Records does not have .');
							return 0;
						}
						var records = data['records'];
						commUniId = records[0][2];
					}
					//console.log('MetricsCommUniqId = '+id);
					var detailsLink = '#/reportDetails?reportId=License%20Summary&app=ADMIN_CONSOLE&nav=' + path;
					self.data.titleDetailsLink = appUtil.appRoot + detailsLink;
					if (cv.isCometApp) {
						self.data.titleDetailsLinkSaved = detailsLink;
						self.data.titleDetailsLink = '';
					}
					getCurrentCapacity(commUniId);
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						self.data.showLoader = false;
						console.log(err);
						reportsDashboardFactory.showErrorMessage(err, self.data.title);
						self.data.noData = { text: cvLoc('dashboard.error.server') };
					}
				});

			function aggregateLicMap(perCommcellLicMap) {
				const aggregatedLicMap = {};
				for (const commcellName in perCommcellLicMap) {
					const licMap = perCommcellLicMap[commcellName];
					for (const gTypeKey in licMap) {
						if (!aggregatedLicMap[gTypeKey]) {
							aggregatedLicMap[gTypeKey] = {};
						}
						const gTypeAggregate = aggregatedLicMap[gTypeKey];
						const gType = licMap[gTypeKey];
						for (const key in gType) {
							switch (key) {
								case 'Name':
								case 'gType':
									// Overwrite, its same data
									gTypeAggregate[key] = gType[key];
									break;
								case 'loanedCapacity':
								case 'currentUsage':
								case 'totalCapacity':
									// Aggregate data
									if (!gTypeAggregate[key]) {
										gTypeAggregate[key] = 0;
									}
									gTypeAggregate[key] += gType[key] || 0;
									if (key == 'loanedCapacity' && gTypeAggregate[key] > 0) {
										gTypeAggregate.loanedCapacityLabel = cvLoc(
											'capacity.label.LoanedCapacity',
											gTypeAggregate[key],
											'TB'
										);
									}
									break;
								case 'capacityPriority':
									// Find the one which will expire first and use its usage text
									if (!gTypeAggregate[key]) {
										gTypeAggregate[key] = DEFAULT_PRIORITY;
										gTypeAggregate.usageNote = gType.usageNote;
									}
									if ((gType[key] || DEFAULT_PRIORITY) < gTypeAggregate[key]) {
										gTypeAggregate[key] = gType[key];
										gTypeAggregate.usageNote = gType.usageNote;
										if (gType[key] != DEFAULT_PRIORITY) {
											gTypeAggregate.usageNote = commcellName + ': ' + gType.usageNote;
										}
									}
									break;
							}
						}
					}
				}
				return aggregatedLicMap;
			}

			function getCometLicMap(multicommcellData) {
				const aggregatedLicMap = {};
				const perCommcellLicMap = {};
				const groupedMulticommcellData = _.groupBy(multicommcellData, 'Data Source');
				for (const key in groupedMulticommcellData) {
					perCommcellLicMap[key] = createLicMap(groupedMulticommcellData[key]);
					_.forEach(perCommcellLicMap[key], lic => {
						processCommCellLicenceType(lic);
					});
				}
				aggregatedLicMap.aggregate = aggregateLicMap(perCommcellLicMap);
				aggregatedLicMap.breakdown = perCommcellLicMap;
				return aggregatedLicMap;
			}

			function createCometBreakdown(perCommcellLicMap) {
				const breakDown = {};
				for (const commcellName in perCommcellLicMap) {
					const licMap = perCommcellLicMap[commcellName];
					for (const gTypeKey in licMap) {
						const gType = licMap[gTypeKey];
						if (!breakDown[gTypeKey]) {
							breakDown[gTypeKey] = {
								perCommcellData: []
							};
						}
						// All sizes are in tb after other functions
						const total = _.round((gType.loanedCapacity || 0) + (gType.totalCapacity || 0), 2);
						const totalString = total + ' TB';

						const rawData = _.round(gType.currentUsage || 0, 2);
						const currentUsageString = rawData + ' TB';

						const value = currentUsageString + ' / ' + totalString;
						const link = cvUtil.modifyUrlForComet(self.data.titleDetailsLinkSaved, commcellName);
						let clickFunction = multiCommcellFactory.redirectToCommcell;
						breakDown[gTypeKey].perCommcellData.push({
							commcellName,
							commcellColumnName: cvLoc('label.commcell'),
							rowData: [
								{
									columnName: gTypeKey,
									value,
									rawData,
									styledValue: true,
									link,
									target: '_blank'
								}
							],
							visible: total > 0,
							clickFunction
						});
					}
				}
				return breakDown;
			}

			function getCurrentCapacity(commUniId) {
				self.data.showLoader = true;
				reportsDashboardService
					.getCurrentCapacity(commUniId)
					.then(function(data) {
						if (data) {
							data = data.data;
						}

						data = reportsDashboardFactory.getJsonFromDSResult(data);

						if (data.length > 0) {
							let licMap;
							if (cv.isCometApp) {
								const cometLicMap = getCometLicMap(data);
								licMap = cometLicMap.aggregate;
								self.data.capacityBreakdown = createCometBreakdown(cometLicMap.breakdown);
							} else {
								licMap = createLicMap(data);
								angular.forEach(licMap, function(lic, name) {
									processCommCellLicenceType(lic);
								});
							}
							self.data.showLoader = false;
							handleCurrentCapacityUsage(licMap);
						} else {
							self.data.showLoader = false;
							self.data.noData = { text: cvLoc('capacity.msg.no_data_available') };
						}
					})
					.catch(function(err) {
						if (err.xhrStatus !== 'abort') {
							self.data.showLoader = false;
							console.log(err);
							reportsDashboardFactory.showErrorMessage(err, self.data.title);
							self.data.noData = { text: cvLoc('dashboard.error.server') };
						}
					});
			}

			function createLicMap(rows) {
				var licMap = {};

				var PERCENT_STR = 0;
				var TOTAL_STR = 1;
				var PEAK_STR = 2;
				var GROWTH_STR = 3;
				var LOAN_STR = 4;
				var CURRENT_STR = 5;

				for (var row in rows) {
					row = rows[row];
					var gType = 'protectionEnterprise';
					var isCALUsage = false;
					if (row.Name == 'Commvault Complete') {
						gType = 'commvaultComplete';
					} else if (row.Name == 'Backup' || row.Name == 'DP Core' || row.Name == 'Data Protection') {
						gType = 'protectionEnterprise';
					} else if (row.Name == 'Archive Enterprise' || row.Name == 'Archive Core' || row.Name == 'Archive') {
						gType = 'archiveEnterprise';
					} else if (row.Name == 'Users' || row.Name == 'Sharing Users' || row.Name == 'eDiscovery Users') {
						gType = 'users';
						isCALUsage = true;
					} else if (row.Name == 'VM Sockets') {
						gType = 'vmsockets';
						isCALUsage = true;
					} else if (row.Name == 'Protected VMs') {
						gType = 'protectedvms';
						isCALUsage = true;
					} else if (row.Name == 'Snapshot') {
						gType = 'snapshot';
					} else if (row.Name == 'Replication') {
						gType = 'replication';
					} else if (row.Name == 'Archived MailBoxes') {
						gType = 'archived_mailboxes';
						isCALUsage = true;
					} else if (row.Name == 'CI Objects') {
						gType = 'ci_objects';
						isCALUsage = true;
					} else if (row.Name == 'Mailbox Archive Solution') {
						gType = 'mailboxe_archive_solution';
						isCALUsage = true;
					}
					var lic = {};
					var arr = row.Label.split(',');
					if (licMap[row.Name]) {
						lic = licMap[row.Name];
					} else {
						licMap[row.Name] = lic;
						lic.details = [];

						lic.gType = gType;
						lic.Name = row.Name;

						lic.totalCapacity = arr[TOTAL_STR];
						lic.totalCapacity = lic.totalCapacity.substr(lic.totalCapacity.indexOf(':') + 2);
						lic.growthRate = arr[GROWTH_STR];
						lic.growthRate = parseFloat(lic.growthRate.substr(lic.growthRate.indexOf(':') + 2));
						lic.loanedCapacity = 0;
						lic.yMax = 0;
						if (isCALUsage) {
							lic.currentUsage = arr[CURRENT_STR - 1];
							lic.currentUsage = lic.currentUsage.substr(lic.currentUsage.indexOf(':') + 2);

							lic.currentUsage = parseFloat(lic.currentUsage);
							lic.totalCapacity = parseFloat(lic.totalCapacity);
							lic.totalCapacity = lic.totalCapacity > 0 ? lic.totalCapacity : -2;
						} else {
							lic.loanedCapacity = arr[LOAN_STR];
							lic.loanedCapacity = lic.loanedCapacity.substr(lic.loanedCapacity.indexOf(':') + 2);

							lic.currentUsage = arr[CURRENT_STR];
							lic.currentUsage = lic.currentUsage.substr(lic.currentUsage.indexOf(':') + 2);

							lic.totalCapacity = parseFloat(lic.totalCapacity.substr(0, lic.totalCapacity.indexOf(' ')));
							lic.loanedCapacity = parseFloat(lic.loanedCapacity.substr(0, lic.loanedCapacity.indexOf(' ')));
							lic.currentUsage = parseFloat(lic.currentUsage.substr(0, lic.currentUsage.indexOf(' ')));

							lic.growthRate = lic.growthRate * 1024 * 1024 * 1024;
							lic.totalCapacity = lic.totalCapacity > 0 ? lic.totalCapacity * 1024 * 1024 * 1024 * 1024 : -2;
						}
					}
					var peak = {};
					peak.startDate = new Date(moment(row['X-Value']).unix() * 1000);
					peak.endDate = new Date(
						moment(row['X-Value'])
							.endOf('month')
							.unix() * 1000
					);
					peak.peakUsage = arr[PEAK_STR];
					peak.peakUsage = peak.peakUsage.substr(peak.peakUsage.indexOf(':') + 2);
					if (isCALUsage) {
						peak.peakUsage = parseFloat(peak.peakUsage);
					} else {
						peak.peakUsage = parseFloat(peak.peakUsage.substr(0, peak.peakUsage.indexOf(' ')));
						peak.peakUsage = peak.peakUsage * 1024 * 1024 * 1024 * 1024;
					}
					if (peak.peakUsage > lic.yMax) {
						lic.yMax = peak.peakUsage;
					}
					lic.details.push(peak);
				}
				return licMap;
			}

			function processCommCellLicenceType(lic) {
				var yMax = lic.yMax;
				var type = lic.gType;
				var capacity = 0;
				var totalCapacity = 0;
				var isAgentBased = false;
				var details = lic.details;
				var currentUsage = lic.currentUsage;
				if (lic.totalCapacity > 0) {
					capacity = lic.totalCapacity;
				} else if (lic.totalCapacity < 0) {
					isAgentBased = true;
					if (lic.growthRate > 0) {
						var bytesPerDay = lic.growthRate;
						var next6Months = bytesPerDay * 6 * 30;
						var baseUsage = 0;
						if (details.length > 0) {
							//get max of current and last full month
							baseUsage = details[details.length - 1].peakUsage;
							var lastFullMonth = 0;
							if (details.length > 1) {
								lastFullMonth = details[details.length - 2].peakUsage;
							}
							baseUsage = Math.max(baseUsage, lastFullMonth);
						}
						capacity = baseUsage + next6Months;
					} else {
						capacity = yMax;
					}
				}
				var ONE_TB = 1024 * 1024 * 1024 * 1024;

				if (!isCALUsage(type) && capacity > 0 && capacity < ONE_TB) {
					capacity = ONE_TB;
				}
				if (yMax < capacity) {
					yMax = capacity;
				}
				var minFactor = ONE_TB;
				var unitStr = 'TB';
				if (isCALUsage(type)) {
					minFactor = reportsDashboardFactory.getCountMinFactor(yMax);
					unitStr = reportsDashboardFactory.getCountUnitStringFull(yMax);
				}
				totalCapacity = capacity;
				yMax /= minFactor;
				yMax += lic.loanedCapacity; //add loaned capacity to max. Loaned capacity is in TB
				capacity /= minFactor;
				if (isAgentBased) {
					capacity = Math.ceil(capacity);
				}

				var UsageNoteVal = 0;
				var usedData = 0;
				lic.capacityPriority = 0;
				if (lic.totalCapacity > 0) {
					//if (data.getMaxEntityValue() > 0) { //Capacity Based

					var thresholdValue = lic.totalCapacity;
					var currentPeriodData = details[details.length - 1].peakUsage;
					var usage2 = 0;
					if (details.length > 1) {
						usage2 = details[details.length - 2].peakUsage;
					}
					currentPeriodData = Math.max(currentPeriodData, usage2);
					usedData = currentPeriodData;
					var available = thresholdValue - currentPeriodData;
					var percent = (currentPeriodData / minFactor / capacity) * 100;
					var bytePerDay = lic.growthRate;
					if (bytePerDay > 0) {
						var currentPeriodDay = details[details.length - 1].endDate;
						if (available > 0) {
							var days = available / bytePerDay;
							if (days > 36500) {
								//check 100 years
								days = 36500;
							}
							var m = moment(currentPeriodDay).add('day', days);
							var numMonths = getNumberOfMonths(m);
							lic.capacityPriority = numMonths * 30;
							if (numMonths > 18) {
								lic.usageNote = cvLoc('capacity.label.EnoughCapacity');
							} else if (numMonths >= 12) {
								lic.usageNote = cvLoc('capacity.label.OutOfCapacityAtLeast12Months');
							} else if (numMonths >= 2 && numMonths < 12) {
								lic.usageNote = cvLoc('capacity.label.OutOfCapacityInMonths', numMonths);
							} else if (numMonths == 1) {
								lic.usageNote = cvLoc('capacity.label.OutOfCapacityInAMonth');
							} else {
								lic.usageNote = cvLoc('capacity.label.OutOfCapacityLessThanMonth');
							}
						} else if (currentUsage != null && currentUsage < capacity) {
							var today = moment();
							if (today.date() > 25) {
								today = today.add('month', 1); //next month
								lic.capacityPriority = 30;
							}
							lic.usageNote = cvLoc('capacity.label.OutOfCapacityMessageByCurrentUsage', today.format('MMM YYYY'));
						} else {
							currentPeriodData += currentPeriodData * 0.35;
							currentPeriodData /= minFactor;
							lic.usageNote = cvLoc('capacity.label.OverCapacityMessage', currentPeriodData, unitStr);
						}
					} else if (available < 0) {
						if (currentUsage != null && currentUsage < capacity) {
							var today = moment();
							if (today.date() > 25) {
								today = today.add('month', 1); //next month
								lic.capacityPriority = 30;
							}
							lic.usageNote = cvLoc('capacity.label.OutOfCapacityMessageByCurrentUsage', today.format('MMM YYYY'));
						} else {
							lic.usageNote = cvLoc('capacity.label.NotEnoughCapacity');
						}
					} else {
						lic.capacityPriority = DEFAULT_PRIORITY;
						lic.usageNote = cvLoc('capacity.label.EnoughCapacity');
					}

					// when there is no positive or negative growth rate, no need to show estimation
					if (Math.abs(bytePerDay) > 0) {
					}
				} else if (lic.totalCapacity < -1) {
					//Agent Based
					lic.capacityPriority = DEFAULT_PRIORITY;
					totalCapacity = 0;
					if (yMax > 0) {
						if (capacity > 0) {
							lic.usageNote = cvLoc('capacity.label.EstimatedCapacity', capacity, unitStr);
						}
					} else {
						lic.usageNote = cvLoc('capacity.label.NoNeedPurchageCapacity');
					}
				}
				if (lic.loanedCapacity > 0) {
					lic.loanedCapacityLabel = cvLoc('capacity.label.LoanedCapacity', lic.loanedCapacity, unitStr);
				}
				lic.totalCapacity = totalCapacity / minFactor;
				lic.details = undefined;
			}

			//--------------Capacity Dials----------------------

			function handleCurrentCapacityUsage(licData) {
				var dialEnum = {
					CommvaultComplete: 'commvaultComplete',
					DataProtection: 'protectionEnterprise',
					Archive: 'archiveEnterprise',
					Sockets: 'vmsockets',
					VMs: 'protectedvms',
					Users: 'users'
				};
				var dials = {
					commvaultComplete: getEmptyCapacityObject(cvLoc('capacity.title.commvaultcomplete'), true),
					protectionEnterprise: getEmptyCapacityObject(cvLoc('capacity.title.backkup'), true),
					archiveEnterprise: getEmptyCapacityObject(cvLoc('capacity.title.archive'), true),
					vmsockets: getEmptyCapacityObject(cvLoc('capacity.title.vMSockets'), false),
					protectedvms: getEmptyCapacityObject(cvLoc('capacity.title.protectedVMs'), false),
					users: getEmptyCapacityObject(cvLoc('capacity.title.users'), false)
				};

				angular.forEach(licData, function(license, name) {
					if (dials[license.gType]) {
						dials[license.gType].gType = name;
						dials[license.gType].used.value = parseFloat(parseFloat(license.currentUsage).toFixed(2));
						dials[license.gType].total.value = parseFloat(license.totalCapacity);
						dials[license.gType].dialTotal.value =
							parseFloat(license.totalCapacity) + parseFloat(license.loanedCapacity);
						dials[license.gType].usageNote = license.usageNote;
						dials[license.gType].loanNote = '';
						dials[license.gType].loanedCapacity = 0;
						if (license.loanedCapacity > 0) {
							dials[license.gType].loanNote = '*' + license.loanedCapacityLabel; //Includes loaned capacity of " +license.loanedCapacity + " TB";
							dials[license.gType].loanedCapacity = license.loanedCapacity;
						}
					}
				});
				angular.forEach(dials, function(dial, index) {
					if (dial.isDataSize) {
						dial.used.formattedValue = _.round(dial.used.value, 2);
						dial.used.unit = 'TB';

						dial.total.formattedValue = _.round(dial.total.value, 2);
						dial.total.unit = 'TB';

						dial.dialTotal.formattedValue = _.round(dial.dialTotal.value, 2);
						dial.dialTotal.unit = 'TB';

						if (dial.loanedCapacity > 0) {
							dial.dialTotal.unit += '*';
						}
					} else {
						var usedObj = reportsDashboardFactory.autoConvertCount(dial.used.value);
						dial.used.formattedValue = usedObj.value;
						dial.used.unit = usedObj.unit ? usedObj.unit : '';

						var totalObj = reportsDashboardFactory.autoConvertCount(dial.total.value);
						dial.total.formattedValue = totalObj.value;
						dial.total.unit = totalObj.unit ? totalObj.unit : '';
					}
				});
				var chartIndex = 0;
				if (dials[dialEnum.CommvaultComplete].total.value > 0 || dials[dialEnum.CommvaultComplete].loanedCapacity > 0) {
					//commvault complete capacity is present
					if (
						dials[dialEnum.CommvaultComplete].total.value <= 0 &&
						!dials[dialEnum.CommvaultComplete].loanedCapacity > 0
					) {
						//Show some messages
						self.dial1 = {
							name: cvLoc('capacity.title.commvaultcomplete'),
							gType: dials[dialEnum.CommvaultComplete].gType
						};
						if (
							dials[dialEnum.CommvaultComplete].usageNote != undefined &&
							dials[dialEnum.CommvaultComplete].usageNote != ''
						) {
							self.dial1.note1 = dials[dialEnum.CommvaultComplete].usageNote;
						} else {
							self.dial1.note1 = cvLoc('capacity.label.CapacityLicenseNotUsed');
						}
					} else {
						chartIndex++;
						var stats = dials[dialEnum.CommvaultComplete];
						stats.isSingleDial = true;
						processWidgetData(stats, chartIndex);
					}
				} else if (
					dials[dialEnum.DataProtection].total.value != 0 ||
					dials[dialEnum.Archive].total.value != 0 ||
					dials[dialEnum.DataProtection].loanedCapacity > 0 ||
					dials[dialEnum.Archive].loanedCapacity > 0
				) {
					//capacity is present, show DP and archive charts just like before
					if (dials[dialEnum.DataProtection].total.value == 0 && !dials[dialEnum.DataProtection].loanedCapacity > 0) {
						//Show some messages
						self.dial1 = {
							name: cvLoc('capacity.title.backkup'),
							gType: dials[dialEnum.DataProtection].gType
						};
						if (
							dials[dialEnum.DataProtection].usageNote != undefined &&
							dials[dialEnum.DataProtection].usageNote != ''
						) {
							self.dial1.note1 = dials[dialEnum.DataProtection].usageNote;
						} else {
							self.dial1.note1 = cvLoc('capacity.label.BackupLicenseNotUsed');
						}
					} else {
						//draw chart
						chartIndex++;
						var stats = dials[dialEnum.DataProtection];
						processWidgetData(stats, chartIndex);
					}
					if (dials[dialEnum.Archive].total.value == 0 && !dials[dialEnum.Archive].loanedCapacity > 0) {
						//Show some messages
						self.dial2 = {
							name: cvLoc('capacity.title.archive'),
							gType: dials[dialEnum.Archive].gType
						};
						if (dials[dialEnum.Archive].usageNote != undefined && dials[dialEnum.Archive].usageNote != '') {
							self.dial2.note2 = dials[dialEnum.Archive].usageNote;
						} else {
							self.dial2.note2 = cvLoc('capacity.label.ArchiveLicenseNotUsed');
						}
					} else {
						//draw chart
						chartIndex++;
						var stats = dials[dialEnum.Archive];
						processWidgetData(stats, chartIndex);
					}
				} else {
					//		No capacity present. Look for E-SIM/C-SIM licenses in following order and award them dials
					//		a.       Sockets
					//		b.      VMs
					//		c.       End users (basic)
					//		d.      Archive mailboxes
					var stats = getLicenseUsage(dials, dialEnum.Sockets);
					if (stats != null) {
						//draw chart
						chartIndex++;
						processWidgetData(stats, chartIndex);
					}
					stats = getLicenseUsage(dials, dialEnum.VMs);
					if (stats != null) {
						//draw chart
						chartIndex++;
						processWidgetData(stats, chartIndex);
					}
					if (chartIndex < 2) {
						stats = getLicenseUsage(dials, dialEnum.Users);
						if (stats != null) {
							//draw chart
							chartIndex++;
							processWidgetData(stats, chartIndex);
						}
					}
					if (chartIndex < 2) {
						stats = getLicenseUsage(dials, dialEnum.Mailboxes);
						if (stats != null) {
							//draw chart
							chartIndex++;
							processWidgetData(stats, chartIndex);
						}
					}
					if (chartIndex == 0) {
						//If none are found,  show predictive capacity dials like today.  (Since we assume this is ancient customer with agent based licensing)
						self.dial1 = {
							name: cvLoc('capacity.title.backkup'),
							gType: dials[dialEnum.DataProtection].gType
						};
						if (dials[dialEnum.DataProtection].usageNote != '') {
							self.dial1.note1 = dials[dialEnum.DataProtection].usageNote;
						} else {
							self.dial1.note1 = cvLoc('capacity.label.CapacityLicenseNotUsed');
						}

						self.dial2 = {
							name: cvLoc('capacity.title.archive'),
							gType: dials[dialEnum.Archive].gType
						};
						if (dials[dialEnum.Archive].usageNote != '') {
							self.dial2.note2 = dials[dialEnum.Archive].usageNote;
						} else {
							self.dial2.note2 = cvLoc('capacity.label.CapacityLicenseNotUsed');
						}
					} else if (chartIndex == 1) {
						//center the chart
					}
				}
			}

			function getLicenseUsage(data, licenseName) {
				var retVal = null;
				angular.forEach(data, function(lic, index) {
					if (lic.name == licenseName) {
						retVal = lic;
					}
				});
				return retVal;
			}

			function processWidgetData(dial, type) {
				var chartData = [];
				var total = dial.total.value - dial.used.value;
				if (total < 0) {
					total = 0;
				}
				var chartTooltip = cvLoc(
					'capacity.tooltip.purchased',
					reportsDashboardFactory.getFormattedNumberNA(dial.total.formattedValue),
					(dial.loanedCapacity ? dial.total.unit.substr(0, 3) : dial.total.unit).trim()
				);
				if (dial.loanedCapacity) {
					chartTooltip =
						chartTooltip +
						'<br/>' +
						cvLoc('capacity.tooltip.loaned', reportsDashboardFactory.getFormattedNumberNA(dial.loanedCapacity), 'TB');
				}
				chartTooltip =
					chartTooltip +
					'<br/>' +
					cvLoc(
						'capacity.tooltip.used',
						reportsDashboardFactory.getFormattedNumberNA(dial.used.formattedValue),
						dial.used.unit
					);

				chartData.push({
					name: 'usage',
					y: dial.used.value
				});
				chartData.push({
					name: 'purchased',
					y: total
				});
				chartData.push({
					name: 'loaned',
					y: dial.loanedCapacity
				});
				var usagePercent = Math.round(((100 * dial.used.value) / dial.total.value) * 100) / 100; //actual usage in percent

				var usageColor = '#e3b56b';
				if (dial.total.value == 0) {
					//New licenses. These are consumption levels only.  Not licensed entities.
					usageColor = ChartColors.LICENSE_NOT_APPLIED;
				} else if (dial.total.value < 0) {
					amount2 = 110;
					usageColor = ChartColors.WITHIN_CAPACITY; // Unlimited
				} else if (usagePercent < 90) {
					usageColor = ChartColors.WITHIN_CAPACITY;
				} else if (usagePercent >= 90 && usagePercent <= 100) {
					usageColor = ChartColors.WARNING;
				} else {
					usageColor = ChartColors.OVER_CAPACITY;
				}
				var totalColor = ChartColors.CAPACITY_TOTAL;
				var loanedColor = dial.loanedCapacity > 0 ? ChartColors.LOANED_CAPACITY : ChartColors.CAPACITY_TOTAL;

				var dialColors = [];
				dialColors.push(usageColor);
				dialColors.push(totalColor);
				dialColors.push(loanedColor);
				dial.dialColors = dialColors;
				var chartConfig = {
					options: {
						chart: {
							type: 'pie',
							margin: [0, 0, 0, 0]
						},
						tooltip: {
							headerFormat: '',
							useHTML: true,
							style: {
								fontSize: '0.75em'
							},
							formatter: function() {
								return chartTooltip;
							}
						},
						plotOptions: {
							pie: {
								startAngle: -150,
								endAngle: 150,
								allowPointSelect: false,
								cursor: 'pointer',
								dataLabels: {
									enabled: false
								},
								showInLegend: true,
								events: {
									click: function() {
										if (!cv.isCometApp) {
											window.location.href = self.data.titleDetailsLink;
										}
									}
								}
							},
							series: {
								states: {
									hover: {
										enabled: false
									}
								}
							}
						},
						legend: {
							enabled: false
						},
						exporting: {
							enabled: false
						},
						colors: dialColors
					},
					title: {
						text: ''
					},
					subtitle: {
						text: ''
					},

					credits: {
						enabled: false
					},

					series: [
						{
							colorByPoint: true,
							animation: true,
							innerSize: '80%',
							data: chartData
						}
					],
					size: {
						width: 110,
						height: 110
					}
				};
				self['chartConfig' + type] = chartConfig;
				self['dial' + type] = dial;
			}

			function getEmptyCapacityObject(name, isDataSize) {
				return {
					name: name,
					usageNote: '',
					isDataSize: isDataSize,
					used: {
						value: 0,
						formattedValue: 0,
						unit: ''
					},
					total: {
						value: 0,
						formattedValue: 0,
						unit: ''
					},
					dialTotal: {
						value: 0,
						formattedValue: 0,
						unit: ''
					}
				};
			}

			function getNumberOfMonths(outOfCapacityCal) {
				var today = moment();
				var outOfCapacityDay = moment(outOfCapacityCal);
				return outOfCapacityDay.diff(today, 'months');
			}

			function isCALUsage(gType) {
				if (
					gType == 'users' ||
					gType == 'vmsockets' ||
					gType == 'protectedvms' ||
					gType == 'archived_mailboxes' ||
					gType == 'ci_objects'
				) {
					return true;
				}
				return false;
			}
		};
	}
]);
