import { reportsDashboardModule } from 'common/js/modules';

import 'modules/reports/js/constants/reports.constants.js';
import 'modules/reports/js/directives/cv-dashboard-tile-directive.js';
import 'modules/reports/js/controllers/dashboardSecurityModal.ctrl.js';

reportsDashboardModule.controller('DashboardController', [
	'$stateParams',
	'$state',
	'$location',
	'$window',
	'$dialogs',
	'$rootScope',
	'$scope',
	'$uibModal',
	'userPrefService',
	'reportsDashboardService',
	'reportService',
	'serverService',
	'settingsService',
	'DashboardSettings',
	'ReportsConstants',
	'reportsDashboardFactory',
	'cvToaster',
	'cvLoc',
	'cvUtil',
	'$filter',
	'$timeout',
	function DashboardController(
		$stateParams,
		$state,
		$location,
		$window,
		$dialogs,
		$rootScope,
		$scope,
		$uibModal,
		userPrefService,
		reportsDashboardService,
		reportService,
		serverService,
		settingsService,
		DashboardSettings,
		ReportsConstants,
		reportsDashboardFactory,
		cvToaster,
		cvLoc,
		cvUtil,
		$filter,
		$timeout
	) {
		var self = this;
		self.dashboardError = false;
		self.dashboardErrorMsg = cvLoc('generic_error');
		self.dashboardEditor = !cvConfig.hideDashboardEdit && cv.isAdminOnCommcellLevel && !cv.isCometApp;
		self.loaded = false;
		self.isEditing = false;
		self.tileChanged = false;
		self.nameChanged = false;
		self.isCustomDashboard = false;
		self.builtInCustom = false;

		var defaultIds = [
			'commcell',
			'hyperscalecommcell',
			'hyperscale',
			'virtualization',
			'endpoint',
			'tenant',
			'activate',
			'Replication'
		];
		self.data = {};
		self.dashboardData = undefined;
		self.dashboardId = $stateParams.dashboardId === 'disasterRecovery' ? 'Replication' : $stateParams.dashboardId;
		const DASHBOARD_TILES = 'dashboardTiles';
		const MAX_COL = 12;
		const blockHeight = 160;
		self.showCommCellInfo = cv.isAdmin && !cv.isCometApp;
		self.tiles = [];
		self.gridsterOpts = {
			maxRows: 999999,
			columns: 12,
			rowHeight: 65,
			margins: [7, 7],
			defaultSizeX: 2,
			defaultSizeY: 1,
			pushing: false,
			floating: true,
			swapping: false,

			allowOverlapping: false,
			mobileBreakPoint: 1000,
			resizable: {
				enabled: false,
				start: function(e, ui, $widget) {
					self.tileChanged = true;
				}
			},
			draggable: {
				enabled: false,
				start: function(event, ui) {
					self.tileChanged = true;
				}
			},

			colWidth: 'auto',
			deviceType: 'desktop',
			isLayoutAvailable: true,
			sparse: true,
			mobileModeEnabled: false
		};

		self.deleteText = cvLoc('dashboard.delete');
		self.deleteTooltip = cvLoc('dashboard.deleteDashboard');

		self.$filter = $filter;

		$scope.$on('cancelEditDashboard', function() {
			self.cancelEdit(false);
		});

		$scope.$on('$locationChangeStart', function(event, nextUrl, current) {
			if (self.isEditing) {
				$dialogs.confirm(cvLoc('label.confirmNavigationTitle'), cvLoc('label.confirmDashboardNavDetails'), {
					yesFunction: function(e, d, f) {
						self.cancelEdit();
						if (nextUrl) {
							$window.location.href = nextUrl;
						}
					},
					noFunction: function() {
						event.preventDefault();
					}
				});
				event.preventDefault();
				return;
			}
		});

		self.$onInit = function() {
			self.overrides = {};
			if (self.dashboardId === 'commcell' && cv.sessionContext.operatorCompanyId) {
				$location.url(appUtil.appRoot);
				return;
			}
			if (cv.sessionContext[ReportsConstants.SESSION_DASHBOARD_OVERRIDES]) {
				self.overrides = cv.sessionContext[ReportsConstants.SESSION_DASHBOARD_OVERRIDES];
				self.loadDashboard();
			} else {
				serverService
					.getGlobalParam('dashboardOverrides', true)
					.then(function(data) {
						if (data.data && data.data !== 'null') {
							self.overrides = data.data;
							cv.sessionContext[ReportsConstants.SESSION_DASHBOARD_OVERRIDES] = self.overrides;
						}
						self.loadDashboard();
					})
					.catch(function(e) {
						if (err.xhrStatus !== 'abort') {
							self.loadDashboard();
							console.log('Failed to retrieve dashboard override.');
						}
					});
			}
		};

		self.loadDashboard = function() {
			console.log('loadDashboard');
			if (self.dashboardId && defaultIds.indexOf(self.dashboardId) === -1) {
				console.log('loading custom report');
				self.loadCustomDashboard(self.dashboardId);
			} else if (self.dashboardId) {
				if (self.overrides[self.dashboardId]) {
					console.log('override found');
					self.loadCustomDashboard(self.overrides[self.dashboardId]);
				} else {
					console.log('built in, no override');
					self.dashboardType = self.dashboardId;
					switch (self.dashboardId) {
						case 'tenant':
							self.dashboardType = ReportsConstants.DASHBOARD_TYPES.SERVER;
							break;
						case 'hyperscale':
							self.dashboardType = ReportsConstants.DASHBOARD_TYPES.APPLIANCE;
							break;
						case 'hyperscalecommcell':
							self.dashboardType = ReportsConstants.DASHBOARD_TYPES.APPLIANCE_COMMCELL;
							break;
					}

					let dashTileData = reportsDashboardFactory.getDefaultDashboardTiles(
						self.dashboardId === 'Replication' ? 'disasterRecovery' : self.dashboardId
					);
					angular.copy(dashTileData.tiles, self.tiles);

					if (
						self.dashboardType === ReportsConstants.DASHBOARD_TYPES.APPLIANCE_COMMCELL ||
						self.dashboardType === ReportsConstants.DASHBOARD_TYPES.COMMCELL
					) {
						self.customClass = 'overview-content';
					}

					_loadData(self.tiles);
					self.origTiles = angular.copy(self.tiles);
					reportsDashboardFactory.applyDashboardSizeOverrides(self.tiles, dashTileData.sizeOverrides);
				}
			} else {
				$location.url(appUtil.appRoot);
			}
		};

		self.loadCustomDashboard = function(userReportId) {
			let params = {
				includeDrafts: false,
				reportName: '',
				reportId: userReportId
			};

			self.isCustomDashboard = true;

			reportService
				.getCustomReport(params)
				.then(function(data) {
					if (data) {
						data = data.data;
					}

					self.dashboardName = data.report.customReportName;

					if (data.builtIn) {
						self.deleteText = cvLoc('dashboard.revert');
						self.deleteTooltip = cvLoc('dashboard.revertTooltip');
						self.builtInCustom = true;
					}
					if (data && data.pages && data.pages.length > 0) {
						self.dashboardData = data;
						let dashboardData = JSON.parse(data.pages[0].body);
						self.dashboardType = dashboardData.dashboardType;
						self.tiles = dashboardData.tiles;
						self.origTiles = angular.copy(self.tiles);
						_loadData(self.tiles);
					} else {
						self.dashboardError = true;
						console.log('Invalid response received - no pages.');
					}
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						self.dashboardError = true;
						self.loaded = true;
						if (err.data) {
							self.dashboardErrorMsg = err.data;
							console.log(err.data);
						} else {
							console.log('Error fetching dashboard data');
						}
					}
				});
		};

		self.setActiveTiles = function() {
			var active = [];

			angular.forEach(self.tiles, function(tile) {
				active.push(tile.tileType);
			});

			$scope.$broadcast('activeTilesChanged', active);
		};

		self.getDashboardData = function() {
			let dData = {};
			let entity = {
				dashboardType: self.dashboardType,
				dashboardName: self.dashboardName,
				tiles: self.tiles
			};

			if (self.dashboardData) {
				///edit
				dData = self.dashboardData;
			} else {
				//new report
				(dData.type = '3'), (dData.reportDescription = 'Command Center Custom Dashboard'), (dData.report = {});
				dData.userReportId = cvUtil.generateUUID();
				dData.guid = dData.userReportId;

				if (!self.isCustomDashboard) {
					dData.visibleAll = true;
					dData.builtIn = true;
				}
			}

			dData.report.customReportName = self.dashboardName;
			dData.reportTitle = {
				text: self.dashboardName
			};
			let tileProp = {
				dashboardType: self.dashboardType
			};
			dData.tileProperties = JSON.stringify(tileProp);
			dData.pages = [
				{
					body: JSON.stringify(entity)
				}
			];
			dData.translations = [
				{
					locale: 'default',
					localeKey: 'RPT_TITLE',
					localeValue: self.dashboardName
				}
			];
			return dData;
		};

		self.saveDashboard = function() {
			let regex = new RegExp('^[0-9]+$');
			if (regex.test(self.dashboardName)) {
				$dialogs.error(cvLoc('label.error'), cvLoc('dashboard.error.dashboardName'));
				return;
			}

			let dData = self.getDashboardData();
			reportService
				.saveReport(dData)
				.then(function(data) {
					self.setGridEditable(false, false);
					cvToaster.showSuccessMessage({
						message: cvLoc('dashboard.layoutSavedMessage')
					});

					reportService
						.deployReport(data.data.report.customReportId)
						.then(function() {
							//overriding a built-in dashboard
							if (!self.isCustomDashboard) {
								self.updateGlobalParams(false, self.dashboardId, dData.userReportId);
							} else {
								$state.reload();
							}
						})
						.catch(function(err) {
							if (err.xhrStatus !== 'abort') {
								console.log('Deploy error');
								console.log(err);
								reportsDashboardFactory.showErrorMessage(cvLoc('dashboard.error.saveAndDeploy'));
							}
						});
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						console.log('Saving error');
						console.log(err);
						reportsDashboardFactory.showErrorMessage(cvLoc('dashboard.error.saveAndDeploy'));
					}
				});
		};

		self.updateGlobalParams = function(deleteOp, dashboardId, userReportId) {
			serverService
				.getGlobalParam('dashboardOverrides', true)
				.then(function(data) {
					let overrides = {};
					if (data.data && data.data !== 'null') {
						overrides = data.data;
					}

					if (!deleteOp) {
						overrides[dashboardId] = userReportId;
					} else {
						delete overrides[dashboardId];
					}

					settingsService
						.setGlobalParam('dashboardOverrides', overrides)
						.then(function(data) {
							cv.sessionContext[ReportsConstants.SESSION_DASHBOARD_OVERRIDES] = overrides;
							console.log('updated global param with dashboard override');
							$state.reload();
						})
						.catch(function(err) {
							if (err.xhrStatus !== 'abort') {
								console.log(err);
								console.log('Failed to update dashboard override');
							}
						});
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						console.log(err);
					}
				});
		};

		self.saveAsDashboard = function() {
			var data = {
				modalTitle: cvLoc('dashboard.clone'),
				cloneMode: true,
				origName: self.dashboardName,
				defaultName: self.dashboardName + ' ' + cvLoc('dashboard.copy'),
				dashboardType: self.dashboardType,
				tiles: self.tiles
			};

			let modalInstance = $uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/reports/partials/newDashboardModal.jsp',
				backdrop: 'static',
				windowClass: 'small-size',
				controller: 'NewDashboardModalController',
				resolve: {
					data: function() {
						return data;
					}
				}
			});

			modalInstance.result.then(function() {
				self.setGridEditable(false, false);
			});
		};

		self.cancelEdit = function(noBroadcast) {
			self.setGridEditable(false, noBroadcast);
			$state.reload();
		};

		self.deleteDashboard = function() {
			if (self.builtInCustom) {
				self.revertToDefault();
				return;
			}
			$dialogs.confirm(cvLoc('dashboard.deleteDashboard'), cvLoc('dashboard.deleteDashboardConfirmation'), {
				yesFunction: function() {
					let params = {
						reportName: self.dashboardData.report.customReportName,
						reportId: self.dashboardData.report.customReportId
					};

					reportService.deleteReport(params).then(function() {
						cvToaster.showSuccessMessage({
							message: cvLoc('dashboard.dashboardDeleted')
						});
						$location.url(appUtil.appRoot);
					});
				}
			});
		};

		self.revertToDefault = function() {
			$dialogs.confirm(cvLoc('dashboard.revert'), cvLoc('dashboard.resetConfirmation'), {
				yesFunction: function() {
					let params = {
						reportName: self.dashboardData.report.customReportName,
						reportId: self.dashboardData.report.customReportId
					};

					reportService.deleteReport(params).then(function() {
						cvToaster.showSuccessMessage({
							message: cvLoc('dashboard.dashboardReset')
						});
						self.updateGlobalParams(true, self.dashboardId);
					});
				}
			});
		};

		self.removeTile = function(index, event) {
			self.tileChanged = true;
			self.tiles.splice(index, 1);
			self.setActiveTiles();
			event.stopPropagation();
		};

		self.tileDropped = function(dragEl, dropEl, event) {
			let dest = document.getElementById(dropEl);
			let src = document.getElementById(dragEl);

			let drag = angular.element(src);
			let tileType = drag.data('tileType');
			let tile = angular.copy(DashboardSettings.allTiles[tileType]);
			let position = self.calculateTilePosition(event);
			self.tileChanged = true;
			tile.row = position[0];
			tile.col = position[1];
			self.tiles.push(tile);
			_loadTileData(tileType);

			self.setActiveTiles();
			$scope.$apply();
		};

		self.calculateTilePosition = function(event) {
			// calculate row and column to place dragged item
			if (event) {
				var srcTop = event.originalEvent.pageY - $('.gridster').offset().top;
				var srcLeft = event.originalEvent.pageX - $('.gridster').offset().left;
			} else {
				var srcTop = $('.gridster').offset().top;
				var srcLeft = $('.gridster').offset().left;
			}

			// get row height and col width of gridster item
			var rowHeight = self.gridsterOpts.rowHeight;
			var colWidth = $('.gridster').width() / self.gridsterOpts.columns;

			// calculate rows and cols
			var row = srcTop < rowHeight ? 0 : Math.floor(srcTop / rowHeight);
			var column = srcLeft < colWidth ? 0 : Math.floor(srcLeft / colWidth);
			return [row, column];
		};

		self.setGridEditable = function(editable, noBroadcast) {
			self.isEditing = editable;
			self.gridsterOpts.draggable.enabled = editable;
			self.gridsterOpts.resizable.enabled = editable;
			let activeTiles = [];
			if (editable) {
				angular.element('.dashboard').addClass('dashboard-edit');
				angular.forEach(self.tiles, function(tile) {
					activeTiles.push(tile.tileType);
				});
			} else {
				angular.element('.dashboard').removeClass('dashboard-edit');
			}

			let editData = {
				isEdit: editable,
				dashboardType: self.dashboardType,
				activeTiles: activeTiles
			};

			if (!noBroadcast) {
				$rootScope.$broadcast('editDashboard', editData);
			}
		};

		self.isEmptyDashboard = function() {
			return self.isEditing && self.loaded && self.tiles.length == 0;
		};

		self.showSecurityModal = function() {
			$uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/reports/partials/dashboardSecurityModal.jsp',
				backdrop: 'static',
				controller: 'DashboardSecurityController',
				controllerAs: 'securityCtrl',
				resolve: {
					data: function() {
						return {
							dashboardReportId: self.dashboardData.report.customReportId,
							dashboardName: self.dashboardData.report.customReportName
						};
					}
				}
			});
		};

		self.doSyncCall = () => {
			reportsDashboardService.doSyncCallForActivateSummary().then(data => {
				if (data.data && data.data.messageName === 'syncInitiated') {
					cvToaster.showSuccessMessage({
						message: cvLoc('message.synDataCompleted')
					});
					//refresh the data after sync operation
					$timeout(() => $state.reload(), 30000);
				} else {
					cvToaster.showErrorMessage({
						message: cvLoc('message.synDataFailed')
					});
				}
			});
		};

		self.getMostRecentCollectionTime = () => {
			const lastSyncTime = sessionStorage.getItem('lastSyncTime');
			let collectionTime;
			//update the data collection time only when both sda and fso data collection time are greater than lastSyncTime
			if (!lastSyncTime || (lastSyncTime < self.sdaLastCollectionTime && lastSyncTime < self.fsoLastCollectionTime)) {
				collectionTime =
					self.sdaLastCollectionTime > self.fsoLastCollectionTime
						? self.sdaLastCollectionTime
						: self.fsoLastCollectionTime;
				sessionStorage.setItem('dataCollectionTime', collectionTime);
			} else {
				collectionTime = sessionStorage.getItem('dataCollectionTime');
			}
			return self.$filter('date')(collectionTime, 'MM/dd/yy, hh:mm:ss a');
		};

		function _loadData(tiles) {
			self.data.entities = {};
			angular.forEach(tiles, function(tile) {
				_loadTileData(tile.tileType);
			});
			self.loaded = true;
			if (cv.sessionContext[ReportsConstants.SESSION_EDITDASHBOARD_MODE]) {
				self.setGridEditable(true, false);
				cv.sessionContext[ReportsConstants.SESSION_EDITDASHBOARD_MODE] = false;
			}
		}

		function _loadTileData(tileType) {
			if (tileType === 'SLA' && !self.data.slaData) {
				self.data.slaData = {
					dashboardType:
						self.dashboardType === ReportsConstants.DASHBOARD_TYPES.SERVER
							? ReportsConstants.DASHBOARD_TYPES.COMMCELL
							: self.dashboardType
				};
			} else if (tileType === 'LAPTOP_COUNT' && !self.data.entities.laptops) {
				self.data.entities.laptops = _getEntityObject(
					cvLoc('entity.label.laptops'),
					0,
					appUtil.appRoot + 'common/img/laptop.svg',
					appUtil.appRoot + '#/devices',
					null,
					null
				);
				_getClientCounts();
			} else if (
				(tileType === 'USER_COUNT' && !self.data.entities.users) ||
				(tileType === 'ALERT_COUNT' && !self.data.entities.alerts)
			) {
				self.data.entities.users = _getEntityObject(
					cvLoc('entity.label.endpointUsers'),
					0,
					appUtil.appRoot + 'common/img/users.svg',
					appUtil.appRoot + '#/users?view=laptopUsers',
					null,
					null
				);
				self.data.entities.alerts = _getEntityObject(
					cvLoc('entity.label.criticalEndpointAlerts'),
					0,
					appUtil.appRoot + 'common/img/critical-alerts.svg',
					appUtil.appRoot + '#/triggeredAlerts?view=critical-alerts',
					cvLoc('entity.critical.alerts.tooltip'),
					null
				);
				_getOverviewStats();
			}
		}

		function _getEntityObject(txt, val, img, link, tooltip, unit) {
			return {
				text: txt,
				value: 0,
				unit: unit,
				imgSrc: img,
				detailsLink: link,
				tooltip: tooltip ? tooltip : txt,
				showLoader: true
			};
		}

		//Enpoint Users and Enpoint Alerts tiles will always show endpoint data
		function _getOverviewStats() {
			reportsDashboardService
				.getOverviewData(ReportsConstants.DASHBOARD_TYPES.ENDPOINT)
				.then(function(data) {
					if (data) {
						data = data.data;
					}
					data = reportsDashboardFactory.getJsonFromDSResult(data);
					angular.forEach(data, function(row, i) {
						var entityName = row['PropertyType'].toLowerCase();
						if (entityName == 'laptops') {
							return; //Will use new API to get client entity counts
						}
						var entity = self.data.entities[entityName];
						if (entity) {
							entity.value = reportsDashboardFactory.getFormattedNumberNA(row['PropertyCount']);
						}
					});
					self.data.entities.users.showLoader = false;
					self.data.entities.alerts.showLoader = false;
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						console.log(err);
						reportsDashboardFactory.showErrorMessage(
							err && err.errorMessage ? err.errorMessage : cvLoc('generic_error')
						);
					}
				});
		}

		function _getClientCounts() {
			var entityTypes = 'laptop';
			reportsDashboardService
				.getClientsCounts(entityTypes)
				.then(function(data) {
					if (data) {
						data = data.data;
					}
					var entity = self.data.entities['laptops'];
					if (data && data.laptopCount) {
						entity.value = reportsDashboardFactory.getFormattedNumberNA(data.laptopCount > 0 ? data.laptopCount : 0);
					}
					entity.showLoader = false;
				})
				.catch(function(err) {
					if (err.xhrStatus !== 'abort') {
						console.log(err);
						reportsDashboardFactory.showErrorMessage(
							err && err.errorMessage ? err.errorMessage : cvLoc('generic_error')
						);
					}
				});
		}
	}
]);

export default reportsDashboardModule;
