import { policyModule } from 'common/js/modules';
import 'modules/policies/schedulePolicies/js/controllers/addScheduleToPolicy.ctrl.js';
import 'modules/policies/schedulePolicies/js/controllers/addSchedulePolicy.ctrl.js';
import 'modules/policies/schedulePolicies/js/services/schedulePolicy.svc.js';
import 'adminConsole/js/services/clientGroups.svc.js';
import 'modules/policies/schedulePolicies/js/constants/backupLevel.constants.js';
import 'modules/policies/schedulePolicies/js/constants/ddbverification.constants.js';
import 'vsa/js/factories/blockLevelReplication.factory.js';

var schedulePolicyMod = policyModule;

schedulePolicyMod.factory('schedulePolicyFactory', [
	'cvUtil',
	'cvLoc',
	'schedulePolicyService',
	'$uibModal',
	'$q',
	'DDBVERIFICATION_LEVEL',
	'BACKUP_TYPES',
	'$state',
	'blrUIFactory',
	function(
		cvUtil,
		cvLoc,
		schedulePolicyService,
		$modal,
		$q,
		DDBVERIFICATION_LEVEL,
		BACKUP_TYPES,
		$state,
		blrUIFactory
	) {
		var factory = {
			//Strategy pattern(s) to avoid multiple if/else blocks
			scheduleOptionsTemplateMap: {
				DATA_PROTECTION: 'modules/policies/schedulePolicies/partials/dataProtectionOptions.jsp',
				AUX_COPY: 'modules/policies/schedulePolicies/partials/auxCopyOptions.jsp',
				SINGLE_INSTANCING: 'modules/policies/schedulePolicies/partials/backupCopyOptions.jsp',
				CONTENT_INDEX: 'modules/policies/schedulePolicies/partials/offlineContentIndexOptions.jsp',
				DATA_VERIFICATION: 'modules/policies/schedulePolicies/partials/dataVerificationOptions.jsp',
				CONTENT_INDEX_SUBCLIENT: 'modules/policies/schedulePolicies/partials/contentIndexingOptions.jsp'
			},
			taskOperationTypeMap: {
				DATA_PROTECTION: 'BACKUP',
				AUX_COPY: 'AUX_COPY',
				SINGLE_INSTANCING: 'SNAP_TO_TAPE',
				CONTENT_INDEX: 'OFFLINE_CONTENT_INDEX',
				CONTENT_INDEX_SUBCLIENT: 'CONTENT_INDEX_FROM_SUBCLIENT',
				DATA_VERIFICATION: 'ARCHIVE_CHECK'
			},
			scheduleModelInitializationMap: {
				DATA_PROTECTION: initModelForDataProtection,
				AUX_COPY: initModelForAuxCopy,
				SINGLE_INSTANCING: initModelBackupCopyModel, //'Single instancing' refers to 'Backup Copy'
				CONTENT_INDEX: initOfflineContentIndexingModel, //'Content_index' refers to "Offline content indexing"
				DATA_VERIFICATION: initModelForDataVerification,
				CONTENT_INDEX_SUBCLIENT: initModelForContentIndexing // 'CONTENT_INDEX_SUBCLIENT' refers to 'Content Indexing'
			},
			subTaskConversionMap: {
				DATA_PROTECTION: convertModelToSubTaskForDataProtection,
				AUX_COPY: convertModelToSubTaskForAuxCopy,
				SINGLE_INSTANCING: convertModelToSubTaskForBackupCopy, //'Single instancing' refers to 'Backup Copy'
				CONTENT_INDEX: convertModelToSubTaskForOfflineContentIndex, //'Content_index' refers to "Offline content indexing"
				DATA_VERIFICATION: convertModelToSubTaskForDataVerification,
				CONTENT_INDEX_SUBCLIENT: convertModelToSubTaskForContentIndexing
			}
		};

		factory.loadSchedulePolicies = function() {
			return schedulePolicyService.loadSchedulePolicies();
		};

		factory.loadAssociatedSchedules = function(taskId) {
			return schedulePolicyService.loadAssociatedSchedules(taskId);
		};

		factory.constructTreeNodesForStoragePolicy = function(storagePolicies) {
			return [
				{
					value: 0,
					label: cvLoc('label.storagePolicies'),
					children: storagePolicies.map(function(sp) {
						return {
							value: sp.storagePolicyId,
							label: sp.storagePolicyName,
							selected: false
						};
					})
				}
			];
		};

		factory.constructTreeNodesForDDBEngines = function(ddbEngines) {
			return [
				{
					value: 0,
					label: cvLoc('label.deduplicationEngines'),
					children: ddbEngines.map(function(eng) {
						return {
							value: eng.sp.id,
							label: eng.sp.name,
							selected: false,
							children: [
								{
									value: eng.copy.id,
									label: eng.copy.name,
									selected: false,
									children: [
										{
											value: eng.storeId,
											label: eng.storeName + ' [ID:' + eng.storeId + ']',
											selected: false
										}
									]
								}
							]
						};
					})
				}
			];
		};

		factory.createSchedulePolicyRequest = function(model) {
			var schedulePolicyInfo = {
				task: model.task,
				schedules: model.schedules,
				associations: []
			};

			if (model.task.policyType == 'DATA_VERIFICATION') {
				schedulePolicyInfo.associations = factory.getAssociationsFromAdvancedTree(model.ddbAssociationsTreeParams);
			} else if (
				model.task.policyType == 'DATA_PROTECTION' ||
				model.task.policyType == 'CONTENT_INDEX' ||
				model.task.policyType == 'AUX_COPY' ||
				model.task.policyType == 'SINGLE_INSTANCING' ||
				model.task.policyType == 'CONTENT_INDEX_SUBCLIENT'
			) {
				var selectedNodes = _.get(model, 'associationsTreeParams.result.selected', []),
					excludedNodes = _.get(model, 'associationsTreeParams.result.excluded', []);

				selectedNodes.forEach(selectedNode => schedulePolicyInfo.associations.push(selectedNode.entity));
				excludedNodes.forEach(excludedNode => {
					_.set(excludedNode, 'entity.flags.exclude', true);
					schedulePolicyInfo.associations.push(excludedNode.entity);
				});

				if (model.task.policyType == 'DATA_PROTECTION') {
					schedulePolicyInfo.appGroup = getSelectedAppGroupsFromTree(model.agentTypeTree);
				}
			}
			return schedulePolicyInfo;
		};

		factory.getAssociationsFromAdvancedTree = function(treeParams) {
			var associations = [];
			var selectedNodes = _.get(treeParams, 'result.selected', []),
				excludedNodes = _.get(treeParams, 'result.excluded', []);

			selectedNodes.forEach(selectedNode => associations.push(selectedNode.entity));
			excludedNodes.forEach(excludedNode => {
				_.set(excludedNode, 'entity.flags.exclude', true);
				associations.push(excludedNode.entity);
			});
			return associations;
		};

		factory.isRequestValid = function(request, model) {
			if (!request.task.taskName) {
				model.serverMessage = cvUtil.errMsgLoc('error.schedulePolicyName');
				return false;
			} else if (request.schedules.length == 0) {
				model.serverMessage = cvUtil.errMsgLoc('label.associatedScheduleDeleteError');
				model.accordionStatus.schedulesTab = true;
				return false;
			} else if (model.task.policyType == 'DATA_PROTECTION' && angular.isUndefined(request.appGroup)) {
				model.serverMessage = cvUtil.errMsgLoc('error.selectAgents');
				return false;
			}
			return true;
		};

		factory.getScheduleOptionsTemplate = function(policyType, subTask) {
			if (policyType === 'DATA_PROTECTION') {
				if (subTask) {
					if (subTask.operationType === 'BACKUP')
						return appUtil.appRoot + factory.scheduleOptionsTemplateMap.DATA_PROTECTION;
				} else {
					return appUtil.appRoot + factory.scheduleOptionsTemplateMap.DATA_PROTECTION;
				}
			}
			if (!factory.scheduleOptionsTemplateMap.hasOwnProperty(policyType)) {
				return '';
			}
			return appUtil.appRoot + factory.scheduleOptionsTemplateMap[policyType];
		};

		factory.getTaskOperationType = function(policyType, subTask) {
			if (subTask && subTask.operationType) {
				return subTask.operationType;
			}
			if (!factory.taskOperationTypeMap.hasOwnProperty(policyType)) {
				return '';
			}
			return factory.taskOperationTypeMap[policyType];
		};

		factory.getSubTaskType = function(policyType, subTask) {
			if (subTask && subTask.subTaskType) {
				return subTask.subTaskType;
			} else if (policyType == 'DATA_PROTECTION') {
				return 'BACKUP';
			} else {
				return 'ADMIN';
			}
		};

		factory.deleteSchedulePolicy = function(taskId) {
			return schedulePolicyService.deleteSchedulePolicy(taskId);
		};

		factory.addScheduleToPolicy = function(inputParams) {
			var editMode = angular.isDefined(inputParams.scheduleToBeEdited),
				existingSchedules = [],
				deferred = $q.defer();

			if (angular.isArray(inputParams.schedules)) {
				existingSchedules = getScheduleNames(inputParams.schedules); //existingSchedules are needed for checking schedule creation with duplicate names
				if (editMode) {
					var $index = inputParams.schedules.indexOf(inputParams.scheduleToBeEdited);
					existingSchedules.splice($index, 1);
				}
			}

			var modalInstance = $modal.open({
				templateUrl: appUtil.appRoot + 'modules/policies/schedulePolicies/partials/addScheduleToPolicy.jsp',
				backdrop: 'static',
				controller: 'addScheduleToPolicyController',
				controllerAs: 'astpCtrl',
				resolve: {
					inputParams: function() {
						inputParams.subTask = angular.copy(inputParams.scheduleToBeEdited);
						inputParams.existingSchedules = existingSchedules;
						delete inputParams.scheduleToBeEdited;
						delete inputParams.schedules;
						return inputParams;
					}
				}
			});

			modalInstance.result.then(function(subTask) {
				deferred.resolve(angular.copy(subTask));
			});

			return deferred.promise;
		};

		factory.initializeScheduleModel = function(subTaskInfo, policyType) {
			let scheduleModel = {
				schedulePattern: _.get(subTaskInfo, 'schedulePattern'),
				subTask: _.get(subTaskInfo, 'subTask'),
				options: _.get(subTaskInfo, 'options')
			};
			if (factory.scheduleModelInitializationMap.hasOwnProperty(policyType)) {
				factory.scheduleModelInitializationMap[policyType](scheduleModel, subTaskInfo);
			}
			_.set(
				scheduleModel,
				'options.backupOpts.runIncrementalBackup',
				_.get(scheduleModel, 'options.backupOpts.runIncrementalBackup', false)
			);
			return scheduleModel;
		};

		factory.getBLRSpecificOptions = function(subTaskInfo, policyType) {
			let subTaskOperationType = _.get(subTaskInfo, 'subTask.operationType');
			if (policyType === 'DATA_PROTECTION' && subTaskOperationType === 'BLR_OPERATION') {
				let blrModel = { operations: _.get(subTaskInfo, 'options.adminOpts.blockOperation.operations[0]') };
				blrModel.expiryTimeInDH = blrUIFactory.convertSecToDaysHours(blrModel.operations.vmBootInfo.lifeTimeInSec);
				return blrModel;
			}
		};

		function initModelForDataProtection(scheduleModel, subTaskInfo) {
			scheduleModel.backupLevel = _.get(subTaskInfo, 'options.backupOpts.backupLevel', 'FULL');
			scheduleModel.cumulative = _.get(subTaskInfo, 'options.backupOpts.oracleOptions.cumulative', false);
		}

		function initModelForAuxCopy(scheduleModel, subTaskInfo) {
			scheduleModel.maxNumberOfStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				1
			);
			scheduleModel.useMaximumStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				true
			);
			scheduleModel.startNewMedia = _.get(subTaskInfo, 'options.backupOpts.mediaOpt.startNewMedia', false);
			scheduleModel.markMediaFullOnSuccess = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.markMediaFullOnSuccess',
				false
			);
			scheduleModel.useMostRecentBackupForAuxcopy = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMostRecentBackupForAuxcopy',
				false
			);
			scheduleModel.waitForAllParallelCopyResources = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.waitForAllParallelCopyResources'
			);
			scheduleModel.useScallableResourceManagement = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useScallableResourceManagement',
				true
			);
			scheduleModel.ignoreDataVerificationFailedJobs = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.ignoreDataVerificationFailedJobs',
				false
			);
		}

		function initModelBackupCopyModel(scheduleModel, subTaskInfo) {
			scheduleModel.noofJobsToRun = _.get(subTaskInfo, 'options.adminOpts.snapToTapeOption.noofJobsToRun', 1);
			scheduleModel.allowMaximum = _.get(subTaskInfo, 'options.adminOpts.snapToTapeOption.allowMaximum', true);
			scheduleModel.startNewMedia = _.get(subTaskInfo, 'options.backupOpts.mediaOpt.startNewMedia', false);
			scheduleModel.markMediaFullOnSuccess = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.markMediaFullOnSuccess',
				false
			);
		}

		function initOfflineContentIndexingModel(scheduleModel, subTaskInfo) {
			scheduleModel.maxNumberOfStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				1
			);
			scheduleModel.useMaximumStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				true
			);
		}

		function initModelForDataVerification(scheduleModel, subTaskInfo) {
			scheduleModel.mediaAgentId = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.mediaAgent.mediaAgentId',
				0
			);
			scheduleModel.runIncrementalVerification =
				_.get(subTaskInfo, 'options.adminOpts.archiveCheckOption.backupLevel', BACKUP_TYPES.INCREMENTAL) ==
				BACKUP_TYPES.INCREMENTAL;
			scheduleModel.deduplicationLevel = _.get(
				subTaskInfo,
				'options.adminOpts.archiveCheckOption.ddbVerificationLevel',
				DDBVERIFICATION_LEVEL.DDB_VERIFICATION
			);
			let tempReclaimLevel = _.get(subTaskInfo, 'options.adminOpts.archiveCheckOption.defragmentationPercentage', 80);
			if (tempReclaimLevel < 20 || tempReclaimLevel > 80) {
				scheduleModel.reclaimLevel = 1;
			} else {
				scheduleModel.reclaimLevel = (100 - tempReclaimLevel) / 20;
			}
			scheduleModel.maxNumberOfStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				1
			);
			scheduleModel.useMaximumStreams = _.get(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				true
			);
		}

		function initModelForContentIndexing(scheduleModel, subTaskInfo) {
			scheduleModel.backupLevel = _.get(subTaskInfo, 'options.backupOpts.backupLevel', 'INCREMENTAL');
			scheduleModel.autoCopy = _.get(subTaskInfo, 'options.backupOpts.dataOpt.autoCopy', false);
			scheduleModel.proxies = _.get(subTaskInfo, 'options.backupOpts.mediaOpt.auxcopyJobOption.proxies');
		}

		factory.convertScheduleModelToSubtask = function(scheduleModel, policyType) {
			let subTaskInfo = {
				schedulePattern: _.get(scheduleModel, 'schedulePattern'),
				options: _.get(scheduleModel, 'options', {}),
				subTask: _.get(scheduleModel, 'subTask')
			};

			if (factory.subTaskConversionMap.hasOwnProperty(policyType)) {
				factory.subTaskConversionMap[policyType](scheduleModel, subTaskInfo);
			}

			//populate the properties for automatic schedule pattern
			if (_.get(scheduleModel, 'schedulePattern.freqType') == 'AUTOMATIC_SCHEDULE') {
				_.set(
					subTaskInfo,
					'options.commonOpts.automaticSchedulePattern',
					extractAutomaticSchedulePattern(scheduleModel)
				);
			}
			return subTaskInfo;
		};

		function convertModelToSubTaskForDataProtection(scheduleModel, subTaskInfo) {
			_.set(subTaskInfo, 'options.backupOpts.backupLevel', scheduleModel.backupLevel);
			_.set(subTaskInfo, 'options.backupOpts.oracleOptions.cumulative', scheduleModel.cumulative);
		}

		function convertModelToSubTaskForAuxCopy(scheduleModel, subTaskInfo) {
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				scheduleModel.maxNumberOfStreams
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				scheduleModel.useMaximumStreams
			);
			_.set(subTaskInfo, 'options.backupOpts.mediaOpt.startNewMedia', scheduleModel.startNewMedia);
			_.set(subTaskInfo, 'options.backupOpts.mediaOpt.markMediaFullOnSuccess', scheduleModel.markMediaFullOnSuccess);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMostRecentBackupForAuxcopy',
				scheduleModel.useMostRecentBackupForAuxcopy
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.waitForAllParallelCopyResources',
				scheduleModel.waitForAllParallelCopyResources
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useScallableResourceManagement',
				scheduleModel.useScallableResourceManagement
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.ignoreDataVerificationFailedJobs',
				scheduleModel.ignoreDataVerificationFailedJobs
			);
		}

		function convertModelToSubTaskForBackupCopy(scheduleModel, subTaskInfo) {
			_.set(subTaskInfo, 'options.adminOpts.snapToTapeOption.noofJobsToRun', scheduleModel.noofJobsToRun);
			_.set(subTaskInfo, 'options.adminOpts.snapToTapeOption.allowMaximum', scheduleModel.allowMaximum);
			_.set(subTaskInfo, 'options.backupOpts.mediaOpt.startNewMedia', scheduleModel.startNewMedia);
			_.set(subTaskInfo, 'options.backupOpts.mediaOpt.markMediaFullOnSuccess', scheduleModel.markMediaFullOnSuccess);
		}

		function convertModelToSubTaskForOfflineContentIndex(scheduleModel, subTaskInfo) {
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				scheduleModel.maxNumberOfStreams
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				scheduleModel.useMaximumStreams
			);
		}

		function convertModelToSubTaskForDataVerification(scheduleModel, subTaskInfo) {
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.mediaAgent.mediaAgentId',
				scheduleModel.mediaAgentId
			);
			_.set(
				subTaskInfo,
				'options.adminOpts.archiveCheckOption.backupLevel',
				scheduleModel.runIncrementalVerification ? BACKUP_TYPES.INCREMENTAL : BACKUP_TYPES.FULL
			);
			_.set(subTaskInfo, 'options.adminOpts.archiveCheckOption.ddbVerificationLevel', scheduleModel.deduplicationLevel);
			_.set(
				subTaskInfo,
				'options.adminOpts.archiveCheckOption.defragmentationPercentage',
				100 - scheduleModel.reclaimLevel * 20
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.maxNumberOfStreams',
				scheduleModel.maxNumberOfStreams
			);
			_.set(
				subTaskInfo,
				'options.backupOpts.mediaOpt.auxcopyJobOption.useMaximumStreams',
				scheduleModel.useMaximumStreams
			);
		}

		function convertModelToSubTaskForContentIndexing(scheduleModel, subTaskInfo) {
			_.set(subTaskInfo, 'options.backupOpts.backupLevel', scheduleModel.backupLevel);
			_.set(subTaskInfo, 'options.backupOpts.dataOpt.autoCopy', scheduleModel.autoCopy);
			var selectedClients = scheduleModel.selectedClients,
				memberServers = [];
			for (var i = 0; i < selectedClients.length; i++) {
				memberServers.push({
					client: {
						clientGroupId: selectedClients[i].clientGroupId,
						clientGroupName: selectedClients[i].clientGroupName
					}
				});
			}
			_.set(subTaskInfo, 'options.backupOpts.mediaOpt.auxcopyJobOption.proxies.memberServers', memberServers);
		}

		function extractAutomaticSchedulePattern(scheduleModel) {
			var automaticSchedulePattern = {
				ignoreOpWindowPastMaxInterval: _.get(scheduleModel, 'schedulePattern.ignoreAtMaxInterval', false),
				maxBackupInterval: _.get(scheduleModel, 'schedulePattern.maxBackupIntervalHours', 72),
				maxBackupIntervalMinutes: _.get(scheduleModel, 'schedulePattern.maxBackupIntervalMinutes', 0),
				minBackupInterval: _.get(scheduleModel, 'schedulePattern.minBackupIntervalHours', 0),
				minBackupIntervalMinutes: _.get(scheduleModel, 'schedulePattern.minBackupIntervalMinutes', 15),
				minSyncInterval: _.get(scheduleModel, 'schedulePattern.minSyncIntervalHours', 0),
				minSyncIntervalMinutes: _.get(scheduleModel, 'schedulePattern.minSyncIntervalMinutes', 2),
				wiredNetworkConnection: {
					enabled: _.get(scheduleModel, 'schedulePattern.useOnlyWiredWork', false)
				},
				minNetworkBandwidth: {
					enabled: _.get(scheduleModel, 'schedulePattern.minNetworkBandwidth', false),
					threshold: _.get(scheduleModel, 'schedulePattern.networkBandwidth', 128)
				},
				specfificNetwork: {
					enabled: _.get(scheduleModel, 'schedulePattern.useSpecificNetwork', false),
					ipAddress: {
						address: _.get(scheduleModel, 'schedulePattern.specificNetworkIp', '0.0.0.0'),
						subnet: _.get(scheduleModel, 'schedulePattern.specificNetworkNumber', 24)
					}
				},
				acPower: {
					enabled: _.get(scheduleModel, 'schedulePattern.startOnlyOnAc', false)
				},
				stopIfOnBattery: {
					enabled: _.get(scheduleModel, 'schedulePattern.stopIfBatteryMode', false)
				},
				stopSleepIfBackUp: {
					enabled: _.get(scheduleModel, 'schedulePattern.preventSleep', false)
				},
				cpuUtilization: {
					enabled: _.get(scheduleModel, 'schedulePattern.cpuBelowThresholdEnabled', false),
					threshold: _.get(scheduleModel, 'schedulePattern.cpuBelowThreshold', 10)
				},
				newOrModifiedFile: {
					enabled: _.get(scheduleModel, 'schedulePattern.startOnlyfileBackUp', false)
				}
			};
			return automaticSchedulePattern;
		}

		function getSelectedAppGroupsFromTree(agentTypeTree) {
			if (!agentTypeTree.tree[0].selected && !agentTypeTree.tree[0].__ivhTreeviewIndeterminate) {
				return; //none of the nodes are selected;
			}
			var appGroup = {
					appGroups: [],
					appTypes: []
				},
				selectedAgents;
			if (!agentTypeTree.tree[0].selected && agentTypeTree.tree[0].__ivhTreeviewIndeterminate) {
				selectedAgents = agentTypeTree.getSelectedNodes();
				if (angular.isArray(selectedAgents) && selectedAgents.length > 0) {
					selectedAgents.forEach(function(agent) {
						if (agent.isAppGroup) {
							appGroup.appGroups.push({
								appGroupId: agent.value
							});
						} else {
							appGroup.appTypes.push({
								appTypeId: agent.value
							});
						}
					});
				}
			}
			return appGroup;
		}

		function getScheduleNames(schedules) {
			if (schedules) {
				let scheduleNames = schedules.map(function(schedule) {
					if (schedule.subTask && schedule.subTask.subTaskName) {
						return schedule.subTask.subTaskName;
					} else if (schedule.schedulePattern && schedule.schedulePattern.name) {
						return schedule.schedulePattern.name;
					} else {
						return '';
					}
				});
				return scheduleNames;
			} else {
				return [];
			}
		}

		factory.loadTaskOperationTypes = function() {
			var deferred = $q.defer();
			if (angular.isUndefined(factory.scheduleTypesMap)) {
				factory.scheduleTypesMap = {};
				schedulePolicyService
					.getTaskOperationTypes()
					.success(function(scheduleTypes) {
						if (angular.isArray(scheduleTypes)) {
							scheduleTypes.forEach(scheduleType => {
								factory.scheduleTypesMap[scheduleType.value] = scheduleType.label;
							});
							factory.scheduleTypesList = scheduleTypes;
							deferred.resolve(factory.scheduleTypesMap);
						}
					})
					.error(function(errMsg) {
						deferred.reject(errMsg);
					});
			} else {
				deferred.resolve(factory.scheduleTypesMap);
			}
			return deferred.promise;
		};

		factory.addSchedulePolicy = function() {
			$modal.open({
				templateUrl: appUtil.appRoot + 'modules/policies/schedulePolicies/partials/addSchedulePolicy.jsp',
				backdrop: 'static',
				controller: 'addSchedulePolicyController',
				controllerAs: 'aspCtrl'
			});
		};

		factory.getColumnFiltersForAgents = function(taskInfoList) {
			let agentFilters = [];

			if (angular.isArray(taskInfoList)) {
				taskInfoList.forEach(taskInfo => {
					if (taskInfo.agents && angular.isString(taskInfo.agents.allAgents)) {
						let agentNames = taskInfo.agents.allAgents.split(', ');
						if (angular.isArray(agentNames)) {
							agentNames.forEach(agentName => {
								if (agentName && !_.find(agentFilters, { value: agentName })) {
									agentFilters.push({
										label: agentName,
										value: agentName
									});
								}
							});
						}
					}
				});
				return cvUtil.sortAscending(agentFilters, 'label');
			}

			return agentFilters;
		};

		factory.editSchedule = function(params) {
			$state.go('scheduleDetails', params);
		};

		return factory;
	}
]);

export default schedulePolicyMod;
