import 'modules/policies/schedulePolicies/js/services/schedulePolicy.svc.js';

import { policyModule } from 'common/js/modules';

policyModule.factory('ddbVerificationTreeComponent', [
	'cvLoc',
	'schedulePolicyService',
	'$q',
	function(cvLoc, schedulePolicyService, $q) {
		var factory = {};

		factory.DDBEnginesRootNode = {
			ItemId: 'ALL_SIDB_STORES_ENTITY-0',
			ItemName: cvLoc('label.deduplicationEngines'),
			entity: {
				entityTypeName: 'ALL_SIDB_STORES_ENTITY'
			}
		};

		factory.populateTreeParams = function(associations, mode) {
			var params = {
				initialSelections: [],
				initialExclusions: [],
				result: {},
				mode: mode || 'edit',
				loadRootNodes: factory.loadRootNodes,
				loadChildren: factory.loadDDBAssociations,
				processChildren: factory.processDDBAssociations
			};
			var selectedEngines = [],
				excludedEngines = [];

			if (angular.isArray(associations)) {
				associations.forEach(entity => {
					var formattedEntity = formatTreeNode(entity);
					if (_.get(entity, 'flags.exclude', false)) {
						excludedEngines.push(formattedEntity);
					} else {
						selectedEngines.push(formattedEntity);
					}
				});
				params.initialSelections = selectedEngines;
				params.initialExclusions = excludedEngines;
			}
			return params;
		};

		factory.getRootNode = function() {
			return [factory.DDBEnginesRootNode];
		};

		/**
		 * Entity ID should be of format: "<ENTITY_TYPE> - <EntityId>"
		 */
		factory.frameTreeNodeId = function(entityId, entityType) {
			return entityType + '-' + entityId;
		};

		factory.getEntityType = function(entity) {
			var entityType = _.get(entity, 'entityTypeName', 'STORAGE_POLICY_ENTITY');
			if (entityType === 'ROOT_ENTITY') {
				return 'STORAGE_POLICY_ENTITY';
			} else {
				return entityType;
			}
		};

		function formatTreeNode(entity) {
			var entityType = factory.getEntityType(entity);

			switch (entityType) {
				case 'ALL_SIDB_STORES_ENTITY':
					return factory.DDBEnginesRootNode;
				case 'STORAGE_POLICY_ENTITY':
					return {
						ItemId: factory.frameTreeNodeId(entity.storagePolicyId, entityType),
						ItemName: entity.storagePolicyName,
						parentNode: factory.DDBEnginesRootNode,
						entity: {
							storagePolicyId: entity.storagePolicyId,
							storagePolicyName: entity.storagePolicyName,
							entityTypeName: 'STORAGE_POLICY_ENTITY'
						}
					};
				case 'STORAGE_POLICY_COPY_ENTITY':
					return {
						ItemId: factory.frameTreeNodeId(entity.copyId, entityType),
						ItemName: entity.copyName,
						parentNode: spNode(entity),
						entity: {
							storagePolicyId: entity.storagePolicyId,
							storagePolicyName: entity.storagePolicyName,
							copyId: entity.copyId,
							copyName: entity.copyName,
							entityTypeName: 'STORAGE_POLICY_COPY_ENTITY'
						}
					};
				case 'SIDB_STORE_ENTITY':
					return {
						ItemId: factory.frameTreeNodeId(entity.sidbStoreId, entityType),
						ItemName: entity.sidbStoreName + ' [ID:' + entity.sidbStoreId + ']',
						parentNode: copyNode(entity),
						entity: {
							storagePolicyId: entity.storagePolicyId,
							storagePolicyName: entity.storagePolicyName,
							copyId: entity.copyId,
							copyName: entity.copyName,
							sidbStoreId: entity.sidbStoreId,
							sidbStoreName: entity.sidbStoreName + ' [ID:' + entity.sidbStoreId + ']',
							entityTypeName: 'SIDB_STORE_ENTITY'
						}
					};
			}
		}

		function spNode(entity) {
			return {
				ItemId: factory.frameTreeNodeId(entity.storagePolicyId, 'STORAGE_POLICY_ENTITY'),
				ItemName: entity.storagePolicyName,
				parentNode: factory.DDBEnginesRootNode,
				entity: entity
			};
		}

		function copyNode(entity) {
			return {
				ItemId: factory.frameTreeNodeId(entity.copyId, 'STORAGE_POLICY_COPY_ENTITY'),
				ItemName: entity.copyName,
				parentNode: spNode(entity),
				entity: entity
			};
		}

		factory.loadRootNodes = function() {
			return factory.getRootNode();
		};

		factory.loadDDBAssociations = function(parent) {
			if (parent.entity.entityTypeName === 'ALL_SIDB_STORES_ENTITY') {
				var storagePolicyId = 0,
					storagePolicyCopyId = 0;
				return schedulePolicyService.getDDBEngines(storagePolicyId, storagePolicyCopyId);
			} else {
				var deferred = $q.defer();
				deferred.resolve(parent.entireNode);
				return deferred.promise;
			}
		};

		factory.processAllSIDBAssociations = function(response, parent) {
			var children = [];
			if (response.data.engines && angular.isArray(response.data.engines)) {
				var DDBEngines = response.data.engines;

				for (var i = 0, len = DDBEngines.length; i < len; i++) {
					var engine = DDBEngines[i];

					var entireNode = {
						spId: engine.sp.id,
						spName: engine.sp.name,
						copyId: engine.copy.id,
						copyName: engine.copy.name,
						storeId: engine.storeId,
						storeName: engine.storeName
					};

					var currentEntity = {
						storagePolicyId: engine.sp.id,
						storagePolicyName: engine.sp.name,
						entityTypeName: 'STORAGE_POLICY_ENTITY'
					};

					children.push({
						ItemId: factory.frameTreeNodeId(engine.sp.id, 'STORAGE_POLICY_ENTITY'),
						ItemName: engine.sp.name,
						parentNode: parent,
						entireNode: entireNode,
						entity: currentEntity,
						forceDisable: false,
						noChild: false
					});
				}
			}
			return children;
		};

		factory.processSPAssociations = function(response, parent) {
			var children = [];
			if (response) {
				var currentEntity = {
					copyId: response.copyId,
					copyName: response.copyName,
					storagePolicyId: response.spId,
					storagePolicyName: response.spName,
					entityTypeName: 'STORAGE_POLICY_COPY_ENTITY'
				};
				children.push({
					ItemId: factory.frameTreeNodeId(response.copyId, 'STORAGE_POLICY_COPY_ENTITY'),
					ItemName: response.copyName,
					parentNode: parent,
					entireNode: response,
					entity: currentEntity,
					forceDisable: false,
					nochild: false
				});
			}
			return children;
		};

		factory.processSPCopyAssociations = function(response, parent) {
			var children = [];
			if (response) {
				var currentEntity = {
					sidbStoreId: response.storeId,
					sidbStoreName: response.storeName + ' [ID:' + response.storeId + ']',
					copyId: response.copyId,
					copyName: response.copyName,
					storagePolicyId: response.spId,
					storagePolicyName: response.spName,
					entityTypeName: 'SIDB_STORE_ENTITY'
				};
				children.push({
					ItemId: factory.frameTreeNodeId(response.storeId, 'SIDB_STORE_ENTITY'),
					ItemName: response.storeName + ' [ID:' + response.storeId + ']',
					parentNode: parent,
					entireNode: response,
					entity: currentEntity,
					forceDisable: false,
					nochild: true
				});
			}
			return children;
		};

		factory.processDDBAssociations = function(response, parent) {
			if (parent.entity.entityTypeName === 'ALL_SIDB_STORES_ENTITY') {
				return factory.processAllSIDBAssociations(response, parent);
			} else if (parent.entity.entityTypeName === 'STORAGE_POLICY_ENTITY') {
				return factory.processSPAssociations(response, parent);
			} else if (parent.entity.entityTypeName === 'STORAGE_POLICY_COPY_ENTITY') {
				return factory.processSPCopyAssociations(response, parent);
			} else {
				return [];
			}
		};

		return factory;
	}
]);

export default policyModule;
