import 'modules/policies/schedulePolicies/js/constants/backupLevel.constants.js';
import 'modules/policies/schedulePolicies/js/constants/supportedBackupOptions.constant.js';

import { policyModule } from 'common/js/modules';

var app = policyModule;

/**
 * Schedule policy of type: "Data Protection" can support multiple agents.
 *
 * Based on the agents selected for a schedule policy, we have to determine what backup-options should be
 * visible in the UI.
 *
 * Agents selection tree can have 3 different types of nodes. (a) a dummy node with id=0, (b) an apptype node
 * with apptypeId & name (c) an appgroup node with appGroupId. Refer to cvAppTypes.js for details about the
 * tree structure.
 *
 * Two different maps are constructed (one with AppTypeId as key & another with AppGroupId)
 * For each agent, these are the supported options as per the behaviour confirmed with Commcell Console.
 * Ex: Exchange database supports only "Full" and "Incremental" and Oracle agent supports "Full", "Incremental" & "Synthetic full".
 *
 * When "Add schedule" dialog is opened, a list of agents will be passed as input to this method:
 * "getSupportedBackupOptions()" that will use this MAP and retrieves an object with boolean fields denoting what are all the options to be shown.
 *
 */

app.factory('backupOptionsMap', [
	'AppTypes',
	'AppGroups',
	'SUPPORTED_BACKUP_OPTIONS',
	function(AppTypes, AppGroups, SUPPORTED_BACKUP_OPTIONS) {
		var MAP = {
			appGroups: {},
			appTypes: {}
		};
		var OPTIONS = SUPPORTED_BACKUP_OPTIONS;
		MAP.appGroups[AppGroups.APPGRP_CLOUDAPPS] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_DB2] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;
		MAP.appTypes[AppTypes.DB2] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;
		MAP.appTypes[AppTypes.DB2_MultiNode] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;
		MAP.appTypes[AppTypes.DB_On_Unix] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;

		MAP.appGroups[AppGroups.APPGRP_XchangeDB] = OPTIONS.FULL | OPTIONS.PRE_SELECT;
		MAP.appTypes[AppTypes.EXCHANGE_DATABASE] = OPTIONS.FULL | OPTIONS.PRE_SELECT;

		MAP.appGroups[AppGroups.APPGRP_AppTypeIndexedBased] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;

		MAP.appGroups[AppGroups.APPGRP_INFORMIX] = OPTIONS.FULL | OPTIONS.INCREMENTAL;
		MAP.appTypes[AppTypes.INFORMIX_DATABASE] = OPTIONS.FULL | OPTIONS.INCREMENTAL;

		MAP.appGroups[AppGroups.APPGRP_MySql] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.MYSQL] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_NotesDb] = OPTIONS.FULL | OPTIONS.INCREMENTAL;
		MAP.appGroups[AppGroups.APPGRP_NotesDb_Transaction_Log] = OPTIONS.FULL | OPTIONS.INCREMENTAL;

		MAP.appGroups[AppGroups.APPGRP_ORACLE] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL | OPTIONS.CUMULATIVE;
		MAP.appTypes[AppTypes.ORACLE] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.ORACLE_RAC] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_POSTGRES] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appGroups[AppGroups.APPGRP_SAP_FOR_ORACLE] = OPTIONS.FULL | OPTIONS.INCREMENTAL;
		MAP.appGroups[AppGroups.APPGRP_SAP_HANA] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appGroups[AppGroups.APPGRP_SQL_POLICY] =
			OPTIONS.FULL | OPTIONS.TRANSACTION_LOG | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_Sybase] = OPTIONS.FULL | OPTIONS.TRANSACTION_LOG | OPTIONS.CUMULATIVE_INCREMENTAL;
		MAP.appTypes[AppTypes.SYBASE_DATABASE] = OPTIONS.FULL | OPTIONS.TRANSACTION_LOG | OPTIONS.CUMULATIVE_INCREMENTAL;

		MAP.appGroups[AppGroups.APPGRP_DISTRIBUTEDAPPS] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appGroups[AppGroups.APPGRP_DOCUMENTUM] = OPTIONS.FULL | OPTIONS.INCREMENTAL;

		MAP.appGroups[AppGroups.APPGRP_ARCHIVED_MAILS] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL | OPTIONS.CLEANUP;
		MAP.appTypes[AppTypes.EXCHANGE_MAILBOX] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL | OPTIONS.CLEANUP;
		MAP.appTypes[AppTypes.EXCHANGE_MAILBOX_ARCHIVER] = 0;
		MAP.appTypes[AppTypes.EXCHANGE_PF_ARCHIVER] = 0;
		MAP.appGroups[AppGroups.APPGRP_JOURNALED_MAILS] = 0;
		MAP.appTypes[AppTypes.EXCHANGE_COMPLIANCE_ARCHIVER] = 0;
		MAP.appGroups[AppGroups.APPGRP_BACKEDUP_MAILS] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.EXCHANGE_MAILBOX_CLASSIC] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.EXCHANGE_PUBLIC_FOLDER] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_ARCHIVED_FILES] = 0;
		MAP.appTypes[AppTypes.FILE_SHARE_ARCHIVER] = 0;
		MAP.appTypes[AppTypes.NETWARE_FILE_ARCHIVER] = 0;
		MAP.appTypes[AppTypes.UNIX_FILE_ARCHIVER] = 0;
		MAP.appTypes[AppTypes.WINDOWS_FILE_ARCHIVER] = 0;

		MAP.appGroups[AppGroups.APPGRP_BACKEDUP_FILES] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.ACTIVE_DIRECTORY] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.AIX_FILE_SYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.DISTRIBUTED_APPS] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.DPM] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.FREE_BSD] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.HP_UX_FILE_SYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.IMAGE_LEVEL] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.IMAGE_LEVEL_ON_UNIX] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.IMAGE_LEVEL_PROXYHOST] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.IMAGE_LEVEL_PROXYHOST_ON_UNIX] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.LINUX_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.MAC_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.NAS] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL;
		MAP.appTypes[AppTypes.NETWARE_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.NOVELL_DIRECTORY_SERVICES] = OPTIONS.FULL;
		MAP.appTypes[AppTypes.OBJECT_LINK] = OPTIONS.INCREMENTAL;
		MAP.appTypes[AppTypes.OBJECT_STORE] = OPTIONS.INCREMENTAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.OES_FILESYSTEM_ON_LINUX] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.OTHER_EXTERNAL_AGENT] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.PROXY_CLIENT_FILE_SYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.PROXYHOST_ON_UNIX] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.SOLARIS_64BIT_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.SOLARIS_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.UNIX_TRU64_64BIT_FILESYSTEM] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.VMWARE] = OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appGroups[AppGroups.APPGRP_W2KFileSystem] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_NotesDocDataMigrator] = 0;
		MAP.appGroups[AppGroups.APPGRP_NotesDoc] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;

		MAP.appGroups[AppGroups.APPGRP_ARCHIVE_DOCS] = 0;
		MAP.appTypes[AppTypes.MS_SHAREPOINT_ARCHIVER] = 0;
		MAP.appGroups[AppGroups.APPGRP_BACKEDUP_DOCS] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;
		MAP.appTypes[AppTypes.SHAREPOINT_SERVER] =
			OPTIONS.FULL | OPTIONS.INCREMENTAL | OPTIONS.DIFFERENTIAL | OPTIONS.SYNTHETIC_FULL;

		MAP.getSupportedBackupOptions = function(selectedAgents) {
			if (!selectedAgents) {
				return;
			}
			var supportedOptionsInt = 0;
			if (angular.isUndefined(selectedAgents['appGroups']) && angular.isUndefined(selectedAgents['appTypes'])) {
				supportedOptionsInt =
					SUPPORTED_BACKUP_OPTIONS.FULL |
					SUPPORTED_BACKUP_OPTIONS.INCREMENTAL |
					SUPPORTED_BACKUP_OPTIONS.DIFFERENTIAL |
					SUPPORTED_BACKUP_OPTIONS.SYNTHETIC_FULL |
					SUPPORTED_BACKUP_OPTIONS.CLEANUP |
					SUPPORTED_BACKUP_OPTIONS.PRE_SELECT |
					SUPPORTED_BACKUP_OPTIONS.TRANSACTION_LOG;
			} else {
				var keys = ['appGroups', 'appTypes'],
					id = ['appGroupId', 'appTypeId'];
				for (var i = 0; i < 2; i++) {
					if (angular.isArray(selectedAgents[keys[i]])) {
						selectedAgents[keys[i]].forEach(agent => {
							var options = MAP[keys[i]][agent[id[i]]];
							supportedOptionsInt = supportedOptionsInt | (angular.isNumber(options) ? options : 0);
						});
					}
				}
			}

			var supportedOptionsObj = {
				backupTypes: supportedOptionsInt != 0,
				full: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.FULL) != 0,
				incremental: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.INCREMENTAL) != 0,
				differential: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.DIFFERENTIAL) != 0,
				synthfull: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.SYNTHETIC_FULL) != 0,
				cleanup: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.CLEANUP) != 0,
				preselected: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.PRE_SELECT) != 0,
				incrementalLog: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.LOG) != 0,
				incrementalData: (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.DATA) != 0
			};

			if (!supportedOptionsObj['incremental']) {
				supportedOptionsObj['transactionLog'] = (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.TRANSACTION_LOG) != 0;
			} else if (!(supportedOptionsObj['incrementalLog'] || supportedOptionsObj['incrementalData'])) {
				supportedOptionsObj['incrementalCumulative'] = (supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.CUMULATIVE) != 0;
			}

			if (!supportedOptionsObj['differential']) {
				supportedOptionsObj['cumulativeIncremental'] =
					(supportedOptionsInt & SUPPORTED_BACKUP_OPTIONS.CUMULATIVE_INCREMENTAL) != 0;
			}

			return supportedOptionsObj;
		};

		MAP.getDefaultBackupOptionsForSchedules = function() {
			return (
				SUPPORTED_BACKUP_OPTIONS.FULL |
				SUPPORTED_BACKUP_OPTIONS.INCREMENTAL |
				SUPPORTED_BACKUP_OPTIONS.DIFFERENTIAL |
				SUPPORTED_BACKUP_OPTIONS.SYNTHETIC_FULL
			);
		};

		return MAP;
	}
]);

export default app;
