import 'modules/policies/schedulePolicies/js/services/schedulePolicy.svc.js';
import 'modules/policies/schedulePolicies/js/factory/schedulePolicy.factory.js';
import 'modules/schedule/js/factory/schedule.factory.js';
import 'adminConsole/js/factories/reports.factory.js';
import 'modules/policies/schedulePolicies/js/constants/freqType.constant.js';
import 'modules/policies/schedulePolicies/js/filters/backupLevel.filter.js';
import 'modules/policies/schedulePolicies/js/filters/associatedScheduleName.filter.js';
import 'modules/policies/schedulePolicies/js/directives/cvScheduleSpecificOptions.directive.js';
import 'modules/policies/schedulePolicies/js/factory/schedulePattern.factory.js';
import 'modules/policies/schedulePolicies/js/filters/backupInterval.filter.js';

import { policyModule } from 'common/js/modules';

(function() {
	'use strict';

	/**
	 * Controller to route #/scheduleDetails/:subTaskId Schedule page displays pattern and backup settings for
	 * a schedule
	 */
	policyModule.controller('scheduleDetailsController', [
		'$dialogs',
		'$stateParams',
		'$filter',
		'$scope',
		'$rootScope',
		'$state',
		'cvLoc',
		'cvUtil',
		'cvToaster',
		'cvBreadcrumbsTabsFactory',
		'scheduleUIFactory',
		'schedulePolicyService',
		'schedulePatternFactory',
		'customReportsFactory',
		'$location',
		ScheduleDetailsController
	]);

	function ScheduleDetailsController(
		$dialogs,
		$stateParams,
		$filter,
		$scope,
		$rootScope,
		$state,
		cvLoc,
		cvUtil,
		cvToaster,
		cvBreadcrumbsTabsFactory,
		scheduleUIFactory,
		schedulePolicyService,
		schedulePatternFactory,
		customReportsFactory,
		$location
	) {
		var self = this,
			editPatternListener;

		self.model = {
			taskId: $stateParams['taskId'],
			taskName: '',
			subTaskId: $stateParams['subTaskId'],
			subTaskName: ''
		};

		self.runSchedule = runSchedule;
		self.deleteSchedule = deleteSchedule;
		self.editSchedulePattern = editSchedulePattern;
		self.isSchedulePartOfSchedulePolicy = isSchedulePartOfSchedulePolicy;
		self.editWebReportSchedule = editWebReportSchedule;
		self.isWebReportSchedule = isWebReportSchedule;

		function initGlobalBreadCrumbs() {
			var nav = $stateParams['nav'];
			if (nav == 'ReportSchedules') {
				var reportScheduleBreadcrumbs = [
					{
						title: cvLoc('label.nav.reports'),
						link: '#/reports'
					},
					{
						title: cvLoc('label.reports.schedule'),
						link: '#/reportSchedules?reset=true'
					}
				];
				cvBreadcrumbsTabsFactory.addBreadCrumbs(reportScheduleBreadcrumbs);
			} else {
				cvBreadcrumbsTabsFactory.addBreadCrumbs({
					title: cvLoc('label.schedulePolicies'),
					link: '#/schedulePolicies'
				});

				if (isSchedulePartOfSchedulePolicy()) {
					cvBreadcrumbsTabsFactory.addBreadCrumbs({
						title: self.model.taskName,
						link: '#/schedulePolicyDetails/' + self.model.taskId
					});
				}
			}
		}

		function isSchedulePartOfSchedulePolicy() {
			let taskType = _.get(self.model, 'taskInfo.task.taskType');
			return taskType === 'SCHEDULE_POLICY';
		}

		function isWebReportSchedule() {
			let type = _.get(self.model, 'subTaskInfo.options.adminOpts.reportOption.commonOpt.reportType');
			return type === 'WEB_REPORT';
		}

		function genericErrorHandler(error) {
			cvToaster.showErrorMessage({
				ttl: '10000',
				message: error || cvLoc('generic_error')
			});
		}

		function runSchedule() {
			schedulePolicyService
				.runSchedule(self.model.subTaskId, self.model.subTaskName, self.model.taskId, self.model.taskName)
				.success(function(jobIdList) {
					if (jobIdList) {
						var responseContent;
						if (jobIdList.length == 1) {
							var jobId = jobIdList[0];
							responseContent =
								cvLoc('label.jobStarted', jobId) +
								'  <a target="_blank" href="' +
								'#/jobs/' +
								jobId +
								'">' +
								cvLoc('notification.viewJobs') +
								'</a>';
						} else if (jobIdList.length > 1) {
							responseContent =
								cvLoc('label.jobsStarted') +
								'. ' +
								'<a target="_blank" href="#/activeJobs" >' +
								cvLoc('notification.viewJobs') +
								'</a>';
						}
						cvToaster.showSuccessMessage({
							ttl: '3000',
							message: responseContent
						});
					} else {
						genericErrorHandler(false);
					}
				})
				.error(genericErrorHandler);
		}

		function deleteSchedule() {
			$dialogs.confirm(
				cvLoc('label.confirmDelete'),
				cvLoc('label.confirmDeleteAssociatedSchedule', '<b>' + self.model.subTaskName + '</b>'),
				{
					noFunction: function() {},
					yesFunction: function() {
						schedulePolicyService
							.deleteSchedule(self.model.taskId)
							.success(function() {
								cvToaster.showSuccessMessage(cvUtil.infoMsgLoc('label.associatedScheduleDeleted'));
								if (isWebReportSchedule()) {
									$state.go('reportSchedules', { reset: 'true' });
								} else {
									$location.path('schedulePolicies');
								}
							})
							.error(genericErrorHandler);
					}
				}
			);
		}

		function editSchedulePattern() {
			schedulePatternFactory.editSchedulePattern(self.model.taskInfo, self.model.subTaskInfo);
		}
		
		editPatternListener = $rootScope.$on(schedulePatternFactory.editPatternEmitListener, function(
			evt,
			addedSchedule
		) {
			schedulePatternFactory.formatRepeatSettingsObject(addedSchedule);
			schedulePatternFactory.formatAutomaticSchedules(self.model.subTaskInfo, addedSchedule);
			self.model.subTaskInfo.schedulePattern = addedSchedule;
			saveSchedulePattern();
		});

		function editWebReportSchedule() {
			customReportsFactory.editSchedule(self.model.taskId);
			$rootScope.$on(customReportsFactory.editScheduleEventEmitListener, $state.forceReload);
		}

		self.toggleScheduleStatus = function() {
			self.model.serverMessage = cvUtil.emptyMsg();

			customReportsFactory
				.toggleScheduleStatus(self.model.taskId, self.model.subTaskId, self.model.isEnabled)
				.then(function() {
					self.model.isEnabled = !self.model.isEnabled;
					$scope.$broadcast('scheduleStatusChanged', self.model.isEnabled);
				}, showErrorMessage);
		};

		function showErrorMessage(errorMessage) {
			if (errorMessage && angular.isString(errorMessage)) {
				self.model.serverMessage = cvUtil.errMsg(errorMessage);
			} else if (errorMessage && angular.isString(errorMessage.data)) {
				self.model.serverMessage = cvUtil.errMsg(errorMessage.data);
			} else {
				self.model.serverMessage = cvUtil.errMsgLoc('generic_error');
			}
		}

		function saveSchedulePattern() {
			schedulePolicyService
				.editAssociatedSchedule({
					taskId: self.model.taskId,
					subTaskId: self.model.subTaskId,
					scheduleOptions: angular.toJson(self.model.subTaskInfo)
				})
				.success(function() {
					cvToaster.showSuccessMessage(cvUtil.infoMsgLoc('label.scheduleUpdatedSuccess'));
					$rootScope.$emit('scheduleDataChanged');
					self.scheduleLoaded = false;
					$state.forceReload();
				})
				.error(function(errorMessage) {
					$rootScope.$emit('editScheduleFailed', errorMessage);
				});
		}

		function initPattern() {
			self.model.pattern = schedulePatternFactory.getPatternInfo(self.model.subTaskInfo);
			self.model.subTaskInfo.schedulePattern = scheduleUIFactory.convertSubTaskInfoToSchedule(self.model.subTaskInfo);
			let repeatEnabled = _.get(self, 'model.subTaskInfo.pattern.freq_subday_interval', 0) !== 0;
			_.set(self.model, 'subTaskInfo.schedulePattern.repeat.enabled', repeatEnabled);
		}

		function initModel(taskInfo) {
			self.model.taskInfo = taskInfo;
			self.model.subTaskInfo = _.get(taskInfo, 'subTasks[0]');
			self.model.taskName = _.get(taskInfo, 'task.taskName', '') || _.get(taskInfo, 'task.task.taskName', '');
			self.model.subTaskName = $filter('associatedScheduleName')(self.model.subTaskInfo);
			self.model.showScheduleSpecificOptions = isScheduleSpecificOptionsSupported(
				self.model.taskInfo,
				self.model.subTaskInfo
			);
			self.model.isEnabled = !taskInfo.task.taskFlags.disabled;
		}

		function initNameChanger() {
			self.model.changeNameObj = {
				id: self.model.subTaskId,
				name: self.model.subTaskName,
				taskId: self.model.taskId
			};
		}

		function initialize(taskInfo) {
			initModel(taskInfo);
			initGlobalBreadCrumbs();
			initPattern();
			initNameChanger();
			self.model.scheduleLoaded = true;
		}

		function isScheduleSpecificOptionsSupported(taskInfo, subTaskInfo) {
			let taskType = _.get(taskInfo, 'task.taskType'),
				policyType = _.get(taskInfo, 'task.policyType'),
				subTaskOperationType = _.get(subTaskInfo, 'subTask.operationType');

			if (isWebReportSchedule()) {
				return true;
			}
			if (taskType === 'SCHEDULE') {
				return (
					policyType === 'DATA_PROTECTION' &&
					(subTaskOperationType === 'BACKUP' || subTaskOperationType === 'BLR_OPERATION')
				);
			}
			return taskType === 'SCHEDULE_POLICY';
		}

		schedulePolicyService
			.loadScheduleDetails(self.model.taskId, self.model.subTaskId)
			.success(initialize)
			.error(genericErrorHandler);

		$scope.$on('$destroy', cleanUp);

		function cleanUp() {
			if (editPatternListener) {
				editPatternListener();
			}
		}
	}
})();
