import 'modules/plan/js/directives/exchangePlanDirectives.js';
import 'vsa/js/controllers/collections.ctrl.js';
import 'dlo/js/services/profile.svc.js';
import 'common/js/cv-time-relative-picker.js';

/**
 * A factory for creating new base Office365 Plan instances.
 */
class office365PlanFactory {
	constructor(
		$q,
		$timeout,
		cvLoc,
		cvUtil,
		profileService,
		name,
		subType,
		exchangeConstants,
		ediscoveryUtility,
		$modal,
		$dialogs
	) {
		this.$modal = $modal;
		this.$q = $q;
		this.$timeout = $timeout;
		this.$dialogs = $dialogs;
		this.cvLoc = cvLoc;
		this.cvUtil = cvUtil;
		this.profileService = profileService;
		this.exchangeConstants = exchangeConstants;
		this.ediscoveryUtility = ediscoveryUtility;
		var _this = this;
		/**
		 * Declare a new instance of a Plan.
		 *
		 * @param {string}
		 *            defaultName - The default name of the plan if the name is not already taken.
		 */
		this.defaultName = name;
		this.planInfo = {
			summary: {
				plan: {
					planName: ''
				},
				type: this.exchangeConstants.plan.TYPE,
				subtype: subType
			},
			office365Info: {
				o365Exchange: {
					mbArchiving: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
						detail: {
							emailPolicy: {
								emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.ARCHIVE,
								archivePolicy: {
									primaryMailbox: true,
									archiveMailbox: false,
									disabledMailbox: true,
									backupStubs: true,
									includeMsgsLargerThan: 0,
									includeMsgsOlderThan: 0,
									contentIndexProps: {
										deferredDays: 0,
										enableContentIndex: false,
										enablePreviewGeneration: false,
										enableExactSearch: false,
										previewPathDir: {
											userAccount: {
												password: [],
												confirmPassword: []
											}
										}
									}
								}
							}
						}
					},
					mbRetention: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
						detail: {
							emailPolicy: {
								emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.RETENTION,
								retentionPolicy: {
									type: '',
									numOfDaysForMediaPruning: -1,
									numOfDaysForDeletedMediaPruning: -1
								}
							}
						}
					},
					mbCleanup: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.EMAIL_POLICY_TYPE,
						detail: {
							emailPolicy: {
								emailPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.CLEANUP,
								cleanupPolicy: {
									primaryMailbox: true,
									archiveMailbox: false,
									disabledMailbox: true,
									createStubs: true,
									collectMsgsLargerThan: 0,
									collectMsgsDaysAfter: 0,
									numOfDaysForSourcePruning: 730,
									truncateBody: false,
									truncateBodyToBytes: 1024
								}
							}
						}
					}
				},
				o365CloudOffice: {
					caBackup: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.CLOUD_APPS_POLICY_TYPE,
						detail: {
							cloudAppPolicy: {
								cloudAppPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.ARCHIVE,
								backupPolicy: {
									enableRules: true,
									ruleDetails: [
										{
											agentType: this.exchangeConstants.plan.AGENT_TYPE.ONEDRIVE,
											ruleType: this.exchangeConstants.plan.RULE_TYPE.FILTER,
											ruleEnabled: false,
											filters: [
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.FOLDER,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												},
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.FILE,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												}
											]
										},
										{
											agentType: this.exchangeConstants.plan.AGENT_TYPE.SHAREPOINT,
											ruleType: this.exchangeConstants.plan.RULE_TYPE.FILTER,
											ruleEnabled: false,
											filters: [
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.WEB,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												}
											]
										}
									],
									spbackupPolicy: {
										backupVersion: true
									},
									onedrivebackupPolicy: {
										backupVersion: true
									}
								}
							}
						}
					},
					caRetention: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.CLOUD_APPS_POLICY_TYPE,
						detail: {
							cloudAppPolicy: {
								cloudAppPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.RETENTION,
								retentionPolicy: {
									type: '',
									numOfDaysForMediaPruning: -1
								}
							}
						}
					},
					caCleanup: {
						policyType: this.exchangeConstants.plan.POLICY_TYPE.OFFICE_365_POLICY_TYPE,
						detail: {
							cloudAppPolicy: {
								cloudAppPolicyType: this.exchangeConstants.plan.EMAIL_POLICY_TYPES.CLEANUP,
								cleanupPolicy: {
									enableRules: true,
									ruleDetails: [
										{
											ruleType: this.exchangeConstants.plan.RULE_TYPE.FILTER,
											ruleEnabled: false,
											filters: [
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.FOLDER,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												},
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.FILE,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												},
												{
													filterType: this.exchangeConstants.plan.FILTER_TYPE.MESSAGE,
													includeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													},
													excludeFilter: {
														patternsAvailable: [],
														patternsSelected: []
													}
												}
											]
										},
										{
											ruleType: this.exchangeConstants.plan.RULE_TYPE.CLEANUP_SIZE,
											ruleEnabled: false,
											ruleThreshold: 1000
										},
										{
											ruleType: this.exchangeConstants.plan.RULE_TYPE.CLEANUP_AGE,
											ruleEnabled: false,
											ruleThreshold: 1000
										}
									],
									addRecallLink: false
								}
							}
						}
					}
				}
			}
		};

		this.enableOneDriveIncludeFolder = true; //Enabling OneDrive Include filter by default in SP23
	}

	/**
	 * Retrieves the object used for changing the plan's name.
	 *
	 * @return {Object}
	 */
	getChangeNameObj() {
		return {
			id: this.originalPlanInfo.summary.plan.planId,
			name: this.originalPlanInfo.summary.plan.planName
		};
	}

	/**
	 * Checks to see if the given default name is already taken. If it's not already taken, then it will be
	 * set as the name of the plan.
	 *
	 * @return {HttpPromise}
	 */
	checkDefaultName() {
		return this.profileService.checkPlanName(encodeURIComponent(this.defaultName)).success(data => {
			if (angular.isUndefined(data.id)) {
				this.planInfo.summary.plan.planName = this.defaultName;
				this.autoselect = true;
			} else {
				this.planInfo.summary.plan.planName = '';
			}
		});
	}

	/**
	 * Retrieves a single exchange plan from the API and loads it into the current instance.
	 *
	 * @return {HttpPromise}
	 */
	load(id) {
		var deferred = this.$q.defer();
		this.profileService.getProfileDetails(id).then(resp => {
			this.permissions = _.get(resp, 'data.permissions', {});
			//this.planInfo.summary =  _.get(resp, 'data.profile.summary', {});
			this.planInfo = _.get(resp, 'data.profile', {});

			this.refactorRetentionValues();

			//Set to show custom backup settings toggle in UI
			_.set(
				this.planInfo,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.showArchiveCustomSettings',
				false
			);
			if (
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan',
					0
				) > 0 ||
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsLargerThan',
					0
				) > 0 ||
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeOnlyMsgsWithAttachemts',
					false
				) === true
			) {
				_.set(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.showArchiveCustomSettings',
					true
				);
			}

			//If no entities are selected, we set the toggle in the UI to false
			let entityDetails = _.get(this.planInfo, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
			_.set(
				this.planInfo,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
				false
			);
			for (let index = 0; index < entityDetails.length; index++) {
				for (let i = 0; i < entityDetails[index].tags.length; i++) {
					_.set(
						this.planInfo,
						'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
						true
					);
					break;
				}
				if (
					_.get(
						this.planInfo,
						'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
						false
					) === true
				) {
					break;
				}
			}

			//Filter values details
			_.set(
				this.planInfo,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter.folderPatternsSelected',
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter.folderPatternsSelected',
					[]
				)
			);
			_.set(
				this.planInfo,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter.folderPatternsSelected',
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter.folderPatternsSelected',
					[]
				)
			);
			_.set(
				this.planInfo,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeMessageClassFilter.folderPatternsSelected',
				_.get(
					this.planInfo,
					'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeMessageClassFilter.folderPatternsSelected',
					[]
				)
			);

			if (
				_.isUndefined(
					this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy
						.excludeMessageClassFilter.folderPatternsAvailable
				)
			) {
				this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeMessageClassFilter.folderPatternsAvailable = [];
			}
			if (
				_.isUndefined(
					this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter
						.folderPatternsAvailable
				)
			) {
				this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeFolderFilter.folderPatternsAvailable = [];
			}
			if (
				_.isUndefined(
					this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter
						.folderPatternsAvailable
				)
			) {
				this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.excludeFolderFilter.folderPatternsAvailable = [];
			}

			//Rule Details
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails',
				_.get(
					this.planInfo,
					'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails',
					[]
				)
			);
			_.set(this.planInfo, 'oneDriveFilterDetails.filters', []);
			_.set(this.planInfo, 'sharepointFilterDetails.filters', []);

			this.planInfo.office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails.forEach(
				rule => {
					if (rule.agentType === 'ONEDRIVE') {
						rule.filters.forEach(filter => {
							let filterValues = {};
							filterValues.filterType = filter.filterType;
							//filterValues.displayName = this.cvLoc(`label.${filter.filterType.toLowerCase()}`);
							filterValues.displayName = filter.filterType.toLowerCase();

							if (
								filter.filterType !== 'FOLDER' ||
								(filter.filterType === 'FOLDER' && this.enableOneDriveIncludeFolder)
							) {
								_.set(
									filterValues,
									'includeFilter.displayName',
									this.cvLoc(`label.include.${filter.filterType.toLowerCase()}`)
								);
								_.set(
									filterValues,
									'includeFilter.patternsAvailable',
									_.get(filter, 'includeFilter.patternsAvailable', [])
								);
								_.set(
									filterValues,
									'includeFilter.patternsSelected',
									_.get(filter, 'includeFilter.patternsSelected', [])
								);
							}
							_.set(
								filterValues,
								'excludeFilter.displayName',
								this.cvLoc(`label.exclude.${filter.filterType.toLowerCase()}`)
							);
							_.set(
								filterValues,
								'excludeFilter.patternsAvailable',
								_.get(filter, 'excludeFilter.patternsAvailable', [])
							);
							_.set(
								filterValues,
								'excludeFilter.patternsSelected',
								_.get(filter, 'excludeFilter.patternsSelected', [])
							);
							this.planInfo.oneDriveFilterDetails.filters.push(filterValues);
						});
					} else if (rule.agentType === 'SHAREPOINT') {
						rule.filters.forEach(filter => {
							let filterValues = {};
							filterValues.filterType = filter.filterType;
							filterValues.displayName = this.cvLoc(`label.${filter.filterType.toLowerCase()}`);
							_.set(
								filterValues,
								'includeFilter.patternsAvailable',
								_.get(filter, 'includeFilter.patternsAvailable', [])
							);
							_.set(
								filterValues,
								'includeFilter.patternsSelected',
								_.get(filter, 'includeFilter.patternsSelected', [])
							);
							_.set(
								filterValues,
								'excludeFilter.patternsAvailable',
								_.get(filter, 'excludeFilter.patternsAvailable', [])
							);
							_.set(
								filterValues,
								'excludeFilter.patternsSelected',
								_.get(filter, 'excludeFilter.patternsSelected', [])
							);
							this.planInfo.sharepointFilterDetails.filters.push(filterValues);
						});
					}
				}
			);

			//OneDrive Details
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.onedrivebackupPolicy.backupVersions',
				_.get(
					this.planInfo,
					'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.onedrivebackupPolicy.backupVersions',
					false
				)
			);

			//sharepoint details
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.spbackupPolicy.skipChildren',
				_.get(
					this.planInfo,
					'office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.spbackupPolicy.skipChildren',
					false
				)
			);
			this.originalPlanInfo = _.cloneDeep(this.planInfo);
			deferred.resolve(resp);
		});
		return deferred.promise;
	}

	refactorRetentionValues() {
		if (
			_.get(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				''
			) === this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
		) {
			//Exchange retention values
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				_.get(
					this.planInfo,
					'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
					0
				)
			);
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				-1
			);
		} else if (
			_.get(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				''
			) === this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
		) {
			//Exchange retention values
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				-1
			);
		} else if (
			_.get(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				''
			) <= 0
		) {
			//Exchange retention values
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				-1
			);
		}
	}

	/**
	 * Loads the master template of the plan to populate default values.
	 *
	 * @return {HttpPromise}
	 */
	loadMaster() {
		return this.profileService
			.getMasterProfile(this.planInfo.summary.type, this.planInfo.summary.subtype)
			.success(data => {
				this.planInfo = data.profile;
			});
	}

	preProcessData() {
		var sendData = _.cloneDeep(this.planInfo);
		sendData.errorProcessing = false;
		if (
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				0
			) === 0 ||
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				0
			) === 0
		) {
			sendData.errorProcessing = true;
			return sendData;
		}
		sendData.eePolicy = this.ediscoveryUtility.getEntityObjectForPlanCreation(this.planInfo.eePolicy);

		// if (sendData.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.enableEntitySearch) {
		// 	sendData.eePolicy = this.ediscoveryUtility.getEntityObjectForPlanCreation(sendData.eePolicy);
		// }
		if (
			_.get(
				sendData,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.archiveMailbox',
				false
			) === true
		) {
			_.set(sendData, 'office365Info.o365Exchange.mbCleanup.detail.emailPolicy.cleanupPolicy.archiveMailbox', true);
		}
		if (
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				-1
			) > 0
		) {
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
			);
			_.set(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
			);

			//OneDrive retention values
			_.set(
				sendData,
				'office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.MESSAGE_RECEIVED_TIME
			);
			_.set(
				sendData,
				'office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning',
				_.get(
					sendData,
					'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning'
				)
			);
			delete sendData.office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;
			delete sendData.office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;
		}
		if (
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				-1
			) > 0
		) {
			_.set(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				_.get(
					sendData,
					'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
					0
				)
			);
			_.set(
				this.planInfo,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
			);
			_.set(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
			);
			//OneDrive retention values
			_.set(
				sendData,
				'office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy.numOfDaysForMediaPruning',
				_.get(
					sendData,
					'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
					0
				)
			);
			_.set(
				sendData,
				'office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy.type',
				this.exchangeConstants.plan.RETENTIONPOLICY.SOURCE_DELETION_TIME
			);
			delete sendData.office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;
			delete sendData.office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;
		}
		if (
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForMediaPruning',
				-1
			) === -1 &&
			_.get(
				sendData,
				'office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.numOfDaysForDeletedMediaPruning',
				-1
			) === -1
		) {
			delete sendData.office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy.type;
			delete sendData.office365Info.o365CloudOffice.caRetention.detail.cloudAppPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;

			delete sendData.office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy.type;
			delete sendData.office365Info.o365Exchange.mbRetention.detail.emailPolicy.retentionPolicy
				.numOfDaysForDeletedMediaPruning;
		}
		const stringPassword = _.get(
			sendData,
			'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.stringPassword',
			''
		);
		const stringConfirmPassword = _.get(
			sendData,
			'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.stringConfirmPassword',
			''
		);
		if (angular.isString(stringPassword) && stringPassword.length > 0) {
			_.set(
				sendData.office365Info.o365Exchange,
				'mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.password',
				this.cvUtil.getBytes(stringPassword)
			);
			delete sendData.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps
				.previewPathDir.userAccount.stringPassword;
			delete sendData.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps
				.previewPathDir.userAccount.stringConfirmPassword;
		} else {
			_.set(
				sendData.office365Info.o365Exchange,
				'mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.password',
				[]
			);
		}
		if (angular.isString(stringConfirmPassword) && stringConfirmPassword.length > 0) {
			_.set(
				sendData.office365Info.o365Exchange,
				'mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword',
				this.cvUtil.getBytes(stringConfirmPassword)
			);
		} else {
			_.set(
				sendData.office365Info.o365Exchange,
				'mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.previewPathDir.userAccount.confirmPassword',
				[]
			);
		}
		if (
			_.get(
				sendData,
				'office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.contentIndexProps.enableEntitySearch',
				false
			) === false
		) {
			_.set(sendData, 'eePolicy.detail.eePolicy.extractionPolicy.extractionPolicyList', []);
		}
		if (_.get(sendData, 'summary.subtype', 'Office365') === 'Dynamics365') {
			delete sendData.office365Info.o365Exchange;
		}

		return sendData;
	}
	/**
	 * Saves the current plan instance in the back-end.
	 *
	 * @return {Promise} Since this method returns a regular Promise, be sure to use the "then" method when
	 *         waiting for the function to complete, rather than the "success" or "error" methods. The reason
	 *         a regular promise is returned is because an HTTP call WILL NOT be called if no data for the
	 *         plan was changed.
	 */
	save() {
		var deferred = this.$q.defer();
		this.error = '';
		if (!angular.equals(this.planInfo, this.originalPlanInfo)) {
			if (
				this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan !==
				this.originalPlanInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy
					.includeMsgsOlderThan
			) {
				const callBackFunctions = {
					noFunction: () => {
						this.planInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan = this.originalPlanInfo.office365Info.o365Exchange.mbArchiving.detail.emailPolicy.archivePolicy.includeMsgsOlderThan;
						return;
					},
					yesFunction: () => {
						this.saveData(deferred);
					}
				};
				this.$dialogs.confirm(
					this.cvLoc('header.changedArchiveRules'),
					this.cvLoc('label.archiveRuleChangeWarningMessage'),
					callBackFunctions
				);
			} else {
				this.saveData(deferred);
			}
		} else {
			deferred.resolve();
		}
		return deferred.promise;
	}

	saveData(deferred) {
		const sendData = this.preProcessData();
		if (sendData.errorProcessing) {
			return deferred.reject(sendData.errorProcessing);
		}
		delete sendData.errorProcessing;
		var UpdatePlanEEPolicyInfo = {
			eePolicy: this.planInfo.eePolicy
		};

		this.profileService
			.editOffice365Plan(this.planInfo.summary.plan.planId, sendData.office365Info, UpdatePlanEEPolicyInfo)
			.then(
				resp => {
					//this.refactorRetentionValues();
					this.originalPlanInfo = _.cloneDeep(this.planInfo);
					deferred.resolve();
				},
				resp => {
					deferred.reject(resp.data);
				}
			);
	}

	/**
	 * Opens the modal used for editing the archive mailbox types of the current User Mailbox Plan.
	 *
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditArchiveSettingsModal() {
		const showData = {
			isOffice365: true,
			accordianTitle: 'label.backupMessages',
			labelCustomArchiveSettings: 'label.customBackupSettings'
		};
		return this.openEditModal(
			this.cvLoc('label.editExchangeSettings'),
			this.exchangeConstants.pathTemplate.ARCHIVE,
			showData
		);
	}

	/**
	 * Open the modal for editing the retention rules of the plan.
	 *
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditRetentionRulesModal(isArchiveMailbox) {
		const showData = {};
		showData.isArchiveMailbox = isArchiveMailbox;
		showData.isOffice365 = true;
		showData.modalWindowType = 'small-size';
		return this.openEditModal(
			this.cvLoc('label.editRetentionSettings'),
			this.exchangeConstants.pathTemplate.RETENTION,
			showData
		);
	}

	openEditSearchSettingsModal(showContent, showExactSearch) {
		const showData = {};
		showData.showExactSearch = showExactSearch;
		if (_.isUndefined(showContent)) {
			showData.showAllSearchEntity = true;
			showData.modalWindowType = '';
		} else {
			showData[showContent] = true;
			showData.modalWindowType = 'small-size';
		}
		return this.openEditModal(
			this.cvLoc('label.editSearchSettings'),
			this.exchangeConstants.pathTemplate.SEARCH_SETTINGS,
			showData
		)
			.result.then(resp => {
				this.originalPlanInfo = _.cloneDeep(this.planInfo);
			})
			.catch(resp => {
				this.planInfo = _.cloneDeep(this.originalPlanInfo);
			});
	}

	openEditOneDriveBackupSettingsModal() {
		this.openEditModal(this.cvLoc('label.editOneDriveSettings'), this.exchangeConstants.pathTemplate.ONEDRIVE_BACKUP);
	}

	openEditSharepointBackupSettingsModal() {
		this.openEditModal(
			this.cvLoc('label.editSharepointSettings'),
			this.exchangeConstants.pathTemplate.SHAREPOINT_BACKUP
		);
	}

	/**
	 * Opens a modal to edit content for an exchange plan.
	 *
	 * @param {string}
	 *            modalTitle - The title of the modal that will be displayed.
	 * @param {string}
	 *            templateUrl - The templateUrl to load within the body of the modal.
	 * @return {Promise} The resulting promise of the modal opened. It will be resolved if the modal is
	 *         closed, and rejected if the modal is dismissed.
	 */
	openEditModal(modalTitle, templateUrl, showData) {
		return this.$modal.open({
			templateUrl: 'modules/plan/partials/exchange/editModal.jsp',
			backdrop: 'static',
			windowClass: _.get(showData, 'modalWindowType', ''),
			controllerAs: 'exchangeEditModalController',
			controller: [
				'$uibModalInstance',
				'plan',
				'enableOneDriveIncludeFolder',
				'$uibModal',
				function($uibModalInstance, plan, enableOneDriveIncludeFolder, $modal) {
					let self = this;
					this.inModal = true;
					this.showData = showData || {};
					this.enableOneDriveIncludeFolder = enableOneDriveIncludeFolder;
					this.isArchiveMailbox = _.get(showData, 'isArchiveMailbox', false);
					this.isOffice365 = _.get(showData, 'isOffice365', false);
					this.plan = _.cloneDeep(plan);
					this.exchangeDetails = _.get(this.plan.planInfo, 'office365Info.o365Exchange', {});
					this.Office365Details = _.get(this.plan.planInfo, 'office365Info.o365CloudOffice', {});
					this.sharepointDetails = _.get(this.plan.planInfo, 'sharepointFilterDetails', {});
					this.oneDriveDetails = _.get(this.plan.planInfo, 'oneDriveFilterDetails', {});

					this.oneDriveDetails.includeFolderFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					this.oneDriveDetails.excludeFolderFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					this.oneDriveDetails.includeFileFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					this.oneDriveDetails.excludeFileFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					const oneDriveFilter = _.get(this.oneDriveDetails, 'filters', []);

					//temporarily assigning patternSelected to folderPatternsSelected so as to re-use add/remove filter
					oneDriveFilter.forEach(filter => {
						if (filter.filterType === 'FOLDER') {
							if (enableOneDriveIncludeFolder) {
								this.oneDriveDetails.includeFolderFilters.folderPatternsSelected = _.get(
									filter,
									'includeFilter.patternsSelected',
									[]
								);
							}
							this.oneDriveDetails.excludeFolderFilters.folderPatternsSelected = _.get(
								filter,
								'excludeFilter.patternsSelected',
								[]
							);
						} else if (filter.filterType === 'FILE') {
							this.oneDriveDetails.includeFileFilters.folderPatternsSelected = _.get(
								filter,
								'includeFilter.patternsSelected',
								[]
							);
							this.oneDriveDetails.excludeFileFilters.folderPatternsSelected = _.get(
								filter,
								'excludeFilter.patternsSelected',
								[]
							);
						}
					});

					this.sharepointDetails.includeWebFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					this.sharepointDetails.excludeWebFilters = {
						folderPatternsSelected: [],
						folderPatternsAvailable: []
					};
					const sharepointFilter = _.get(this.sharepointDetails, 'filters', []);

					sharepointFilter.forEach(filter => {
						if (filter.filterType === 'WEB') {
							this.sharepointDetails.includeWebFilters.folderPatternsSelected = _.get(
								filter,
								'includeFilter.patternsSelected',
								[]
							);
							this.sharepointDetails.excludeWebFilters.folderPatternsSelected = _.get(
								filter,
								'excludeFilter.patternsSelected',
								[]
							);
						}
					});

					this.modalTitle = modalTitle;
					this.templateUrl = templateUrl;

					this.addFilterValues = function(name, value) {
						//name 					: names of the filed to be updated: name[0] = patternSelected, name[1]: patternAvailable
						//value					: new value being added to patternSelected and removed from patternAvailable
						this[value] = this[value].trim();
						let dataArray = _.get(this, name[0], {});
						if (this[value].length > 0) {
							if (dataArray.folderPatternsSelected.length === 1 && dataArray.folderPatternsSelected[0] === '*') {
								dataArray.folderPatternsSelected = [];
							}
							dataArray.folderPatternsSelected.push(this[value]);
						}
						let pos = dataArray.folderPatternsAvailable.indexOf(this[value]);
						if (pos >= 0) {
							dataArray.folderPatternsAvailable.splice(pos, 1);
						}
						if (name.length > 1) {
							let updateArray = _.get(this, name[1], {});
							pos = updateArray.folderPatternsAvailable.indexOf(this[value]);
							if (pos >= 0) {
								updateArray.folderPatternsAvailable.splice(pos, 1);
								_.set(this, name[1], updateArray);
							}
						}
						_.set(this, name[0], dataArray);
						this[value] = '';
					};

					this.removeFilterValues = function(name, value, deleteAll) {
						let dataArray = _.get(this, name[0], {});
						if (deleteAll === false) {
							dataArray.folderPatternsSelected.splice(dataArray.folderPatternsSelected.indexOf(value), 1);
							dataArray.folderPatternsAvailable.push(value);
							if (name.length > 1) {
								let updateArray = _.get(this, name[1], {});
								updateArray.folderPatternsAvailable.push(value);
							}
						} else {
							_.set(
								dataArray,
								'folderPatternsAvailable',
								dataArray.folderPatternsAvailable.concat(dataArray.folderPatternsSelected)
							);
							if (name.length > 1) {
								let updateArray = _.get(this, name[1], {});
								_.set(
									updateArray,
									'folderPatternsAvailable',
									updateArray.folderPatternsAvailable.concat(dataArray.folderPatternsSelected)
								);
								_.set(this, name[1], updateArray);
							}
						}

						if (deleteAll === true || dataArray.folderPatternsSelected.length === 0) {
							dataArray.folderPatternsSelected = [];
						}
						_.set(this, name[0], dataArray);
					};

					/**
					 * When the cancel button is pressed.
					 */
					this.cancel = function() {
						$uibModalInstance.dismiss();
					};

					/**
					 * When the save button is pressed.
					 */
					this.save = function() {
						// _.set(
						// 	this.oneDriveFilterDetails,
						// 	'.filters',
						// 	_.get(this.oneDriveDetails, 'filters', [])
						// );

						this.oneDriveDetails.filters.forEach(filter => {
							if (filter.filterType === 'FOLDER') {
								if (this.enableOneDriveIncludeFolder) {
									_.set(
										filter,
										'includeFilter.patternsSelected',
										this.oneDriveDetails.includeFolderFilters.folderPatternsSelected
									);
								}
								_.set(
									filter,
									'excludeFilter.patternsSelected',
									this.oneDriveDetails.excludeFolderFilters.folderPatternsSelected
								);
							} else if (filter.filterType === 'FILE') {
								_.set(
									filter,
									'includeFilter.patternsSelected',
									this.oneDriveDetails.includeFileFilters.folderPatternsSelected
								);
								_.set(
									filter,
									'excludeFilter.patternsSelected',
									this.oneDriveDetails.excludeFileFilters.folderPatternsSelected
								);
							}
						});

						// _.set(
						// 	this.sharepointDetails,
						// 	'caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails[0].filters',
						// 	_.get(this.sharepointDetails, 'caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails[0].filters', [])
						// );

						this.sharepointDetails.filters.forEach(filter => {
							if (filter.filterType === 'WEB') {
								_.set(
									filter,
									'includeFilter.patternsSelected',
									this.sharepointDetails.includeWebFilters.folderPatternsSelected
								);
								_.set(
									filter,
									'excludeFilter.patternsSelected',
									this.sharepointDetails.excludeWebFilters.folderPatternsSelected
								);
							}
						});

						_.set(this.plan.planInfo, 'office365Info.o365Exchange', this.exchangeDetails);

						this.plan.planInfo.office365Info.o365CloudOffice.caBackup.detail.cloudAppPolicy.backupPolicy.ruleDetails.forEach(
							rule => {
								if (rule.agentType === 'ONEDRIVE') {
									_.set(rule, 'filters', this.oneDriveDetails.filters);
									if (
										this.oneDriveDetails.excludeFolderFilters.folderPatternsSelected.length > 0 ||
										this.oneDriveDetails.includeFolderFilters.folderPatternsSelected.length > 0 ||
										this.oneDriveDetails.includeFileFilters.folderPatternsSelected.length > 0 ||
										this.oneDriveDetails.excludeFileFilters.folderPatternsSelected.length > 0
									) {
										rule.ruleEnabled = true;
									} else {
										rule.ruleEnabled = false;
									}
								} else if (rule.agentType === 'SHAREPOINT') {
									_.set(rule, 'filters', this.sharepointDetails.filters);
									if (
										this.sharepointDetails.includeWebFilters.folderPatternsSelected.length > 0 ||
										this.sharepointDetails.excludeWebFilters.folderPatternsSelected.length > 0
									) {
										rule.ruleEnabled = true;
									} else {
										rule.ruleEnabled = false;
									}
								}
							}
						);

						//only when a user clicks the save button, the changed planinfo moves to original plan object
						plan.planInfo = _.cloneDeep(this.plan.planInfo);
						plan.save().then(() => {
							$uibModalInstance.close(this.plan);
						});
					};
				}
			],
			resolve: {
				plan: this,
				enableOneDriveIncludeFolder: this.enableOneDriveIncludeFolder
			}
		});
	}
}
export { office365PlanFactory };
