import { commonPlanModule } from 'common/js/modules';

var app = commonPlanModule;
/*
 * Wrapper to isteven-multi-select to accommodate html template in place of an item in the dropdown. Every
 * item will have a name and a summary of the plan in the next line, like this:
 *
 * Gold plan RPO: 4 hours | Number of copies: 1 | Associated entities: 3
 *
 * Silver plan RPO: 12 hours | Number of copies: 3 | Associated entities: 9
 *
 */

app.directive('cvPlanSummaryDropdown', [
	function() {
		return {
			restrict: 'E',
			scope: {
				//required attributes
				inputModel: '=', //type: Array
				outputModel: '=', //Note: output-model should always be an array regardless of selection-mode
				buttonLabel: '=', //rest all attribute types except callback functions should be a string
				itemLabel: '=',

				//optional attributes
				directiveId: '=?',
				tickProperty: '=?',
				searchProperty: '=?',
				disableProperty: '=?',
				groupProperty: '=?',
				maxLabels: '=?',
				selectionMode: '=?',
				helperElements: '=?',
				searchPlaceholder: '=?',
				customConfig: '=?', //Enables custom template for description {locVar: <val>, label: <val>}
				showNone: '=?',
				translation: '=?',
				//callback functions
				onItemClicked: '&',
				onSelectNone: '&',
				onSelectAll: '&',
				onReset: '&',
				onClose: '&'
			},
			templateUrl: appUtil.appRoot + 'modules/plan/partials/cvPlanSummaryDropdown.jsp',
			controllerAs: 'ctrl',
			controller: [
				'$scope',
				'cvUtil',
				'cvLoc',
				'$filter',
				'$sce',
				function($scope, cvUtil, cvLoc, $filter, $sce) {
					var self = this;
					self.localLang = $scope.translation ? $scope.translation : cvUtil.getIStevenLocLabels();

					//Override default search placeholder, if searchPlaceholder is provided
					if ($scope.searchPlaceholder) {
						self.localLang.search = $scope.searchPlaceholder;
					}

					initDefaultOptions();
					let defaultPlan = {
						planName: cvLoc('label.none'),
						planId: -1, // To dissociate plan
						isNone: true
					};

					$scope.$watch('inputModel.length', () => {
						if (angular.isArray($scope.inputModel)) {
							// Seperate groups, sort & re-group
							let _planTypes = _.uniq(
									_.map(
										_.filter($scope.inputModel, plan => plan.type),
										plan => plan.type
									)
								),
								_plansWithoutGroups = _.filter(
									$scope.inputModel,
									plan => plan[$scope.groupProperty || 'msGroup'] === undefined
								),
								_sortedPlansWithGroups = [];

							_plansWithoutGroups.sort((a, b) => {
								let AssociatedEntitiesA = a.numAssocEntities || 0;
								let AssociatedEntitiesB = b.numAssocEntities || 0;
								if (b.planId === -1) {
									return 1;
								}
								return AssociatedEntitiesB - AssociatedEntitiesA;
							});
							//show none option for grouped plan list
							if ($scope.showNone && $scope.groupProperty) {
								_sortedPlansWithGroups.push(defaultPlan);
							}

							// Group only if $scope.groupProperty is defined
							if ($scope.groupProperty) {
								_.each(_planTypes, planType => {
									let groupSt = {},
										groupEnd = {};
									groupSt[$scope.itemLabel] = planType;
									groupSt[$scope.groupProperty] = true;
									groupEnd[$scope.groupProperty] = false;
									_sortedPlansWithGroups.push(groupSt);

									_sortedPlansWithGroups = _sortedPlansWithGroups.concat(
										_.filter(_plansWithoutGroups, { type: planType })
									);
									_sortedPlansWithGroups.push(groupEnd);
								});
							}

							//show none option in the list for non-grouped plan list
							if ($scope.showNone && !_sortedPlansWithGroups.length) {
								$scope.showNone = false;
								_plansWithoutGroups.unshift(defaultPlan);
							}
							$scope.inputModel = _sortedPlansWithGroups.length ? _sortedPlansWithGroups : _plansWithoutGroups;
							_.each($scope.inputModel, initPlanTemplate);
						}
					});

					function initDefaultOptions() {
						$scope.selectionMode = $scope.selectionMode || 'multiple';
						$scope.maxLabels = $scope.maxLabels || 1;
						$scope.helperElements = $scope.helperElements || '';
						$scope.directiveId = $scope.directiveId || 'plans';
						if (angular.isDefined($scope.outputModel)) {
							$scope.selectedItems = [];
						}
					}

					function initPlanTemplate(planEntity) {
						const itemText = _.escape(_.get(planEntity, $scope.itemLabel));
						if (planEntity[$scope.groupProperty]) {
							planEntity.template = `
					<div class="isteven-html-content" title="${itemText}">
						<div class="title">${itemText}</div>
					</div>`;
						} else if (planEntity.isNone || planEntity.hideDescription) {
							planEntity.template = `
						<div class="isteven-html-content" title="${itemText}">
							<div class="title padding-top-5 ">${itemText}</div>
						</div>`;
						} else {
							planEntity.template = `
					<div class="isteven-html-content" title="${itemText}">
						<div>${itemText}</div>
						<div class="description">
							<label>${_.escape(planSummaryFilter(planEntity, $scope.customConfig))}</label>
						</div>
					</div>`;
						}
					}

					function planSummaryFilter(planEntity, customConfig) {
						return $filter('cvPlanSummary')(planEntity, customConfig);
					}

					$scope.onItemClickCallBack = function(selectedPlan) {
						if (angular.isArray($scope.outputModel)) {
							$scope.outputModel.length = 0;
							$scope.outputModel.push(...$scope.selectedItems);
						}

						if (angular.isFunction($scope.onItemClicked)) {
							$scope.onItemClicked({
								selectedPlan: selectedPlan
							});
						}
					};

					$scope.onSelectNoneCallBack = function(e) {
						if (angular.isFunction($scope.onSelectNone)) {
							$scope.onSelectNone(e);
						}
					};

					$scope.onSelectAllCallBack = function(e) {
						if (angular.isFunction($scope.onSelectAll)) {
							$scope.onSelectAll(e);
						}
					};

					$scope.onResetCallBack = function(e) {
						if (angular.isFunction($scope.onReset)) {
							$scope.onReset(e);
						}
					};

					$scope.onCloseCallBack = function(e) {
						if (angular.isFunction($scope.onClose)) {
							$scope.onClose(e);
						}
					};
				}
			]
		};
	}
]);

export default app;
