import { acAppNetworkTopologyModule } from 'common/js/modules';

import 'networkTopology/js/services/NetworkTopology.svc';
import 'modules/network/js/constants/nwtopology.constant.js';
import 'common/js/cv-dialog-factory.js';

/*
 * Terminology for mapping firstClientGroup = FwGroupType.EXTERNAL.id thirdClientGroup =
 * FwGroupType.PROXIED.id fourthClientGroup = FwGroupType.CASCADINGGATEWAY.id
 */

acAppNetworkTopologyModule.factory('NetworkTopologyFactory', [
	'cvLoc',
	'NetworkTopologyService',
	'cvToaster',
	'$uibModal',
	'NW_TOPOLOGY_TYPE',
	'$state',
	'dialogFactory',
	function(cvLoc, NetworkTopologyService, cvToaster, $modal, NW_TOPOLOGY_TYPE, $state, dialogFactory) {
		var NetworkTopologyFactory = {};

		NetworkTopologyFactory.xmlToJson = function(xml) {
			// Create the return object
			var obj = {};

			if (xml.nodeType === 1) {
				// element
				// do attributes
				if (xml.attributes.length > 0) {
					obj['attributes'] = {};
					for (var j = 0; j < xml.attributes.length; j++) {
						var attribute = xml.attributes.item(j);
						obj['attributes'][attribute.nodeName] = attribute.nodeValue;
					}
				}
			} else if (xml.nodeType === 3) {
				// text
				obj = xml.nodeValue;
			}

			// do children
			// If just one text node inside
			if (xml.hasChildNodes() && xml.childNodes.length === 1 && xml.childNodes[0].nodeType === 3) {
				obj = xml.childNodes[0].nodeValue;
			} else if (xml.hasChildNodes()) {
				for (var i = 0; i < xml.childNodes.length; i++) {
					var item = xml.childNodes.item(i);
					var nodeName = item.nodeName;
					if (typeof obj[nodeName] == 'undefined') {
						obj[nodeName] = this.xmlToJson(item);
					} else {
						if (typeof obj[nodeName].push == 'undefined') {
							var old = obj[nodeName];
							obj[nodeName] = [];
							obj[nodeName].push(old);
						}
						obj[nodeName].push(this.xmlToJson(item));
					}
				}
			}
			return obj;
		};

		NetworkTopologyFactory.topologySpecificActionitems = {
			PUSH_CONFIGURATION: {
				label: cvLoc('action.push'),
				value: 'PUSH_CONFIGURATION'
			}
		};

		NetworkTopologyFactory.deleteTopology = function(topologyId) {
			var callBackFunctions = {
				noFunction: function() {},
				yesFunction: function() {
					NetworkTopologyService.deleteFWTopology(topologyId)
						.success(function() {
							$state.forceReload();
						})
						.error(function(e) {
							cvToaster.showErrorMessage({
								ttl: '10000',
								message: e ? e : cvLoc('error.deleteTopology')
							});
						});
				}
			};
			dialogFactory
				.deleteConfirmDialog(cvLoc('label.confirmDelete'), cvLoc('label.confirmDeleteTopology'))
				.result.then(callBackFunctions.yesFunction, callBackFunctions.noFunction);
		};

		NetworkTopologyFactory.pushFWTopology = function(topologyId, topologyName) {
			NetworkTopologyService.pushFWTopology(topologyId)
				.success(function() {
					cvToaster.showSuccessMessage({
						message: cvLoc('success.fwpush', topologyName)
					});
					//location.reload();
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: '10000',
						message: e ? e.errorMessage : cvLoc('error.fwpush') + topologyName
					});
				});
		};

		NetworkTopologyFactory.openAddToplogyModal = function(topology) {
			$modal.open({
				templateUrl: appUtil.appRoot + 'modules/networkTopology/partials/addTopologyModal.jsp',
				backdrop: 'static',
				controller: 'topologyModalController as tmc',
				resolve: {
					topology: function() {
						if (topology) {
							return NetworkTopologyService.getFWTopologyDetails(topology.topologyEntity.topologyId)
								.success(function(data) {
									return data.topologyInfo;
								})
								.error(function(e) {
									self.serverMessage = {
										type: 'error',
										message: e
									};
								});
						}
					}
				}
			});
		};

		NetworkTopologyFactory.openFWDownloadSummaryModal = function() {
			$modal.open({
				templateUrl: appUtil.appRoot + 'modules/networkTopology/partials/downloadFWSummary.jsp',
				windowClass: 'small-size',
				backdrop: false,
				controller: 'fwDownloadModalController as fdc',
				resolve: {
					fwEnabledClients: function() {
						return NetworkTopologyService.getFWEnabledClients()
							.success(function(data) {
								return data.clients;
							})
							.error(function(e) {
								self.serverMessage = {
									type: 'error',
									message: e
								};
							});
					}
				}
			});
		};

		NetworkTopologyFactory.getClientType = function(entity) {
			var NW_CLIENT_TYPE = {
				0: 'Server',
				1: 'Laptop'
			};
			var xmlDoc = jQuery.parseXML(entity.extendedProperties);
			var ePropJson = this.xmlToJson(xmlDoc);
			return NW_CLIENT_TYPE[ePropJson.App_TopologyExtendedProperties.attributes.displayType];
		};

		NetworkTopologyFactory.getNWTopologyType = function(entity) {
			if (entity == NW_TOPOLOGY_TYPE.PROXIED.id) {
				return NW_TOPOLOGY_TYPE.PROXIED.name;
			}

			if (entity == NW_TOPOLOGY_TYPE.ONEWAY.id) {
				return NW_TOPOLOGY_TYPE.ONEWAY.name;
			}

			if (entity == NW_TOPOLOGY_TYPE.TWOWAY.id) {
				return NW_TOPOLOGY_TYPE.TWOWAY.name;
			}

			if (entity == NW_TOPOLOGY_TYPE.CASCADINGGATEWAY.id) {
				return NW_TOPOLOGY_TYPE.CASCADINGGATEWAY.name;
			}

			return '';
		};

		return NetworkTopologyFactory;
	}
]);

// Modal Controller
acAppNetworkTopologyModule.controller('topologyModalController', [
	'$uibModalInstance',
	'$state',
	'topology',
	function($modalInstance, $state, topology) {
		var self = this;
		self.topology = topology;
		self.close = function(refresh) {
			$modalInstance.dismiss();
			if (refresh === true) {
				$state.forceReload();
			}
		};
	}
]);

//Modal Controller
acAppNetworkTopologyModule.controller('fwDownloadModalController', [
	'$uibModalInstance',
	'$state',
	'fwEnabledClients',
	function($modalInstance, $state, fwEnabledClients) {
		var self = this;
		self.fwEnabledClients = fwEnabledClients;
		self.close = function(refresh) {
			$modalInstance.dismiss();
		};
	}
]);

//Download FW Summary controller
acAppNetworkTopologyModule.controller('downloadFWSummaryController', [
	'NetworkTopologyService',
	'cvLoc',
	'cvUtil',
	'$scope',
	'$filter',
	'$document',
	'cvToaster',
	function(NetworkTopologyService, cvLoc, cvUtil, $scope, $filter, $document, cvToaster) {
		var self = this;
		self.downloadMessage = cvUtil.emptyMsg();

		self.uiModel = {
			listFWEnabledClients: [],
			selectedClient: undefined,
			translation: cvUtil.getIStevenLocLabels(),
			showError: false,
			showFWClientsError: false,
			errorMessage: undefined
		};

		self.init = function(fwEnabledClients) {
			if (fwEnabledClients && fwEnabledClients.data && fwEnabledClients.data.clients) {
				self.uiModel.listFWEnabledClients = angular.copy(fwEnabledClients.data.clients);
			} else {
				self.uiModel.showFWClientsError = true;
				var errorMessage = cvLoc('error.noFWClientsFound');
				self.downloadMessage = {
					type: 'error',
					message: errorMessage
				};
			}
		};

		self.clearErrors = function() {
			self.uiModel.errorMessage = cvUtil.emptyMsg();
			self.uiModel.showError = false;
			self.uiModel.showFWClientsError = false;
		};

		self.validateInputs = function() {
			self.clearErrors();

			if (self.uiModel.selectedClient === undefined || self.uiModel.selectedClient.length < 1) {
				self.uiModel.showError = true;
				return false;
			}

			return true;
		};

		self.downloadFWSummary = function(doneCallback) {
			if (!self.validateInputs()) {
				return;
			}

			NetworkTopologyService.downloadFWSummaryForClient(self.uiModel.selectedClient[0].clientId)
				.success(function(resp) {
					var blob = new Blob([resp], { type: 'text/plain, charset=UTF-8' });
					if (window.navigator.msSaveOrOpenBlob) {
						window.navigator.msSaveBlob(blob, 'FwConfig_' + self.uiModel.selectedClient[0].displayName + '.txt');
					} else {
						var hiddenElement = $document[0].createElement('a');
						hiddenElement.href = window.URL.createObjectURL(blob);
						$document[0].body.appendChild(hiddenElement);
						hiddenElement.target = '_blank';
						hiddenElement.download = 'FwConfig_' + self.uiModel.selectedClient[0].displayName + '.txt';
						hiddenElement.click();
						$document[0].body.removeChild(hiddenElement);
					}
					doneCallback(false);
					cvToaster.showSuccessMessage({
						message: cvLoc('success.downloadFWSummary', self.uiModel.selectedClient[0].displayName)
					});
				})
				.error(function(e) {
					var errorMessage = cvLoc('error.downloadFWSummary', self.uiModel.selectedClient[0].displayName);
					self.downloadMessage = {
						type: 'error',
						message: errorMessage
					};
				});
		};
	}
]);

// Add/Edit Controller
acAppNetworkTopologyModule.controller('topologyAddEditController', [
	'NetworkTopologyService',
	'clientGroupService',
	'cvLoc',
	'cvUtil',
	'$scope',
	'NW_CLIENT_TYPE',
	'NW_TOPOLOGY_TYPE',
	'SMART_TOPOLOGY_MNEMONICS',
	'FW_GROUP_TYPE',
	'NW_TOPOLOGY_PROTOCOL_TYPE',
	'$filter',
	'cvToaster',
	'NetworkTopologyFactory',
	function(
		NetworkTopologyService,
		clientGroupService,
		cvLoc,
		cvUtil,
		$scope,
		NWClientType,
		NWTopologyType,
		SmartTopologies,
		FwGroupType,
		NWTopologyProtocolType,
		$filter,
		cvToaster,
		NetworkTopologyFactory
	) {
		var self = this;

		self.clientType = NWClientType.SERVER.id;
		self.topologyType = NWTopologyType.PROXIED.id;
		self.defaultTunnelPort = 8403;
		self.defaultTunnelPortStr = cvLoc('label.defaultTunnelPort');
		self.defaultKeepAliveInterval = 180;
		self.defaultNumberOfStreams = 1;
		self.defaultConfigureAutomatically = 1;
		self.defaultTunnelProtocol = cvLoc('label.protocolAuthenticated');
		self.clientGroupNetworkOptions = {};

		self.localLang = cvUtil.getIStevenLocLabels();

		self.uiModel = {
			listClientGroups: [],
			firstClientGroup: undefined,
			secondClientGroup: undefined,
			thirdClientGroup: undefined,
			translation: self.localLang,
			firstClientGroupLabel: {},
			secondClientGroupLabel: {},
			thirdClientGroupLabel: {},
			fourthClientGroupLabel: {},
			showThirdClientGroup: true,
			showFourthClientGroup: false,
			showErrorFirstClientGroup: false,
			showErrorSecondClientGroup: false,
			showErrorThirdClientGroup: false,
			showErrorFourthClientGroup: false,
			topologyModes: [
				{ title: cvLoc('label.viaNetworkGateway'), id: NWTopologyType.PROXIED.id },
				{ title: cvLoc('label.oneway'), id: NWTopologyType.ONEWAY.id },
				{ title: cvLoc('label.twoway'), id: NWTopologyType.TWOWAY.id },
				{ title: cvLoc('label.cascadingGateways'), id: NWTopologyType.CASCADINGGATEWAY.id }
			],
			showAdvanced: false,
			tunnelProtocolList: [
				cvLoc('label.protocolEncrypted'),
				cvLoc('label.protocolAuthenticated'),
				cvLoc('label.protocolRegular'),
				cvLoc('label.protocolRaw')
			],
			numberOfStreamsList: [1, 2, 3, 4, 5, 6, 7, 8], // Todo : Figure out a cleaner approach
			//tunnelProtocolList : [{title : cvLoc("label.protocolEncrypted"), id : 1}  , {title : cvLoc("label.protocolAuthenticated"), id : 2}, {title : cvLoc("label.protocolRegular"), id : 3}, {title : cvLoc("label.protocolRaw"), id : 4}]
			showFirstGroupNetworkOptions: false,
			showSecondGroupNetworkOptions: false,
			showThirdGroupNetworkOptions: false,
			showFourthGroupNetworkOptions: false,
			firstGroupOptions: {
				configureAutomatically: self.defaultConfigureAutomatically,
				keepAliveInterval: self.defaultKeepAliveInterval,
				keepAliveIntervalLabel: self.defaultKeepAliveInterval,
				tunnelPort: '',
				tunnelPortLabel: self.defaultTunnelPortStr
			},
			secondGroupOptions: {
				configureAutomatically: self.defaultConfigureAutomatically,
				keepAliveInterval: self.defaultKeepAliveInterval,
				keepAliveIntervalLabel: self.defaultKeepAliveInterval,
				tunnelPort: '',
				tunnelPortLabel: self.defaultTunnelPortStr
			},
			thirdGroupOptions: {
				configureAutomatically: self.defaultConfigureAutomatically,
				keepAliveInterval: self.defaultKeepAliveInterval,
				keepAliveIntervalLabel: self.defaultKeepAliveInterval,
				tunnelPort: '',
				tunnelPortLabel: self.defaultTunnelPortStr
			},
			fourthGroupOptions: {
				configureAutomatically: self.defaultConfigureAutomatically,
				keepAliveInterval: self.defaultKeepAliveInterval,
				keepAliveIntervalLabel: self.defaultKeepAliveInterval,
				tunnelPort: '',
				tunnelPortLabel: self.defaultTunnelPortStr
			},
			advancedOptions: {
				tunnelProtocol: self.defaultTunnelProtocol,
				numberOfStreams: self.defaultNumberOfStreams,
				encryptTraffic: false
			}
		};

		self.init = function(inTopology) {
			// We need to fetch all group properties as we have to set it in the add/update topology request
			NetworkTopologyService.fetchClientGroupNetworkProperties()
				.success(function(resp) {
					self.initializeClientGroupNetworkProperties(resp);
				})
				.error(function(e) {
					var errorMessage = cvLoc('error.addtopology', newTopology.firewallTopology.topologyEntity.topologyName);
					self.loading = false;
					self.topologiesMessage = {
						type: 'error',
						message: errorMessage
					};
				});

			if (inTopology) {
				self.createTopology = false;
				self.topology = inTopology.data.topologyInfo;

				// Get the client type
				var xmlDoc = jQuery.parseXML(self.topology.extendedProperties);
				self.topologyExtendedProps = NetworkTopologyFactory.xmlToJson(xmlDoc);

				self.topology.clientType = parseInt(
					self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.displayType
				);
			} else {
				self.createTopology = true;
			}

			self.showCorrectLabelsForUI(self.topology);

			self.initializeGroupOptions();
			self.initializeTopoSpecificAdvancedOptions();
			clientGroupService.getClientGroups().then(initializeClientGroups, showErrorMessage);
		};

		// Return back list of grouped client groups and smart topology mnemonics
		function getGroupedClientGroups(clientGroupsList) {
			var SMART_TOPOLOGY_MNEMONICS = [
				SmartTopologies.MY_COMMSERVE,
				SmartTopologies.MY_MEDIAAGENTS,
				SmartTopologies.MY_COMMSERVE_AND_MEDIAAGENTS
			];

			var groupedClientGroups = [];

			groupedClientGroups.push({
				name: cvLoc('heading.smarttopology'),
				group: true
			});
			Array.prototype.push.apply(groupedClientGroups, SMART_TOPOLOGY_MNEMONICS);
			groupedClientGroups.push({
				group: false
			});

			groupedClientGroups.push({
				name: cvLoc('heading.ClientGroups'),
				group: true
			});
			Array.prototype.push.apply(groupedClientGroups, angular.copy(_.sortBy(clientGroupsList, 'name')));
			groupedClientGroups.push({
				group: false
			});

			return groupedClientGroups;
		}

		function preselectClientGroupForDropdown(topologyData, clientGroupList, groupType) {
			var clientGroup = $filter('filter')(
				clientGroupList,
				{
					Id: $filter('filter')(topologyData.firewallGroups, { fwGroupType: groupType })[0].clientGroup.clientGroupId
				},
				true
			);
			clientGroup[0].selected = true;
			clientGroup[0].ticked = true;
		}

		function initializeClientGroups(clientGroupsList) {
			var groupedClientGroups = getGroupedClientGroups(clientGroupsList.data);

			if (self.topology !== undefined && self.topology.clientType === NWClientType.SERVER.id) {
				self.uiModel.listFirstClientGroups = angular.copy(groupedClientGroups);
			} else {
				self.uiModel.listFirstClientGroups = angular.copy(_.sortBy(clientGroupsList.data, 'name'));
			}

			self.uiModel.listSecondClientGroups = angular.copy(groupedClientGroups);
			self.uiModel.listThirdClientGroups = angular.copy(_.sortBy(clientGroupsList.data, 'name'));
			self.uiModel.listFourthClientGroups = angular.copy(_.sortBy(clientGroupsList.data, 'name'));
			self.uiModel.firstClientGroup = [];
			self.uiModel.secondClientGroup = [];
			self.uiModel.thirdClientGroup = [];
			self.uiModel.fourthClientGroup = [];

			if (self.topology.firewallGroups) {
				// first client group - fwGroupType = 2
				preselectClientGroupForDropdown(self.topology, self.uiModel.listFirstClientGroups, FwGroupType.EXTERNAL.id);

				// second client group - fwGroupType = 1
				preselectClientGroupForDropdown(self.topology, self.uiModel.listSecondClientGroups, FwGroupType.INTERNAL.id);

				// via proxy type or via cascading gateway
				if (
					self.topology.topologyType === NWTopologyType.PROXIED.id ||
					self.topology.topologyType == NWTopologyType.CASCADINGGATEWAY.id
				) {
					// thrid client group - fwGroupType = 3
					preselectClientGroupForDropdown(self.topology, self.uiModel.listThirdClientGroups, FwGroupType.PROXIED.id);
				}

				// via cascading gateway
				if (self.topology.topologyType == NWTopologyType.CASCADINGGATEWAY.id) {
					preselectClientGroupForDropdown(
						self.topology,
						self.uiModel.listFourthClientGroups,
						FwGroupType.CASCADINGGATEWAY.id
					);
				}
			}
		}

		function showErrorMessage(errorMessage) {
			if (errorMessage) {
				if (angular.isString(errorMessage)) {
					self.uiModel.serverMessage = cvUtil.errMsg(errorMessage);
				} else if (angular.isString(errorMessage.data)) {
					self.uiModel.serverMessage = cvUtil.errMsg(errorMessage.data);
				} else {
					self.uiModel.serverMessage = cvLoc('generic_error');
				}
			}
		}

		// smart ui checks
		self.updateTopology = function(data) {
			self.clearErrors();

			// clear all disabled properties and disable the not chosen one

			// first client group
			var firstcg = $filter('filter')(self.uiModel.listFirstClientGroups, {
				disabled: true
			});
			if (firstcg != undefined && firstcg.length > 0) {
				firstcg.forEach(function(i) {
					i.disabled = false;
				});
			}

			// second client group
			var secondcg = $filter('filter')(self.uiModel.listSecondClientGroups, {
				disabled: true
			});
			if (secondcg != undefined && secondcg.length > 0) {
				secondcg.forEach(function(j) {
					j.disabled = false;
				});
			}

			// third client group
			var thirdcg = $filter('filter')(self.uiModel.listThirdClientGroups, {
				disabled: true
			});
			if (thirdcg != undefined && thirdcg.length > 0) {
				thirdcg[0].disabled = false;
			}

			// fourth client group
			var fourthcg = $filter('filter')(self.uiModel.listFourthClientGroups, {
				disabled: true
			});
			if (fourthcg != undefined && fourthcg.length > 0) {
				fourthcg[0].disabled = false;
			}

			// Get all the selected ids
			var firstcgselectedId;
			var firstcgselected = $filter('filter')(self.uiModel.listFirstClientGroups, {
				selected: true
			});
			if (firstcgselected != undefined && firstcgselected.length > 0) {
				firstcgselectedId = firstcgselected[0].Id;
			} else {
				firstcgselectedId = 0;
			}

			var secondcgselectedId;
			var secondcgselected = $filter('filter')(self.uiModel.listSecondClientGroups, {
				selected: true
			});
			if (secondcgselected != undefined && secondcgselected.length > 0) {
				secondcgselectedId = secondcgselected[0].Id;
			} else {
				secondcgselectedId = 0;
			}

			var thirdcgselectedId;
			var thirdcgselected = $filter('filter')(self.uiModel.listThirdClientGroups, {
				selected: true
			});
			if (thirdcgselected != undefined && thirdcgselected.length > 0) {
				thirdcgselectedId = thirdcgselected[0].Id;
			} else {
				thirdcgselectedId = 0;
			}

			var fourthcgselectedId;
			var fourthcgselected = $filter('filter')(self.uiModel.listFourthClientGroups, {
				selected: true
			});
			if (fourthcgselected != undefined && fourthcgselected.length > 0) {
				fourthcgselectedId = fourthcgselected[0].Id;
			} else {
				fourthcgselectedId = 0;
			}

			// data 1 - first scg, 2 - 2nd scg, 3 - 3rd scg
			if (data === 1) {
				self.uiModel.listFirstClientGroups.forEach(function(j) {
					if (j.Id === secondcgselectedId || j.Id === thirdcgselectedId || j.Id === fourthcgselectedId) {
						j.disabled = true;
					}
					// only 1 smart group can be selected
					if (secondcgselectedId < 0 && j.Id < 0) {
						j.disabled = true;
					}
				});
			}

			if (data === 2) {
				self.uiModel.listSecondClientGroups.forEach(function(j) {
					if (j.Id === firstcgselectedId || j.Id === thirdcgselectedId || j.Id === fourthcgselectedId) {
						j.disabled = true;
					}
					// only 1 smart group can be selected
					if (firstcgselectedId < 0 && j.Id < 0) {
						j.disabled = true;
					}
				});
			}

			if (data === 3) {
				self.uiModel.listThirdClientGroups.forEach(function(j) {
					if (j.Id === secondcgselectedId || j.Id === firstcgselectedId || j.Id === fourthcgselectedId) {
						j.disabled = true;
					}
				});
			}

			if (data === 4) {
				self.uiModel.listFourthClientGroups.forEach(function(j) {
					if (j.Id === secondcgselectedId || j.Id === firstcgselectedId || j.Id === thirdcgselectedId) {
						j.disabled = true;
					}
				});
			}
		};

		self.clearErrors = function() {
			self.topologiesMessage = cvUtil.emptyMsg();
			self.uiModel.showErrorFirstClientGroup = false;
			self.uiModel.showErrorSecondClientGroup = false;
			self.uiModel.showErrorThirdClientGroup = false;
			self.uiModel.showErrorFourthClientGroup = false;
		};

		self.showCorrectLabelsForUI = function(topology) {
			self.clearErrors();

			if (topology === undefined) {
				// Setting the defaults
				self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.servers');
				self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
				self.uiModel.thirdClientGroupLabel = cvLoc('label.commonType.infrastructureGateways');
				self.uiModel.showThirdClientGroup = true;
				self.topology = {
					clientType: NWClientType.SERVER.id, // server
					topologyType: NWTopologyType.PROXIED.id // proxy
				};
				return;
			}

			// Client type - server
			if (topology.clientType === NWClientType.SERVER.id) {
				self.uiModel.topologyModes = [
					{ title: cvLoc('label.viaNetworkGateway'), id: 1 },
					{ title: cvLoc('label.oneway'), id: 2 },
					{ title: cvLoc('label.twoway'), id: 3 },
					{ title: cvLoc('label.cascadingGateways'), id: 4 }
				];

				self.uiModel.showThirdClientGroup = false;
				self.uiModel.showFourthClientGroup = false;
				switch (parseInt(topology.topologyType)) {
					case NWTopologyType.PROXIED.id /* via proxy */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.servers');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						self.uiModel.thirdClientGroupLabel = cvLoc('label.commonType.infrastructureGateways');
						self.uiModel.showThirdClientGroup = true;
						break;
					case NWTopologyType.ONEWAY.id /* oneway */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.servers');
						self.uiModel.secondClientGroupLabel = cvLoc('label.dmzServers');
						break;
					case NWTopologyType.TWOWAY.id /* twoway */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.servers');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						break;
					case NWTopologyType.CASCADINGGATEWAY.id /* cascading gateways */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.servers');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						self.uiModel.thirdClientGroupLabel = cvLoc('label.serverGateways');
						self.uiModel.fourthClientGroupLabel = cvLoc('label.commonType.infrastructureGateways');
						self.uiModel.showThirdClientGroup = true;
						self.uiModel.showFourthClientGroup = true;
						break;
				}

				// Check if the firstcg group has smart topology, if not, add it
				if (self.uiModel.listFirstClientGroups !== undefined) {
					var isGroupPresent = $filter('filter')(self.uiModel.listFirstClientGroups, {
						group: true
					});

					if (isGroupPresent === undefined || (isGroupPresent !== undefined && isGroupPresent.length === 0)) {
						self.uiModel.listFirstClientGroups = angular.copy(
							getGroupedClientGroups(self.uiModel.listFirstClientGroups)
						);
					}
				}
			} else if (topology.clientType === NWClientType.LAPTOP.id) {
				// two way not an option so hide it
				self.uiModel.topologyModes = [
					{ title: cvLoc('label.viaNetworkGateway'), id: 1 },
					{ title: cvLoc('label.oneway'), id: 2 },
					{ title: cvLoc('label.cascadingGateways'), id: 4 }
				];

				self.uiModel.showFourthClientGroup = false;

				switch (parseInt(topology.topologyType)) {
					case NWTopologyType.PROXIED.id /* via proxy */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.laptops');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						self.uiModel.thirdClientGroupLabel = cvLoc('label.commonType.infrastructureGateways');
						self.uiModel.showThirdClientGroup = true;
						break;
					case NWTopologyType.ONEWAY.id /* oneway */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.laptops');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						self.uiModel.showThirdClientGroup = false;
						break;
					case NWTopologyType.CASCADINGGATEWAY.id /* cascading gateways */:
						self.uiModel.firstClientGroupLabel = cvLoc('label.clientType.laptops');
						self.uiModel.secondClientGroupLabel = cvLoc('label.commonType.infrastructureMachines');
						self.uiModel.thirdClientGroupLabel = cvLoc('label.laptopGateways');
						self.uiModel.fourthClientGroupLabel = cvLoc('label.commonType.infrastructureGateways');
						self.uiModel.showThirdClientGroup = true;
						self.uiModel.showFourthClientGroup = true;
						break;
				}

				// first the first client group to not show smart topology
				if (self.uiModel.listFirstClientGroups !== undefined) {
					self.uiModel.listFirstClientGroups = self.uiModel.listFirstClientGroups.filter(function(obj) {
						return obj.Id > 0;
					});
				}
			}
		};

		self.validateInputs = function() {
			self.clearErrors();
			var errorFound = false;
			if (self.uiModel.firstClientGroup === undefined || self.uiModel.firstClientGroup.length < 1) {
				self.uiModel.showErrorFirstClientGroup = true;
				errorFound = true;
			}

			if (self.uiModel.secondClientGroup === undefined || self.uiModel.secondClientGroup.length < 1) {
				self.uiModel.showErrorSecondClientGroup = true;
				errorFound = true;
			}

			if (
				(self.topology.topologyType === NWTopologyType.PROXIED.id ||
					self.topology.topologyType === NWTopologyType.CASCADINGGATEWAY.id) &&
				(self.uiModel.thirdClientGroup === undefined ||
					self.uiModel.thirdClientGroup.length < 1 ||
					self.uiModel.thirdClientGroup[0].Id < 1)
			) {
				self.uiModel.showErrorThirdClientGroup = true;
				errorFound = true;
			}

			if (
				self.topology.topologyType === NWTopologyType.CASCADINGGATEWAY.id &&
				(self.uiModel.fourthClientGroup === undefined ||
					self.uiModel.fourthClientGroup.length < 1 ||
					self.uiModel.fourthClientGroup[0].Id < 1)
			) {
				self.uiModel.showErrorFourthClientGroup = true;
				errorFound = true;
			}

			return errorFound;
		};

		self.sanitizeInputs = function() {};

		self.getSerialNumber = function() {
			var baseSerialNumber = 2;
			self.uiModel.showThirdClientGroup && baseSerialNumber++;
			self.uiModel.showFourthClientGroup && baseSerialNumber++;
			return baseSerialNumber;
		};

		// Function to create/save topology
		self.createOrSaveTopology = function(doneCallback) {
			// Check if all things are good.
			if (self.validateInputs()) {
				return;
			}

			self.sanitizeInputs();

			var encryptTraffic = self.uiModel.advancedOptions.encryptTraffic == true ? 1 : 0;

			var tunnelProtocol = self.getTunnelProtocolID(self.uiModel.advancedOptions.tunnelProtocol);

			var extendedPropertiesXML =
				'<App_TopologyExtendedProperties displayType="' +
				self.topology.clientType +
				'" ' +
				'encryptTraffic="' +
				encryptTraffic +
				'" ' +
				'numberOfStreams="' +
				self.uiModel.advancedOptions.numberOfStreams +
				'" ' +
				'connectionProtocol="' +
				tunnelProtocol +
				'" ' +
				'/>';

			if (self.createTopology) {
				var newTopology = {
					firewallTopology: {
						topologyEntity: {
							topologyName: self.topology.topologyEntity.topologyName
						},
						topologyType: self.topology.topologyType,
						useWildcardProxy: false,
						extendedProperties: extendedPropertiesXML
					}
				};

				if (self.topology.useWildcardProxy !== undefined) {
					newTopology.firewallTopology.useWildcardProxy = self.topology.useWildcardProxy;
				}

				newTopology.firewallTopology.firewallGroups = [
					{
						fwGroupType: FwGroupType.EXTERNAL.id,
						isMnemonic: self.uiModel.firstClientGroup[0].Id < 0 ? true : false,
						clientGroup: {
							clientGroupId: self.uiModel.firstClientGroup[0].Id
						},
						options: {
							configureAutomatically: self.uiModel.firstGroupOptions.configureAutomatically == 0 ? false : true,
							tunnelConnectionPort:
								self.uiModel.firstGroupOptions.tunnelPort == ''
									? self.defaultTunnelPort
									: parseInt(self.uiModel.firstGroupOptions.tunnelPort),
							keepAliveSeconds: parseInt(self.uiModel.firstGroupOptions.keepAliveInterval)
						}
					},
					{
						fwGroupType: FwGroupType.INTERNAL.id,
						isMnemonic: self.uiModel.secondClientGroup[0].Id < 0 ? true : false,
						clientGroup: {
							clientGroupId: self.uiModel.secondClientGroup[0].Id
						},
						options: {
							configureAutomatically: self.uiModel.secondGroupOptions.configureAutomatically == 0 ? false : true,
							tunnelConnectionPort:
								self.uiModel.secondGroupOptions.tunnelPort == ''
									? self.defaultTunnelPort
									: parseInt(self.uiModel.secondGroupOptions.tunnelPort),
							keepAliveSeconds: parseInt(self.uiModel.secondGroupOptions.keepAliveInterval)
						}
					}
				];

				// Only for proxy push the data about 3rd client group
				if (self.topology.topologyType === NWTopologyType.PROXIED.id) {
					newTopology.firewallTopology.firewallGroups.push({
						fwGroupType: FwGroupType.PROXIED.id,
						isMnemonic: false,
						clientGroup: {
							clientGroupId: self.uiModel.thirdClientGroup[0].Id
						},
						options: {
							configureAutomatically: self.uiModel.thirdGroupOptions.configureAutomatically == 0 ? false : true,
							tunnelConnectionPort:
								self.uiModel.thirdGroupOptions.tunnelPort == ''
									? self.defaultTunnelPort
									: parseInt(self.uiModel.thirdGroupOptions.tunnelPort),
							keepAliveSeconds: parseInt(self.uiModel.thirdGroupOptions.keepAliveInterval)
						}
					});
				} else if (self.topology.topologyType === NWTopologyType.CASCADINGGATEWAY.id) {
					newTopology.firewallTopology.firewallGroups.push(
						{
							fwGroupType: FwGroupType.PROXIED.id,
							isMnemonic: false,
							clientGroup: {
								clientGroupId: self.uiModel.thirdClientGroup[0].Id
							},
							options: {
								configureAutomatically: self.uiModel.thirdGroupOptions.configureAutomatically == 0 ? false : true,
								tunnelConnectionPort:
									self.uiModel.thirdGroupOptions.tunnelPort == ''
										? self.defaultTunnelPort
										: parseInt(self.uiModel.thirdGroupOptions.tunnelPort),
								keepAliveSeconds: parseInt(self.uiModel.thirdGroupOptions.keepAliveInterval)
							}
						},
						{
							fwGroupType: FwGroupType.CASCADINGGATEWAY.id,
							isMnemonic: false,
							clientGroup: {
								clientGroupId: self.uiModel.fourthClientGroup[0].Id
							},
							options: {
								configureAutomatically: self.uiModel.fourthGroupOptions.configureAutomatically == 0 ? false : true,
								tunnelConnectionPort:
									self.uiModel.fourthGroupOptions.tunnelPort == ''
										? self.defaultTunnelPort
										: parseInt(self.uiModel.fourthGroupOptions.tunnelPort),
								keepAliveSeconds: parseInt(self.uiModel.fourthGroupOptions.keepAliveInterval)
							}
						}
					);
				}

				// Set smart topology flag
				if (self.uiModel.firstClientGroup[0].Id < 0 || self.uiModel.secondClientGroup[0].Id < 0) {
					newTopology.firewallTopology.isSmartTopology = true;
				}

				NetworkTopologyService.saveTopology(newTopology)
					.success(function(resp) {
						var errorMessage = cvLoc('error.addtopology', newTopology.firewallTopology.topologyEntity.topologyName);
						if (resp !== undefined && resp.error !== undefined) {
							if (resp.error.errorCode === 0) {
								doneCallback(true);
								cvToaster.showSuccessMessage({
									message: cvLoc('success.addtopology', newTopology.firewallTopology.topologyEntity.topologyName)
								});
							} else if (resp.error.errorCode > 0) {
								if (resp.error.errorMessage !== undefined && resp.error.errorMessage.length > 0) {
									errorMessage = resp.error.errorMessage;
								}

								self.loading = false;
								self.topologiesMessage = {
									type: 'error',
									message: errorMessage
								};
							}
						} else {
							doneCallback(true);
						}
					})
					.error(function(e) {
						var errorMessage = cvLoc('error.addtopology', newTopology.firewallTopology.topologyEntity.topologyName);
						self.loading = false;
						self.topologiesMessage = {
							type: 'error',
							message: errorMessage
						};
					});
			} else {
				// Update the client groups
				var firewallGroupFilter = new Set([FwGroupType.EXTERNAL.id, FwGroupType.INTERNAL.id]);
				var firstcg = $filter('filter')(self.topology.firewallGroups, { fwGroupType: FwGroupType.EXTERNAL.id });
				if (firstcg !== undefined && firstcg.length > 0) {
					firstcg[0].clientGroup.clientGroupId = self.uiModel.firstClientGroup[0].Id;
					if (self.uiModel.firstClientGroup[0].Id < 0) {
						firstcg[0].isMnemonic = true;
					} else {
						firstcg[0].isMnemonic = false;
					}

					var tempConfig = {
						configureAutomatically: self.uiModel.firstGroupOptions.configureAutomatically == 0 ? false : true,
						tunnelConnectionPort:
							self.uiModel.firstGroupOptions.tunnelPort == ''
								? self.defaultTunnelPort
								: parseInt(self.uiModel.firstGroupOptions.tunnelPort),
						keepAliveSeconds: parseInt(self.uiModel.firstGroupOptions.keepAliveInterval)
					};
					firstcg[0].options = tempConfig;
				}

				var secondcg = $filter('filter')(self.topology.firewallGroups, { fwGroupType: FwGroupType.INTERNAL.id });
				if (secondcg !== undefined && secondcg.length > 0) {
					secondcg[0].clientGroup.clientGroupId = self.uiModel.secondClientGroup[0].Id;

					if (self.uiModel.secondClientGroup[0].Id < 0) {
						secondcg[0].isMnemonic = true;
					} else {
						secondcg[0].isMnemonic = false;
					}

					var tempConfig = {
						configureAutomatically: self.uiModel.secondGroupOptions.configureAutomatically == 0 ? false : true,
						tunnelConnectionPort:
							self.uiModel.secondGroupOptions.tunnelPort == ''
								? self.defaultTunnelPort
								: parseInt(self.uiModel.secondGroupOptions.tunnelPort),
						keepAliveSeconds: parseInt(self.uiModel.secondGroupOptions.keepAliveInterval)
					};
					secondcg[0].options = tempConfig;
				}
				if (self.topology.topologyType === NWTopologyType.PROXIED.id) {
					firewallGroupFilter.add(FwGroupType.PROXIED.id);
					var thirdcg = $filter('filter')(self.topology.firewallGroups, { fwGroupType: FwGroupType.PROXIED.id });

					var isMnemonic = self.uiModel.thirdClientGroup[0].Id < 0;

					var tempConfig = {
						configureAutomatically: self.uiModel.thirdGroupOptions.configureAutomatically == 0 ? false : true,
						tunnelConnectionPort:
							self.uiModel.thirdGroupOptions.tunnelPort == ''
								? self.defaultTunnelPort
								: parseInt(self.uiModel.thirdGroupOptions.tunnelPort),
						keepAliveSeconds: parseInt(self.uiModel.thirdGroupOptions.keepAliveInterval)
					};

					if (thirdcg.length > 0) {
						thirdcg[0].clientGroup.clientGroupId = self.uiModel.thirdClientGroup[0].Id;
						thirdcg[0].isMnemonic = isMnemonic;
						thirdcg[0].options = tempConfig;
					} else {
						self.topology.firewallGroups.push({
							fwGroupType: FwGroupType.PROXIED.id,
							isMnemonic,
							clientGroup: {
								clientGroupId: self.uiModel.thirdClientGroup[0].Id
							},
							options: tempConfig
						});
					}
				}

				if (self.topology.topologyType === NWTopologyType.CASCADINGGATEWAY.id) {
					firewallGroupFilter.add(FwGroupType.CASCADINGGATEWAY.id);
					firewallGroupFilter.add(FwGroupType.PROXIED.id);
					var thirdcg = $filter('filter')(self.topology.firewallGroups, { fwGroupType: FwGroupType.PROXIED.id });
					var isMnemonic = self.uiModel.thirdClientGroup[0].Id < 0;

					var tempConfig = {
						configureAutomatically: self.uiModel.thirdGroupOptions.configureAutomatically == 0 ? false : true,
						tunnelConnectionPort:
							self.uiModel.thirdGroupOptions.tunnelPort == ''
								? self.defaultTunnelPort
								: parseInt(self.uiModel.thirdGroupOptions.tunnelPort),
						keepAliveSeconds: parseInt(self.uiModel.thirdGroupOptions.keepAliveInterval)
					};

					if (thirdcg.length > 0) {
						thirdcg[0].clientGroup.clientGroupId = self.uiModel.thirdClientGroup[0].Id;
						thirdcg[0].isMnemonic = isMnemonic;
						thirdcg[0].options = tempConfig;
					} else {
						self.topology.firewallGroups.push({
							fwGroupType: FwGroupType.PROXIED.id,
							isMnemonic,
							clientGroup: {
								clientGroupId: self.uiModel.thirdClientGroup[0].Id
							},
							options: tempConfig
						});
					}

					var fourthcg = $filter('filter')(self.topology.firewallGroups, {
						fwGroupType: FwGroupType.CASCADINGGATEWAY.id
					});
					isMnemonic = self.uiModel.fourthClientGroup[0].Id < 0;

					tempConfig = {
						configureAutomatically: self.uiModel.fourthGroupOptions.configureAutomatically == 0 ? false : true,
						tunnelConnectionPort:
							self.uiModel.fourthGroupOptions.tunnelPort == ''
								? self.defaultTunnelPort
								: parseInt(self.uiModel.fourthGroupOptions.tunnelPort),
						keepAliveSeconds: parseInt(self.uiModel.fourthGroupOptions.keepAliveInterval)
					};

					if (fourthcg.length > 0) {
						fourthcg[0].clientGroup.clientGroupId = self.uiModel.fourthClientGroup[0].Id;
						fourthcg[0].isMnemonic = isMnemonic;
						fourthcg[0].options = tempConfig;
					} else {
						self.topology.firewallGroups.push({
							fwGroupType: FwGroupType.CASCADINGGATEWAY.id,
							isMnemonic,
							clientGroup: {
								clientGroupId: self.uiModel.fourthClientGroup[0].Id
							},
							options: tempConfig
						});
					}
				}
				// Groups present in current topology only should be sent back in request. Other groups should be discarded else leads to stale db entries.
				self.topology.firewallGroups = self.topology.firewallGroups.filter(fwGroup => {
					return firewallGroupFilter.has(fwGroup.fwGroupType);
				});

				var updateTopology = {
					firewallTopology: self.topology
				};

				updateTopology.firewallTopology.extendedProperties = extendedPropertiesXML;

				if (self.uiModel.firstClientGroup[0].Id < 0 || self.uiModel.secondClientGroup[0].Id < 0) {
					updateTopology.firewallTopology.isSmartTopology = true;
				} else {
					updateTopology.firewallTopology.isSmartTopology = false;
				}

				NetworkTopologyService.updateTopology(updateTopology)
					.success(function(resp) {
						var errorMessage = cvLoc(
							'error.updatetopology',
							updateTopology.firewallTopology.topologyEntity.topologyName
						);
						if (resp !== undefined && resp.error !== undefined) {
							if (resp.error.errorCode === 0) {
								doneCallback(true);
								cvToaster.showSuccessMessage({
									message: cvLoc('success.updatetopology', updateTopology.firewallTopology.topologyEntity.topologyName)
								});
							} else if (resp.error.errorCode > 0) {
								if (resp.error.errorMessage !== undefined && resp.error.errorMessage.length > 0) {
									errorMessage = resp.error.errorMessage;
								}

								self.loading = false;
								self.topologiesMessage = {
									type: 'error',
									message: errorMessage
								};
							}
						} else {
							doneCallback(true);
						}
					})
					.error(function(e) {
						var errorMessage = cvLoc(
							'error.updatetopology',
							updateTopology.firewallTopology.topologyEntity.topologyName
						);
						self.loading = false;
						self.topologiesMessage = {
							type: 'error',
							message: errorMessage
						};
					});
			}
		};

		self.previousOptionExpand = false;
		self.previousSelection = 0;
		self.showNetworkOptions = function(data) {
			var advancedOptionsDiv;
			if (self.previousOptionExpand == false || (self.previousOptionExpand == true && self.previousSelection != data)) {
				self.uiModel.showFirstGroupNetworkOptions = false;
				self.uiModel.showSecondGroupNetworkOptions = false;
				self.uiModel.showThirdGroupNetworkOptions = false;
				self.uiModel.showFourthGroupNetworkOptions = false;

				if (data == 1) {
					self.uiModel.showFirstGroupNetworkOptions = true;
				} else if (data == 2) {
					self.uiModel.showSecondGroupNetworkOptions = true;
				} else if (data == 3) {
					self.uiModel.showThirdGroupNetworkOptions = true;
				} else if (data == 4) {
					self.uiModel.showFourthGroupNetworkOptions = true;
				}
				self.previousOptionExpand = true;
				self.previousSelection = data;
			} else {
				if (data == 1) {
					self.uiModel.showFirstGroupNetworkOptions = false;
				} else if (data == 2) {
					self.uiModel.showSecondGroupNetworkOptions = false;
				} else if (data == 3) {
					self.uiModel.showThirdGroupNetworkOptions = false;
				} else if (data == 4) {
					self.uiModel.showFourthGroupNetworkOptions = false;
				}
				self.previousOptionExpand = false;
				self.previousSelection = data;
			}
		};

		self.isPortValid = function(port) {
			if (port > 0 && port <= 65535) {
				return true;
			}

			return false;
		};

		self.updateTunnelPort = function(data) {
			if (data == 1) {
				var port = parseInt(self.uiModel.firstGroupOptions.tunnelPort);
				if (!isNaN(port) && self.isPortValid(port)) {
					self.uiModel.firstGroupOptions.tunnelPortLabel = self.uiModel.firstGroupOptions.tunnelPort = port;
					self.uiModel.firstGroupOptions.configureAutomatically = 0;
				} else {
					self.uiModel.firstGroupOptions.tunnelPortLabel = self.defaultTunnelPortStr;
					self.uiModel.firstGroupOptions.tunnelPort = '';
					self.uiModel.firstGroupOptions.configureAutomatically = 1;
				}
			} else if (data == 2) {
				var port = parseInt(self.uiModel.secondGroupOptions.tunnelPort);
				if (!isNaN(port) && self.isPortValid(port)) {
					self.uiModel.secondGroupOptions.tunnelPortLabel = self.uiModel.secondGroupOptions.tunnelPort = port;
					self.uiModel.secondGroupOptions.configureAutomatically = 0;
				} else {
					self.uiModel.secondGroupOptions.tunnelPortLabel = self.defaultTunnelPortStr;
					self.uiModel.secondGroupOptions.tunnelPort = '';
					self.uiModel.secondGroupOptions.configureAutomatically = 1;
				}
			} else if (data == 3) {
				var port = parseInt(self.uiModel.thirdGroupOptions.tunnelPort);
				if (!isNaN(port) && self.isPortValid(port)) {
					self.uiModel.thirdGroupOptions.tunnelPortLabel = self.uiModel.thirdGroupOptions.tunnelPort = port;
					self.uiModel.thirdGroupOptions.configureAutomatically = 0;
				} else {
					self.uiModel.thirdGroupOptions.tunnelPortLabel = self.defaultTunnelPortStr;
					self.uiModel.thirdGroupOptions.tunnelPort = '';
					self.uiModel.thirdGroupOptions.configureAutomatically = 1;
				}
			} else if (data == 4) {
				var port = parseInt(self.uiModel.fourthGroupOptions.tunnelPort);
				if (!isNaN(port) && self.isPortValid(port)) {
					self.uiModel.fourthGroupOptions.tunnelPortLabel = self.uiModel.fourthGroupOptions.tunnelPort = port;
					self.uiModel.fourthGroupOptions.configureAutomatically = 0;
				} else {
					self.uiModel.fourthGroupOptions.tunnelPortLabel = self.defaultTunnelPortStr;
					self.uiModel.fourthGroupOptions.tunnelPort = '';
					self.uiModel.fourthGroupOptions.configureAutomatically = 1;
				}
			}
		};

		self.isValidPositiveInteger = function(data) {
			return data > 0 ? true : false;
		};

		self.updateKeepAlive = function(data) {
			if (data == 1) {
				var keepAlive = parseInt(self.uiModel.firstGroupOptions.keepAliveInterval);
				if (self.isValidPositiveInteger(keepAlive)) {
					self.uiModel.firstGroupOptions.keepAliveIntervalLabel = self.uiModel.firstGroupOptions.keepAliveInterval = keepAlive;
				} else {
					self.uiModel.firstGroupOptions.keepAliveInterval = self.uiModel.firstGroupOptions.keepAliveIntervalLabel;
				}
			} else if (data == 2) {
				var keepAlive = parseInt(self.uiModel.secondGroupOptions.keepAliveInterval);
				if (self.isValidPositiveInteger(keepAlive)) {
					self.uiModel.secondGroupOptions.keepAliveIntervalLabel = self.uiModel.secondGroupOptions.keepAliveInterval = keepAlive;
				} else {
					self.uiModel.secondGroupOptions.keepAliveInterval = self.uiModel.secondGroupOptions.keepAliveIntervalLabel;
				}
			} else if (data == 3) {
				var keepAlive = parseInt(self.uiModel.thirdGroupOptions.keepAliveInterval);
				if (self.isValidPositiveInteger(keepAlive)) {
					self.uiModel.thirdGroupOptions.keepAliveIntervalLabel = self.uiModel.thirdGroupOptions.keepAliveInterval = keepAlive;
				} else {
					self.uiModel.thirdGroupOptions.keepAliveInterval = self.uiModel.thirdGroupOptions.keepAliveIntervalLabel;
				}
			} else if (data == 4) {
				var keepAlive = parseInt(self.uiModel.fourthGroupOptions.keepAliveInterval);
				if (self.isValidPositiveInteger(keepAlive)) {
					self.uiModel.fourthGroupOptions.keepAliveIntervalLabel = self.uiModel.fourthGroupOptions.keepAliveInterval = keepAlive;
				} else {
					self.uiModel.fourthGroupOptions.keepAliveInterval = self.uiModel.fourthGroupOptions.keepAliveIntervalLabel;
				}
			}
		};

		self.initializeSpecificGroupOption = function(groupType, groupOptions) {
			var group = $filter('filter')(self.topology.firewallGroups, { fwGroupType: groupType });

			if (
				groupOptions == undefined ||
				groupOptions == null ||
				!group.length ||
				group[0].options == undefined ||
				group[0].options == null
			) {
				return;
			}

			var fwConfig = group[0].options;

			if (fwConfig.keepAliveSeconds != undefined) {
				groupOptions.keepAliveInterval = groupOptions.keepAliveIntervalLabel = fwConfig.keepAliveSeconds;
			}

			if (fwConfig.configureAutomatically != undefined && fwConfig.configureAutomatically == 0) {
				groupOptions.tunnelPort = groupOptions.tunnelPortLabel = fwConfig.tunnelConnectionPort;
				groupOptions.configureAutomatically = 0;
			}
		};

		// Initialize group specific options
		self.initializeGroupOptions = function() {
			if (self.createTopology == true) {
				// Initialize to default values
			} else {
				self.initializeSpecificGroupOption(FwGroupType.EXTERNAL.id, self.uiModel.firstGroupOptions);

				self.initializeSpecificGroupOption(FwGroupType.PROXIED.id, self.uiModel.thirdGroupOptions);

				self.initializeSpecificGroupOption(FwGroupType.CASCADINGGATEWAY.id, self.uiModel.fourthGroupOptions);

				self.initializeSpecificGroupOption(FwGroupType.INTERNAL.id, self.uiModel.secondGroupOptions);
			}
		};

		// Initialize topology specific advanced options
		self.initializeTopoSpecificAdvancedOptions = function() {
			if (self.createTopology != true) {
				// Update advanced options only when it is edit mode.
				if (self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.encryptTraffic != undefined) {
					self.uiModel.advancedOptions.encryptTraffic =
						self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.encryptTraffic == 0 ? false : true;
				}

				if (self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.numberOfStreams != undefined) {
					self.uiModel.advancedOptions.numberOfStreams = parseInt(
						self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.numberOfStreams
					);
				}

				if (self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.connectionProtocol != undefined) {
					self.uiModel.advancedOptions.tunnelProtocol = self.getTunnelProtocolLabel(
						self.topologyExtendedProps.App_TopologyExtendedProperties.attributes.connectionProtocol
					);
				}
			}
		};

		self.getTunnelProtocolID = function(protocol) {
			var retValue = $filter('filter')(NWTopologyProtocolType, { name: protocol });
			return retValue[0].id;
		};

		self.getTunnelProtocolLabel = function(protocolId) {
			var retValue = $filter('filter')(NWTopologyProtocolType, { id: protocolId });
			if (retValue == undefined || !retValue.length) {
				return self.defaultTunnelProtocol;
			}
			return retValue[0].name;
		};

		self.initializeClientGroupNetworkProperties = function(networkInfo) {
			for (var groupOptions of networkInfo.groupNetworkProperties) {
				self.clientGroupNetworkOptions[groupOptions.group.clientGroupId] = {
					name: groupOptions.group.clientGroupName,
					options: groupOptions.options
				};
			}
		};

		self.updateDisplayGroupNetworkOptions = function(Id, group) {
			if (group === undefined) {
				return;
			}

			var groupOptions = self.clientGroupNetworkOptions[Id];
			if (groupOptions != undefined && groupOptions.options != undefined) {
				if (groupOptions.options.keepAliveSeconds != undefined) {
					group.keepAliveInterval = group.keepAliveIntervalLabel = groupOptions.options.keepAliveSeconds;
				}

				if (
					groupOptions.options.configureAutomatically != undefined &&
					groupOptions.options.configureAutomatically == 0
				) {
					group.tunnelPort = group.tunnelPortLabel = groupOptions.options.tunnelConnectionPort;
					group.configureAutomatically = 0;
				} else {
					group.tunnelPortLabel = self.defaultTunnelPortStr;
					group.tunnelPort = '';
					group.configureAutomatically = 1;
				}
			} else {
				// Set to defaults
				group.configureAutomatically = self.defaultConfigureAutomatically;
				group.keepAliveInterval = self.defaultKeepAliveInterval;
				group.keepAliveIntervalLabel = self.defaultKeepAliveInterval;
				group.tunnelPort = '';
				group.tunnelPortLabel = self.defaultTunnelPortStr;
			}
		};

		// Update group network options when a group is selected
		self.updateGroupNetworkOptions = function(data) {
			if (data == 1) {
				if (self.uiModel.firstClientGroup != undefined && self.uiModel.firstClientGroup.length > 0) {
					self.updateDisplayGroupNetworkOptions(self.uiModel.firstClientGroup[0].Id, self.uiModel.firstGroupOptions);
				}
			} else if (data == 2) {
				if (self.uiModel.secondClientGroup != undefined && self.uiModel.secondClientGroup.length > 0) {
					self.updateDisplayGroupNetworkOptions(self.uiModel.secondClientGroup[0].Id, self.uiModel.secondGroupOptions);
				}
			} else if (data == 3) {
				if (self.uiModel.thirdClientGroup != undefined && self.uiModel.thirdClientGroup.length > 0) {
					self.updateDisplayGroupNetworkOptions(self.uiModel.thirdClientGroup[0].Id, self.uiModel.thirdGroupOptions);
				}
			} else if (data == 4) {
				if (self.uiModel.fourthClientGroup != undefined && self.uiModel.fourthClientGroup.length > 0) {
					self.updateDisplayGroupNetworkOptions(self.uiModel.fourthClientGroup[0].Id, self.uiModel.fourthGroupOptions);
				}
			}
		};
	}
]);

export default acAppNetworkTopologyModule;
