(function() {
	'use strict';

	var navigationMod = angular.module('cvModule.navigation');
	navigationMod.factory('cvNavigationFactory', [
		'cvLoc',
		'$state',
		'LIST_STATENAMES',
		'DETAILS_STATENAMES',
		'cvUtil',
		function(cvLoc, $state, LIST_STATENAMES, DETAILS_STATENAMES, cvUtil) {
			var cvNavigationFactoryObjectToReturn = {};
			var _routes = angular.copy(cv.nav.routes);
			var _routeStateToNavItemMap = new Map();
			var _availableStatesForLoggedUser = new Set();
			var _fullScreen = false;
			var _navHidden = false;
			var _isZealBrand = cvApp.oemId === 118;

			const ServiceTypes = Object.freeze({
				CaseManager: '1',
				GDPR: '2',
				Analytics: '3'
			});

			cvNavigationFactoryObjectToReturn['modernViewData'] = angular.copy(cv.nav.routes);

			var _setupRouteStateToNavItemMap = function(routes) {
				routes.forEach(function(route) {
					if (route.showNavItem) {
						if (
							!angular.isObject(_routeStateToNavItemMap.get(route.state)) ||
							angular.isUndefined(_routeStateToNavItemMap.get(route.state))
						) {
							if (angular.isDefined(route.cvTitle) && route.cvTitle !== '') {
								route.cvTitle = route.titleLocalized ? route.cvTitle : cvLoc(route.cvTitle);
							}
							if (angular.isDefined(route.description) && route.description !== '') {
								route.description = cvLoc(route.description);
							}
							if (angular.isDefined(route.pageTitle) && route.pageTitle !== '') {
								route.pageTitle = cvLoc(route.pageTitle);
							}
							_routeStateToNavItemMap.set(route.state, route);
						}
					}

					_availableStatesForLoggedUser.add(route.state);

					if (angular.isArray(route.children)) {
						_setupRouteStateToNavItemMap(route.children);
					}
				});
			};

			_setupRouteStateToNavItemMap(_routes);

			cvNavigationFactoryObjectToReturn.putItemInRouteStateToNavItemMap = function(stateName, stateObject) {
				if (
					angular.isDefined(stateName) &&
					stateName !== '' &&
					angular.isObject(stateObject) &&
					angular.isDefined(stateObject)
				) {
					_routeStateToNavItemMap.set(stateName, stateObject);
				}
			};

			/**
			 * Given a route object, it checks all of the children under it to ensure that
			 * there is at least 1 child to show.
			 */
			cvNavigationFactoryObjectToReturn.hasChildrenToShow = function(route) {
				if (route && (route.state === 'my-apps' || route.isApp === true)) {
					return true;
				}

				if (angular.isArray(route.children)) {
					var returnValue = false;
					for (var i in route.children) {
						/*
						 * If we made it here, that means that there is at least 1 child
						 * that doesn't require a setup to be completed in order to show,
						 * meaning we can show the parent nav.
						 */

						if (angular.isArray(route.children[i].children)) {
							returnValue = returnValue || cvNavigationFactoryObjectToReturn.hasChildrenToShow(route.children[i]);
						} else {
							//If lowest Level child return true for its parent
							returnValue = returnValue || (angular.isDefined(route.children[i].url) && route.children[i].url !== '');
						}
					}
					return returnValue;
				}
				return false;
			};

			/**
			 * Filters the routes based on the title of the route containing the given
			 * search term at the start of the string.
			 *
			 * Recursion is used to filter through all child routes of any parents that
			 * have at least 1 child.
			 *
			 * This method also takes into account that if a nav link is only shown when a
			 * certain setup is complete, it shouldn't be searchable.
			 *
			 * @param {string}
			 *            search - The search term to filter the routes by.
			 * @return {array} filteredRoutes The routes that have been filtered by the
			 *         search term.
			 */
			cvNavigationFactoryObjectToReturn.filterRoutes = function(routes, search) {
				var filteredRoutes = [];
				if (angular.isArray(routes) && angular.isString(search) && search.length > 0) {
					angular.forEach(routes, function(route) {
						var added = false;

						/*
						 * If the route has the attribute "requiresSetupComplete", we need
						 * to check the given setup name to see if it is complete before
						 * being able to search it. If the setup is complete, we can
						 * continue along as planned.
						 */
						/*
						 * If the route is a parent to children, but those children are
						 * all hidden for other reasons, then we this route should not be
						 * searchable since we won't show it on the nav.
						 */
						if (angular.isDefined(route.children) && !cvNavigationFactoryObjectToReturn.hasChildrenToShow(route)) {
							return;
						}

						/*
						 * If any part of the title of the route or any keywords defined
						 * contains the given search term, it will be added to the
						 * filtered routes.
						 */
						if (hasSearchMatch(route.cvTitle, search) || hasSearchMatch(route._keywords, search)) {
							filteredRoutes.push(route);
							added = true;
						}

						/*
						 * If the current route has children, we need to filter those as
						 * well. Whatever filtered routes that were returned will be added
						 * as the route's new children.
						 *
						 * If the route has already been added, we don't need to run this
						 * condition since we will be displaying everything under the
						 * route anyway.
						 */
						if (angular.isArray(route.children)) {
							var filteredChildRoutes = cvNavigationFactoryObjectToReturn.filterRoutes(route.children, search);
							if (filteredChildRoutes.length > 0) {
								/*
								 * We add the below attribute because if a route has only
								 * 1 child when it is filtered, it will be displayed
								 * different in the nav. Therefore, we have to explicitly
								 * say that the route has more than 1 child so it can
								 * still display normally.
								 */
								if (route.children.length > 1) {
									route.hasMoreThanOneChild = true;
								}

								/*
								 * If the route has already been added, then we want to
								 * show all of its children. If the route hasn't been
								 * added, then we want to replace the route's children
								 * with the filtered routes that we found.
								 */
								if (!added) {
									route.children = filteredChildRoutes;
									route.isOpen = true;
								}

								/*
								 * Make sure the route wasn't already added or else we
								 * will get errors on the front-end saying we have
								 * duplicate instances.
								 */
								if (!added) {
									filteredRoutes.push(route);
								}
							}
						}
					});
				}
				return filteredRoutes;

				function hasSearchMatch(haystack, needle) {
					return (
						angular.isString(haystack) &&
						haystack.length > 0 &&
						haystack.toLowerCase().indexOf(needle.toLowerCase()) >= 0
					);
				}
			};

			cvNavigationFactoryObjectToReturn.getRouteForState = function(stateName) {
				return _routeStateToNavItemMap.get(stateName);
			};

			cvNavigationFactoryObjectToReturn.isStateAvailable = function(stateName) {
				return (
					_availableStatesForLoggedUser &&
					_availableStatesForLoggedUser.size > 0 &&
					_availableStatesForLoggedUser.has(stateName)
				);
			};

			var _redirectToState = function(stateName, paramsObject) {
				$state.transitionTo(stateName, _.isUndefined(paramsObject) ? {} : paramsObject, {
					reload: true,
					inherit: false,
					notify: true
				});
			};

			cvNavigationFactoryObjectToReturn.redirectToListPage = function(stateName, paramsObject) {
				_redirectToState(stateName, paramsObject);
			};

			cvNavigationFactoryObjectToReturn.redirectToDetailsPage = function(stateName, paramsObject) {
				_redirectToState(stateName, paramsObject);
			};

			/**
			 * @param {boolean}
			 *            hidden
			 */
			cvNavigationFactoryObjectToReturn.setNavHidden = function(hidden) {
				_navHidden = hidden;
			};

			/**
			 * @returns {boolean}
			 */
			cvNavigationFactoryObjectToReturn.isNavHidden = function() {
				return _navHidden;
			};

			/**
			 * @param {boolean}
			 *            fullScreen
			 */
			cvNavigationFactoryObjectToReturn.setFullScreen = function(fullScreen) {
				_fullScreen = fullScreen;
			};

			/**
			 * @returns {boolean}
			 */
			cvNavigationFactoryObjectToReturn.isFullScreen = function() {
				return _fullScreen;
			};

			cvNavigationFactoryObjectToReturn.LIST_STATENAMES = LIST_STATENAMES;

			cvNavigationFactoryObjectToReturn.DETAILS_STATENAMES = DETAILS_STATENAMES;

			var showAppTile = function(bitPosition) {
				if (cv.isTenantAdmin || cv.isTenantOperator) {
					var bitmaskValue = angular.isDefined(cv.sessionContext.supportedSolutions)
						? cv.sessionContext.supportedSolutions
						: 4127; //supported solutions bitmask value obtained from backend or 4127(0001000000011111)(FS, VSA, ENDPOINT, DBS, EXCHANGE, O365)
					var appMask = 1 << bitPosition;
					if ((bitmaskValue & appMask) !== 0) {
						return true;
					} else {
						return false;
					}
				} else {
					return false;
				}
			};

			var _appSteps = null;

			var _appsShowCriterion = {
				EXCHANGE: cvConfig && cvConfig.showExchange,
				ARCHIVING: cvConfig && cvConfig.showArchivingSolution,
				EBS: false,
				VNA: cvConfig && cvConfig.showVNA,
				NOTES_DATABASE: cvConfig && cvConfig.showNotesDatabase,
				DYNAMICS365: cvConfig && cvConfig.showOffice365Agent_D365
			};

			cvNavigationFactoryObjectToReturn.isAppVisible = function(solutionId) {
				return _.isNil(_appsShowCriterion[solutionId]) ? true : _appsShowCriterion[solutionId];
			};

			/**
			 * Below are the apps that will be generate the setup tiles that will be
			 * available to the user. Below are a list of properties that are used
			 * throughout the controller.
			 *
			 * @property {object} APP_NAME - The name of the app you are creating an
			 *           object for.
			 * @property {string} APP_NAME.appName - The display name of the app.
			 * @property {string[]} APP_NAME.appFeatures - The features that the app
			 *           offers.
			 * @property {boolean=} APP_NAME.appSetupCompleted - Whether or not your setup
			 *           is complete. If your solution does not have a setup, set this to
			 *           TRUE.
			 * @property {string=} APP_NAME.appSetupState - The state name of the setup
			 *           page for the app. If your solution does not have a setup, you can
			 *           remove this property, but you have to set
			 *           APP_NAME.appSetupCompleted = true.
			 * @property {string} APP_NAME.appDashboardState - The state of the page that
			 *           the user will be taken to if the setup is clicked on when
			 *           complete.
			 * @property {string=} APP_NAME.appHelpLink - The link to the help page for
			 *           your app. You can remove this if you don't want to display a help
			 *           link.
			 * @property {string=} APP_NAME.appSubApps - Subapps within an app. They will
			 *           be listed and a link available to take the user to respective
			 *           setup.
			 * @property {string=} APP_NAME.setupCompletedText - Custom text to display
			 *           when the setup is completed. The default text will be used if
			 *           this property is absent.
			 * @property {number=} APP_NAME.order - The order in which the app should be
			 *           displayed on the Getting Started page. If no number is given, it
			 *           will be showed after all other apps that have an order. (Does not
			 *           have any effect on sub apps.)
			 * @property {boolean} SHOW - Show hide the Setup tile
			 */

			var _apps = {
				ADMIN_CONSOLE: {
					appName: cvUtil.lookupEnumConstant('ADMIN_CONSOLE'),
					appFeatures: [cvLoc('label.serversFeature1'), cvLoc('label.serversFeature2'), cvLoc('label.serversFeature3')],
					appSetupState: 'acSetup',
					appDashboardState: 'fsServersList',
					appHelpLink: cvConfig.helpLink + '?p=87294.htm',
					isSubApp: false,
					//Show property is used to show hide the app in TenantAdmin/TenantOperatorMode
					show: cv.isMspAdmin ? true : showAppTile(0),
					order: 1
				},
				VSA: {
					appName: cvUtil.lookupEnumConstant('VSA'),
					appFeatures: [
						cvLoc('label.virtualizationFeature1'),
						cvLoc('label.virtualizationFeature2'),
						cvLoc('label.virtualizationFeature3')
					],
					appHelpLink: cvConfig.helpLink + '?p=119353.htm',
					appSetupState: 'vsaSetup',
					appDashboardState: 'hypervisors',
					show: cv.isMspAdmin ? true : showAppTile(1),
					isSubApp: false,
					order: 2
				},
				ENDPOINT: {
					appName: cvUtil.lookupEnumConstant('ENDPOINT'),
					appFeatures: [
						cvLoc('label.endpointFeature1'),
						cvLoc('label.endpointFeature2'),
						cvLoc('label.endpointFeature3')
					],
					appHelpLink: cvConfig.helpLink + '?p=86671.htm',
					appSetupState: 'dloSetup',
					appDashboardState: 'devices',
					show: cv.isMspAdmin ? true : showAppTile(2),
					isSubApp: false,
					order: 3
				},
				DBS: {
					appName: cvUtil.lookupEnumConstant('DBS'),
					appFeatures: [cvLoc('label.databaseFeature1'), cvLoc('label.databaseFeature2')],
					appHelpLink: cvConfig.helpLink + '?p=86674.htm',
					appSetupState: 'dbsSetup',
					appDashboardState: 'instances',
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(3),
					order: 4
				},
				OFFICE365: {
					appName: cvUtil.lookupEnumConstant('OFFICE365'),
					appDashboardState: 'office365V2',
					appSetupState: 'exchangeOnlineSetup',
					isSubApp: false,
					keyWords: [
						cvLoc('label.nav.exchangeOnline'),
						cvLoc('label.nav.sharepointOnline'),
						cvLoc('label.nav.oneDriveForBusiness')
					],
					appFeatures: [cvLoc('label.office365.feature')],
					show: cv.isMspAdmin ? true : showAppTile(12),
					appHelpLink: cvConfig.helpLink + '?p=106096.htm',
					order: 5
				},
				EXCHANGE: {
					appName: cvUtil.lookupEnumConstant('EXCHANGE'),
					appHelpLink: cvConfig.helpLink + '?p=93214.htm',
					appSetupState: 'exchangeMailboxSetup',
					setupCompletedText: cvLoc('label.clickToContinue'),
					appDashboardState: 'exchange',
					keyWords: [cvLoc('label.exchangeMailbox')],
					isSubApp: false,
					show: _appsShowCriterion['EXCHANGE'] && (cv.isMspAdmin ? true : showAppTile(4)),
					order: 6
				},
				ARCHIVING: {
					appName: cvUtil.lookupEnumConstant('ARCHIVING'),
					appFeatures: [
						cvLoc('label.archivingFeature1'),
						cvLoc('label.archivingFeature2'),
						cvLoc('label.archivingFeature3')
					],
					appHelpLink: cvConfig.helpLink + '?p=24643.htm',
					appSetupState: 'archiveSetup',
					appDashboardState: 'archiveFileServers',
					isSubApp: false,
					show: _appsShowCriterion['ARCHIVING'] && (cv.isMspAdmin === true ? true : showAppTile(5)),
					order: 7
				},
				/*
				 * EXCHANGE_DAG : { appName : cvLoc('lable.nav.exchange.dag'),
				 * appSetupState : 'exchangeDagSetup', isSubApp : true },
				 */
				CAPPS: {
					appName: cvLoc('label.devOps'),
					appHelpLink: cvConfig.helpLink + '?p=86685.htm',
					//the state to which redirect will happen on click cloudapps tile. Tiles of apps taht have subApps are not clickable if setup is not complete.
					appSetupState: 'gitAppsAccountsSetup',
					appDashboardState: 'gitAppsAccounts',
					isSubApp: false,
					keyWords: [cvLoc('label.clientType.amazonSql')],
					show: cv.isMspAdmin ? true : showAppTile(6),
					order: 8,
					redirectLink: 'cappsClients?view&isGuidedSetup=true'
				},
				VNA: {
					appName: cvUtil.lookupEnumConstant('VNA'),
					appFeatures: [
						cvLoc('label.healthcareFeature1'),
						cvLoc('label.healthcareFeature2'),
						cvLoc('label.healthcareFeature3')
					],
					appSetupCompleted: true,
					appDashboardState: 'vnaConfig',
					isSubApp: false,
					show: _appsShowCriterion['VNA'] && (cv.isMspAdmin === true ? true : showAppTile(8)),
					order: 10
				},
				SHAREPOINT: {
					appName: cvUtil.lookupEnumConstant('SHAREPOINT'),
					appHelpLink: cvConfig.helpLink + '?p=17813.htm',
					appFeatures: [cvLoc('label.sharePointFeature1'), cvLoc('label.sharePointFeature2')],
					setupCompletedText: cvLoc('label.clickToContinue'),
					appSetupState: 'sharepointSetup',
					appDashboardState: 'sharepoint',
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(9),
					order: 11
				},
				GDPR: {
					appName: cvLoc('label.gdpr'),
					appFeatures: [cvLoc('label.gdpr.description')],
					show: cv.isMspAdmin ? true : showAppTile(10),
					redirectLink:
						_.has(cv, 'sessionContext.completedSetups.GDPR') && cv.sessionContext.completedSetups.GDPR === true
							? 'sdg'
							: 'activateSetup?type=' + ServiceTypes.GDPR
				},
				FSO: {
					appName: cvLoc('label.analytics'),
					appFeatures: [cvLoc('label.analytics.description')],
					show: cv.isMspAdmin ? true : showAppTile(10),
					redirectLink:
						_.has(cv, 'sessionContext.completedSetups.FSO') && cv.sessionContext.completedSetups.FSO === true
							? 'analytics'
							: 'activateSetup?type=' + ServiceTypes.Analytics
				},
				CASE_MANAGER: {
					appName: cvLoc('label.casemanager'),
					appFeatures: [cvLoc('label.casemanager.description')],
					show: cv.isMspAdmin ? true : showAppTile(10),
					redirectLink:
						_.has(cv, 'sessionContext.completedSetups.CASE_MANAGER') &&
						cv.sessionContext.completedSetups.CASE_MANAGER === true
							? 'caseManager'
							: 'activateSetup?type=' + ServiceTypes.CaseManager
				},
				ACTIVE_DIRECTORY: {
					appName: cvUtil.lookupEnumConstant('ACTIVE_DIRECTORY'),
					appFeatures: [cvLoc('label.activeDirectoryFeature1'), cvLoc('label.activeDirectoryFeature2')],
					appSetupState: 'activeDirectorySetup',
					appDashboardState: 'activeDirectory',
					isSubApp: false,
					appHelpLink: cvConfig.helpLink + '?p=105386.htm',
					show: cv.isMspAdmin ? true : showAppTile(11),
					order: 13
				},
				BIG_DATA_APPS: {
					appName: cvUtil.lookupEnumConstant('BIG_DATA_APPS'),
					appDashboardState: 'bigDataApps',
					appSetupState: 'bigDataAppsSetup',
					appSetupCompleted: false,
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(13),
					appHelpLink: cvConfig.helpLink + '?p=30428.htm',
					keyWords: [
						cvLoc('label.clusterType.cassandra'),
						cvLoc('label.clusterType.hadoop'),
						cvLoc('label.nav.bigDataApps'),
						cvLoc('label.clusterType.splunk')
					],
					order: 14
				},
				NOTES_DATABASE: {
					appName: cvUtil.lookupEnumConstant('NOTES_DATABASE'),
					appFeatures: [cvLoc('label.notesDatabaseFeature1'), cvLoc('label.notesDatabaseFeature2')],
					appSetupState: 'notesDatabaseSetup',
					appDashboardState: 'notesDatabase',
					isSubApp: false,
					show: _appsShowCriterion['NOTES_DATABASE'] && (cv.isMspAdmin === true ? true : showAppTile(14)),
					order: 15
				},
				GSUITE: {
					appName: cvUtil.lookupEnumConstant('GSUITE'),
					appHelpLink: cvConfig.helpLink + '?p=86685.htm',
					appSetupState: 'cappsGDriveSetup',
					appDashboardState: 'gsuiteApps',
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(15),
					keyWords: [cvLoc('label.clientType.gmail'), cvLoc('label.clientType.gdrive')],
					order: 16
				},
				SALESFORCE: {
					appName: cvUtil.lookupEnumConstant('SALESFORCE'),
					appSetupState: 'cappsConfigurationSF',
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(16),
					order: 17,
					appHelpLink: cvConfig.helpLink + '?p=87053.htm',
					redirectLink: 'salesforceOrganizations'
					//appDashboardState: 'cappsClients'
				},
				REPLICATION: {
					appName: cvUtil.lookupEnumConstant('REPLICATION'),
					appSetupState: 'replicationGroups',
					isSubApp: false,
					show: cv.isMspAdmin || (_isZealBrand && cv.isTenantAdmin) ? true : showAppTile(17),
					order: 18,
					appHelpLink: cvConfig.helpLink + '?p=87228.htm',
					appDashboardState: _isZealBrand ? 'replicationGroups' : 'disasterRecoveryDashboard'
				},
				METALLIC: {
					appName: cvLoc('label.nav.Metallic'),
					appFeatures: [cvLoc('label.MetallicFeatures')],
					appSetupState: 'metallicSetup',
					isSubApp: false,
					appDashboardState: 'metallicSetup',
					show: cv.isMspAdmin || cv.isTenantAdmin ? true : false,
					order: 19
				},
				KUBERNETES: {
					appName: cvLoc('label.nav.kubernetes'),
					appFeatures: [
						cvLoc('label.kubernetesFeature1'),
						cvLoc('label.kubernetesFeature2'),
						cvLoc('label.kubernetesFeature3')
					],
					//appHelpLink: cvConfig.helpLink + '?p=86644.htm', //TO DO: after Documentation
					appSetupState: 'kubernetesSetup',
					appDashboardState: 'clusters',
					show: cv.isMspAdmin ? true : showAppTile(18),
					isSubApp: false,
					order: 20
				},
				OBJECT_STORAGE: {
					appName: cvLoc('label.nav.cloudStorage'),
					appFeatures: [
						cvLoc('label.objectStorageFeature1'),
						cvLoc('label.objectStorageFeature2'),
						cvLoc('label.objectStorageFeature3')
					],
					//appHelpLink: cvConfig.helpLink + '?p=86644.htm', //TO DO: after Documentation
					appSetupState: 'objectStorageSetup',
					appDashboardState: 'cloudStorageApp',
					show: cv.isMspAdmin ? true : showAppTile(19),
					isSubApp: false,
					order: 21
				},
				EPIC: {
					appName: cvLoc('label.nav.epic'),
					appFeatures: [cvLoc('label.epicFeature1')],
					appSetupState: 'epicSetup',
					appDashboardState: 'epicServersList',
					isSubApp: false,
					show: cv.isMspAdmin ? true : showAppTile(20),
					order: 22
				},
				DYNAMICS365: {
					appName: cvUtil.lookupEnumConstant('DYNAMICS365'),
					appDashboardState: 'dynamics365',
					appSetupState: 'dynamics365Setup',
					isSubApp: false,
					keyWords: [cvLoc('label.nav.dynamics365')],
					appFeatures: [cvLoc('label.d365.feature')],
					show: _appsShowCriterion['DYNAMICS365'] && cv.isMspAdmin ? true : showAppTile(21),
					order: 23
				}
			};

			cvNavigationFactoryObjectToReturn.getAppSteps = function() {
				if (!_.isUndefined(_appSteps) && !_.isNull(_appSteps)) {
					return _appSteps;
				} else {
					_appSteps = _apps;
					return _appSteps;
				}
			};

			cvNavigationFactoryObjectToReturn.isSetupComplete = function(setup) {
				if (angular.isString(setup)) {
					setup = setup.trim();
					return (
						angular.isDefined(cv.sessionContext.completedSetups[setup]) && cv.sessionContext.completedSetups[setup]
					);
				}
				return false;
			};

			cvNavigationFactoryObjectToReturn.getIncompletedSetups = function() {
				var incompleteSetups = [];
				var excludeKeys = ['VNA', 'GDPR', 'FSO', 'CASE_MANAGER'];
				angular.forEach(cvNavigationFactoryObjectToReturn.getAppSteps(), function(step, key) {
					if (!step.isSubApp && !excludeKeys.includes(key)) {
						if (
							angular.isDefined(step.show) &&
							step.show === true &&
							!cvNavigationFactoryObjectToReturn.isSetupComplete(key)
						) {
							incompleteSetups.push({
								state: step.appSetupState ? step.appSetupState : null,
								cvTitle: step.appName ? step.appName : '',
								keyWords: step.keyWords ? step.keyWords : [],
								appDashboardState: step.appDashboardState ? step.appDashboardState : ''
								//"url": step.redirectLink ? step.redirectLink : "" // This will be required for setups related to COMPLIANCE
							});
						}
					}
				});
				return incompleteSetups;
			};

			return cvNavigationFactoryObjectToReturn;
		}
	]);
})();
