import { commonAllAgentsModule } from 'common/js/modules';
import 'modules/ida/js/services/idaService.svc.js';
import 'modules/ida/js/services/idas.factory.js';
import 'adminConsole/js/services/instances.svc.js';
import 'adminConsole/js/services/subclients.svc.js';

/**
 * Use this directive for DB2BPF Streams
 */
const app = commonAllAgentsModule;

app.directive('cvDb2dpfManageStreams', function() {
	return {
		restrict: 'E',
		templateUrl: `${appUtil.appRoot}modules/ida/partials/cv-db2dpf-manage-streams.jsp`,
		bindToController: {
			entity: '=',
			db2dpfdata: '='
		},
		controller: 'db2dpfManageStreamsController',
		controllerAs: 'db2dpfMS'
	};
});

app.controller('db2dpfManageStreamsController', [
	'$timeout',
	'cvUtil',
	'cvToaster',
	'cvLoc',
	'idasFactory',
	'cvTableOptions',
	'$uibModal',
	'instanceService',
	'subclientService',
	'$state',
	function(
		$timeout,
		cvUtil,
		cvToaster,
		cvLoc,
		idasFactory,
		cvTableOptions,
		$uibModal,
		instanceService,
		subclientService,
		$state
	) {
		const self = this;
		self.$state = $state;
		this.$onInit = function() {
			/* GRID CONTSTRUCTION */
			const globalGridOptions = angular.copy(cvTableOptions.commonNgGridOptions);

			const streamsForExistingSubclient = self.entity.subclientId > 0;

			self.partitionNodes = [];

			function init(entity) {
				if (entity) {
					self.entity = entity;
					self.grid.grid.dataSource.read();
				}
			}

			function getNodeName(partitionId) {
				let nodeNumber = 'NODE000';
				if (partitionId > 9 && partitionId <= 99) {
					nodeNumber = 'NODE00';
				} else if (partitionId > 99 && partitionId <= 999) {
					nodeNumber = 'NODE0';
				} else if (partitionId > 999 && partitionId <= 9999) {
					nodeNumber = 'NODE';
				}
				return nodeNumber + partitionId;
			}

			function updateStream(node, streamString) {
				//Sample String db2DpfStreams: "0,1 1,1"
				let nodes = streamString.split(' ');
				let partitions = [];
				for (let i = 0; i < nodes.length; i++) {
					let data = nodes[i];
					let dataItem = data.split(',');
					if (dataItem[0] == node.entity.partitionName) {
						node.dataStream = parseInt(dataItem[1]);
						break;
					}
				}
			}

			function getPartitions(streamString) {
				//Sample String db2DpfStreams: "0,1 1,1"
				let nodes = streamString.split();
				let partitions = [];
				angular.forEach(nodes, function(data) {
					let dataItem = data.split(',');
					let node = {};
					node.partitionName = getNodeName(dataItem[0]);
					node.dataStream = dataItem[1];
					partitions.push(node);
				});
				return partitions;
			}

			function fetchData(options) {
				if (self.partitionNodes && self.partitionNodes.length > 0) {
					$timeout(options.success(self.partitionNodes));
				} else {
					// Get Nodes from instance Details
					instanceService
						.getInstanceDetails(self.entity.instanceId)
						.success(function(data) {
							self.partitionNodes = [];
							data.db2Instance.dpfpartitionclients.forEach(function(node) {
								const partitionNode = { entity: {} };
								partitionNode.entity.clientName = node.nodeClient.clientName;
								partitionNode.entity.partitionName = node.nodenum;
								partitionNode.dataStream = 1; //default value
								if (streamsForExistingSubclient && self.db2dpfdata.db2DpfStreams) {
									//updateStream(partitionNode,'0,1 1,1');//For Testing
									updateStream(partitionNode, self.db2dpfdata.db2DpfStreams);
								}
								partitionNode.entity.partitionName = getNodeName(node.nodenum);
								self.partitionNodes.push(partitionNode);
							});
							if (!streamsForExistingSubclient) {
								self.db2dpfdata.db2DpfStreams = getStreamString(self.partitionNodes);
								self.db2dpfdata.numberOfBackupStreams = getTotalDataStreams(self.partitionNodes);
							}
							options.success(self.partitionNodes);
						})
						.error(function(err) {
							options.error(err);
						});
				}
			}

			function onGridDataBound(dataItem, row) {
				let partitionName = _.get(dataItem, 'entity.partitionName', '');
				if (partitionName == 'NODE0000') {
					self.grid.grid.select(row);
				}
			}

			/* Editing in place in grid cell */
			self.startEditing = function(rowEntity) {
				rowEntity.dataStreamNew = rowEntity.dataStream;
			};

			function refreshGrid() {
				self.grid.grid.dataSource.read();
				self.grid.grid.clearSelection();
				// self.grid.refreshGrid();
			}

			function getStreamString(partitionNodes) {
				let streamString = '';
				let nodes = [];
				partitionNodes.forEach(function(node) {
					let nodeNum = parseInt(node.entity.partitionName.substring(4));
					nodes.push(nodeNum + ',' + node.dataStream);
				});
				streamString = nodes.join(' ');
				return streamString;
			}

			function getTotalDataStreams(partitionNodes) {
				let streams = 0;
				partitionNodes.forEach(function(node) {
					streams = streams + parseInt(node.dataStream);
				});
				return streams;
			}

			/* Send request to update for existing subclient only */
			function sendRequest(newInstances, callBack) {
				self.sendingRequest = true;
				const streamsObject = getStreamString(newInstances);
				const backupStreams = getTotalDataStreams(newInstances);
				const subclientInfo = {
					db2SubclientProp: {
						db2DpfStreams: streamsObject,
						numberOfBackupStreams: backupStreams
					}
				};
				subclientService
					.updateSubclient({
						subClientEntity: angular.toJson(self.entity),
						subclientInfo: angular.toJson(subclientInfo)
					})
					.success(function() {
						if (callBack) {
							callBack();
						}
						self.sendingRequest = false;
					})
					.error(function(err) {
						cvToaster.showErrorMessage({ ttl: 10000, message: err });
						self.sendingRequest = false;
					});
			}

			/* Check if stream is valid */
			function isValidStream(stream) {
				return (stream || stream === 0) && Math.round(stream) === stream && stream >= 0 && stream <= 50;
			}

			/* Check if all nodes have 0 stream */
			function hasPositiveStream(rowEntity) {
				let hasPositiveDataStream = false;
				for (let i = 0; i < self.partitionNodes.length; ++i) {
					const dataStream =
						rowEntity === self.partitionNodes[i]
							? self.partitionNodes[i].dataStreamNew
							: self.partitionNodes[i].dataStream;
					if (dataStream > 0) {
						hasPositiveDataStream = true;
					}
				}
				return hasPositiveDataStream;
			}

			/* Stop editing */
			self.stopEditing = function(rowEntity, callBackFunc) {
				if (!isValidStream(rowEntity.dataStreamNew)) {
					cvToaster.showErrorMessage({
						message: cvLoc('error.streamRange')
					});
					return;
				}
				if (!hasPositiveStream(rowEntity)) {
					cvToaster.showErrorMessage({
						message: cvLoc('error.allNodesZeroStream')
					});
					return;
				}
				const newInstances = [];
				//update all nodes with same number
				for (let i = 0; i < self.partitionNodes.length; ++i) {
					const newInst = angular.copy(self.partitionNodes[i]);
					newInst.dataStream = rowEntity.dataStreamNew;
					newInst.editing = false;
					newInstances.push(newInst);
				}
				// Send update request for existing subclient
				if (streamsForExistingSubclient) {
					sendRequest(newInstances, function() {
						self.partitionNodes = newInstances;
						refreshGrid();
						callBackFunc();
						//refresh subclient details page to update number of streams
						self.$state.forceReload();
					});
				} else {
					self.partitionNodes = newInstances;
					self.db2dpfdata.db2DpfStreams = getStreamString(newInstances);
					self.db2dpfdata.numberOfBackupStreams = getTotalDataStreams(newInstances);
					refreshGrid();
					callBackFunc();
				}
			};

			function getColumns() {
				const columns = {
					'entity.clientName': {
						title: cvLoc('label.server'),
						type: 'string',
						width: '30%',
						template: `<span class="crop" cv-toggle-content="#=entity.clientName#" cv-toggle="tooltip">#=entity.clientName#</span>`
					},
					'entity.partitionName': {
						title: cvLoc('label.db2partition'),
						type: 'string',
						width: '30%',
						sort: {
							direction: 'asc',
							priority: 0
						},
						template: `<span class="crop" cv-toggle-content="#=entity.partitionName#" cv-toggle="tooltip">#=entity.partitionName#</span>`
					},
					dataStream: {
						title: cvLoc('label.stream'),
						type: 'number',
						width: '20%',
						cellTemplate: `<span class="crop">#=entity.dataStream#</span>`
					}
				};
				return columns;
			}

			function actionClicked(e) {
				const selectedInst = e.selectedRowValues[0];
				switch (e.optionId) {
					case 'Edit':
						self.openEditModal(selectedInst);
						break;
					default:
						break;
				}
			}

			self.openEditModal = function(inst) {
				$uibModal.open({
					templateUrl: `${appUtil.appRoot}modules/ida/partials/db2dpfEditStream.jsp`,
					windowClass: 'small-size',
					backdrop: 'static',
					controllerAs: 'db2dpfESM',
					controller: [
						'$uibModalInstance',
						'$scope',
						function($uibModalInstance, $scope) {
							$scope.isReadyForAPICall = false;
							let modSelf = this;
							modSelf.inst = inst;
							modSelf.streamsForExistingSubclient = self.streamsForExistingSubclient;

							self.startEditing(modSelf.inst);
							modSelf.submit = function() {
								self.stopEditing(modSelf.inst, function() {
									$uibModalInstance.close();
								});
							};

							modSelf.cancel = function() {
								$uibModalInstance.dismiss();
							};
						}
					]
				});
			};

			function onGridSelectionChange(e) {
				if (e.rows.length >= 1) {
					self.grid.showActionMenuOption('Edit');
				} else {
					self.grid.hideActionMenuOption('Edit');
				}
			}

			// Use Kendo grid
			function initGrid() {
				self.gridOptions = {
					enableCheckBoxColumn: false,
					enableVirtualization: true,
					enableFiltering: false,
					enableSorting: false,
					tableName: 'db2dpfManageStreamsTable',
					columns: getColumns(),
					url: fetchData,
					onGridDataBound: onGridDataBound,
					gridTitle: streamsForExistingSubclient
						? cvLoc('label.db2PartitionStreams') + ': ' + self.db2dpfdata.numberOfBackupStreams
						: cvLoc('label.db2PartitionStreams'),
					hasViews: false,
					hasDefaultView: false,
					onGridSelectionChange,
					beforeGridInitialize: ({ grid }) => {
						self.grid = grid;
					},
					afterGridInitialize: () => {
						$('#streams .cv-k-grid-header').hide();
					}
				};

				//For create subclient table scroll bar fix
				if (!streamsForExistingSubclient) {
					self.gridOptions.height = 250;
				}
				self.gridOptions.actionMenu = [
					{
						id: 'Edit',
						label: cvLoc('label.inlineEdit'),
						onSelect: actionClicked
					}
				];
			}
			initGrid();
			init();
		};
	}
]);

export default app;
