import 'modules/ida/js/directives/cv-browse-facets.js';
import 'adminConsole/js/directives/acDirectives.js';
import 'adminConsole/js/directives/cv-search-box.directive.js';
import 'adminConsole/js/factories/backupAndRestore.factory.js';

import { commonAllAgentsModule } from 'common/js/modules';

const FileOrFolderType = {
	FOLDER: 'FOLDER',
	FILE: 'FILE',
	MAILBOX: 'MAILBOX'
};

/*
 * TODO: -- Instead of just replacing the grid headers for a change in the different entities that are
 * browsed, a new grid should be made. No problems occur because of this as of now, but it could prevent
 * issues that arise later. -- See if there's a better way to handle making a browse request to fill the
 * browse tree with the remaining results after clicking "More" 3 times. Right now, a request is made to
 * retrieve everything, and the duplicates are removed. -- Right now, when the item being browsed in the table
 * causes the directive to request for the table headers to be shown by the defined
 * "options.getBrowseTableHeaders" hook. This overwrites the table's current headers to use the new headers.
 * Instead, a new grid should be made with new headers.
 */

/**
 * Specifies the height of each row in the browse table.
 *
 * @var {number} ROW_HEIGHT
 */
var ROW_HEIGHT = 39;

var mod = commonAllAgentsModule;

/**
 * A directive used for building all components to browsing different types of file structures. This includes
 * a browse tree on the left hande side, and a browse table on the right hand side
 *
 * @param {string}
 *            entityType - The type of entity you are browsing.
 * @param {number}
 *            entityId - The primary key value of the entity you are browsing.
 * @param {Object=}
 *            options - Different options the directive uses to handle the functionality of the browse
 *            components.
 * @param {boolean=}
 *            options.showTable - Whether or not to show the browse table with the directive. By default, the
 *            table will show.
 * @param {boolean=}
 *            options.useSingleCallToGetData - Whether or not to use single browse request to render both tree
 *            and table content
 * @param {boolean=}
 *            options.showDateFilter - Whether or not to show the dropdown for filtering by the date backups
 *            occured. By default, the filter will show. The browse table must be shown as well in order for
 *            the filter to appear.
 * @param {Object=}
 *            options.forEntity - The entity to focus on restoring. When included, this entity will be the
 *            only one available to browse in the tree.
 * @param {string}
 *            options.forEntity.id - A unique identifier to use for the entity to focus on. This is required
 *            if options.forEntity is given.
 * @param {string}
 *            options.forEntity.label - The display name of the entity to focus on. This is required if
 *            options.forEntity is given.
 * @param {string=}
 *            options.restoreText - Optional restore text to show INSTEAD of the default "Restore" text.
 * @param {boolean=}
 *            options.showCollapseTreeButton - Whether or not you want the option to (expand/close) the left
 *            tree when viewing the browse results.
 * @param {string=|Function=}
 *            options.rootLabel - The label of the root item of the tree. If a Function is given, then the
 *            root node will be passed to the function, and the string that is returned will be used as the
 *            root's label. By default, the root label will be the name of the subclient that is being
 *            browsed.
 * @param {Function=}
 *            options.onRestore - A function called when the restore button is pressed after the user has
 *            selected different items to restore. An array of the selected items will be passed as an
 *            argument.
 * @param {Function=}
 *            options.onEntityRetrieval - A function called when the root node of the entity being browsed is
 *            retrieved. The root node will be passed as an argument, and it allows you to handle page
 *            specific functionality, such as building breadcrumbs.
 * @param {Function=}
 *            options.onTreeItemSelected - A function called when an item on the tree is selected. The item
 *            itself will be passed as an argument.
 * @param {Function=}
 *            options.onBeforeBrowseTreeRequest - A function called before a request is sent to build the
 *            browse tree of the component. The item the user selected to browse will be passed as an
 *            argument. If the function returns an object of browse parameters, those parameters will be used
 *            in the request sent to the back-end.
 * @param {Function=}
 *            options.onBeforeBrowseTableRequest - A function called before a request is sent to build the
 *            browse table of the component. The item the user selected to browse will be passed as an
 *            argument. As a second argument, the function will be passed if the request is being made for a
 *            search by the user. If the function returns an object of browse parameters, those parameters
 *            will be used in the request sent to the back-end.
 * @param {Function=}
 *            options.onAfterBrowseTableRequest - A function called after a request to build the browse table
 *            component has finished. The item that was selected to be browsed will be passed as the first
 *            argument, and the browse data that was retrieved will be passed as the second argument, allowing
 *            you to alter the data before it is used in the table.
 * @param {Function=}
 *            options.onBrowseFailure - A function called when there is a browse failure when either trying to
 *            retrieve the node entity, performing a tree browse, or performing a table browse.
 * @param {Function=}
 *            options.onAfterBrowseTreeRequest - A function called after a request to build the browse tree
 *            component has finished. The item that was selected to be browsed will be passed as the first
 *            argument, and the browse data that was retrieved will be passed as the second argument, allowing
 *            you to alter the data before it is used in the tree.
 * @param {Function=}
 *            options.onBeforeBrowseTableBuilt - A function called before the browse table is built after each
 *            browse table request. This can be used to manipulate all of the grid options as the integrator
 *            sees fit. An object will be given to the hook with the the following parameters: tableEntity,
 *            gridOptions, and forSearch.
 * @param {Function=}
 *            (DEPRECATED - Please use options.onBeforeBrowseTableBuilt) options.getBrowseTableHeaders - A
 *            function called when the component needs to build the browse table headers. The item the user
 *            selected to browse will be passed as an argument. As a second argument, the function will be
 *            passed whether or not the table response has been for a search request or not. If true, a search
 *            was made for the tree response, and false otherwise. This is helpful in the event that different
 *            table headers are used to display the results ona search request. If the function returns an
 *            array of cv-grid column header definitions, those headers will be used to build the table.
 * @param {Function=}
 *            options.noneSelected - A function called when user has to decide on what logic will the restore
 *            be disabled. If this function is not implementd by user, the default behaviour will be to
 *            disable when no rows are selected in the browseTable. To use this method, browseTable must be
 *            present.
 * @param {boolean=}
 *            options.hideTimeRange - Whether or not to hide the date range in the drop-down of date options,
 *            by default it will be shown.
 * @param {boolean=}
 *            options.skipEncodePath - If true skips encoding of path by default it will be false
 * @param {Function=}
 *            options.getEntity - Function to fetch entity node
 * @param {Function=}
 *            options.getBrowseData - Function to fetch browse data
 * @param {Function=}
 *            options.gridSelectionHandlerForActions - A function to allow dev to specify user actions
 *            behavior for grid row(s) selection
 *
 */
mod.directive('cvBrowse', [
	'cvLoc',
	'$timeout',
	function(cvLoc, $timeout) {
		return {
			restrict: 'E',
			scope: {
				entityType: '@',
				entityId: '@',
				options: '=',
				onLoadSearch: '@'
			},
			transclude: {
				browseActions: '?cvBrowseActions'
			},
			templateUrl: appUtil.appRoot + 'modules/ida/partials/cv-browse.jsp',
			controllerAs: 'cvBrowseController',
			controller: [
				'$q',
				'$scope',
				'$filter',
				'cvTableOptions',
				'uiGridConstants',
				'$stateParams',
				'$state',
				'backupAndRestoreFactory',
				function($q, $scope, $filter, cvTableOptions, uiGridConstants, $stateParams, $state, backupAndRestoreFactory) {
					var self = this;
					self.entityType = $scope.entityType;
					self.entityId = $scope.entityId;
					self.options = $scope.options;
					self.onLoadSearch = $scope.onLoadSearch;
					self.epicServerState = $scope.options.epicServerState;

					if (self.entityType.length <= 0 || self.entityId.length <= 0) {
						return;
					}

					self.options = angular.isObject(self.options) ? self.options : {};
					self.options.showTable =
						angular.isDefined(self.options.showTable) && self.options.showTable === false ? false : true;
					self.options.useSingleCallToGetData =
						angular.isDefined(self.options.useSingleCallToGetData) && self.options.useSingleCallToGetData === true
							? true
							: false;
					self.options.showDownload = self.options.showDownload || false;
					self.options.disableSearchActions = self.options.disableSearchActions || false;
					self.options.showRestoreGranular = self.options.showRestoreGranular || false;

					self.lastTablePageSize = 15;
					self.options = self.options;
					self.entityType = self.entityType;
					self.entityId = self.entityId;
					self.breadcrumbs = [];

					self.restore = self.options.onRestore;
					self.serverMessage = undefined;

					//if it is an onLoadSearch then hide the tree by default, check this
					if (angular.isDefined(self.onLoadSearch)) {
						self.hasSearced = true;
					}

					/**
					 * Encodes the given path before sending it to the server.
					 *
					 * @param {string}
					 *            path - The path to encode.
					 * @return {string} The encoded string.
					 */
					var encodePath = function(path) {
						return encodeURI(path)
							.replace(/%5C/g, '\\')
							.replace(/%20/g, ' ')
							.replace(/%7B/g, '{')
							.replace(/%7D/g, '}');
					};

					/**
					 * Performs a browse request on the given item. If a browse request was already performed
					 * on the given item, it is assumed we want to increase the paging. Therefore, we can use
					 * the previous parameters used to make the browse request, but increase the paging.
					 *
					 * @param {Object}
					 *            item - The item selected to browse.
					 */
					self.performTreeBrowse = function(item) {
						var params = {};
						if (angular.isFunction(self.options.onBeforeBrowseTreeRequest)) {
							/*
							 * A hook has been defined to run before a browse request for the tree is sent.
							 * This hook can return parameters to be sent with the request.
							 */
							var result = self.options.onBeforeBrowseTreeRequest(item);
							if (angular.isObject(result)) {
								params = result;
							}
						}

						if (!angular.isObject(item.params) || self.doReload) {
							/*
							 * If the item we're browsing doesn't already have pre-existing parameters
							 * attached to it, then that means we haven't performed a browse on it yet.
							 * Therefore, we start at the first page.
							 *
							 * Also, if on reload, say due to timerange change, go back to page 1 in tree too.
							 */
							params.pagingInfo = '0,20';
						} else if (angular.isDefined(item.params.pagingInfo)) {
							/*
							 * The item already has pre-existing browsing parameters attached to it, so we
							 * need to continue browsing it at the next page.
							 */
							var pages = item.params.pagingInfo.split(',');
							var page = parseInt(pages[0]);
							var size = parseInt(pages[1]);

							if ((page + 2) * size >= 3 * size) {
								/*
								 * Each browse request for the tree retrieves 20 results. On the third browse
								 * request, we want to get every entity for the tree, so if that's the case,
								 * we delete the paging info to get everything.
								 */
								delete params.pagingInfo;
							} else {
								params.pagingInfo = page + 1 + ',' + size;
								//Fix for table page size set as default instead of derived from paging info
								self.lastTablePageSize = size;
							}
						}

						item.params = params;

						/*
						 * If a path is being given to browse, we must encode it so that it doesn't cause
						 * problems with the back-end. However, backward slashes are okay still, so we need to
						 * convert those back after using "encodeUri".
						 */
						if (angular.isString(item.params.path) && !self.options.skipEncodePath) {
							//not sure why we need to check decode string equals original string, but instead removing it, add try catch block to avoid js error
							try {
								if (decodeURIComponent(item.params.path) === item.params.path) {
									item.params.path = encodePath(item.params.path);
								}
							} catch (error) {
								//if path string has %, decodeURIComponent function will throw exception, thus don't do decodeURIComponent if path string has %
							}
						}

						if (_.isString(self.lastToDateString)) {
							item.params.toTime = self.lastToDateString;
						}

						if (_.isString(self.lastFromDateString)) {
							item.params.fromTime = self.lastFromDateString;
						}

						return self.options.getBrowseData(item.params).then(function(data) {
							if (angular.isFunction(self.options.onAfterBrowseTreeRequest)) {
								/*
								 * A hook has been defined to pass the retrieved data to. This hook allows the
								 * developer to alter the data before it is used in the tree.
								 */
								self.options.onAfterBrowseTreeRequest(item, data.browseData);
								//reset totalItems to browse data which removed leaf node
								data.totalItems = data && data.browseData ? data.browseData.length : 0;
							}

							/*
							 * If the paging info for the item that's being browsed is undefined, then we
							 * didn't send any paging info. What this means is that everything will be
							 * returned to us. Therefore, we remove the duplicates from the retrieved data
							 * that we already have before operating on it.
							 */
							if (angular.isUndefined(item.params.pagingInfo) && angular.isArray(item.children)) {
								data.browseData.splice(0, item.children.length);
							}
							if (self.options.useSingleCallToGetData || self.options.isExchangeBrowse) {
								angular.forEach(data.browseData, function(dataItem) {
									dataItem.parent = item;
								});
							}
							return data;
						}, onError);
					};

					/**
					 * Makes the given item the selected item in the browse tree component.
					 *
					 * If the item does not have any children defined yet (different from having an array of
					 * empty children), the method will perform a request to get the item's children to
					 * populate the tree if it has any.
					 */
					self.makeSelectedItem = function(item) {
						if (angular.isDefined(self.selectedItem)) {
							self.selectedItem.selected = false;
						}
						self.selectedItem = item;
						self.selectedItem.selected = self.selectedItem.expanded = true;

						if (!item.browsePerformed && !angular.isArray(item.children)) {
							self.performTreeBrowse(item).then(function(data) {
								self.selectedItem.children = data.browseData;
							});
						}
					};

					/**
					 * This can be used to handle the clear action from the search box directive
					 */
					self.onClearSearch = function() {
						//clear the facet information
						self.facets = [];
						self.lastFacets = [];
						self.onCollapseButtonClick(true);
					};

					/**
					 * This method is used to dynamically collapse or open left tree/facet view
					 *
					 * @param {boolean}
					 *            action - Action to either open or collapse
					 */
					self.onCollapseButtonClick = function(action) {
						self.collapseTree = action;
					};

					if (self.options.showTable) {
						self.lastSearchTerm = '';
						/**
						 * The method that is called upon for performing a search.
						 *
						 * @param {string}
						 *            searcTerm - The search term the user has entered.
						 */
						self.onSearch = function(searchTerm) {
							//open the Facet view on left when search is initiated
							self.onCollapseButtonClick(false);
							if (searchTerm !== self.lastSearchTerm) {
								//clearing the facet information.
								//Currently, this option "clearFacetInfo" is set only for exchange but should be set for all idas using cv-browse.
								if (angular.isDefined(self.options.clearFacetInfo) && self.options.clearFacetInfo === true) {
									self.facets = [];
									self.lastFacets = [];
								}

								if (searchTerm.length > 0) {
									//since this is going to be a new search, remove the facet information from memory so that the new search is fresh.
									//currently checking for onLoadSearch for VSA but should be generalized to others as well
									if (angular.isDefined(self.onLoadSearch)) {
										//clear the facet information
										self.facets = [];
										self.lastFacets = [];
									}

									/*
									 * If a search was made, we will perform the search at the root level of
									 * the tree. Therefore, we perform a browse on the table with the root
									 * node of the tree as the argument.
									 */
									self.lastSearchTerm = searchTerm;
									if (angular.isFunction(self.changeGridPage)) {
										self.changeGridPage(1);
									}
									this.performTableBrowse(self.tree, 0)
										.then(function(response) {
											//for onLoadSearch for VSA this is first time call with direct search so assign the browse results need to be assigned here
											if (angular.isDefined(self.onLoadSearch)) {
												//update the onLoadSearch with the new term
												self.onLoadSearch = searchTerm;
												if (self.options.showTable) {
													self.data = response;
												} else {
													self.dontSpin = true;
												}
												//update time range label
												//in case we are doing default search, on change of backup time range
												//loadEntity is called and we need to update timeRangeLabel here
												self.updateTimeRangeLabel();
											}
											if (angular.isDefined(response.facets)) {
												self.facets = response.facets.facetResult;
											} else {
												self.facets = [];
											}
										})
										['finally'](function() {
											self.hasSearched = true;
											self.breadcrumbs.push({
												label: cvLoc('label.searchResults')
											});
										});
								} else {
									/*
									 * A search was made with an empty string, so we need to reset back to our
									 * default state.
									 */
									self.lastSearchTerm = undefined;
									if (angular.isFunction(self.changeGridPage)) {
										self.changeGridPage(1);
									}

									//in case it was an onLoadSearch for VSA we don't want to trigger a browse, show an empty grid in this case
									if (angular.isDefined(self.onLoadSearch)) {
										self.dontSpin = self.loaded = true;
										if (angular.isFunction(self.changeGridPage)) {
											self.gridOptions.gridOptions.data = [];
											self.gridOptions.gridOptions.totalItems = 0;
										}
										self.visibleItems = 0;
									} else {
										this.performTableBrowse(self.tree, 0)['finally'](function() {
											self.hasSearched = false;
										});
									}
								}
							}
						};

						/**
						 * Handles the functionality of the user clicking on a breadcrumb in the browse table.
						 *
						 * @param {Object}
						 *            item - The item selected in the breadcrumbs.
						 */
						self.breadcrumbSelected = function(item) {
							self.lastSearchTerm = undefined;
							self.performTableBrowse(item)['finally'](function() {
								self.searchTerm = self.lastFacets = undefined;
								self.hasSearched = false;
							});
						};

						/**
						 * The method that is called upon when facets are changed.
						 *
						 * The method will reference the existing "facets" object in the controller to find
						 * the changes, and will filter out any deselected facets since the browse API is only
						 * looking for the selected facets when the request is sent.
						 */
						self.onFacetsChanged = function() {
							self.lastFacets = angular.copy(self.facets);
							self.lastFacets = $filter('filter')(self.lastFacets, function(item) {
								if (angular.isDefined(item.stringParameter)) {
									item.stringParameter = $filter('filter')(item.stringParameter, {
										selected: true
									});
									return item.stringParameter.length > 0;
								} else if (angular.isDefined(item.dateParameter)) {
									item.dateParameter = $filter('filter')(item.dateParameter, {
										selected: true
									});
									return item.dateParameter.length > 0;
								} else if (angular.isDefined(item.rangeParameter)) {
									item.rangeParameter = $filter('filter')(item.rangeParameter, {
										selected: true
									});
									return item.rangeParameter.length > 0;
								}

								return false;
							});

							this.performTableBrowse(self.tree).then(function(response) {
								if (angular.isDefined(response.facets)) {
									self.facets = response.facets.facetResult;
								} else {
									self.facets = [];
								}
							});
						};

						/*
						 * If we are loading the table for the first time, we want to populate the breadcrumbs
						 * with a "Loading..." label. This will be removed after the first table browse is
						 * completed.
						 */
						var firstTimeTableBrowse = true;
						self.breadcrumbs.push({
							label: cvLoc('label.bePatient')
						});

						/**
						 * Searches the browse tree to find and select an item with the matching ID.
						 *
						 * @param {string}
						 *            id - The ID used to find the item to select.
						 * @param {Object=}
						 *            tree - The tree to search through. If left blank, the function will
						 *            start at the root of the tree, and will recursively search through its
						 *            children.
						 * @return boolean True if the item was found; False otherwise.
						 */
						var findAndSelectItemWithId = function(id, tree) {
							var tree = angular.isDefined(tree) ? tree : self.tree;

							/*
							 * If the tree is still not defined, then it must be the first time we're
							 * browsing, so don't select anything.
							 */
							if (!angular.isDefined(tree)) {
								return false;
							}

							if (tree.id === id) {
								self.makeSelectedItem(tree);
								return true;
							} else if (angular.isArray(tree.children)) {
								for (var i in tree.children) {
									if (findAndSelectItemWithId(id, tree.children[i])) {
										return true;
										break;
									}
								}
							}
						};

						/**
						 * Updates the field that is being sorted in the grid.
						 *
						 * @param {string}
						 *            param - The field to sort.
						 * @param {string?}
						 *            direction - The direction to sort. Is ASC by default.
						 */
						self.updateSort = function(param, direction) {
							self.lastSort = {
								direction: angular.isString(direction) ? direction : 'asc',
								param: param
							};

							// Updating the sort parameters updates the paging
							self.changeGridPage(1);
							self.performTableBrowse(self.tableEntity, 0);
						};

						/**
						 * Formats the given date GMT date into a date that is understood by the Browse API.
						 *
						 * @param {string}
						 *            date - A GMT date structed like the following: Wed Jun 07 2017 15:26:15
						 *            GMT-0400 (Eastern Daylight Time)
						 * @return {string} A date that is understood by the browse API.
						 */
						var formatDate = function(date) {
							return $filter('date')(date, 'yyyy-MM-ddTHH:mm:ssZ');
						};

						/**
						 * Fornats the given GMT date into a date that can be displayed to the user.
						 *
						 * @param {string}
						 *            date - A GMT date structed like the following: Wed Jun 07 2017 15:26:15
						 *            GMT-0400 (Eastern Daylight Time)
						 * @return {string} A date that can be displayed to the user.
						 */
						var formatDisplayDate = function(date) {
							var utcDate = $filter('date')(date, 'yyyy-MM-dd HH:mm:ss');
							// return $filter('date')(new Date(utcDate + 'Z'), 'M/dd/yyyy h:mma');
							return moment.utc(utcDate).format('MM/DD/YYYY h:mma');
						};

						self.timeRangeLabel = cvLoc('Showing_latest_backups');

						/**
						 * Resets the time range filter on the browse request for the table.
						 */
						self.resetTimeRange = function() {
							self.lastFromDateString = undefined;
							self.lastToDateString = undefined;
							delete self.tableEntity.params.toTime;
							delete self.tableEntity.params.fromTime;

							/*
							 * Resetting the time range resets the table back to page 0 for the current
							 * entity.
							 */
							if (angular.isFunction(self.changeGridPage)) {
								self.changeGridPage(1);
							}
							self.reload();
						};

						/**
						 * Updates the browse table to only show records of a given date range.
						 *
						 * @param {Object}
						 *            result - The resulting object given by the cv-date-time directive.
						 */
						self.updateTimeRange = function(result) {
							if (angular.isObject(result)) {
								if (angular.isDefined(result.fromDateString)) {
									self.lastFromDateString = formatDate(result.fromDateString);
								} else {
									self.lastFromDateString = undefined;
								}

								if (angular.isDefined(result.toDateString)) {
									self.lastToDateString = formatDate(result.toDateString);
								} else {
									self.lastToDateString = undefined;
								}
								/*
								 * Updating the time range resets the table back to page 0 for the current
								 * entity.
								 */
								if (angular.isFunction(self.changeGridPage)) {
									self.changeGridPage(1);
								}
								self.reload();
							}
						};

						/**
						 * Performs a browse request on the given item to populate the table on the right hand
						 * side.
						 *
						 * @param {Object}
						 *            item - The item selected to browse.
						 * @param {number}
						 *            page - The page to start at.
						 * @param {number}
						 *            pageSize - The size of the pages to retrieve.
						 * @param {string}
						 *            searchTerm - query to perform search
						 */
						self.performTableBrowse = function(item, page, pageSize, searchTerm, useClientSidePaging) {
							if (self.tableEntity !== item && !searchTerm) {
								// Changing the table entity being browsed resets sorting
								self.lastSort = undefined;

								/*
								 * Since we're changing the table entity that is being browsed, we need to
								 * find and select the new entity in the tree if it exists. If it doesn't
								 * exist, just make sure that nothing is selected.
								 */
								//This might trigger a tree browse request with table browse. This should be prevented in case we are doing an onLoadSearch for VSA
								if (!angular.isDefined(self.onLoadSearch)) {
									if (!findAndSelectItemWithId(item.id)) {
										if (angular.isDefined(self.selectedItem)) {
											self.selectedItem.selected = false;
											self.selectedItem = undefined;
										}
									}
								}

								self.tableEntity = angular.copy(item);
							}

							var params = {};
							if (angular.isFunction(self.options.onBeforeBrowseTableRequest)) {
								/*
								 * A hook has been defined to run before the browse request is sent to the
								 * API. This hook can return back the browsing options that should be used for
								 * the request.
								 */
								var result = self.options.onBeforeBrowseTableRequest(
									self.tableEntity,
									angular.isString(self.lastSearchTerm) && self.lastSearchTerm.length > 0
								);
								if (angular.isObject(result)) {
									params = result;

									if (!searchTerm && params.keywords) {
										searchTerm = params.keywords;
									}
								}
							}

							self.tableEntity.params = params;

							/*
							 * Define some of the additional parameters for the request, such as paging,
							 * searching, sorting, and facets.
							 */
							var pageParam = angular.isDefined(page) ? page : 0;
							var pageSizeParam = angular.isDefined(pageSize) ? pageSize : self.lastTablePageSize;
							self.lastTablePageSize = pageSizeParam;
							self.tableEntity.params.pagingInfo = pageParam + ',' + pageSizeParam;

							if (angular.isString(self.lastSearchTerm) && self.lastSearchTerm.length > 0) {
								self.tableEntity.params.keywords = self.lastSearchTerm;
								self.dontSpin = false;
							} else {
								delete self.tableEntity.params.keywords;
							}

							if (searchTerm) {
								self.tableEntity.params.keywords = searchTerm;
							}

							if (angular.isArray(self.lastFacets) && self.lastFacets.length > 0) {
								self.tableEntity.params.facets = angular.toJson(self.lastFacets);
							} else {
								delete self.tableEntity.params.facets;
							}

							if (angular.isObject(self.lastSort)) {
								self.tableEntity.params.sortingInfo = self.lastSort.direction + ':' + self.lastSort.param;
							}

							if (angular.isString(self.lastToDateString)) {
								self.tableEntity.params.toTime = self.lastToDateString;
							}

							if (angular.isString(self.lastFromDateString)) {
								self.tableEntity.params.fromTime = self.lastFromDateString;
							}

							/*
							 * If a path is being given to browse, we must encode it so that it doesn't cause
							 * problems with the back-end. However, backward slashes are okay still, so we
							 * need to convert those back after using "encodeUri".
							 */
							if (angular.isString(self.tableEntity.params.path) && !self.options.skipEncodePath) {
								try {
									//not sure why we need to check decode string equals original string, but instead removing it, add try catch block to avoid js error
									if (decodeURIComponent(self.tableEntity.params.path) === self.tableEntity.params.path) {
										self.tableEntity.params.path = encodePath(self.tableEntity.params.path);
									}
								} catch (error) {
									//if path string has %, decodeURIComponent function will throw exception, thus don't do decodeURIComponent if path string has %
								}
							}

							var newBreadcrumbs = self.breadcrumbs;
							if (self.tableEntity !== item && !searchTerm) {
								if (firstTimeTableBrowse || self.doReload) {
									newBreadcrumbs = [];
									firstTimeTableBrowse = false;
									self.doReload = false;
								}

								if (angular.isDefined(item.parent)) {
									/*
									 * If the item has a parent, that means it came from the browse tree.
									 * Therefore, we can build the breadcrumbs over again.
									 */
									newBreadcrumbs = [];
									newBreadcrumbs.unshift(item);
									var parent = item.parent;
									while (angular.isDefined(parent)) {
										newBreadcrumbs.unshift(parent);
										parent = parent.parent;
									}
								} else {
									/*
									 * If the given item already exists in our breadcrumbs list, we simply
									 * remove every breadcrumb after the existing item.
									 */
									var index = newBreadcrumbs.indexOf(item);
									if (index >= 0) {
										newBreadcrumbs.splice(index + 1, newBreadcrumbs.length);
									} else {
										newBreadcrumbs.push(item);
									}
								}
							}

							self.serverError = undefined;
							self.serverMessage = self.options.serverMessage;
							if (useClientSidePaging) {
								var begin = page * self.gridOptions.gridOptions.paginationPageSize;
								var end =
									self.tableEntity.data.length > begin + self.gridOptions.gridOptions.paginationPageSize
										? begin + self.gridOptions.gridOptions.paginationPageSize
										: self.tableEntity.data.length;
								self.gridOptions.gridOptions.data = [];
								self.gridOptions.gridOptions.data = self.tableEntity.data.slice(begin, end);
							} else {
								return self.options.getBrowseData(self.tableEntity.params).then(
									function(data) {
										self.dontSpin = self.loaded = true;
										self.serverMessage = undefined;
										var diff = 0;
										if (angular.isFunction(self.options.onAfterBrowseTableRequest)) {
											var copyOfBrowseData = angular.copy(data.browseData);
											/*
											 * A hook has been defined to pass the retrieved data to. This hook
											 * allows the developer to alter the data before it is used in the
											 * table.
											 */
											self.options.onAfterBrowseTableRequest(self.tableEntity, copyOfBrowseData);

											diff = Math.abs(copyOfBrowseData.length - data.browseData.length);
											//table data is parsed to copyOfBrowseData when useSingleCallToGetData
											if (diff > 0 || self.options.useSingleCallToGetData) {
												self.updatedBrowseData = copyOfBrowseData;
											}
										}

										if (self.tableEntity !== item) {
											if (angular.isFunction(self.options.getBrowseTableHeaders)) {
												/*
												 * WARNING: This section is deprecated, and will be removed
												 * eventually in favor of the hook "onBeforeBrowseTableBuilt".
												 *
												 * A hook has been defined to generate the table headers that
												 * should be used for the table. The hook should return an array
												 * of column definitions to use for cv-grid.
												 */
												self.gridOptions.gridOptions.columnDefs = self.options.getBrowseTableHeaders(
													self.tableEntity,
													angular.isString(self.lastSearchTerm) && self.lastSearchTerm.length > 0
												);
											}

											if (angular.isFunction(self.options.onBeforeBrowseTableBuilt)) {
												/*
												 * A hook has been defined signifying that the integrator wants to
												 * change some of the grid options in the table. The hook will be
												 * given the table entity the grid needs to show, the entire grid
												 * options for the integrator to manipulate as they wish, and
												 * whether or not the table is being built for a search request.
												 */
												self.options.onBeforeBrowseTableBuilt({
													tableEntity: self.tableEntity,
													gridOptions: self.gridOptions,
													forSearch: angular.isString(self.lastSearchTerm) && self.lastSearchTerm.length > 0
												});
											}

											angular.forEach(self.gridOptions.gridOptions.columnDefs, function(column) {
												column.suppressRemoveSort = true;
											});
										}

										//self.breadcrumbs = newBreadcrumbs;
										self.breadcrumbs = self.options.useSingleCallToGetData ? self.breadcrumbs : newBreadcrumbs;
										self.gridOptions.gridOptions.data =
											diff == 0 && !self.options.useSingleCallToGetData ? data.browseData : self.updatedBrowseData;

										if (self.epicServerState) {
											//format label on table
											angular.forEach(self.gridOptions.gridOptions.data, function(dataItem) {
												if (dataItem.userObject && dataItem.userObject.displayName) {
													dataItem.userObject.displayName = backupAndRestoreFactory.formatContent(
														dataItem.userObject.displayName
													);
												}
											});
										}

										//if during server side browse the webservice returned the flag then switch to client side paging/sorting
										if (data.switchToClient) {
											self.options.useSingleCallToGetData = true;
										}
										//for client side pagination
										if (self.options.useSingleCallToGetData) {
											self.tableEntity.data = angular.copy(self.gridOptions.gridOptions.data);
											var end =
												self.tableEntity.data.length > self.gridOptions.gridOptions.paginationPageSize
													? self.gridOptions.gridOptions.paginationPageSize
													: self.tableEntity.data.length;
											self.gridOptions.gridOptions.data = [];
											self.gridOptions.gridOptions.data = self.tableEntity.data.slice(0, end);
											self.gridOptions.gridOptions.totalItems = self.tableEntity.data.length;
											self.visibleItems = self.tableEntity.data.length;
										} else {
											self.gridOptions.gridOptions.totalItems = data.totalItems - diff;
											self.visibleItems = data.browseData.length;
										}
										angular.forEach(data.browseData, function(dataItem) {
											dataItem.parent = item;
										});
										return data;
									},
									function(response) {
										self.breadcrumbs = newBreadcrumbs;
										self.dontSpin = self.loaded = true;
										self.serverMessage = undefined;
										self.serverError = response || cvLoc('generic_error');
										self.gridOptions.gridOptions.data = [];
										self.gridOptions.gridOptions.totalItems = 0;
										self.visibleItems = 0;
									}
								); //ENDSW
							} //END ELSE
						};

						self.performSearch = function(query, fields) {
							var includesFromTime = _.includes(query, 'FromTime:'),
								includesToTime = _.includes(query, 'ToTime:'),
								fromTimeField = _.find(fields, { tag: 'FromTime:' }),
								toTimeField = _.find(fields, { tag: 'ToTime:' }),
								fromTime = moment(fromTimeField.model).format('MM/DD/YYYY'),
								toTime = moment(toTimeField.model).format('MM/DD/YYYY');

							if (includesFromTime) {
								if (fromTimeField) {
									query = query.replace('FromTime:' + fromTime, cvLoc('fs.keyword.modTime') + fromTime);
								}

								if (includesToTime) {
									query = query.replace('ToTime:' + toTime, ' to ' + toTime);
								}
							} else if (includesToTime) {
								query = query.replace('ToTime:' + toTime, cvLoc('fs.keyword.modTime') + 'to ' + toTime);
							}

							self.performTableBrowse(self.tableEntity, undefined, undefined, query);
						};

						/*
						 * Define the grid options for showing the browse table.
						 */
						self.gridOptions = {
							cvIsPageTitle: false,
							cvGridCssClass: 'grid-style users-grid',
							cvIsSearchable: false,
							cvIsAdvSearch: false,
							cvAdvSearchOptions: {
								query: '',
								fieldDefs: [],
								onSearch: self.performSearch
							},
							cvHasTitle: false,
							cvAppScope: self,
							cvGridDirectives: {
								uiGridSelection: true
							},
							gridOptions: angular.extend(angular.copy(cvTableOptions.commonNgGridOptions), {
								data: [],
								totalItems: 0,
								useExternalPagination: true,
								useExternalSorting: true,
								paginationPageSizes: [15, 30, 45],
								paginationPageSize: 15,
								rowHeight: ROW_HEIGHT,
								enableGridMenu: false,
								columnDefs: [],
								onRegisterApi: function(gridApi) {
									/**
									 * Changes the page on the grid.
									 *
									 * @param {number}
									 *            page - The page to go to on the grid.
									 */
									self.changeGridPage = function(page) {
										gridApi.pagination.seek(page);
									};

									/*
									 * Runs if the user changes pagination. If that's the case, we subtract
									 * the page the current user is on by 1 when making the API request. This
									 * is because the API starts at page 0, while gridApi starts at page 1.
									 */
									gridApi.pagination.on.paginationChanged(null, function(newPage, pageSize) {
										if (!self.onLoadSearch || (self.onLoadSearch && self.lastSearchTerm)) {
											self.performTableBrowse(self.tableEntity, newPage - 1, pageSize);
										}
									});

									/*
									 * Runs if the user selects to sort on a column header. If selected, a
									 * request to browse the table will be sent with the new sorting
									 * parameter.
									 */
									gridApi.core.on.sortChanged(null, function(grid, sortColumns) {
										if (sortColumns.length > 1) {
											// When there's more than on column that's sorted, that means a user selected
											// on another column to sort while one was already sorted. Since we don't want to allow
											// multi column sorting, we need to unsort the previous sorted column.
											var column;
											if (angular.isUndefined(self.lastSort)) {
												// If the lastSort variable is undefined, then the second sorted column will be the one to remove
												column = sortColumns[sortColumns.length - 1];
											} else {
												// The lastSort variable is defined, so the first sorted column will be the one to remove.
												column = sortColumns[0];
											}
											column.unsort();
											column.sort.priority = 1;
										} else if (sortColumns.length === 1) {
											var column = sortColumns[0];
											self.lastSort = {
												direction: column.sort.direction,
												param: column.colDef.sortParam
											};

											//in case this is the default search flow, then don't trigger a browse request on sort in case the search term is empty
											if (!self.onLoadSearch || (self.onLoadSearch && self.lastSearchTerm)) {
												self.performTableBrowse(self.tableEntity, 0, self.lastTablePageSize);
											}
										} else {
											self.lastSort = undefined;
										}
									});

									/*
									 * This hook allows the ability to override behavior for the selection of
									 * a row in the grid
									 */
									gridApi.selection.on.rowSelectionChanged($scope, function(row) {
										if (angular.isFunction(self.options.gridSelectionHandlerForActions)) {
											var selections = row.grid.api.selection.getSelectedRows();
											self.disableSearchActions = self.options.gridSelectionHandlerForActions(selections);
										}
									});

									/*
									 * This hook allows the ability to override behavior for the selection of
									 * a batch of rows in the grid
									 */
									gridApi.selection.on.rowSelectionChangedBatch($scope, function(rows) {
										if (angular.isFunction(self.options.gridSelectionHandlerForActions)) {
											var selections =
												rows && rows.length > 0 ? rows[0].grid.api.selection.getSelectedRows() : undefined;
											self.disableSearchActions = self.options.gridSelectionHandlerForActions(selections);
										}
									});

									/**
									 * Returns whether or not any rows are selected in the grid.
									 *
									 * @return {boolean} True if no rows are selected; False otherwise.
									 */
									self.noneSelected = function() {
										if (angular.isFunction(self.options.noneSelected)) {
											return self.options.noneSelected(gridApi.selection.getSelectedRows());
										} else {
											return gridApi.selection.getSelectedRows().length <= 0;
										}
									};

									/**
									 * Returns whether or not all rows are selected in the grid.
									 *
									 * @return {boolean} True if all rows are selected; False otherwise.
									 */
									self.allSelected = function() {
										return gridApi.selection.getSelectAllState();
									};

									/**
									 * Open the modal to restore all of the user's selected items.
									 */
									self.restore = function() {
										if (angular.isFunction(self.options.onRestore)) {
											var restoreParams = {
												items: gridApi.selection.getSelectedRows(),
												toTime: self.tableEntity.params.toTime,
												fromTime: self.tableEntity.params.fromTime
											};

											//Setting the path of folder/mailbox being browsed when there  is no selection on the grid
											if (restoreParams.items.length === 0) {
												restoreParams.folderRestore = true;
												restoreParams.folderPath = self.selectedItem.params.path;
											}
											self.options.onRestore(restoreParams);
										}
									};

									/**
									 * Restore all the items matching the search criteria.
									 */
									self.restoreAllMatching = function() {
										if (angular.isFunction(self.options.onRestore)) {
											var restoreAllMatchingParams = {
												path: self.tableEntity.params.path,
												toTime: self.tableEntity.params.toTime,
												fromTime: self.tableEntity.params.fromTime,
												keywords: self.tableEntity.params.keywords,
												facets: self.tableEntity.params.facets,
												restoreAllMatching: true
											};
											self.options.onRestore(restoreAllMatchingParams);
										}
									};
									/**
									 * List clones
									 */

									self.listClones = function() {
										$state.go('dbInstancesClone');
									};

									/**
									 * Start download of user's selected items.
									 */
									self.granularRestore = function(event) {
										if (angular.isFunction(self.options.onGranularRestore)) {
											var restoreParams = {
												items: gridApi.selection.getSelectedRows(),
												toTime: self.tableEntity.params.toTime,
												fromTime: self.tableEntity.params.fromTime
											};

											if (restoreParams.items.length !== 1 || restoreParams.items[0].objectType !== 1011) {
												//show error message to select content DB
												self.serverError = cvLoc('error.select.contentDatabase');
											} else {
												//Setting the path of folder/mailbox being browsed when there  is no selection on the grid
												if (restoreParams.items.length === 0) {
													restoreParams.folderRestore = true;
													restoreParams.folderPath = self.selectedItem.params.path;
												}
												restoreParams.dbName = restoreParams.items[0].label;
												self.options.onGranularRestore(restoreParams);
											}
										}
									};

									/**
									 * Start download of user's selected items.
									 */
									self.download = function(event) {
										if (angular.isFunction(self.options.onDownload)) {
											var downloadParams = {
												path: self.tableEntity.params.path,
												items: gridApi.selection.getSelectedRows(),
												exportType: _.get(event, 'target.id', undefined),
												gridTotal: self.gridOptions.gridOptions.totalItems,
												fileOrFolder: self.tableEntity.params.fileOrFolder,
												keywords: self.tableEntity.params.keywords,
												facets: self.tableEntity.params.facets
											};

											if (downloadParams.items.length === 0 && _.isEmpty(downloadParams.keywords)) {
												downloadParams.folderBrowse = true;
												downloadParams.parentObjectGUID = self.selectedItem.id;
												downloadParams.fileOrFolder = self.selectedItem.params.fileOrFolder;
											} else if (
												!(
													_.get(self.selectedItem, 'userObject') === undefined ||
													_.get(
														self.selectedItem.userObject,
														'advancedData.browseMetaData.indexing.parentObjectGUID'
													) === '00000000000000000000000000000001'
												) &&
												_.isEmpty(downloadParams.keywords)
											) {
												downloadParams.folderExportWithoutSubFolders = true;
												downloadParams.parentObjectGUID = self.selectedItem.id;
											}

											self.options.onDownload(downloadParams);
										}
									};

									/**
									 * Delete user's selected items.
									 */
									self.ondelete = function() {
										if (angular.isFunction(self.options.onDelete)) {
											var deleteParams = {
												items: gridApi.selection.getSelectedRows(),
												opType: '13', //BrowseOperation enum value in dataBrowse.x
												fileOrFolder: self.tableEntity.params.fileOrFolder
											};

											if (deleteParams.items.length === 0) {
												deleteParams.items = self.selectedItem.params.path;
												deleteParams.fileOrFolder = self.selectedItem.params.fileOrFolder;
												deleteParams.deleteFromBrowseTree = true;
												deleteParams.parentObjectGUID =
													self.selectedItem.userObject.advancedData.browseMetaData.indexing.parentObjectGUID;
												deleteParams.folderName = self.selectedItem.label;
											}

											self.options.onDelete(deleteParams).then(function(response) {
												//For folder delete, refresh both left pane and table
												if (angular.isDefined(self.tableEntity.params.keywords)) {
													self.onFacetsChanged();
												} else if (
													deleteParams.fileOrFolder === FileOrFolderType.FOLDER ||
													deleteParams.fileOrFolder === FileOrFolderType.MAILBOX
												) {
													self.reload();
												} else {
													self.performTableBrowse(self.tableEntity);
												}
											}, onError);
										}
									};

									/**
									 * Delete all the items matching the search criteria.
									 */
									self.deleteAllMatching = function() {
										if (angular.isFunction(self.options.onDelete)) {
											var deleteParams = {
												keywords: self.tableEntity.params.keywords,
												facets: self.tableEntity.params.facets,
												fileOrFolder: self.tableEntity.params.fileOrFolder,
												opType: '13', //BrowseOperation enum value in dataBrowse.x
												bulkOperation: true,
												path: self.tableEntity.params.path
											};

											self.options.onDelete(deleteParams).then(function(response) {
												self.performTableBrowse(self.tableEntity);
											}, onError);
										}
									};

									/**
									 * Show/Hide deleted items based on the option selected
									 */
									self.showDeletedItems = function(action) {
										self.options.isShowDeletedItems = action;
										self.options.requestInProgress = true;
										if (angular.isDefined(self.tableEntity.params.keywords)) {
											self.onFacetsChanged();
										} else {
											const selectedItemId = _.get(self.selectedItem, 'id');

											if (
												!_.isEqual(_.get(self.tree, 'id'), selectedItemId) &&
												!_.isEqual(_.get(self.selectedItem.userObject.flags, 'deleted'), true)
											) {
												self.doReload = true;
												self.performTreeBrowse(self.tree);
												self.performTreeBrowse(self.selectedItem).then(function(data) {
													self.selectedItem.children = data.browseData;
												});
												self.performTableBrowse(self.tableEntity);
											} else {
												self.doReload = true;
												self.selectedItem = self.tree;
												self.performTreeBrowse(self.tree).then(function(data) {
													self.selectedItem.children = data.browseData;
												});

												self.performTableBrowse(self.tree);
											}
										}
										self.options.requestInProgress = false;
									};

									/**
									 * Browses the selected item to rebuild the tree's data.
									 */
									self.browse = function(item) {
										self.performTableBrowse(item);
									};
								}
							})
						};
					}

					/**
					 * Called when there is an error retrieving the required data.
					 */
					var onError = function(errorResponse) {
						if (self.options.showTable) {
							self.tree = {};
							self.serverError = errorResponse || cvLoc('generic_error');
						} else {
							self.tree = {
								error: cvLoc('generic_error')
							};
						}
						self.dontSpin = true;

						if (angular.isFunction(self.options.onBrowseFailure)) {
							self.options.onBrowseFailure();
						}
					};

					self.updateTimeRangeLabel = function() {
						if (angular.isDefined(self.lastToDateString) && angular.isDefined(self.lastFromDateString)) {
							self.timeRangeLabel = cvLoc(
								'Showing_backups_range',
								formatDisplayDate(self.lastFromDateString),
								formatDisplayDate(self.lastToDateString)
							);
						} else if (angular.isDefined(self.lastToDateString)) {
							self.timeRangeLabel = cvLoc('Showing_backups_asof', formatDisplayDate(self.lastToDateString));
						} else {
							self.timeRangeLabel = cvLoc('Showing_latest_backups');
						}
					};

					/**
					 * This method extracts the file path from display path
					 *
					 * @param {string}
					 *            sample input path - \Ajay-HotaddVM1\C:\Windows\WinSxS\test.txt output -
					 *            C:\Windows\WinSxS\test.txt
					 */
					self.extractFilePath = function(path) {
						if (path) {
							var index = path.indexOf('\\', 1);
							return path.substring(index + 1);
						}
						return '';
					};

					self.loadEntity = function() {
						/*
						 * Make the API call to get the root entity that we will be browsing.
						 */
						self.options.getEntity().then(function(response) {
							if (angular.isFunction(self.options.onEntityRetrieval)) {
								self.options.onEntityRetrieval(response.data);
							}

							if (angular.isUndefined(self.options.forEntity)) {
								/*
								 * If no "forEntity" argument was given, then we will be starting with the
								 * root node we retrieved. We will make a table browse request, along with a
								 * tree browse request to populate the browse page.
								 */
								var promises = [];

								/*
								 * prevent default tree and table browse in case it is an onLoadSearch.
								 * Trigger a search directly on the searched term
								 */
								if (angular.isDefined(self.onLoadSearch)) {
									self.searchTerm = self.onLoadSearch;
									self.lastSearchTerm = undefined;
									self.tree = response.data;
									self.onSearch(self.searchTerm);
								} else {
									if (!self.options.useSingleCallToGetData) {
										promises.push(self.performTreeBrowse(response.data));
									}

									if (self.options.showTable) {
										promises.push(self.performTableBrowse(response.data));
									}

									$q.all(promises).then(function(responses) {
										self.tree = response.data;
										self.tree.children = responses[0] ? responses[0].browseData : [];
										self.tree.totalChildren = responses[0] ? responses[0].totalItems : 0;
										self.tree.label = self.tree.subclientName;
										self.tree.selected = self.tree.expanded = true;
										self.selectedItem = self.tree;

										if (angular.isDefined(self.options.rootLabel)) {
											if (angular.isString(self.options.rootLabel)) {
												self.tree.label = self.options.rootLabel;
											} else if (angular.isFunction(self.options.rootLabel)) {
												var labelName = self.options.rootLabel(self.tree);
												if (angular.isString(labelName)) {
													self.tree.label = labelName;
												}
											}
										}
										self.updateTimeRangeLabel();
										if (self.options.showTable && !self.options.useSingleCallToGetData) {
											self.data = responses[1] ? responses[1] : [];
										} else {
											self.dontSpin = true;
										}
									}, onError);
								}
							} else {
								/*
								 * A "forEntity" argument was given to the directive, so instead of browsing
								 * the root entity, we will be browsing the "forEntity" object, and will limit
								 * all browsing to the contents of that entity.
								 */
								var promises = [];
								if (!self.options.useSingleCallToGetData) {
									promises.push(self.performTreeBrowse(self.options.forEntity));
								}
								if (self.options.showTable) {
									promises.push(self.performTableBrowse(self.options.forEntity));
								}

								$q.all(promises).then(function(responses) {
									self.tree = self.options.forEntity;
									self.tree.children = responses[0] ? responses[0].browseData : [];
									self.tree.totalChildren = responses[0] ? responses[0].totalItems : 0;
									self.tree.selected = self.tree.expanded = true;
									self.selectedItem = self.tree;
									self.updateTimeRangeLabel();
									if (self.options.showTable) {
										if (!self.options.useSingleCallToGetData) {
											self.data = responses[1] ? responses[1] : [];
										} else {
											self.data = responses[0] ? responses[0] : [];
											if (self.tree.totalChildren > 0) {
												self.tree.selected = false;
												self.selectedItem = self.tree.children[0];
												self.tree.children[0].selected = true;
											}
										}
									}

									if (self.epicServerState) {
										//format label on tree
										self.tree.label = backupAndRestoreFactory.formatContent(self.tree.label);
										angular.forEach(self.tree.children, function(dataItem) {
											dataItem.label = backupAndRestoreFactory.formatContent(dataItem.label);
										});
									}
								}, onError);
							}
						}, onError);
					};

					self.loadEntity();

					self.reload = function() {
						self.doReload = true;
						self.loadEntity();
					};

					//Hide left tree irrespective of which agent is when search triggered from global search page
					if ($stateParams.globalSearch == 'true') {
						self.collapseTree = true;
					}
				}
			]
		};
	}
]);

/**
 * When applied to an element, the element will sync its height to match up with the browse table so that it
 * naturally grows and collapses with it as the user request to see more results per page.
 */
mod.directive('cvBrowseSyncHeight', [
	function() {
		return {
			require: '^^cvBrowse',
			restrict: 'A',
			link: function($scope, element, attr, cvBrowse) {
				$scope.$watch(
					function() {
						return cvBrowse.lastTablePageSize + cvBrowse.visibleItems;
					},
					function() {
						$scope.$watch(
							function() {
								return cvBrowse.lastTablePageSize + cvBrowse.visibleItems;
							},
							function() {
								/**
								 * When pagination is changed the visible items might be less that pagesize,
								 * (fixes scroll issue) So assining visibleitems as noOfRows if they are less
								 * than lastTablePageSize
								 */
								var noOfRows =
									cvBrowse.visibleItems < cvBrowse.lastTablePageSize
										? cvBrowse.visibleItems
										: cvBrowse.lastTablePageSize;
								// The 31 is the height of the table headers
								// The 36 is the height of the row that holds the breadcrumbs
								element.height(noOfRows * ROW_HEIGHT + 36 + 31);
							}
						);
					}
				);
			}
		};
	}
]);

/**
 * A directive used for creating and displaying the browse tree for searching through files.
 *
 * @param {Object}
 *            tree - The data used to build the tree.
 * @param {string}
 *            tree.id - A unique identifier for the tree object.
 * @param {string}
 *            tree.label - The display name for the item in the tree.
 * @param {boolean=}
 *            tree.expanded - Whether or not the item should be expanded in the tree.
 * @param {boolean=}
 *            tree.selected - Whether or not the item should be selected on the tree.
 * @param {Object[]=}
 *            tree.children - An array of other trees that are children to the instance.
 */
mod.directive('cvBrowseTree', [
	'$compile',
	'cvLoc',
	function($compile, cvLoc) {
		return {
			require: '^^cvBrowse',
			restrict: 'E',
			scope: {
				tree: '='
			},
			templateUrl: appUtil.appRoot + 'modules/ida/partials/browseTree.jsp',
			link: function($scope, element, attr, cvBrowse) {
				/**
				 * Returns whether or not the directive's tree is empty.
				 *
				 * @return {boolean}
				 */
				$scope.isTreeEmpty = function() {
					return angular.equals($scope.tree, {});
				};

				/*
				 * Listen for the tree being given to the directive.
				 */
				var deregWatch = $scope.$watch('tree.children', function(children) {
					if (angular.isArray(children) && children.length > 0) {
						deregWatch();

						$compile(
							'<div class="children" data-uib-collapse="!tree.expanded">\
							<cv-browse-tree data-tree="item" data-ng-repeat="item in tree.children"></cv-browse-tree>\
							<a data-ng-click="onMore(tree)" data-ng-if="tree.totalChildren > tree.children.length">{{\'label.showMore\' | cvLoc}}</a>\
					</div>'
						)($scope, function(cloned, scope) {
							element.find('.browse-item:first-child').append(cloned);
						});
					}
				});

				/**
				 * Performs a browse to be ran on the tree of the component when the user clicks the More
				 * button.
				 */
				$scope.onMore = function() {
					cvBrowse.performTreeBrowse($scope.tree).then(function(data) {
						$scope.tree.children = $scope.tree.children.concat(data.browseData);
					});
				};

				/**
				 * Expands the tree after clicking the expand arrow. If the tree is already expanded, then the
				 * tree will be collapsed.
				 */
				$scope.expandTree = function() {
					if (!angular.isFunction(cvBrowse.performTableBrowse)) {
						if (!angular.isArray($scope.tree.children)) {
							$scope.tree.browsePerformed = true;
							cvBrowse.performTreeBrowse($scope.tree).then(function(data) {
								$scope.tree.children = data.browseData;
								$scope.tree.totalChildren = data.totalItems;
								$scope.tree.expanded = true;
							});
						} else {
							$scope.tree.expanded = !$scope.tree.expanded;
						}
					} else {
						$scope.onSelect();
					}
				};

				/**
				 * Performs a browse to be ran on the selection of an item. If the item already has children,
				 * then we don't need to perform a browse any more, but we still need to set it as the
				 * selected item, and expand its contents.
				 */
				$scope.onSelect = function() {
					if (cvBrowse.options.useSingleCallToGetData && cvBrowse.options.showTable) {
						if (cvBrowse.selectedItem !== $scope.tree) {
							var tables = [];
							var tableData;
							if (angular.isDefined($scope.tree.userObject)) {
								tableData = $scope.tree.userObject;
							} else if (angular.isDefined($scope.tree.children)) {
								tableData = $scope.tree.children;
							} //which means click the entityNode

							angular.forEach(tableData, function(tableBrowseItem) {
								tables.push(tableBrowseItem);
								tableBrowseItem.parent = $scope.tree;
							});
							$scope.tree.browsePerformed = true;

							var newBreadcrumbs = [];
							newBreadcrumbs.unshift($scope.tree);
							var parent = $scope.tree.parent;
							while (angular.isDefined(parent)) {
								newBreadcrumbs.unshift(parent);
								parent = parent.parent;
							}

							cvBrowse.breadcrumbs = newBreadcrumbs;

							cvBrowse.tableEntity.data = tables;

							var pageSize = tables.length;
							if (
								cvBrowse.options.forEntity &&
								cvBrowse.options.forEntity.pageSize &&
								cvBrowse.options.forEntity.pageSize < tables.length
							) {
								pageSize = cvBrowse.options.forEntity.pageSize;
							}
							cvBrowse.gridOptions.gridOptions.data = tables.slice(0, pageSize);
							cvBrowse.gridOptions.gridOptions.totalItems = tables.length;

							cvBrowse.gridOptions.gridOptions.paginationCurrentPage = 1;

							cvBrowse.makeSelectedItem($scope.tree);
							if (angular.isFunction(cvBrowse.options.onBeforeBrowseTableBuilt)) {
								cvBrowse.options.onBeforeBrowseTableBuilt({
									tableEntity: cvBrowse.tableEntity,
									gridOptions: cvBrowse.gridOptions,
									forSearch: angular.isString(cvBrowse.lastSearchTerm) && cvBrowse.lastSearchTerm.length > 0
								});
							}
						}
						return;
					}

					if (angular.isFunction(cvBrowse.options.onTreeItemSelected) && cvBrowse.selectedItem !== $scope.tree) {
						cvBrowse.options.onTreeItemSelected($scope.tree);
					}

					if (!angular.isArray($scope.tree.children)) {
						$scope.tree.browsePerformed = true;
						cvBrowse.performTreeBrowse($scope.tree).then(function(data) {
							$scope.tree.children = data.browseData;
							$scope.tree.totalChildren = data.totalItems;
							cvBrowse.makeSelectedItem($scope.tree);
						});
						if (angular.isFunction(cvBrowse.performTableBrowse)) {
							cvBrowse.performTableBrowse($scope.tree);
						}
					} else {
						if (angular.isFunction(cvBrowse.performTableBrowse) && cvBrowse.selectedItem !== $scope.tree) {
							cvBrowse.performTableBrowse($scope.tree).then(function() {
								cvBrowse.makeSelectedItem($scope.tree);
							});
						} else if (cvBrowse.selectedItem !== $scope.tree) {
							cvBrowse.selectedItem.selected = false;
							cvBrowse.selectedItem = $scope.tree;
							cvBrowse.selectedItem.selected = true;
							cvBrowse.selectedItem.expanded = true;
						} else {
							cvBrowse.selectedItem.expanded = !cvBrowse.selectedItem.expanded;
						}
					}
				};
			}
		};
	}
]);

/**
 * Note:- Two directives communication should be through require A directive which holds browse actions.
 * Access all cvBrowse methods and properties using `cvBrowseController.<method()/property>`
 */
mod.directive('cvBrowseActions', [
	function() {
		return {
			restrict: 'E',
			transclude: true,
			template: '<span class="browse-actions" ng-transclude></span>',
			require: '^cvBrowse',
			link: function($scope, $el, attrs, ctrl) {
				$scope.cvBrowseController = ctrl;
			}
		};
	}
]);

export default mod;
