import 'capps/js/cappsUtil.js';
import 'common/js/newGrid/cv-kendo-grid.js';
import 'adminConsole/js/services/instances.svc.js';
import 'modules/servers/js/services/servers.svc.js';
import 'adminConsole/js/directives/cv-isteven-single-select.js';
import 'capps/js/services/cappsClients.svc.js';
import 'capps/js/services/cloudInstanceTypes.factory.js';
import 'adminConsole/js/services/agents.svc.js';

import { commonAllAgentsModule } from 'common/js/modules';

export class sapForOracleRestoreOptionsController {
	constructor(
		cvLoc,
		cvUtil,
		AppTypes,
		idaService,
		instanceService,
		$filter,
		$uibModal,
		$uibModalInstance,
		data,
		serverService,
		agentService,
		$dialogs,
		dbsUtil,
		cvToaster
	) {
		this.cvLoc = cvLoc;
		this.cvUtil = cvUtil;
		this.AppTypes = AppTypes;
		this.idaService = idaService;
		this.instanceService = instanceService;
		this.$filter = $filter;
		this.$uibModal = $uibModal;
		this.$uibModalInstance = $uibModalInstance;
		this.serverService = serverService;
		this.agentService = agentService;
		this.$dialogs = $dialogs;
		this.dbsUtil = dbsUtil;
		this.cvToaster = cvToaster;
		this.data = data;

		this.defaultTab = 'owner';
		this._init();
	}

	_init() {
		const self = this;
		this.serverLoc = this.cvUtil.getIStevenLocLabels();
		this.serverLoc.nothingSelected = this.cvLoc('placeholder.selectDestinationServer');
		this.instanceLoc = this.cvUtil.getIStevenLocLabels();
		this.instanceLoc.nothingSelected = this.cvLoc('option.destInstance');

		this.tabControls = {};
		this.pitTimeFormat = 'yyyy-MM-dd HH:mm:ss';

		this.entity = this.data.entity;
		this.isFullDataRestore = this.data.isFullDataRestore;

		this.model = {
			streams: 2,
			database: true,
			controlFile: true,
			archiveLog: true,
			useSnapRevert: false,
			destinationServer: {},
			destinationInstance: {},
			pitType: 'CURRENT_TIME',
			pitDate: new Date()
		};

		if (this.data.toTime) {
			this.model.pitType = 'PIT';
			this.model.pitDate = new Date(this.data.toTime * 1000);
		}

		this.savedDate = new Date(this.model.pitDate);

		this.destinationServerList = [];
		this.destinationInstanceList = [];
		this.instanceCache = {};

		this.onTabChangedBind = this.onTabChanged.bind(this);
		this.loadDestinationServerList();

		this.onTabChanged();
	}

	onTabChanged() {
		if (!this.tabControls.getCurrentTabName) {
			this.isInplace = false;
			return;
		} else {
			this.isInplace = this.tabControls.getCurrentTabName() === 'IN_PLACE';
			this.model.destinationServer = this.isInplace ? angular.copy(this.data.entity) : undefined;
			this.model.destinationInstance = this.isInplace ? angular.copy(this.data.entity) : undefined;

			this.model.database = true;
			this.model.controlFile = true;
			this.model.archiveLog = true;
			this._clearServerSelection();
			this._clearInstanceSelection();
		}
	}

	destinationServerChanged() {
		this.model.destinationServer = angular.copy(this._selectedDestinationServers[0]);
		this.loadDestinationInstanceList();
	}

	destinationInstanceChanged() {
		this.model.destinationInstance = angular.copy(this._selectedDestinationInstances[0]);
	}
	_clearInstanceSelection() {
		this.destinationInstanceList.forEach(instance => {
			instance.selected = false;
		});
	}

	_clearServerSelection() {
		this.destinationServerList.forEach(server => {
			server.selected = false;
		});
	}

	loadDestinationServerList() {
		const self = this;
		this.idaService
			.getRestoreDestinations(
				this.entity.clientId,
				this.entity.applicationId,
				this.entity.instanceName,
				this.entity.instanceId
			)
			.success(function(data) {
				data.sort((a, b) => a.clientName.localeCompare(b.clientName));
				self.destinationServerList = data;
			})
			.error(function(err) {
				self.serverMessage = self.cvUtil.errMsg(err);
			});
	}

	loadDestinationInstanceList() {
		const self = this;

		const cacheId = _.get(this, 'model.destinationServer.clientId');
		if (this.instanceCache[cacheId]) {
			self.destinationInstanceList = this.instanceCache[cacheId];
			self._clearInstanceSelection();
		} else {
			this.destinationInstanceList = [];
			if (_.get(this, 'model.destinationServer.clientId')) {
				this.idaService
					.getInstances(this.model.destinationServer.clientId, this.entity.applicationId)
					.success(function(data) {
						data.sort((a, b) => a.instance.instanceName.localeCompare(b.instance.instanceName));
						self.destinationInstanceList = data
							.filter(info => info.instance.instanceId !== self.entity.instanceId)
							.map(info => info.instance);
						self.instanceCache[cacheId] = self.destinationInstanceList;
						self._clearInstanceSelection();
					})
					.error(function(err) {
						self.serverMessage = self.cvUtil.errMsg(err);
					});
			}
		}
	}

	pitTypeChanged() {
		if (this.model.pitType === 'CURRENT_TIME') {
			this.model.pitDate = new Date(this.savedDate);
			this.pitError = false;
		}
	}

	cancel() {
		this.$uibModalInstance.dismiss();
	}

	close() {
		this.$uibModalInstance.close();
	}

	_confirmAndRestore(restoreParam, yesFunction) {
		const destination = angular.fromJson(restoreParam.destination);
		const destClient = destination.destClient.clientName;
		const destInstance = destination.destinationInstance.instanceName;
		const message = this.isInplace
			? this.cvLoc('label.confirmOracleRestoreInPlace', destClient, destInstance)
			: this.cvLoc(
					'label.confirmOracleRestoreOutOfPlace',
					this.entity.clientName,
					this.entity.instanceName,
					destClient,
					destInstance,
					destInstance
			  );

		this.$dialogs.confirm(this.cvLoc('label.confirm'), message, {
			yesFunction: yesFunction,
			noFunction: function() {}
		});
	}
	submit() {
		if (this.model.pitType === 'PIT' && (!this.model.pitDate || this.model.pitDate.toString() === 'Invalid Date')) {
			console.log('invalid inputs');
			this.pitError = true;
			return;
		}
		this.pitError = false;

		const restoreParam = {
			entityType: this.data.entityType,
			genericEntity: angular.toJson(this.data.entity),
			browseOption: angular.toJson(this._composeBrowseOptions()),
			destination: angular.toJson(this._composeDestination()),
			sourceItemsToRestore: angular.toJson(this._composeSourceItems()),
			agentSpecificOptions: angular.toJson(this._composeAgentSpecificOptions()),
			commonOptions: angular.toJson(this._composeCommonOptions())
		};

		const self = this;
		this._confirmAndRestore(restoreParam, function() {
			self.idaService
				.submitRestoreJob(restoreParam)
				.success(function(data) {
					self.dbsUtil.showRestoreToaster(data);
					self.close();
				})
				.error(function(err) {
					self.cvToaster.showErrorMessage({
						message: err
					});
				});
		});
	}

	_composeBrowseOptions() {
		const browseOption = {
			commCellId: 2,
			timeRange: {
				fromTime: this.data.fromTime ? this.data.fromTime : 0,
				toTime: this.data.toTime ? this.data.toTime : 0
			}
		};
		if (this.data.copyPrecedence) {
			angular.extend(browseOption, {
				mediaOption: {
					copyPrecedence: {
						copyPrecedenceApplicable: true,
						copyPrecedence: this.data.copyPrecedence
					}
				}
			});
		}
		return browseOption;
	}

	_composeDestination() {
		const destination = {
			destClient: this.model.destinationServer,
			destinationInstance: this.model.destinationInstance
		};
		return destination;
	}

	_composeSourceItems() {
		const sourceItems = [];
		this.data.selectedRows.forEach(function(r) {
			if (r.fileType === 'DATAFILE') {
				//To have same format as Java GUI.
				sourceItems.push(`${r.path}:0`);
			} else {
				sourceItems.push(r.path);
			}
		});
		return sourceItems;
	}

	_composeAgentSpecificOptions() {
		const oracleOpt = {};

		const oraSourcePath = [];
		this.data.selectedRows.forEach(function(r) {
			//To have same format as Java GUI.
			if (r.fileType === 'DATAFILE') {
				oraSourcePath.push(`${r.parent.id}${r.path[0]}${r.id}`);
			} else if (r.fileType === 'TABLESPACE') {
				oraSourcePath.push(r.id);
			}
		});
		oracleOpt.sourcePaths = oraSourcePath;

		if (!this.isInplace) {
			oracleOpt.databaseCopy = true;
		}
		if (!this.isFullDataRestore) {
			oracleOpt.partialRestore = true;
		}
		oracleOpt.restoreStream = this.model.streams;
		oracleOpt.restoreData = this.model.database;
		oracleOpt.restoreControlFile = this.model.controlFile;
		oracleOpt.archiveLog = this.model.archiveLog;
		oracleOpt.archiveLogBy = 0;
		if (this.model.pitType === 'PIT') {
			oracleOpt.recoverFrom = 1;
			const unixTimeStamp = Math.floor(this.model.pitDate.getTime() / 1000);
			oracleOpt.recoverTime = {
				time: unixTimeStamp
			};

			oracleOpt.restoreFrom = 1;
			oracleOpt.restoreTime = {
				time: unixTimeStamp
			};
		} else {
			oracleOpt.recoverFrom = 0;
			oracleOpt.restoreFrom = 0;
		}
		return oracleOpt;
	}

	_composeCommonOptions() {
		const commonOpts = {};
		commonOpts.revert = this.model.useSnapRevert;
		return commonOpts;
	}
}

sapForOracleRestoreOptionsController.$inject = [
	'cvLoc',
	'cvUtil',
	'AppTypes',
	'idaService',
	'instanceService',
	'$filter',
	'$uibModal',
	'$uibModalInstance',
	'data',
	'serverService',
	'agentService',
	'$dialogs',
	'dbsUtil',
	'cvToaster'
];

commonAllAgentsModule.controller('sapForOracleRestoreOptions', sapForOracleRestoreOptionsController);
export default commonAllAgentsModule;
