import { disasterRecoveryModule } from 'common/js/modules';

// Service imports
import 'adminConsole/js/services/clients.svc.js';
import 'adminConsole/js/services/agents.svc.js';
import 'adminConsole/js/services/backupSets.svc.js';
import 'modules/ida/js/services/disasterRecovery.svc.js';

export class IbmiDisasterRecoveryController {
	constructor(
		$log,
		$uibModalInstance,
		sourceClient,
		clientService,
		agentService,
		jobService,
		ibmiDisasterRecoveryService,
		AppTypes,
		cvLoc,
		cvUtil,
		cvToaster
	) {
		this.log = $log;
		this.cvLoc = cvLoc;
		this.cvToaster = cvToaster;
		this.localLang = cvUtil.getIStevenLocLabels();
		this.convertUTCDateToLocalDate = cvUtil.convertUTCDateToLocalDate;
		this.uibModalInstance = $uibModalInstance;
		this.ibmiDisasterRecoveryService = ibmiDisasterRecoveryService;

		this._initialize();
		this.storeDVDImagePath = this.storeDVDImagePath.bind(this);
		this.ibmiDisasterRecoveryService.sourceClient = sourceClient;

		clientService
			.getClientsByAppId(AppTypes.PROXY_CLIENT_FILE_SYSTEM)
			.success(this._fetchIbmiClients.bind(this))
			.error(this._onError.bind(this));

		agentService
			.getBackupSets(sourceClient.clientId, AppTypes.PROXY_CLIENT_FILE_SYSTEM)
			.success(this._fetchBackupsets.bind(this))
			.error(this._onError);

		clientService
			.getFSClients()
			.success(this._fetchAllClients.bind(this))
			.error(this._onError);

		jobService.getCREjobs(
			{
				clientId: sourceClient.clientId,
				opType: 'sysrecoverybackup',
				sortOptions: {
					directions: [0],
					fields: ['jobId']
				},
				pagingInfo: {
					pageSize: 10000,
					currentPage: 1
				},
				completedJobLookupTime: 7776000
			},
			this._onSuccessfulJobRetrieval.bind(this),
			this._onError
		);
	}

	_initialize() {
		this.destinationClientsInputModel = [];
		this.backupSetsInputModel = [];
		this.fsClientsInputModal = [];
		this.jobsInputModal = [];

		this.destinationClientOutputModel = [];
		this.backupSetOutputModel = [];
		this.fsClientOutputModal = [];
		this.jobOutputModal = [];

		this.enableOutputLog = false;
		this.noJobFound = false;
		this.splfdta = false;
		this.logOutput = '';
		this.logPath = '';
		this.copyPrecedence = 0;
	}

	_fetchIbmiClients(clientList) {
		let currentClientId = _.get(this.ibmiDisasterRecoveryService, 'workFlowInputs.sourceClient.clientId', -1);

		//flattens the object to a single level
		this.destinationClientsInputModel = clientList.map(({ client: { clientEntity: client } }) => {
			return {
				displayName: client.displayName,
				clientName: client.clientName,
				clientId: client.clientId,
				hostName: client.hostName,
				// Select the current client by default
				selected: currentClientId === client.clientId
			};
		});
	}

	_fetchBackupsets(backupSetList) {
		//flattens the object to a single level
		this.backupSetsInputModel = backupSetList.map(({ backupSetEntity: backupSet }) => {
			return {
				backupSetName: backupSet.backupsetName,
				backupsetId: backupSet.backupsetId
			};
		});

		//make default selection
		this.backupSetsInputModel[0].selected = true;
	}

	_fetchAllClients(clientList) {
		//flattens the object to a single level
		this.fsClientsInputModal = clientList.map(({ client: { clientEntity: client } }) => {
			return {
				displayName: client.displayName,
				clientName: client.clientName,
				clientId: client.clientId,
				hostName: client.hostName
			};
		});
	}

	_onSuccessfulJobRetrieval(jobList) {
		//flattens the object to a single level
		this.jobsInputModal = jobList.jobs
			.filter(({ jobSummary: { status } }) => {
				return status === 'Completed';
			})
			.map(({ jobSummary }) => {
				let startDate = new Date(parseInt(jobSummary.jobStartTime) * 1000);
				let startDateString = this.convertUTCDateToLocalDate(startDate);
				let endDate = new Date(parseInt(jobSummary.jobEndTime) * 1000);
				let endDateString = this.convertUTCDateToLocalDate(endDate);
				let jobDescription = `
					<div class="isteven-html-content" 
							title="${_.escape(this.cvLoc('label.jobId', jobSummary.jobId))}">
						<div>${jobSummary.jobId}</div>
						<div class="description">
							<label>${_.escape(this.cvLoc('label.jobSummary', jobSummary.backupLevelName, startDateString, endDateString))}</label>
						</div>
					</div>`;
				return {
					jobId: jobSummary.jobId,
					status: jobSummary.status,
					jobDescription
				};
			});

		if (this.jobsInputModal.length) {
			this.jobsInputModal[0].selected = true;
		} else {
			this.noJobFound = true;
		}
	}

	_successfulSubmissionOfJob(response) {
		if (_.has(response, 'jobId')) {
			this.cvToaster.showSuccessMessage({
				ttl: '15000',
				message: `<a target="_blank" href="#/jobs/${response.jobId}">${this.cvLoc('label.redirectMsg')} ${
					response.jobId
				}</a>`
			});
			this.close();
		} else {
			this._onError(response);
		}
	}

	_onError(error) {
		this.serverMessage = {
			message: `${error.errorMessage.charAt(0).toUpperCase()}${error.errorMessage.slice(1)}`,
			type: 'error'
		};
	}

	storeDVDImagePath(result) {
		this.logPath = result.path;
	}

	submit() {
		this.ibmiDisasterRecoveryService.destinationClient = this.destinationClientOutputModel[0];
		this.ibmiDisasterRecoveryService.dvdTargetClient = this.fsClientOutputModal[0];
		this.ibmiDisasterRecoveryService.backupSetName = this.backupSetOutputModel[0].backupSetName;
		this.ibmiDisasterRecoveryService.oneTouchBackupJob = this.jobOutputModal[0].jobId;
		this.ibmiDisasterRecoveryService.copyPrecedence = this.copyPrecedence;
		this.ibmiDisasterRecoveryService.splfdta = this.splfdta;
		this.ibmiDisasterRecoveryService.dvdImagePath = this.logPath;
		if (this.logOutput && this.logOutput.length > 0) {
			this.ibmiDisasterRecoveryService.logOutput = this.logOutput;
		}
		this.ibmiDisasterRecoveryService
			.startDisasterRecovery()
			.success(this._successfulSubmissionOfJob.bind(this))
			.error(this._onError.bind(this));
	}

	cancel() {
		this.uibModalInstance.dismiss();
	}

	close() {
		this.uibModalInstance.close();
	}
}
IbmiDisasterRecoveryController.$inject = [
	'$log',
	'$uibModalInstance',
	'sourceClient',
	'clientService',
	'agentService',
	'jobService',
	'ibmiDisasterRecoveryService',
	'AppTypes',
	'cvLoc',
	'cvUtil',
	'cvToaster'
];
disasterRecoveryModule.controller('ibmiDisasterRecoveryCtrl', IbmiDisasterRecoveryController);
