import 'modules/ida/js/directives/cv-browse-mails-search.js';
import 'modules/ida/js/directives/cv-browse-directives.js';
import 'modules/ida/js/services/idaService.svc.js';
import 'ediscovery/common/js/services/ediscovery.svc.js';
import 'exchange/js/controllers/exchMailboxRestoreOptions.ctrl.js';
import 'exchangeV2/js/services/exchange.svc.js';
import { exchangeFactoryV2 } from 'exchangeV2/js/factory/exchange.factory.js';
import { office365AppsFactory } from 'office365/js/factory/office365Apps.factory.js';
import 'ediscovery/common/js/controllers/createExportSet.ctrl.js';
import { exchangeConstants } from 'exchangeV2/js/constants/exchangeConstants.js';
import { ExchangeTabsClass, ExchangeO365TabsClass } from 'exchangeV2/js/baseClasses/ExchangeTabsClass.js';
import { commonAllAgentsModule } from 'common/js/modules';
import { DownloadParams } from 'ediscovery/common/js/directives/models/downloadParams.js';
import { CVSearchRequest, SearchParams } from 'cvSearchRequest.js';

/**
 * A controller for handling the functionality on the exchange browse page.
 */
commonAllAgentsModule.controller('exchangeBrowseController', [
	'$q',
	'$uibModal',
	'idaService',
	'$stateParams',
	'cvLoc',
	'cvUtil',
	'cvBreadcrumbsTabsFactory',
	'uiGridConstants',
	'$state',
	'$dialogs',
	'tabService',
	'exchangeServiceV2',
	'AppTypes',
	'cvToaster',
	'ediscoveryService',
	'$scope',
	'$injector',
	'$filter',
	function(
		$q,
		$uibModal,
		idaService,
		$stateParams,
		cvLoc,
		cvUtil,
		cvBreadcrumbsTabsFactory,
		uiGridConstants,
		$state,
		$dialogs,
		tabService,
		exchangeService,
		AppTypes,
		cvToaster,
		ediscoveryService,
		$scope,
		$injector,
		$filter
	) {
		var self = this;
		var subclient = null;
		self.cvUtil = cvUtil;
		self.exchangeFactory = new exchangeFactoryV2($uibModal, exchangeConstants);
		self.office365AppsFactory = new office365AppsFactory($uibModal, exchangeConstants);
		self.exchangeConstants = exchangeConstants;
		self.entityType = 'SUBCLIENT_ENTITY';
		self.entityId = $stateParams.subclientId;
		self.clientGUID = undefined;
		self.isMultiMailboxRestore = _.get($stateParams, 'isMultiMailboxRestore', false) === 'true' ? true : false;
		self.mailboxesToBrowse = _.get($stateParams, 'mailboxes', '');
		self.mailboxType = '';

		let cloudId;

		self.mailSortFields = [
			{
				key: 'ModifiedDate',
				label: cvLoc('label.date')
			},
			{
				key: 'From',
				label: cvLoc('label.from')
			},
			{
				key: 'MailSize',
				label: cvLoc('label.size')
			},
			{
				key: 'Subject',
				label: cvLoc('label.subject')
			}
		];

		self.browseOptions = {
			getEntity: idaService.getNodeEntity.bind(undefined, self.entityType, self.entityId),
			getBrowseData: idaService.getIdaBrowseData.bind(undefined, self.entityId, self.entityType),
			/*
			 * Gets the permission list
			 */
			getPermissions: idaService.getPermissions.bind(undefined, self.entityType, self.entityId),
			/*
			 * Shows the button for deleting the files/folders
			 */
			showDelete: false,
			disableActions: true,

			isShowDeletedItems: false,
			requestInProgress: false,
			/*
			 * Shows the button for restoring all the files matching the search criteria
			 */
			showRestoreAllMatching: false,
			/*
			 * Shows the button for deleting all the files matching the search criteria
			 */
			showDeleteAllMatching: false,

			isUserMailbox: false,

			/*
			 * Clear facet info on new search
			 */
			clearFacetInfo: true,

			/*
			 * Hides the common "Restore" button
			 */
			useExternalRestore: true,
			/*
			 * Show the "Restore" options applicable
			 */
			showRestoreActions: false,
			/*
			 * Shows the button for collapsing the left tree.
			 */
			showCollapseTreeButton: true,

			/*
			 * Filtering by the date of backups is not supported for the Exchange One Pass app type.
			 */
			showDateFilter: false,
			/*
			 * Show the "Export" options to create/view export set
			 */

			//Need to default to false after adding permission check for exportset
			showExportActions: false,

			//To identify exchange browse (for fixing breadcrumb issue)
			isExchangeBrowse: true,

			/**
			 * Hook for customizing the root label's name.
			 *
			 * @param {Object}
			 *            root - The root entity retrieved for the browse page.
			 */
			rootLabel: function(root) {
				return root.backupsetName;
			},

			/**
			 * Called when the user selects the "New" option under "Export"
			 *
			 * @param {Object[]}
			 *            items - An array of items that the user wants to export.
			 */
			onDownload: function(downloadParams) {
				let exportSetName = '';

				const fromSubclient = angular.copy(subclient);
				let indexServerClientId = 0;
				exchangeService.getExchangeMailboxDetails(subclient.clientId).then(response => {
					if (!_.isNull(response)) {
						indexServerClientId = _.get(response, 'data.onePassProperties.indexServer.mediaAgentId', 0);
					}
					if (!_.isUndefined($stateParams.indexId)) {
						indexServerClientId = parseInt(_.get($stateParams, 'indexId', 0), 10);
					}
					let browseRequest = {
						opType: 20, //Need to change this later to constant
						entity: {
							subclientId: fromSubclient.subclientId,
							applicationId: fromSubclient.applicationId,
							instanceId: fromSubclient.instanceId,
							clientId: fromSubclient.clientId,
							clientName: fromSubclient.clientName,
							backupsetId: fromSubclient.backupsetId
						},
						options: {
							folderDelete: false,
							file: false,
							mailbox: false,
							showDeletedFiles: self.browseOptions.isShowDeletedItems,
							bulkOperation: _.get(downloadParams.bulkOperation),
							facets: _.get(downloadParams, 'facets'),
							keywords: _.get(downloadParams, 'keywords')
						},
						queries: [{}],
						paths: []
					};

					const folderRestore =
						downloadParams.fileOrFolder === 'FOLDER' || downloadParams.fileOrFolder === 'MAILBOX' ? true : false;

					/*
					 * Below condition is to handle selection of folders on left tree and "All" selection
					 * range on the create export modal at second level of browse (i.e. which has only
					 * messages on grid). This will allow the users to export only messages within a folder
					 * without its subfolders.
					 */
					if (_.get(downloadParams, 'folderExportWithoutSubFolders', false) == true) {
						let query = {
							queryId: 0,
							type: exchangeConstants.query.TYPE.DATA,
							whereClause: [
								{
									connector: exchangeConstants.query.CRITERIA.CONNECTOR_AND,
									criteria: {
										dataOperator: exchangeConstants.query.CRITERIA.DATAOPERATOR_IN,
										field: exchangeConstants.query.CRITERIA.BROWSEFIELD_PARENTOBJGUID,
										values: []
									}
								}
							]
						};
						query.whereClause[0].criteria.values.push(downloadParams.parentObjectGUID);
						browseRequest.queries[0] = query;
					} else {
						if (folderRestore || !_.isEmpty(downloadParams.keywords)) {
							browseRequest.paths.push({
								path: downloadParams.path
							});
						}

						if (_.isEmpty(downloadParams.keywords)) {
							angular.forEach(downloadParams.items, function(item) {
								browseRequest.paths.push({
									path: item.userObject.path
								});
							});
						}
					}

					downloadParams.mailboxType = subclient.backupsetName;
					self.openExportModal(downloadParams, indexServerClientId, browseRequest);
				});
			},

			/**
			 * Called when the user selects the "Restore" button above the browse table.
			 *
			 * @param {Object[]}
			 *            items - An array of items that the user wants to restore.
			 */
			onRestore: function(restoreParams) {
				var fromSubclient = angular.copy(subclient);
				delete fromSubclient.children;

				var selectedPaths = [];
				let selectedItems = [];
				let onlyExchangeEmailsSelected;

				//Setting the path of the mailbox GUID for restore all items matching the search criteria across all pages.
				if (angular.isDefined(restoreParams.restoreAllMatching) && restoreParams.restoreAllMatching === true) {
					// Replace all the un-encoded % characters with %25
					restoreParams.path = restoreParams.path.replace(/%(?![0-9a-fA-F]{2})/g, '%25');

					selectedPaths.push(restoreParams.path);
				} else if (_.get(restoreParams, 'folderRestore', false)) {
					// Replace all the un-encoded % characters with %25
					restoreParams.folderPath = restoreParams.folderPath.replace(/%(?![0-9a-fA-F]{2})/g, '%25');

					//Setting the path of the folder/mailbox being browsed on the left tree
					selectedPaths.push(restoreParams.folderPath);
				} else {
					onlyExchangeEmailsSelected = _.isEmpty(
						_.get(restoreParams, 'items', []).filter(
							item => !_.get(item, 'userObject.flags.file', false) && !_.get(item, 'userObject.flags.stub', false)
						)
					);

					angular.forEach(restoreParams.items, function(item) {
						// Replace all the un-encoded % characters with %25
						item.userObject.path = item.userObject.path.replace(/%(?![0-9a-fA-F]{2})/g, '%25');
						selectedPaths.push(item.userObject.path);
					});

					if (onlyExchangeEmailsSelected) {
						selectedItems = restoreParams.items.map(item => ({
							mailFrom: _.get(item, 'userObject.advancedData.browseMetaData.exchDataV2.from.name', ''),
							mailSubject: _.get(item, 'userObject.advancedData.browseMetaData.exchDataV2.subject', '')
						}));

						// set the from and subject as n/a if they are empty strings
						selectedItems = selectedItems.map(item => ({
							mailFrom: item.mailFrom === '' ? cvLoc('label.noApplicable') : item.mailFrom,
							mailSubject: item.mailSubject === '' ? cvLoc('label.noApplicable') : item.mailSubject
						}));
					} else {
						selectedItems = restoreParams.items.map(item => ({
							displayName: item.label,
							type: item.fileType
						}));
					}
				}
				if (
					$state.current.name === 'exchangeBrowseRecoveryPoint' ||
					$state.current.name === 'O365BrowseRecoveryPoint'
				) {
					restoreParams.recoveryPointID = $stateParams.rpId;
				}
				//Setting the include/exclude deleted items option based on user selection
				restoreParams.showDeletedItems = self.browseOptions.isShowDeletedItems;

				/* Old restore modal
				var modalInstance = $uibModal.open({
					templateUrl: appUtil.appRoot + 'exchange/partials/exchMailboxRestoreOptions.jsp',
					controller: 'exchMailboxRestoreOptionsController as exchMailboxRestoreOptionsController',
					backdrop: 'static',
					resolve: {
						subclient: fromSubclient,
						itemsToRestore: function() {
							return selectedPaths;
						},
						rstParams: function() {
							return restoreParams;
						}
					}
				});
				*/

				// New restore modal
				const exchangeRestoreParams = {
					...restoreParams,
					entity: fromSubclient,
					rawItemsToRestore: selectedItems,
					itemsToRestore: selectedPaths,
					entityType: self.entityType,
					isExchangeBrowseRestore: true,
					onlyExchangeEmailsSelected
				};
				self.office365AppsFactory.getOffice365RestoreOptions(exchangeRestoreParams).result.then(
					data => {
						//Restore submitted!
					},
					e => {
						cvToaster.showErrorMessage({
							ttl: '3000', //3 sec
							message: e.data
						});
					}
				);
			},

			/**
			 * Called when the admin user selects the "Delete" button above the browse table.
			 *
			 * @param {Object[]}
			 *            items - An array of items that the user wants to delete.
			 */
			onDelete: function(deleteParams) {
				var fromSubclient = angular.copy(subclient);
				const deferred = $q.defer();

				var selectedPaths = [];
				const isBulkOperation =
					angular.isDefined(deleteParams.bulkOperation) && deleteParams.bulkOperation === true ? true : false;
				let confirmationText = isBulkOperation
					? cvLoc('label.eraseAll.confirmText')
					: cvLoc('label.eraseMsg.confirmText');
				//Folder and sub folder delete from left pane
				if (angular.isDefined(deleteParams.deleteFromBrowseTree) && deleteParams.deleteFromBrowseTree) {
					confirmationText = cvLoc('label.eraseFolder.confirmText', `<b>${deleteParams.folderName}</b>`);
				}
				//Mailbox level delete
				if (
					angular.isDefined(deleteParams.parentObjectGUID) &&
					deleteParams.parentObjectGUID === '00000000000000000000000000000001'
				) {
					confirmationText = cvLoc('label.eraseFolder.confirmText', `<b>${deleteParams.folderName}</b>`);
				}

				var callBackFunctions = {
					noFunction: function() {},
					yesFunction: function() {
						let browseRequest = {
							opType: deleteParams.opType,
							entity: {
								subclientId: fromSubclient.subclientId,
								applicationId: fromSubclient.applicationId,
								instanceId: fromSubclient.instanceId,
								clientId: fromSubclient.clientId,
								clientName: fromSubclient.clientName,
								backupsetId: fromSubclient.backupsetId
							},
							options: {
								folderDelete: false,
								file: false,
								mailbox: false,
								showDeletedFiles: self.browseOptions.isShowDeletedItems,
								bulkOperation: angular.isDefined(deleteParams.bulkOperation) ? deleteParams.bulkOperation : null
							},
							queries: [{}],
							paths: []
						};

						//Setting the path of the mailbox GUID for bulk delete.
						if (isBulkOperation) {
							browseRequest.paths.push({
								path: deleteParams.path
							});
						} else {
							//Setting the path for the selected item on the left browse tree (Folder/Mailbox delete)
							if (angular.isDefined(deleteParams.deleteFromBrowseTree) && deleteParams.deleteFromBrowseTree) {
								browseRequest.paths.push({
									path: deleteParams.items
								});
								if (
									angular.isDefined(deleteParams.parentObjectGUID) &&
									deleteParams.parentObjectGUID === '00000000000000000000000000000001'
								) {
									browseRequest.options.mailbox = true;
								} else {
									browseRequest.options.folderDelete = true;
								}
							} else {
								//Setting the path of all selected items
								angular.forEach(deleteParams.items, function(item) {
									browseRequest.paths.push({
										path: item.userObject.path
									});

									browseRequest.options.folderDelete = _.get(item.userObject, 'flags.directory', false);
									browseRequest.options.file = _.get(item.userObject, 'flags.file', false);
									browseRequest.options.mailbox = _.get(item.userObject, 'flags.mailbox', false);
								});
							}

							if (browseRequest.options.mailbox == true) {
								//Setting itemType as Mailbox as in enum itemType of databrowse request
								browseRequest.queries[0].fileOrFolder = 3;
							}
						}

						return exchangeService
							.doAdminUserErase({
								browseRequest: angular.toJson(browseRequest),

								options: isBulkOperation ? angular.toJson(deleteParams) : null
							})
							.then(
								response => {
									const { data } = response;
									if (isBulkOperation) {
										if (angular.isDefined(data.browseResponses) && data.browseResponses.length > 0) {
											const jobId = data.browseResponses[0].transactionId;
											const jobMsg = `${cvLoc('notification.bulkdelete.started')}  ${jobId}`;

											const viewMsg = cvLoc('notification.jobDetails');
											cvToaster.showInfoMessage({
												ttl: '10000',
												message: `${jobMsg} <br><a href=#/jobs/${jobId}>${viewMsg}</a>`
											});
										} else {
											cvToaster.showErrorMessage({
												ttl: '10000',
												message: cvLoc('error.bulkdelete.jobError')
											});
										}
									}

									deferred.resolve(response);
								},
								e => {
									deferred.reject(cvLoc('label.eraseMsg.failureText'));
								}
							);
					}
				};
				$dialogs.confirm(cvLoc('label.confirmDelete'), confirmationText, callBackFunctions);
				return deferred.promise;
			},

			/**
			 * Called when the entity from the cv-browse directive is retrieved. We can use this hook to build
			 * our breadcrumbs.
			 *
			 * @param {Object}
			 *            The entity that was retrieved.
			 */
			onEntityRetrieval: function(entity) {
				self.recoveryPointBrowse = false;
				self.getClientGUID(entity.clientId).then(resp => {
					self.clientGUID = resp;
				});
				subclient = entity;
				self.clientName = entity.clientName;
				self.browseOptions.isUserMailbox = self.exchangeConstants.mailboxType.user === subclient.backupsetName;
				self.mailboxType = subclient.backupsetName;
				if (
					$state.current.name === 'exchangeBrowseRecoveryPoint' ||
					$state.current.name === 'O365BrowseRecoveryPoint'
				) {
					self.recoveryPointBrowse = true;
				}
				self.setApplicableActions().then(resp => {
					subclient.showViewExport = self.browseOptions.showExportActions;
					if ($state.current.name === 'exchangeBrowseClassic') {
						cvBreadcrumbsTabsFactory.addBreadCrumbs([
							{
								title: cvLoc('label.exchangeServers'),
								link: '#/exchange'
							},
							{
								title: entity.clientName,
								link: `#/exchangeDetailsClassic/${subclient.clientId}`
							},
							{
								title: entity.appName,
								link: `#/mailboxDetailsClassic/${subclient.clientId}`
							},
							{
								title: entity.backupsetName,
								link: `#/mailboxBackupSetDetailsClassic/${subclient.clientId}/${subclient.backupsetId}/${subclient.subclientId}/${subclient.backupsetName}`
							}
						]);
					} else if ($state.current.name === 'exchangeBrowse') {
						var breadCrumbs = [
							{
								title: cvLoc('label.exchange'),
								link: '#/exchange'
							}
						];
						//if (angular.isDefined($stateParams.forMailbox)) {
						breadCrumbs.push({
							title: entity.clientName,
							link: `#exchange/clientDetails/${subclient.clientName}/${subclient.clientId}/${subclient.subclientId}/users?backupsetName=${subclient.backupsetName}&backupsetId=${subclient.backupsetId}`
						});
						//}
						cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
					} else if ($state.current.name === '365exchangeBrowse') {
						cvBreadcrumbsTabsFactory.addBreadCrumbs([
							{
								title: cvLoc('label.nav.office365'),
								link: '#/office365Dashboard/true'
							},
							{
								title: cvLoc('label.exchangeServers'),
								link: '#/365exchange'
							},
							{
								title: entity.clientName,
								link: `#/365exchangeDetails/${subclient.clientId}`
							},
							{
								title: entity.appName,
								link: `#/365mailboxDetails/${subclient.clientId}`
							},
							{
								title: entity.backupsetName,
								link: `#/365mailboxBackupSetDetails/${subclient.clientId}/${subclient.backupsetId}/${subclient.subclientId}/${subclient.backupsetName}`
							}
						]);
					} else if ($state.current.name === 'O365exchangeBrowseV2') {
						var breadCrumbs = [
							{
								title: cvLoc('label.office365Servers'),
								link: '#/office365V2'
							}
						];
						breadCrumbs.push({
							title: entity.clientName,
							link: `#office365/clientDetails/${subclient.clientName}/${subclient.clientId}/${subclient.subclientId}/users?backupsetId=${subclient.backupsetId}`
						});

						cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
					} else if ($state.current.name === 'exchangeBrowseRecoveryPoint') {
						self.recoveryPointBrowse = true;
						self.fullFormattedRecoveryTime = $filter('date')(
							$stateParams.recoveryTime * 1000,
							exchangeConstants.recoveryPointsConstants.EXCH_MB_DATE_FORMAT
						);
						self.recoveryPointBrowseHelpText = cvLoc(
							'help.recoveryPointBrowse',
							$stateParams.name,
							self.fullFormattedRecoveryTime
						);
						var breadCrumbs = [
							{
								title: cvLoc('label.exchange'),
								link: '#/exchange'
							}
						];
						breadCrumbs.push({
							title: entity.clientName,
							link: `#/exchange/clientDetails/${subclient.clientName}/${subclient.clientId}/${subclient.subclientId}/recoveryPoints?backupsetName=${subclient.backupsetName}&backupsetId=${subclient.backupsetId}`
						});
						cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
					} else if ($state.current.name === 'O365BrowseRecoveryPoint') {
						self.recoveryPointBrowse = true;
						self.fullFormattedRecoveryTime = $filter('date')(
							$stateParams.recoveryTime * 1000,
							exchangeConstants.recoveryPointsConstants.EXCH_MB_DATE_FORMAT
						);
						self.recoveryPointBrowseHelpText = cvLoc(
							'help.recoveryPointBrowse',
							$stateParams.name,
							self.fullFormattedRecoveryTime
						);
						var breadCrumbs = [
							{
								title: cvLoc('label.office365Servers'),
								link: '#/office365V2'
							}
						];
						breadCrumbs.push({
							title: entity.clientName,
							link: `#office365/clientDetails/${subclient.clientName}/${subclient.clientId}/${subclient.subclientId}/recoveryPoints?backupsetId=${subclient.backupsetId}`
						});
						cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
					}
				});
			},

			/**
			 * Called before a request is sent to the Browse service for populating the left-hand tree in the
			 * browse directive.
			 *
			 * @param {Object}
			 *            item - An optional item that the user has clicked on and wants to browse.
			 * @return {Object} The parameters to be sent to the browse request.
			 */
			onBeforeBrowseTreeRequest: function(item) {
				var params = {
					applicationId: AppTypes.EXCHANGE_MAILBOX,
					subclientId: $stateParams.subclientId,
					entityType: self.entityType,
					//To show the soft deleted items from web console
					showDeletedFiles: self.browseOptions.isShowDeletedItems
				};

				/*
				 * If the item has a "userObject" property, then it's not the root node.
				 */
				if (angular.isObject(item.userObject)) {
					self.browseOptions.disableActions = false;
					let paramOptions = {
						path: item.userObject.path,
						criteria: 'ParentObjectGUID:' + item.id,
						sortingInfo: 'asc:FileName',
						fileOrFolder: 'FOLDER'
					};
					if (
						$state.current.name === 'exchangeBrowseRecoveryPoint' ||
						$state.current.name === 'O365BrowseRecoveryPoint'
					) {
						paramOptions.recoveryPoint = true;
						paramOptions.rpId = $stateParams.rpId;
					}
					return angular.extend(params, paramOptions);
				}

				if (
					$state.current.name === 'exchangeBrowseRecoveryPoint' ||
					$state.current.name === 'O365BrowseRecoveryPoint'
				) {
					return angular.extend(params, {
						path: '\\',
						sortingInfo: 'asc:MailboxName',
						fileOrFolder: 'MAILBOX',
						criteria: 'ParentObjectGUID:00000000000000000000000000000001',
						recoveryPoint: true,
						rpId: $stateParams.rpId
					});
				}
				return angular.extend(params, {
					path: '\\',
					sortingInfo: 'asc:MailboxName',
					fileOrFolder: 'MAILBOX',
					criteria: self.isMultiMailboxRestore
						? `MailboxName:${self.mailboxesToBrowse}`
						: `ParentObjectGUID:00000000000000000000000000000001`
				});
			},

			/**
			 * Called before a request is sent to the Browse service for populating the right-hand table in
			 * the browse directive.
			 *
			 * @param {Object}
			 *            item - An optional item that the user has clicked on and wants to browse.
			 * @param {boolean}
			 *            forSearch - Whether or not the request was for a search.
			 * @return {Object} The parameters to be sent to the browse request.
			 */
			onBeforeBrowseTableRequest: function(item, forSearch) {
				var params = {
					applicationId: AppTypes.EXCHANGE_MAILBOX,
					subclientId: $stateParams.subclientId,
					entityType: self.entityType,
					//To show the soft deleted items from web console
					showDeletedFiles: self.browseOptions.isShowDeletedItems
				};

				//Added to show "Restore all matching" button
				self.browseOptions.showRestoreAllMatching = angular.isDefined(forSearch) ? forSearch : false;

				//Added to show "Delete all matching" button
				self.browseOptions.showDeleteAllMatching = angular.isDefined(forSearch) ? forSearch : false;

				/*
				 * If the item has a "userObject" property, then it's not the root node.
				 */
				if (angular.isObject(item.userObject)) {
					self.browseOptions.disableActions = false;
					let paramOptions = {
						path: item.userObject.path,
						criteria: 'ParentObjectGUID:' + item.id
					};
					if (
						$state.current.name === 'exchangeBrowseRecoveryPoint' ||
						$state.current.name === 'O365BrowseRecoveryPoint'
					) {
						paramOptions.recoveryPoint = true;
						paramOptions.rpId = $stateParams.rpId;
					}
					angular.extend(params, paramOptions);

					if (
						item.userObject.advancedData.browseMetaData.indexing.parentObjectGUID === '00000000000000000000000000000001'
					) {
						self.browseOptions.disableActions = forSearch ? true : false;
						return angular.extend(params, {
							sortingInfo: forSearch ? 'desc:ModifiedDate' : 'asc:FileName',
							fileOrFolder: forSearch ? 'FILE' : 'FOLDER'
						});
					} else {
						return angular.extend(params, {
							sortingInfo: 'desc:ModifiedDate',
							fileOrFolder: 'FILE'
						});
					}
				}
				self.browseOptions.disableActions = true;
				if (
					$state.current.name === 'exchangeBrowseRecoveryPoint' ||
					$state.current.name === 'O365BrowseRecoveryPoint'
				) {
					return angular.extend(params, {
						path: '\\',
						sortingInfo: forSearch ? 'desc:ModifiedDate' : 'asc:MailboxName',
						fileOrFolder: forSearch ? 'FILE' : 'MAILBOX',
						criteria: 'ParentObjectGUID:00000000000000000000000000000001',
						recoveryPoint: true,
						rpId: $stateParams.rpId
					});
				}

				return angular.extend(params, {
					path: '\\',
					sortingInfo: forSearch ? 'desc:ModifiedDate' : 'asc:MailboxName',
					fileOrFolder: forSearch ? 'FILE' : 'MAILBOX',
					criteria: self.isMultiMailboxRestore
						? `MailboxName:${self.mailboxesToBrowse}`
						: `ParentObjectGUID:00000000000000000000000000000001`
				});
			},

			/**
			 * Called after the request sent to the Browse service for populating the right-hand table in the
			 * browse directive has completed.
			 *
			 * @param {Object}
			 *            item - The item that was selected by the user to browse.
			 * @param {Object[]}
			 *            data - The data that was retrieved from the request.
			 */
			onAfterBrowseTableRequest: function(item, data) {
				if (!angular.isObject(item.userObject)) {
					/*
					 * If the item being browsed does not have a userObject property, then it must be the root
					 * node, so this is the only circumstance where we want to alter the display data.
					 */
					angular.forEach(data, function(item) {
						var mailbox = item.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name;
						mailbox = mailbox.substring(mailbox.indexOf('(') + 1, mailbox.indexOf(')'));
						item.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name = mailbox;
					});
				}
			},

			/**
			 * Called when the browse table on the right hand side is going to be built.
			 *
			 * @param {Object}
			 *            params - The parameters for the function
			 * @param {Object}
			 *            params.tableEntity - The entity that was selected to browse for the table.
			 * @param {Object}
			 *            params.gridOptions - The grid options to manipulate if you see fit.
			 * @param {boolean}
			 *            params.forSearch - Whether or not the table will be used for a search response.
			 *
			 *
			 */
			onBeforeBrowseTableBuilt: function(params) {
				var item = params.tableEntity;
				var gridOptions = params.gridOptions;
				var forSearch = params.forSearch;

				if (forSearch || angular.isObject(item.userObject)) {
					if (
						!forSearch &&
						item.userObject.advancedData.browseMetaData.indexing.parentObjectGUID === '00000000000000000000000000000001'
					) {
						/*
						 * The item's parent is the root node. If this is the case, then the next set of data
						 * we will be showing is folders within the root.
						 */
						gridOptions.gridOptions.columnDefs = [
							{
								field: 'label',
								displayName: cvLoc('Name'),
								headerTooltip: true,
								cellTemplate:
									'<a data-ng-click="grid.appScope.performTableBrowse(row.entity)" class="crop" title="{{row.entity.label}} "><i data-ng-if="row.entity.userObject.flags.deleted === true" title="' +
									cvLoc('label.userDeletedItem') +
									'" class="ion-trash-b"></i>{{row.entity.label}}</a>',
								width: '*',
								sortParam: 'FileName',
								sort: {
									direction: uiGridConstants.ASC
								}
							}
						];
					} else {
						/*
						 * The item's parent is a folder. If this is the case, then the next set of data we
						 * will be showing is individual emails.
						 */
						gridOptions.gridOptions.columnDefs = [
							{
								name: cvLoc('label.hasAttachment'),
								displayName: ' ',
								width: '2%',
								enableHiding: false,
								enableFiltering: false,
								enableSorting: false,
								cellTemplate:
									'<span><i data-ng-if="row.entity.userObject.flags.deleted === true" title="' +
									cvLoc('label.userDeletedItem') +
									'" class="ion-trash-b"></i></span><span class="padding-left-5"><i data-ng-if="row.entity.userObject.advancedData.browseMetaData.exchDataV2.hasAttachment" title="' +
									cvLoc('label.hasAttachment') +
									'" class="ion-paperclip"></i></span>'
							},
							{
								field: 'mail',
								displayName: cvLoc('label.from'),
								enableSorting: true,
								sortParam: 'From',
								sort: {
									direction: uiGridConstants.ASC
								},
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="crop" data-ng-attr-title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.from.name}}">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.from.name}}</span>' +
									'</div>',
								width: '20%'
							},
							{
								field: 'mail',
								displayName: cvLoc('label.to'),
								enableSorting: false,
								sortParam: 'To',
								sort: {
									direction: uiGridConstants.ASC
								},
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="crop" data-ng-attr-title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.to.name}}">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.to.name}}</span>' +
									'</div>',
								width: '20%'
							},
							{
								field: 'mail',
								displayName: cvLoc('label.subject'),
								enableSorting: true,
								sortParam: 'Subject',
								sort: {
									direction: uiGridConstants.ASC
								},
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="crop" data-ng-attr-title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.subject}}"> <span class="ui-grid-cell-contents">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.subject}}</span></span>' +
									'</div>',
								width: '*'
							},
							{
								field: 'mail',
								displayName: cvLoc('label.date'),
								enableSorting: true,
								sortParam: 'ModifiedDate',
								sort: {
									direction: uiGridConstants.ASC
								},
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="crop" data-ng-attr-title="{{row.entity.userObject.modificationTime * 1000 | date : "medium"}}">{{row.entity.userObject.modificationTime * 1000 | date : "medium"}}</span>' +
									'</div>',
								width: '15%'
							},
							{
								field: 'mail',
								displayName: cvLoc('label.size'),
								enableSorting: true,
								sortParam: 'MailSize',
								sort: {
									direction: uiGridConstants.ASC
								},
								headerTooltip: false,
								cellTemplate:
									'<div class="gmailRow">' +
									'<span class="exchnageEmailSize crop" data-ng-attr-title="{{row.entity.userObject.size|capacity}}">{{row.entity.userObject.size|capacity}}</span>' +
									'</div>',
								width: '7%'
							}
						];
					}
				} else {
					gridOptions.gridOptions.columnDefs = [
						{
							field: 'label',
							displayName: cvLoc('Name'),
							headerTooltip: true,
							cellTemplate:
								'<a data-ng-click="grid.appScope.performTableBrowse(row.entity)" class="crop" title="{{row.entity.label}}">{{row.entity.label}}</a>',
							width: '*',
							sortParam: 'MailboxName',
							sort: {
								direction: uiGridConstants.ASC
							}
						},
						{
							field: 'userObject.displayName',
							displayName: cvLoc('label.emailAddress'),
							headerTooltip: true,
							cellTemplate:
								'<div class="crop" title="{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name}}">{{row.entity.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name}}</div>',
							width: '*',
							enableSorting: false
						}
					];
				}
			}
		};

		self.viewExport = function() {
			self.exchangeFactory.viewExports(self.clientGUID);
		};

		self.getClientGUID = function(clientId) {
			return new Promise(async (resolve, reject) => {
				idaService.getClientProps(clientId).then(resp => {
					if (!_.isUndefined(resp)) {
						resolve(_.get(resp, 'data.clientGUID'));
					}
				});
			});
		};
		self.getExportSetName = function(clientGUID) {
			return `exchange_${clientGUID}_exportset`;
		};

		//Setting parameter to show various actions based on security permissions applicable
		self.setApplicableActions = async resp => {
			await self.checkPermissions().then(permissionList => {
				if (angular.isDefined(permissionList)) {
					permissionList.forEach(permission => {
						switch (permission.permissionId) {
							case self.exchangeConstants.permissions.DELETE:
								if (!self.recoveryPointBrowse) {
									self.browseOptions.showDelete = true;
								}
								break;
							case self.exchangeConstants.permissions.DOWNLOAD:
								//TO DO: Set the value to true once the additional setting is removed and export is shown by default
								self.browseOptions.showExportActions = true;
								break;
							case self.exchangeConstants.permissions.IN_PLACE_RESTORE:
							case self.exchangeConstants.permissions.OUT_PLACE_RESTORE:
								self.browseOptions.showRestoreActions = true;
								break;
						}
					});
				}
			});
		};

		self.checkPermissions = function() {
			return new Promise(async (resolve, reject) => {
				self.browseOptions.getPermissions().then(response => {
					const permissionList = _.get(response, 'data.permissions', []);

					const hasBrowse = permissionList.find(obj => obj.permissionId === self.exchangeConstants.permissions.BROWSE);

					if (!_.isUndefined(hasBrowse)) {
						resolve(permissionList);
					}
				});
			});
		};

		self.openExportModal = function(downloadParams, indexServerClientId, browseRequest) {
			const self = this;
			if (indexServerClientId != 0) {
				ediscoveryService.getAnalyticsEngines(false).success(function(analyticEngines) {
					if (!_.isUndefined(analyticEngines) && analyticEngines.length > 0) {
						let analyticsEngine = analyticEngines.find(function(engine) {
							return engine.indexServerClientId === indexServerClientId;
						});
						if (!_.isUndefined(analyticsEngine)) {
							self.cloudId = analyticsEngine.cloudID;
						}
						idaService.getClientProps(browseRequest.entity.clientId).then(resp => {
							self.clientGUID = _.get(resp, 'data.clientGUID');
							const gResultItems = self.convertToSearchRequestItems(downloadParams.items, self.cloudId);
							const folderRestore =
								downloadParams.fileOrFolder === 'FOLDER' || downloadParams.fileOrFolder === 'MAILBOX' ? true : false;

							var modalInstance = $uibModal.open({
								templateUrl: `${appUtil.appRoot}ediscovery/common/partials/createExportSet.jsp`,
								controller: 'createExportSetController',
								controllerAs: 'createExportSetCtrl',
								backdrop: 'static',
								windowClass: 'small-size',
								resolve: {
									downloadParams: new DownloadParams(
										self.getExportSetName(self.clientGUID),
										browseRequest,
										downloadParams.items,
										downloadParams.exportType,
										downloadParams.gridTotal,
										folderRestore,
										gResultItems,
										true,
										undefined,
										downloadParams.folderBrowse,
										downloadParams.mailboxType
									)
								}
							});
						});
					}
				});
			}
		};

		self.convertToSearchRequestItems = function(selectedItems, cloudId) {
			return selectedItems.map(row => {
				const resultItem = {};
				if (angular.isDefined(row.userObject)) {
					resultItem.appSpecific = {
						eMail: {
							links: row.userObject.path,
							mailBox: row.userObject.advancedData.browseMetaData.exchDataV2.mbxInfo.name,
							subject: row.userObject.advancedData.browseMetaData.exchDataV2.subject,
							from: _.get(row, 'userObject.advancedData.browseMetaData.exchDataV2.from.name'),
							fromHTML: _.get(row, 'userObject.advancedData.browseMetaData.exchDataV2.from.name'),
							to: _.get(row, 'userObject.advancedData.browseMetaData.exchDataV2.to.name'),
							toHTML: _.get(row, 'userObject.advancedData.browseMetaData.exchDataV2.to.name'),
							importance: row.userObject.advancedData.browseMetaData.exchDataV2.msgImportance,
							msgModifiedTime: row.userObject.advancedData.browseMetaData.exchDataV2.msgModifiedTime
						},
						appSpecificFlags: {
							deleted: _.get(row, 'userObject.flags.deleted', false),
							directory: _.get(row, 'userObject.flags.directory', false)
						}
					};

					if (angular.isDefined(subclient.clientId)) {
						resultItem.clientId = subclient.clientId;
					}

					if (angular.isDefined(cloudId)) {
						resultItem.cloudId = cloudId;
					}

					if (angular.isDefined(row.userObject.advancedData.subclient.applicationId)) {
						resultItem.appId = row.userObject.advancedData.subclient.applicationId;
					}
					if (angular.isDefined(row.id)) {
						resultItem.documentId = row.id;
					}
					// If cvturboguid is not present then content id is the turbo guid (Atleast for emails this is working)
					if (angular.isDefined(row.cvturboguid)) {
						resultItem.turboGuid = row.cvturboguid;
					} else {
						resultItem.turboGuid = row.id;
					}
					if (angular.isDefined(row.JobId)) {
						resultItem.jobId = row.JobId;
					}
					if (angular.isDefined(row.userObject.advancedData.offset)) {
						resultItem.aFileOffset = row.userObject.advancedData.offset;
					}
					// Not in schema
					if (angular.isDefined(row.userObject.advancedData.archiveFileId)) {
						resultItem.aFileId = row.userObject.advancedData.archiveFileId;
					}
					//if (angular.isDefined(row.AppType)) {
					resultItem.appType = AppTypes.EXCHANGE_MAILBOX;
					//}
					if (angular.isDefined(row.userObject.advancedData.advConfig.browseAdvancedConfigResp.commcellNumber)) {
						resultItem.commcellNo = row.userObject.advancedData.advConfig.browseAdvancedConfigResp.commcellNumber;
					}
					if (angular.isDefined(row.userObject.path)) {
						resultItem.filePath = row.userObject.path;
					}
					if (angular.isDefined(row.FileName)) {
						resultItem.fileName = row.FileName;
					}
					if (angular.isDefined(row.data_source)) {
						resultItem.dataSourceId = row.data_source;
					}
				}
				return resultItem;
			});
		};

		if (angular.isDefined($stateParams.forMailbox)) {
			try {
				self.browseOptions.forEntity = angular.fromJson($stateParams.forMailbox);
				self.browseOptions.forEntity = angular.extend(
					{
						userObject: {
							size: 0,
							path: '\\MB\\{' + self.browseOptions.forEntity.id + '}',
							advancedData: {
								browseMetaData: {
									indexing: {
										parentObjectGUID: '00000000000000000000000000000001'
									},
									exchDataV2: {
										mbxInfo: {
											name: self.browseOptions.forEntity.emailAddress
										}
									}
								}
							}
						}
					},
					self.browseOptions.forEntity
				);
			} catch (e) {}
		}
	}
]);

export default commonAllAgentsModule;
