import { acAppInstanceModule } from 'common/js/modules';
import 'modules/ida/js/services/sqlService.svc.js';
import 'capps/js/services/cappsClients.svc.js';
import 'vsa/js/factories/vendorTypes.factory.js';
import 'common/js/directives/cv-selected-items.directives.js';

var instanceMod = acAppInstanceModule;

class ComputeStorage {
	constructor(id, name, vCores, allocStorageRange) {
		this.id = id;
		this.name = name;
		this.vCores = vCores;
		this.allocStorageRange = allocStorageRange;
	}
}

class AllocatedStorageRange {
	constructor(min, max, defaultValue) {
		this.min = min;
		this.max = max;
		this.default = defaultValue;
	}
}

instanceMod.controller('cloudMigrateCtrl', [
	'cvLoc',
	'sqlService',
	'cappsClientService',
	'cvUtil',
	'cvToaster',
	'migrateParams',
	'$uibModal',
	'cvTableOptions',
	'cvBreadcrumbsTabsFactory',
	'idaService',
	'AppTypes',
	'dataCachingFactory',
	'CACHING_TYPE',
	'serverService',
	'dbsUtil',
	'$uibModalInstance',
	'INVENTORY_PATH',
	'CAPPS_CONSTANTS',
	'uiGridConstants',
	function(
		cvLoc,
		sqlService,
		cappsClientService,
		cvUtil,
		cvToaster,
		migrateParams,
		$uibModal,
		cvTableOptions,
		cvBreadcrumbsTabsFactory,
		idaService,
		AppTypes,
		dataCachingFactory,
		CACHING_TYPE,
		serverService,
		dbsUtil,
		$modalInstance,
		INVENTORY_PATH,
		CAPPS_CONSTANTS,
		uiGridConstants
	) {
		this.cvTableOptions = cvTableOptions;
		this.cvToaster = cvToaster;
		const self = this;

		self.params = {
			applicationId: migrateParams['applicationId'] ? parseInt(migrateParams['applicationId']) : 0,
			entityId: migrateParams['entityId'] ? parseInt(migrateParams['entityId']) : -1,
			toTimeValue: migrateParams['toTimeValue'] ? migrateParams['toTimeValue'] : '',
			fromTimeValue: migrateParams['fromTimeValue'] ? migrateParams['fromTimeValue'] : '',
			sourceItems: migrateParams['sourceItems'],
			backupsetId: migrateParams['backupsetId'] ? parseInt(migrateParams['backupsetId']) : null,
			entityType: migrateParams['entityType'] ? migrateParams['entityType'] : '',
			copyPrecedence: migrateParams['copyPrecedence'] ? migrateParams['copyPrecedence'] : ''
		};

		self.isSQLServer = self.params.applicationId === AppTypes.SQL_SERVER;

		self.serverMessage = {
			message: '',
			type: ''
		};

		self.azureContainerLoaded = true;
		self.clientListLoaded = true;
		self.instanceListLoaded = true;
		self.regionListLoaded = true;

		self.destinationInstancesMap = null;

		//only ipv4 address is supported  format: x . x . x . x where x must be a decimal value between 0 and 255
		self.IPPattern =
			'^([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5])$';

		self.userNameHelpText = self.isSQLServer ? cvLoc('help.adminUserName2') : cvLoc('help.adminUserName');
		self.passwordHelpText = self.isSQLServer ? cvLoc('help.adminPassword2') : cvLoc('help.adminPassword');

		self.model = {
			selectedVendor: '',
			selectedClient: null,
			selectedInstance: null,
			inputInstance: {
				name: '',
				size: 10
			},
			newOrExistingInstance: '0',
			selectedRegion: null,
			selectedResourceGroup: null,
			selectedAllocatedStorage: 50,
			selectedPricingTier: null,
			selectedvCore: null,
			dbCredentials: {
				userName: '',
				passwordString: '',
				confirmPasswordString: ''
			},
			firewallRules: [],
			startIP: '',
			endIP: ''
		};

		self.validateStartIPAddress = function(formController) {
			self.IPErrorMsg = '';
			if (formController.startIP.$error && formController.startIP.$error.pattern) {
				self.startIPError = cvLoc('error.startIP');
			} else {
				self.startIPError = '';
			}
		};

		self.validateEndIPAddress = function(formController) {
			self.IPErrorMsg = '';
			if (formController.endIP.$error && formController.endIP.$error.pattern) {
				self.endIPError = cvLoc('error.endIP');
			} else {
				self.endIPError = '';
			}
		};

		self.addIPAddress = function() {
			if (self.model.startIP.length === 0 || self.model.endIP.length === 0) {
				return;
			}
			if (
				self.model.firewallRules.findIndex(
					obj => obj.startIP === self.model.startIP && obj.endIP === self.model.endIP
				) >= 0
			) {
				self.IPErrorMsg = cvLoc('error.IPadressAlreadyAdded');
				return;
			}

			if (self.model.firewallRules.length >= 16) {
				self.IPErrorMsg = cvLoc('error.ipPairReachMaximum');
				return;
			}

			const startIPInt = self.model.startIP.split('.').reduce(function(ipInt, octet) {
				return ipInt * 256 + parseInt(octet, 10);
			}, 0);
			const endIPInt = self.model.endIP.split('.').reduce(function(ipInt, octet) {
				return ipInt * 256 + parseInt(octet, 10);
			}, 0);

			if (startIPInt > endIPInt) {
				self.IPErrorMsg = cvLoc('error.startIPNoGreaterThanEnd');
				return;
			}

			self.model.firewallRules.push({
				startIP: self.model.startIP,
				endIP: self.model.endIP
			});
			self.model.startIP = '';
			self.model.endIP = '';
		};

		self.instanceLst = [];

		function init() {
			self.vendorTypes = [
				{
					name: cvLoc('label.vendor.azure_v2'),
					value: 'AZURE_V2',
					regionGroup: cvLoc('label.azure')
				}
				//			,
				//			{
				//			name: cvLoc('label.vendor.amazon'),
				//			value: 'AMAZON',
				//			regionGroup: cvLoc('label.AWS')
				//			}
			];
			//_.sortBy(vendorTypes.getVendorTypes({}), 'name');

			self.addedInstanceId = -1;

			const basicTier = new ComputeStorage(
				'Basic',
				cvLoc('label.computeClassBasic'),
				[{ id: 1 }, { id: 2 }],
				new AllocatedStorageRange(5, 1024, 50)
			);

			const generalTier = new ComputeStorage(
				'General purpose',
				cvLoc('label.computeClassGePurpose'),
				[{ id: 2 }, { id: 4 }, { id: 8 }, { id: 16 }, { id: 32 }, { id: 64 }],
				new AllocatedStorageRange(5, 16384, 100)
			);

			const memoryOptimizedTier = new ComputeStorage(
				'Memory optimized',
				cvLoc('label.computeClassMemOptimized'),
				[{ id: 2 }, { id: 4 }, { id: 8 }, { id: 16 }, { id: 32 }],
				new AllocatedStorageRange(5, 16384, 100)
			);

			const sqlGeneralTier = new ComputeStorage(
				'General purpose',
				cvLoc('label.computeClassGePurpose'),
				[
					{ id: 2 },
					{ id: 4 },
					{ id: 6 },
					{ id: 8 },
					{ id: 10 },
					{ id: 12 },
					{ id: 14 },
					{ id: 16 },
					{ id: 18 },
					{ id: 20 },
					{ id: 24 },
					{ id: 32 },
					{ id: 40 },
					{ id: 80 }
				],
				new AllocatedStorageRange(1, 1024, 32)
			);

			self.pricingTierList = [basicTier, generalTier, memoryOptimizedTier];

			self.model.selectedPricingTier = self.pricingTierList[0];
			self.pricingTierChanged();

			idaService
				.getInstanceDetails(self.params.entityId)
				.success(function(instanceData) {
					self.entity = instanceData.instance;
					if (instanceData.clientAppType === 'CLOUDDB') {
						self.entity.isCloudDBInstance = true;
					}
					updateBreadCrumbs();
				})
				.error(function(err) {
					self.serverMessage = {
						message: err ? err : cvLoc('generic_error'),
						type: 'error'
					};
				});

			if (self.isSQLServer) {
				self.params.sourceItems.forEach(function(item) {
					item.displayName = item.databaseName;
				});
			}
		}

		self.checkAllocStorageValue = function() {
			if (
				self.model.selectedAllocatedStorage <= self.model.selectedPricingTier.allocStorageRange.max &&
				self.model.selectedAllocatedStorage >= self.model.selectedPricingTier.allocStorageRange.min
			) {
				self.model.savedAllocatedStorage = self.model.selectedAllocatedStorage;
			} else {
				self.model.selectedAllocatedStorage = self.model.savedAllocatedStorage;
			}
		};

		function updateBreadCrumbs() {
			let entityId = self.params.entityId;
			if (self.params.backupsetId >= 0) {
				entityId = self.params.backupsetId;
			}

			const instanceDetailsLink = {
				title: self.entity.displayName,
				link:
					'#dbsIdaBrowse' +
					'/' +
					self.params.applicationId +
					'/' +
					entityId +
					'?entityType=' +
					self.params.entityType +
					'&restoreType=0&isMigrate=true'
			};

			const instanceLink = {
				title: cvLoc('pageHeader.instances'),
				link: '#instances'
			};

			const breadCrumbs = [];
			breadCrumbs.push(instanceLink);
			breadCrumbs.push(instanceDetailsLink);

			cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
		}

		self.vendorTypeChanged = function() {
			self.clientListLoaded = false;
			self.clientList = [];
			loadClients();
		};

		self.pricingTierChanged = function() {
			self.vCoreList = self.model.selectedPricingTier.vCores;
			self.model.selectedvCore = self.vCoreList[0];
			self.allocStorageRange = self.model.selectedPricingTier.allocStorageRange;
			self.model.selectedAllocatedStorage = self.model.selectedPricingTier.allocStorageRange.default;
			self.model.savedAllocatedStorage = self.model.selectedAllocatedStorage;
		};

		self.cancel = function() {
			$modalInstance.dismiss();
		};

		function loadClients() {
			self.clientListLoaded = false;
			let instanceType = null;

			//eventually cloud account can be shared across database agents
			if (self.isSQLServer) {
				if (self.model.selectedVendor.value === 'AZURE_V2') {
					instanceType = CAPPS_CONSTANTS.CLOUD_APP_INSTANCE_TYPE.AZURE_SQL.key;
				}

				if (self.model.selectedVendor.value === 'AMAZON') {
					instanceType = CAPPS_CONSTANTS.CLOUD_APP_INSTANCE_TYPE.AMAZON_SQL.key;
				}
			}

			const params = {
				vendor: self.model.selectedVendor.value,
				instanceType: instanceType
			};
			self.clientListError = '';
			cappsClientService
				.getCloudAccounts(params)
				.success(function(data) {
					if (data && data.CloudAppsClientsList) {
						if (data.CloudAppsClientsList.length === 0) {
							self.clientListError = cvLoc('error.noCloudAccount');
							return;
						}

						angular.forEach(data.CloudAppsClientsList, function(client) {
							if (self.addedServerId && self.addedServerId === client.client.clientId) {
								client.client.selected = true;
								self.model.selectedClient = client.client;
								self.clientSelectionChanged();
							}
							self.clientList.push(client.client);
						});
						self.clientList.sort((a, b) => {
							return a.clientName.localeCompare(b.clientName);
						});
					}
					self.clientListLoaded = true;
				})
				.error(function(err) {
					self.clientListError = err ? err : cvLoc('generic_error');
					self.clientListLoaded = true;
				});
		}

		function getInventoryPath() {
			if (self.model.selectedVendor.value === 'AZURE_V2') {
				return INVENTORY_PATH.AZURE_V2;
			}
			if (self.model.selectedVendor.value === 'AMAZON') {
				return INVENTORY_PATH.AMAZON;
			}
		}

		function browseInventory(serverId) {
			// Reset data
			self.azureContainerLoaded = false;
			const inventoryPath = getInventoryPath();
			self.azureContainerList = [];
			self.resourceGroupError = '';
			if (angular.isDefined(serverId)) {
				const azureContainerList = dataCachingFactory.getCacheData(CACHING_TYPE.RESOURCE_GROUP, serverId);
				if (azureContainerList && azureContainerList.length) {
					self.azureContainerList = azureContainerList;
					self.azureContainerLoaded = true;
				} else {
					self.azureContainerList = [];
					serverService
						.vsaInventoryBrowse(serverId, inventoryPath)
						.success(function(data) {
							self.azureContainerList = data;
							dataCachingFactory.setCacheData(CACHING_TYPE.RESOURCE_GROUP, serverId, self.azureContainerList);
							self.azureContainerLoaded = true;
						})
						.error(function(err) {
							self.resourceGroupError = err ? err : cvLoc('error.azureContainer');
							self.azureContainerLoaded = true;
						});
				}
			}
		}

		self.addCloudAccountDialog = function() {
			const self = this;
			const modalInstance = $uibModal.open({
				templateUrl: appUtil.appRoot + 'vsa/partials/addServerContent.jsp',
				windowClass: 'addServerModal',
				backdrop: 'static',
				controller: [
					'$scope',
					'$uibModalInstance',
					function($scope, $modalInstance) {
						$scope.isSubModal = true;
						$scope.forCloudAccount = self.model.selectedVendor.value;
						$scope.cloudDbAppType = self.params.applicationId;
						$scope.$on('serverAdded', function(evt, serverId) {
							self.addedServerId = serverId;
							self.loadClients();
							$modalInstance.dismiss();
						});
						$scope.$on('serverAddCancelled', function() {
							$modalInstance.dismiss();
						});
					}
				]
			});
		};

		self.discoverInstanceDialog = function() {
			const self = this;

			const modalInstance = $uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/ida/partials/cloudMigrateDiscoverInstance.jsp',
				windowClass: 'small-size',
				backdrop: 'static',
				controller: [
					'cloudMigrateData',
					'$scope',
					'$window',
					'$log',
					'$uibModalInstance',
					function(cloudMigrateData, $scope, $window, $log, $modalInstance) {
						$scope.cloudMigrateData = cloudMigrateData;
						$scope.$modalInstance = $modalInstance;
						$scope.$on('instanceAdded', function(evt, instanceId) {
							self.addedInstanceId = instanceId;
							self.loadInstances();
						});

						$scope.$on('instanceAddFailed', function() {
							$log.debug('instanceAdd Failed....');
						});
						$scope.$on('instanceAddCancelled', function() {
							$log.debug('instanceAddCancelled....');
						});
					}
				],
				resolve: {
					cloudMigrateData: function() {
						self.idaEntity = {
							vendor: self.model.selectedVendor.value,
							clientId: self.model.selectedClient.clientId,
							clientName: self.model.selectedClient.clientName,
							applicationId: self.params.applicationId
						};
						return {
							idaEntity: self.idaEntity,
							editDialog: 'Add_Instance'
						};
					}
				}
			});
		};

		self.clientSelectionChanged = function() {
			if (self.model.selectedClient.clientId) {
				self.loadInstances();

				if (self.isSQLServer) {
					loadSQLServerResource();
				} else {
					if (self.model.selectedVendor.value === 'AZURE_V2') {
						browseInventory(self.model.selectedClient.clientId);
					}
					getRegionList(self.model.selectedClient.clientId);
				}
			}
		};

		function loadSQLServerResource() {
			self.azureContainerList = [];
			self.resourceGroupError = '';
			self.azureContainerLoaded = false;
			self.regionListLoaded = false;
			self.regionList = [];
			self.regionErrorMessage = '';
			sqlService
				.getSqlAzureResourceGroupAndRegions(self.model.selectedClient.clientId)
				.success(function(data) {
					self.azureContainerLoaded = true;
					self.regionListLoaded = true;
					if (data && data.azureInventoryList) {
						self.azureContainerList = data.azureInventoryList.resourceGroups.map(
							item => new Object({ displayName: item })
						);
						self.regionList = data.azureInventoryList.locations.map(
							item => new Object({ displayName: item, guid: item })
						);
					}
				})
				.error(function(err) {
					self.azureContainerLoaded = true;
					self.resourceGroupError = err ? err : cvLoc('error.azureContainer');
					self.regionListLoaded = true;
					self.regionErrorMessage = cvLoc('error.regionLoadError');
				});
		}

		self.loadInstances = function() {
			if (self.isSQLServer) {
				getOOPRestoreOptions();
			} else {
				loadDestinationClientsAndInstances();
			}
		};

		function loadDestinationClientsAndInstances() {
			self.instanceList = [];
			self.instanceErrorMessage = '';
			if (
				self.addedInstanceId === -1 &&
				self.destinationInstancesMap &&
				self.destinationInstancesMap.has(self.model.selectedClient.clientId)
			) {
				self.instanceList = cvUtil.sortAscending(
					self.destinationInstancesMap.get(self.model.selectedClient.clientId),
					'instanceName'
				);
			} else {
				self.instanceListLoaded = false;
				idaService.getRestoreDestinationsForEntity(self.entity).then(
					function(result) {
						if (result && result.data) {
							let instancesMap = new Map();
							for (let i = 0; i < result.data.length; i++) {
								//disable source instance selection
								if (self.entity.instanceId === result.data[i].instanceId) {
									result.data[i].disabled = true;
								}
								if (self.addedInstanceId === result.data[i].instanceId) {
									result.data[i].selected = true;
									self.model.selectedInstance = result.data[i];
									self.addedInstanceId = -1;
								}
								if (!instancesMap.has(result.data[i].clientId)) {
									instancesMap.set(result.data[i].clientId, []);
								}
								instancesMap.get(result.data[i].clientId).push(result.data[i]);
							}
							self.destinationInstancesMap = instancesMap;
							if (self.destinationInstancesMap.has(self.model.selectedClient.clientId)) {
								self.instanceList = cvUtil.sortAscending(
									self.destinationInstancesMap.get(self.model.selectedClient.clientId),
									'instanceName'
								);
							} else {
								self.instanceErrorMessage = cvLoc('error.noInstancesFound');
							}
							self.instanceListLoaded = true;
						}
					},
					function(err) {
						self.instanceErrorMessage = err ? err : cvLoc('generic_error');
						self.instanceListLoaded = true;
					}
				);
			}
		}

		function getOOPRestoreOptions() {
			self.instanceList = [];
			self.instanceErrorMessage = '';

			if (
				self.addedInstanceId === -1 &&
				self.destinationInstancesMap &&
				self.destinationInstancesMap.has(self.model.selectedClient.clientId)
			) {
				self.instanceList = cvUtil.sortAscending(
					self.destinationInstancesMap.get(self.model.selectedClient.clientId),
					'instanceName'
				);
			} else {
				const params = {
					instanceId: self.params.entityId,
					sourceItems: angular.toJson(self.params.sourceItems),
					pointInTime: self.params.toTimeValue,
					restoreType: self.entity.isCloudDBInstance ? 'REGULAR' : 'CLOUD_APPS_WINDOWS'
				};
				self.instanceListLoaded = false;
				sqlService.getSQLServerRestoreOptions(params).then(
					function(data) {
						if (data && data.data) {
							const sqlRestoreOptions = data.data;
							let instancesMap = new Map();
							if (sqlRestoreOptions.sqlDbdeviceItem) {
								self.model.sqlDbDeviceItems = sqlRestoreOptions.sqlDbdeviceItem;
							}
							for (var i = 0; i < sqlRestoreOptions.sqlDestinationInstances.length; i++) {
								if (sqlRestoreOptions.sqlDestinationInstances[i].genericEntity.instanceId === self.entity.instanceId) {
									sqlRestoreOptions.sqlDestinationInstances[i].disabled = true;
								}

								if (self.addedInstanceId === sqlRestoreOptions.sqlDestinationInstances[i].genericEntity.instanceId) {
									sqlRestoreOptions.sqlDestinationInstances[i].selected = true;
									self.model.selectedInstance = sqlRestoreOptions.sqlDestinationInstances[i];
									self.addedInstanceId = -1;
								}

								if (!instancesMap.has(sqlRestoreOptions.sqlDestinationInstances[i].genericEntity.clientId)) {
									instancesMap.set(sqlRestoreOptions.sqlDestinationInstances[i].genericEntity.clientId, []);
								}
								instancesMap
									.get(sqlRestoreOptions.sqlDestinationInstances[i].genericEntity.clientId)
									.push(sqlRestoreOptions.sqlDestinationInstances[i].genericEntity);
							}

							self.destinationInstancesMap = instancesMap;
							if (self.destinationInstancesMap.has(self.model.selectedClient.clientId)) {
								self.instanceList = cvUtil.sortAscending(
									self.destinationInstancesMap.get(self.model.selectedClient.clientId),
									'instanceName'
								);
							} else {
								self.instanceErrorMessage = cvLoc('error.noInstancesFound');
							}
							self.instanceListLoaded = true;
						}
					},
					function(err) {
						self.instanceErrorMessage = err ? err : cvLoc('generic_error');
						self.instanceListLoaded = true;
					}
				);
			}
		}

		function getRegionList(serverId) {
			self.regionList = [];
			self.regionListLoaded = false;
			self.regionErrorMessage = '';

			if (angular.isDefined(serverId)) {
				const regionList = dataCachingFactory.getCacheData(CACHING_TYPE.REGION, serverId);
				if (regionList && regionList.length) {
					self.regionList = regionList;
					self.regionListLoaded = true;
				} else {
					const inventoryPath = INVENTORY_PATH.REGION;
					self.regionList = [];
					serverService
						.vsaInventoryBrowse(serverId, inventoryPath)
						.success(function(data) {
							self.regionList = data;
							dataCachingFactory.setCacheData(CACHING_TYPE.REGION, serverId, self.regionList);
							self.regionListLoaded = true;
						})
						.error(function(err) {
							self.regionErrorMessage = cvLoc('error.regionLoadError');
							self.regionListLoaded = true;
						});
				}
			}
		}

		const firewallRulesTableColumns = [
			{
				name: 'startIP',
				displayName: cvLoc('label.startIP'),
				enableSorting: false,
				cellTemplate: '<span title="{{row.entity.startIP}}">{{row.entity.startIP}}</span>',
				width: '45%'
			},
			{
				name: 'endIP',
				displayName: cvLoc('label.endIP'),
				enableSorting: false,
				cellTemplate: '<span title="{{row.entity.endIP}}">{{row.entity.endIP}}</span>',
				width: '45%'
			},
			{
				name: 'actionMenu',
				enableSorting: false,
				displayName: '',
				cellTemplate:
					'<span title="' +
					cvLoc('label.remove') +
					'"><a data-ng-click="grid.appScope.removeContent(row.entity)"><i class="glyphicon glyphicon-remove"></i></a></span>',
				width: '10%'
			}
		];

		const globalGridOptions = angular.copy(this.cvTableOptions.commonNgGridOptions);
		angular.extend(globalGridOptions, {
			data: 'model.firewallRules',
			enableGridMenu: false,
			columnDefs: firewallRulesTableColumns,
			onRegisterApi: function(gridApi) {
				self.gridApi = gridApi;

				self.refreshGrid = function() {
					gridApi.core.notifyDataChange(uiGridConstants.dataChange.ALL);
				};
			}
		});

		self.contentGridOptions = {
			cvHasTitle: false,
			cvTableName: 'firewallRulesTable',
			cvIsSearchable: false,
			cvHasViews: false,
			cvGridCssClass: {
				'modal-grid': true,
				'margin-bottom-10': true,
				'min-height-auto': true
			},
			cvAppScope: self,
			gridOptions: globalGridOptions
		};

		self.removeContent = function(entity) {
			const index = self.model.firewallRules.findIndex(
				obj => obj.startIP === entity.startIP && obj.endIP === entity.endIP
			);
			if (index >= 0) {
				self.model.firewallRules.splice(index, 1);
				self.refreshGrid();
			}
		};

		let composeBrowseOption = function() {
			let browseOption = {
				commCellId: 2,
				timeRange: {
					fromTimeValue: self.params.fromTimeValue ? self.params.fromTimeValue : null,
					toTimeValue: self.params.toTimeValue ? self.params.toTimeValue : null
				}
			};
			if (self.params.backupsetId) {
				let backupset = {
					clientId: self.entity.clientId,
					backupsetId: self.params.backupsetId
				};
				browseOption.backupset = backupset;
			}
			if (self.params.copyPrecedence) {
				angular.extend(browseOption, {
					mediaOption: {
						copyPrecedence: {
							copyPrecedenceApplicable: true,
							copyPrecedence: self.params.copyPrecedence
						}
					}
				});
			}
			return browseOption;
		};

		let composeDestination = function() {
			let destination = {};
			let destClient = {};
			let destinationInstance = {};

			destClient = angular.copy(self.model.selectedClient);
			destination.destClient = destClient;

			if (self.model.newOrExistingInstance === '1') {
				let selectedInstance = self.model.selectedInstance;
				destinationInstance = {
					clientId: selectedInstance.clientId,
					clientName: selectedInstance.clientName,
					applicationId: selectedInstance.applicationId,
					instanceId: selectedInstance.instanceId,
					instanceName: selectedInstance.instanceName
				};

				destination.destinationInstance = destinationInstance;
			}
			return destination;
		};

		let composeAgentSpecificOptions = function() {
			let agentSpecificOptions = {};
			switch (self.params.applicationId) {
				case AppTypes.POSTGRESQL:
					agentSpecificOptions = {
						fsBackupSetRestore: false,
						instanceRestore: false,
						isCloneRestore: false,
						pointInTime: false,
						restoreToSameServer: false,
						startServer: false,
						tableLevelRestore: false,
						refTime: {
							time: 0
						},
						fromTime: {
							time: 0
						},
						pointOfTime: {
							time: 0
						}
					};
					break;
				case AppTypes.MYSQL:
					agentSpecificOptions = {
						dataStagingLocation: '',
						instanceRestore: false,
						isCloneRestore: false,
						pointofTime: false,
						refTime: {
							time: 0
						},
						fromTime: {
							time: 0
						},
						pointInTime: {
							time: 0
						},
						tableLevelRestore: false,
						destinationServer: {},
						//MySQL Log Restore Options
						mySqlDatabase: [],
						data: true,
						log: true,
						recurringRestore: false,
						logRestoreType: 0, //0 - for Recover ; 1- Do not recover
						//which is on selected destination client ,used to store Logs.
						destinationFolder: '',
						temporaryStagingLocation: ''
					};

					if (self.model.newOrExistingInstance === '1') {
						agentSpecificOptions.destinationServer = {
							id: self.model.selectedInstance.instanceId,
							name: self.model.selectedInstance.instanceName
						};
					}
					break;

				case AppTypes.SQL_SERVER:
					agentSpecificOptions = {
						overWrite: true,
						dropConnectionsToDatabase: true,
						pointOfTimeRst: false,
						latestBkpData: true,
						sqlRecoverType: '0'
					};
			}
			return agentSpecificOptions;
		};

		let composeCloudMigrateOptions = function() {
			if (self.model.newOrExistingInstance === '1') {
				return {};
			}
			let cloudDBMigrationOptions = {
				isCloudDBMigration: true,
				azureMigrationOptions: {
					serverName: self.model.inputInstance.name,
					firewallRules: self.model.firewallRules.length > 0 ? self.model.firewallRules : null,
					region: self.model.selectedRegion.guid,
					resourceGroup: self.model.selectedResourceGroup.displayName,
					dbCredentials: {
						userName: self.model.dbCredentials.userName,
						password: cvUtil.getBytes(self.model.dbCredentials.passwordString)
					}
				}
			};

			if (!self.isSQLServer) {
				cloudDBMigrationOptions.azureMigrationOptions.skuTier = self.model.selectedPricingTier.id;
				cloudDBMigrationOptions.azureMigrationOptions.storageSize = self.model.selectedAllocatedStorage * 1024;
				cloudDBMigrationOptions.azureMigrationOptions.vCores = self.model.selectedvCore.id;
				cloudDBMigrationOptions.azureMigrationOptions.computeClass = 'Gen5';
			}
			return cloudDBMigrationOptions;
		};

		let getSourceItems = function() {
			let sourceItems = [];

			if (self.isSQLServer) {
				angular.forEach(self.params.sourceItems, function(item, key) {
					sourceItems.push(item.databaseName);
				});
			} else {
				angular.forEach(self.params.sourceItems, function(item) {
					sourceItems.push('/' + item.displayName);
				});
			}
			return sourceItems;
		};

		let isInputValid = function() {
			self.attempted = true;

			if (self.model.newOrExistingInstance === '0') {
				if (
					!self.model.selectedVendor ||
					!self.model.selectedClient ||
					!self.model.inputInstance.name ||
					!self.model.selectedRegion ||
					!self.model.selectedResourceGroup ||
					!self.model.dbCredentials.userName ||
					!self.model.dbCredentials.passwordString ||
					self.model.dbCredentials.passwordString != self.model.dbCredentials.confirmPasswordString
				) {
					return false;
				}
				switch (self.params.applicationId) {
					case AppTypes.SQL_SERVER:
						if (self.model.selectedVendor.value === 'AZURE' && !self.model.s3BucketName) {
							return false;
						}
				}
			} else {
				if (!self.model.selectedVendor || !self.model.selectedClient || !self.model.selectedInstance) {
					return false;
				}
			}
			return true;
		};

		self.doMigrate = function() {
			if (isInputValid()) {
				let genericEntity = angular.copy(self.entity);
				if (self.params.backupsetId >= 0) {
					genericEntity.backupsetId = self.params.backupsetId;
				}

				let browseOption = composeBrowseOption();
				let destination = composeDestination();
				let commonOptions = {};
				let agentSpecificOptions = composeAgentSpecificOptions();
				let rdsOption = {};
				let sourceItems = getSourceItems();
				let cloudMigrateOptions = composeCloudMigrateOptions();

				idaService
					.submitRestoreJob({
						entityType: self.params.entityType,
						genericEntity: angular.toJson(genericEntity),
						browseOption: angular.toJson(browseOption),
						destination: angular.toJson(destination),
						commonOptions: angular.toJson(commonOptions),
						sourceItemsToRestore: angular.toJson(sourceItems),
						agentSpecificOptions: angular.toJson(agentSpecificOptions),
						rdsOption: angular.toJson(rdsOption),
						cloudDBMigrationRstOption: angular.toJson(cloudMigrateOptions)
					})
					.success(function(jobId) {
						let jobStartString = cvLoc('notification.migrateJobStarted', jobId);
						dbsUtil.showRestoreToaster(jobId, jobStartString);
						self.cancel();
					})
					.error(function(err) {
						cvToaster.showErrorMessage({
							message: err
						});
					});

				return;
			}
		};

		init();
	}
]);

export default instanceMod;
