import 'modules/emailTemplates/js/services/emailTemplates.svc.js';
import 'modules/emailTemplates/js/services/emailPreviewModal.svc.js';

import { emailTemplatesModule } from 'common/js/modules';

import { cellNameTemplate, cellActionTemplate, cellBooleanTemplate } from './../util';

import { IS_TENANT_ADMIN, COMPANY_COMCELL } from './../constants';

// temporary localization Email Template strings
//import './../temp.localization';

// dependencies injected/need by Angular
import '../services/emailTemplates.svc';
import '../services/emailPreviewModal.svc';

// dependency for cvGrid option `cvCompanyDropdown`
import 'vsa/js/services/subscriptions.svc';

// list of emails sent by Admin Console which are configurable
emailTemplatesModule.controller('EmailTemplatesCtrl', [
	'EmailTemplatesService',
	'EmailPreviewModalService',
	'cvLoc',
	'cvTableOptions',
	'cvBreadcrumbsTabsFactory',
	'$state',
	'$stateParams',
	'$dialogs',
	'cvToaster',
	'$q',
	function EmailTemplatesCtrl(
		EmailTemplatesService,
		EmailPreviewModalService,
		cvLoc,
		cvTableOptions,
		cvBreadcrumbsTabsFactory,
		$state,
		$stateParams,
		$dialogs,
		cvToaster,
		$q
	) {
		let vm = this;
		let companyId = $stateParams.companyId;

		vm.isLoading = true;

		cvBreadcrumbsTabsFactory.addTabs([
			{
				title: cvLoc('emailTemplates.label.emailTemplates'),
				link: '#emailTemplates',
				selected: true
			},
			{
				title: cvLoc('label.nav.emailHeaderFooterConfig'),
				link: '#emailHeaderFooterConfig'
			}
		]);

		// ideally the initial value should be in the route resolver
		vm.emailTemplates = [];

		controllerInit();

		vm.handleEdit = function(emailTemplate) {
			// #emailTemplates/{id}
			let params = { emailTemplateId: emailTemplate.id };
			$state.go('emailTemplateDetail', params);
		};

		vm.handleMakeDefault = function(emailTemplate) {
			emailTemplate.isDefault = true;
			emailTemplate.save().then(res => {
				fetchEmailTemplates();
			});
		};

		vm.handleClone = function(emailTemplate) {
			// #emailTemplates/new-{id}
			let params = { emailTemplateId: `new-${emailTemplate.id}` };
			$state.go('emailTemplateDetail', params);
		};

		vm.handleDelete = function(emailTemplate) {
			let templateName = emailTemplate.name;

			$dialogs.confirm(
				cvLoc('emailTemplates.label.confirmDelete'),
				cvLoc('emailTemplates.msg.confirmDelete', `<b>${templateName}</b>`),
				{
					noFunction: function() {},
					yesFunction: function() {
						emailTemplate.remove().then(onDeleteSuccess, onDeleteFailure);
					}
				}
			);

			function onDeleteSuccess(res) {
				fetchEmailTemplates();
				cvToaster.showSuccessMessage({
					message: cvLoc('emailTemplates.msg.deleteSuccessful', `<b>${templateName}</b>`)
				});
			}

			function onDeleteFailure(err) {
				if (err) {
					cvToaster.showErrorMessage({
						message: cvLoc('emailTemplates.msg.deleteFailure', `<b>${templateName}</b>`)
					});
				}
			}
		};

		let columnDefs = [
			{
				field: 'name',
				sort: { direction: 'asc', priority: 0 },
				displayName: cvLoc('emailTemplates.table.col.name'),
				enableHiding: false,
				cellTemplate: cellNameTemplate('row.entity.name', 'emailTemplateDetail({ emailTemplateId: row.entity.id })'),
				width: '15%'
			},
			{
				field: 'description',
				displayName: cvLoc('emailTemplates.table.col.desc'),
				width: '25%'
			},
			{
				field: 'typeName',
				displayName: cvLoc('emailTemplates.table.col.type'),
				width: '15%'
			},
			{
				field: 'isDefault',
				displayName: cvLoc('emailTemplates.table.col.default'),
				cellTemplate: cellBooleanTemplate('row.entity.isDefault')
			},
			{
				field: 'localeName',
				displayName: cvLoc('emailTemplates.table.col.locale')
			},
			{
				name: 'action',
				displayName: cvLoc('emailTemplates.table.col.actions'),
				enableHiding: false,
				enableFiltering: false,
				enableSorting: false,
				width: '5%',
				cellTemplate: cellActionTemplate([
					{
						label: cvLoc('emailTemplates.label.edit'),
						action: 'grid.appScope.handleEdit(row.entity)'
					},
					{
						label: cvLoc('emailTemplates.label.makeDefault'),
						action: 'grid.appScope.handleMakeDefault(row.entity)',
						condition: 'row.entity.isDefault !== true'
					},
					{
						label: cvLoc('emailTemplates.label.clone'),
						action: 'grid.appScope.handleClone(row.entity)'
					},
					{
						label: cvLoc('emailTemplates.label.delete'),
						action: 'grid.appScope.handleDelete(row.entity)'
					}
				])
			}
		];

		if (!IS_TENANT_ADMIN) {
			// the company column will only be shown if the user is not a tenant admin
			columnDefs.splice(3, 0, {
				field: 'companyName',
				displayName: cvLoc('emailTemplates.table.col.company'),
				width: '15%'
			});
		}

		vm.gridOptions = {
			cvGridTitle: cvLoc('emailTemplates.label.emailTemplates'),
			cvTableName: 'emailTemplatesTable',
			cvSearchFields: ['name', 'description'],
			cvOnGridEmpty: {
				message: cvLoc('emailTemplates.msg.noTemplates'),
				links: [
					{
						label: cvLoc('emailTemplates.label.addTemplate'),
						href: '#emailTemplates/new'
					}
				]
			},
			cvPageLinks: [
				{
					label: cvLoc('emailTemplates.label.addTemplate'),
					href: '#emailTemplates/new',
					disableMultiCommcell: true
				}
			],
			cvAppScope: vm,
			// we removed the company dropdown for now, company will eventually be an app level option
			// which can be selected to scope the page contents for a particular company
			// cvCompanyDropdown: true,
			gridOptions: Object.assign(
				{
					data: 'emailTemplates',
					showGridFooter: true,
					columnDefs
				},
				cvTableOptions.commonNgGridOptions
			)
		};

		// in the absence of using route resolvers, this is probably the best approach
		// to "pre-fetch" the data required for this page
		function controllerInit() {
			// we need to fetch types first, ideally we should prefetch these in the service
			fetchEmailTemplates();
		}

		function fetchEmailTemplates() {
			vm.isLoading = true;
			$q.all([
				EmailTemplatesService.getEmailTemplates({ propertyLevel: 30 }),
				EmailTemplatesService.getSupportedLocales()
			]).then(res => {
				let emailTemplates = res[0];
				let locales = res[1];
				emailTemplates.forEach(function(part, i, etArray) {
					let templateLocale = locales.find(itemLocale => itemLocale.localeId === etArray[i]._locale);
					if (templateLocale) {
						etArray[i].localeName = templateLocale.displayString;
					}
					if (!IS_TENANT_ADMIN && etArray[i].companyId === 0) {
						etArray[i].companyName = COMPANY_COMCELL;
					}
				});
				vm.emailTemplates = emailTemplates;
				vm.isLoading = false;
			});
		}

		// Add breadcrumbs

		var breadCrumbs = [];

		breadCrumbs.push({
			title: cvLoc('label.nav.masterCustomization'),
			link: '#nav/masterCustomization'
		});

		cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
	}
]);

export default emailTemplatesModule;
