import { acAppDistributedStorageModule } from 'common/js/modules';
import 'modules/distributedStorage/js/distributedStorage.svc.js';

acAppDistributedStorageModule.factory('distributedStorageFactory', [
	'cvLoc',
	'cvUtil',
	'cvToaster',
	'$dialogs',
	'$state',
	'cvPermissionFactory',
	'$uibModal',
	'distributedStorageService',
	'settingsService',
	'PERMISSIONS',
	'UI_ROUTE',
	'cvActionsEventFactory',
	function(
		cvLoc,
		cvUtil,
		cvToaster,
		$dialogs,
		$state,
		cvPermissionFactory,
		$modal,
		distributedStorageService,
		settingsService,
		PERMISSIONS,
		UI_ROUTE,
		cvActionsEventFactory
	) {
		var factory = {};

		factory.getDistributedStorageId = function(entity) {
			return _.get(entity, 'distributedStorage.distributedStorageId', 0);
		};

		factory.getDistributedStorageName = function(entity) {
			return _.get(entity, 'distributedStorage.distributedStorageName', '');
		};

		factory.getUserName = function(entity) {
			return _.get(entity, 'credential.record.userName', '');
		};

		factory.getURL = function(entity) {
			return _.get(entity, 'url', '');
		};

		factory.openClusterInNewTab = function(distributedStorageObj) {
			settingsService
				.getSAMLTokenForHedvig()
				.then(function(data) {
					distributedStorageService
						.loginCluster(distributedStorageObj.url, data.data)
						.then(function(data) {
							// Open redirect url in new tab
							let url = _.get(data.data, 'result.redirectUrl', '');
							if (url) {
								window.open(url, '_blank');
							} else {
								cvToaster.showErrorMessage({
									message: cvLoc('error.redirectUrl'),
									ttl: '5000'
								});
							}
						})
						.catch(function(error) {
							cvToaster.showErrorMessage({
								message: error.data,
								ttl: '5000'
							});
						});
					return data.data;
				})
				.catch(function(error) {
					cvToaster.showErrorMessage({
						message: error.data,
						ttl: '5000'
					});
				});
		};

		factory.deleteCluster = function(distributedStorageInfo) {
			let distributedStorageId = factory.getDistributedStorageId(distributedStorageInfo);
			let distributedStorageName = factory.getDistributedStorageName(distributedStorageInfo);
			var callBackFunctions = {
				noFunction: function() {},
				yesFunction: function() {
					distributedStorageService
						.deleteCluster(distributedStorageId)
						.success(function(data) {
							if (data.error && data.error.errorCode !== 0) {
								let errorMessage = data.error.errorMessage
									? data.error.errorMessage
									: cvLoc('error.clusterOperationFailed');
								cvToaster.showErrorMessage({
									message: errorMessage,
									ttl: '5000'
								});
							} else if (typeof data === 'string') {
								cvToaster.showErrorMessage({
									message: data,
									ttl: '5000'
								});
							} else {
								let successMessage = cvLoc('success.deleteCluster', '<b>' + distributedStorageName + '</b>');
								cvToaster.showSuccessMessage({
									message: successMessage,
									ttl: '5000'
								});
								cvActionsEventFactory.removeActionRedirect(UI_ROUTE.DIST_STORAGE);
							}
						})
						.error(function(error) {
							cvToaster.showErrorMessage({
								message: error,
								ttl: '5000'
							});
						});
				}
			};
			$dialogs.confirm(
				cvLoc('label.confirmUnregister'),
				cvLoc('label.globalActions.confirmDeleteDistStorageText', '<b>' + distributedStorageName + '</b>'),
				callBackFunctions
			);
		};

		factory.registerOrEditCluster = function(distributedStorageInfo, callBackMode) {
			var modalInstance = $modal.open({
				templateUrl: appUtil.appRoot + 'modules/distributedStorage/partials/registerOrEditCluster.jsp',
				backdrop: 'static',
				windowClass: 'small-size',
				controllerAs: 'registerCtrl',
				controller: [
					'$scope',
					'cvUtil',
					'$uibModalInstance',
					'cvLoc',
					'cvToaster',
					'$filter',
					'$state',
					'distributedStorageFactory',
					'distributedStorageService',
					'distributedStorageInfo',
					'callBackMode',
					function(
						$scope,
						cvUtil,
						$modalInstance,
						cvLoc,
						cvToaster,
						$filter,
						$state,
						distributedStorageFactory,
						distributedStorageService,
						distributedStorageInfo,
						callBackMode
					) {
						var self = this;
						let init = function() {
							self.passwordPlaceHolder = '';
							if (distributedStorageInfo) {
								self.mode = 'edit';
								self.passwordPlaceHolder = cvLoc('label.keepSamePassword');
							}
							self.entityId = distributedStorageFactory.getDistributedStorageId(distributedStorageInfo);
							self.entityName = distributedStorageFactory.getDistributedStorageName(distributedStorageInfo);
							self.userName = distributedStorageFactory.getUserName(distributedStorageInfo);
							self.url = distributedStorageFactory.getURL(distributedStorageInfo);
						};

						let getDistributedStorageInfo = function(entityId, entityName, url, userName, password) {
							let obj = {
								distributedStorage: {
									distributedStorageId: entityId,
									distributedStorageName: entityName
								},
								credential: {
									record: {
										userName: userName
									}
								},
								url: url
							};
							if (password) {
								obj.credential.record.password = Base64.encode(password);
							}
							return obj;
						};

						self.setErrorMessage = function(message) {
							self.serverMessage = {
								type: 'error',
								message: message
							};
						};

						self.setMessage = function(message) {
							self.serverMessage = {
								type: 'loader',
								message: message
							};
						};

						self.clearMessage = function() {
							self.serverMessage = null;
						};

						self.submit = function() {
							let distributedStorageInfo = getDistributedStorageInfo(
								self.entityId,
								self.entityName,
								self.url,
								self.userName,
								self.password
							);
							self.setMessage(cvLoc('Saving'));
							let promise;
							if (self.mode === 'edit') {
								promise = distributedStorageService.updateCluster(self.entityId, distributedStorageInfo);
							} else {
								promise = distributedStorageService.registerCluster(distributedStorageInfo);
							}

							promise
								.success(function(data) {
									if (data.error && data.error.errorCode !== 0) {
										if (data.error.errorMessage) {
											self.setErrorMessage(data.error.errorMessage);
										} else {
											self.setErrorMessage(cvLoc('error.clusterOperationFailed'));
										}
									} else if (typeof data === 'string') {
										self.setErrorMessage(data);
									} else {
										let successMessage =
											self.mode === 'edit'
												? cvLoc('success.editCluster', '<b>' + self.entityName + '</b>')
												: cvLoc('success.registerCluster', '<b>' + self.entityName + '</b>');
										cvToaster.showSuccessMessage({
											message: successMessage,
											ttl: '5000'
										});
										if (callBackMode) {
											$modalInstance.close(self.entityName);
										} else {
											$modalInstance.close();
											$state.reload();
										}
									}
								})
								.error(function(error) {
									self.setErrorMessage(error);
								});
						};

						self.cancel = function() {
							$modalInstance.close();
						};

						init();
					}
				],
				resolve: {
					distributedStorageInfo: function() {
						return distributedStorageInfo;
					},
					callBackMode: function() {
						return callBackMode;
					}
				}
			});
			return modalInstance;
		};

		factory.sync = function(distributedStorageInfo) {
			let distributedStorageId = factory.getDistributedStorageId(distributedStorageInfo);
			let distributedStorageName = factory.getDistributedStorageName(distributedStorageInfo);
			let callBackFunctions = {
				noFunction: function() {},
				yesFunction: function() {
					distributedStorageService
						.sync(distributedStorageId)
						.success(function(data) {
							if (data && data.error && data.error.errorCode) {
								cvToaster.showErrorMessage({
									message: data.error.errorMessage,
									ttl: '5000'
								});
							} else {
								cvToaster.showSuccessMessage({
									message: cvLoc('success.sync.distributedStorage', distributedStorageName),
									ttl: '5000'
								});
							}
						})
						.error(function(error) {
							cvToaster.showErrorMessage({
								message: error,
								ttl: '5000'
							});
						});
				}
			};

			$dialogs.confirm(
				cvLoc('label.confirmSync'),
				cvLoc('label.globalActions.confirmSyncText', '<b>' + distributedStorageName + '</b>'),
				callBackFunctions
			);
		};

		factory.getActions = function(entity) {
			let distStorageSpecificActions = [
				/*
				{
					EDIT: {
						label: cvLoc('label.globalActions.edit'),
						show: true,
						onClick: factory.registerOrEditCluster.bind(this, entity),
						groupId: 1,
						rank: 2
					}
				},*/
				{
					DELETE: {
						label: cvLoc('label.globalActions.unregister'),
						show: _.get(entity, 'canManage', false),
						permissionIds: [],
						onClick: factory.deleteCluster.bind(this, entity),
						groupId: 1,
						rank: 2
					}
				},
				{
					SYNC: {
						label: cvLoc('label.globalActions.sync'),
						show: true,
						permissionIds: [],
						onClick: factory.sync.bind(this, entity),
						groupId: 1,
						rank: 2
					}
				}
			];

			return cvPermissionFactory.updateAndGetCommonActionsList(
				distStorageSpecificActions,
				false,
				factory.distStorageSpecificActionsPermittedList()
			);
		};

		factory.distStorageSpecificActionsPermittedList = function() {
			return {
				EDIT: {
					label: cvLoc('label.globalActions.edit'),
					value: 'EDIT',
					permissionIds: []
				},
				DELETE: {
					label: cvLoc('label.globalActions.delete'),
					value: 'DELETE',
					permissionIds: []
				},
				SYNC: {
					label: cvLoc('label.globalActions.sync'),
					value: 'SYNC',
					permissionIds: []
				}
			};
		};

		return factory;
	}
]);

export default acAppDistributedStorageModule;
