import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';

import { drAppFailoverModule } from 'common/js/modules';

/**
 * Factory functions for handling DRGroup Operations (failover / failback / test boot / test failover)
 */

var failoverMod = drAppFailoverModule;

failoverMod.factory('drOperationsFactory', [
	'$uibModal',
	'cvToaster',
	'cvLoc',
	'failoverService',
	'FailoverConstants',
	function drOperationsFactory($uibModal, cvToaster, cvLoc, failoverService, FailoverConstants) {
		var factory = {
			// public functions
			areMachinesFromSameSchedule: areMachinesFromSameSchedule,
			failback: failback,
			isFailbackAllowed: isFailbackAllowed,
			isFailbackAllowedOnMachine: isFailbackAllowedOnMachine,
			isFailoverAllowed: isFailoverAllowed,
			isFailoverAllowedOnMachine: isFailoverAllowedOnMachine,
			isTestbootAllowedOnMachine: isTestbootAllowedOnMachine,
			isTestbootSupportedVendor: isTestbootSupportedVendor,
			plannedFailover: plannedFailover,
			testBoot: testBoot,
			undoFailover: undoFailover,
			unPlannedFailover: unPlannedFailover,
			testFailover: testFailover,
			undoTestFailover: undoTestFailover
		};

		// private variable
		var toasterTTL = 10000;
		var testbootSupportedVendors = [FailoverConstants.Vendors['VMW']];
		var imgRootPath = appUtil.appRoot + 'common/img/drSteps';

		return factory;

		function failback(vApp, subclientId) {
			$uibModal.open({
				windowClass: 'small-size',
				templateUrl: appUtil.appRoot + 'vsa/partials/modal.jsp',
				backdrop: 'static',
				controller: [
					'$scope',
					'$uibModalInstance',
					'cvLoc',
					'failoverService',
					'cvUtil',
					function($scope, $modalInstance, cvLoc, failoverService, cvUtil) {
						$scope.showUndoFailoverOptions = false;
						$scope.discardChanges = false;

						for (var i = 0; i < vApp.machinesList.length; i++) {
							if (vApp.machinesList && vApp.machinesList[i].destVendor === 'AZURE_V2') {
								$scope.showSourceVMOverwrittenNote = true;
							}

							if (vApp.machinesList[i].destVendor === 'MICROSOFT') {
								$scope.showUndoFailoverOptions = true;
							}
						}
						if ($scope.showUndoFailoverOptions) {
							var bodyText = '';
							var imgSource = imgRootPath + '/undo-failover-steps.png';
							var imageClass = 'dr4StepsImg';
							var imageText =
								'<div class="dr4StepsTxt">' +
								'<div class="drStepsTxt">' +
								cvLoc('label.discardChanges') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.powerOffDRVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.powerONProductionVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.enableLiveSync') +
								'</div>' +
								'</div>';
						} else {
							var bodyText = cvLoc('label.confirmFailback');
							var imgSource = imgRootPath + '/failback-steps.png';
							var imageClass = 'dr5StepsImg marginTop10px';
							var imageText =
								'<div class="dr5StepsTxt marginTop10px">' +
								'<div class="drStepsTxt">' +
								cvLoc('label.shutdownDRVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.incrementalBackupDRVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.restoreChangestoProductionVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.powerOnSourceVM') +
								'</div>' +
								'<div class="drStepsTxt">' +
								cvLoc('label.enableSync') +
								'</div>' +
								'</div>';
							if (!vApp.machinesFromSameSchedule) {
								var bodyText = cvLoc('label.differentSchedule') + ' ' + cvLoc('label.vmFailbackFromMonitor');
								var imgSource = null;
								var imageText = null;
								var imageClass = null;
								var hideOkButton = true;
								var closeButtonText = cvLoc('OK');
							}
						}

						$scope.setDiscardChanges = function() {
							$scope.discardChanges = $scope.discardChanges === false ? true : false;
						};

						$scope.successCallBackOfUndoFailover = function(data) {
							if (data && data.jobIds && data.jobIds.length > 0) {
								var jobID = data.jobIds[0];
								var jobURL = '#/jobs/' + jobID;
								var responseContent = ' <a href="' + jobURL + '">' + cvLoc('msg.jobDetails') + ' ' + jobID + '</a>';
								showSuccessMessage(responseContent);
							} else {
								showErrorMessage();
							}
							$modalInstance.dismiss();
						};

						$scope.modalOptions = {
							headerText: cvLoc('label.failback'),
							bodyText: bodyText,
							imgSrc: imgSource,
							imgText: imageText,
							imgClass: imageClass,
							closeButtonText: closeButtonText || cvLoc('button.no'),
							actionButtonText: cvLoc('button.yes'),
							hideOkButton: hideOkButton || false,
							ok: function() {
								subclientId = subclientId ? subclientId : 0;
								if ($scope.showUndoFailoverOptions) {
									if (!$scope.discardChanges) {
										$scope.missingDiscardChanges = cvUtil.errMsg(cvLoc('error.discardChangesSelection'));
										return;
									}
									failoverService
										.performUndoFailover(JSON.stringify(vApp), 6, subclientId)
										.success(function(data) {
											$scope.successCallBackOfUndoFailover(data);
										})
										.error(function(e) {
											showErrorMessage(e);
											$modalInstance.dismiss();
										});
								} else {
									failoverService
										.performFailback(JSON.stringify(vApp), subclientId)
										.success(function(data) {
											$scope.successCallBackOfUndoFailover(data);
										})
										.error(function(e) {
											showErrorMessage(e);
											$modalInstance.dismiss();
										});
								}
							},
							close: function() {
								$modalInstance.dismiss();
							}
						};
					}
				]
			});
		}

		function plannedFailover(vApp) {
			var imgSource = imgRootPath + '/planned-failover-steps.png';
			var imageClass = 'dr5StepsImg';
			var imageText =
				'<div class="dr5StepsTxt">' +
				'<div class="drStepsTxt">' +
				cvLoc('label.powerOffProductionVM') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.runProductionVMbackup') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.performReplication') +
				'</div>' +
				'<div class="drStepsTxt padding0505px">' +
				cvLoc('label.disableLiveSync') +
				'</div>' +
				'<div class="drStepsTxt padding0505px">' +
				cvLoc('label.powerOnDRVM') +
				'</div>' +
				'</div>';

			let modalOptions = {
				headerText: cvLoc('label.plannedFailover'),
				bodyText: cvLoc('label.confirmPlannedFailover'),
				imgSrc: imgSource,
				imgText: imageText,
				imgClass: imageClass,
				promise: failoverService.performPlannedFailover,
				params: [JSON.stringify(vApp)]
			};

			openModal(modalOptions);
		}

		function unPlannedFailover(vApp, bodyText, additionalConfirm) {
			var imgSource = imgRootPath + '/unplanned-failover-steps.png';
			var imageClass = 'dr3StepsImg';
			var imageText =
				'<div class="dr3StepsTxt marginLeft85px">' +
				'<div class="drStepsTxt">' +
				cvLoc('label.stopProductionVM') +
				'</div>' +
				'<div class="drStepsTxt padding0505px">' +
				cvLoc('label.disableLiveSync') +
				'</div>' +
				'<div class="drStepsTxt padding05022px">' +
				cvLoc('label.powerOnDRVM') +
				'</div>' +
				'</div>';

			let modalOptions = {
				headerText: cvLoc('label.unplannedFailover'),
				bodyText: bodyText || cvLoc('label.confirmUnplannedFailover'),
				imgSrc: imgSource,
				imgText: imageText,
				imgClass: imageClass,
				additionalConfirm: additionalConfirm,
				promise: failoverService.performUnplannedFailover,
				params: [JSON.stringify(vApp)]
			};

			openModal(modalOptions);
		}

		function undoFailover(vApp) {
			var imgSource = imgRootPath + '/undo-failover-steps.png';
			var imageClass = 'dr4StepsImg';
			var imageText =
				'<div class="dr4StepsTxt">' +
				'<div class="drStepsTxt">' +
				cvLoc('label.discardChanges') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.powerOffDRVM') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.powerONProductionVM') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.enableLiveSync') +
				'</div>' +
				'</div>';

			let modalOptions = {
				headerText: cvLoc('label.undoFailover'),
				bodyText: cvLoc('label.confirmUndoFailover'),
				imgSrc: imgSource,
				imgText: imageText,
				imgClass: imageClass,
				promise: failoverService.performUndoFailover,
				params: [JSON.stringify(vApp), 6]
			};

			openModal(modalOptions);
		}

		function testBoot(vApp, enableNetwork) {
			var imgSource = enableNetwork
				? imgRootPath + '/testboot-step2-deselected.png'
				: imgRootPath + '/testboot-steps.png';
			var imageClass = 'dr6StepsImg';
			var imageText =
				'<div class="dr6StepsTxt">' +
				'<div class="drStepsTxt">' +
				cvLoc('label.takeSnapshot') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.disableNetworkAdapters') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.powerON') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.verifyVMToolStatus') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.revertSnapshot') +
				'</div>' +
				'<div class="drStepsTxt">' +
				cvLoc('label.deleteSnapshot') +
				'</div>' +
				'</div>';
			let modalOptions = {
				headerText: cvLoc('label.testBoot'),
				bodyText: cvLoc('label.confirmTestBoot'),
				imgSrc: imgSource,
				imgText: imageText,
				imgClass: imageClass,
				promise: failoverService.performTestBoot,
				params: [JSON.stringify(vApp), enableNetwork]
			};

			openModal(modalOptions);
		}

		function testFailover(vApp, repIds) {
			let modalOptions = {
				headerText: cvLoc('label.testFailover'),
				bodyText: repIds ? cvLoc('label.confirmTestFailoverMachines') : cvLoc('label.confirmTestFailover'),
				promise: failoverService.performTestFailover,
				params: [JSON.stringify(vApp), JSON.stringify(repIds)]
			};

			openModal(modalOptions);
		}

		function undoTestFailover(vApp, repIds) {
			let modalOptions = {
				headerText: cvLoc('label.undoTestFailover'),
				bodyText: repIds ? cvLoc('label.confirmUndoTestFailoverMachines') : cvLoc('label.confirmUndoTestFailover'),
				promise: failoverService.performUndoTestFailover,
				params: [JSON.stringify(vApp), JSON.stringify(repIds)]
			};

			openModal(modalOptions);
		}

		function openModal(modalOptions) {
			$uibModal.open({
				windowClass: 'small-size',
				templateUrl: appUtil.appRoot + 'vsa/partials/modal.jsp',
				backdrop: 'static',
				controller: [
					'$scope',
					'$uibModalInstance',
					'cvLoc',
					'failoverService',
					function($scope, $modalInstance, cvLoc, failoverService) {
						$scope.modalOptions = modalOptions;
						$scope.modalOptions.closeButtonText = cvLoc('button.no');
						$scope.modalOptions.actionButtonText = cvLoc('button.yes');
						$scope.modalOptions.ok = function() {
							modalOptions
								.promise(...modalOptions.params)
								.success(function(data) {
									if (data && data.jobIds && data.jobIds.length > 0) {
										var jobID = data.jobIds[0];
										var jobURL = '#/jobs/' + jobID;
										var responseContent = ' <a href="' + jobURL + '">' + cvLoc('msg.jobDetails') + ' ' + jobID + '</a>';
										showSuccessMessage(responseContent);
									} else {
										showErrorMessage();
									}
									$modalInstance.dismiss();
								})
								.error(function(e) {
									showErrorMessage(e);
									$modalInstance.dismiss();
								});
						};
						$scope.modalOptions.close = function() {
							$modalInstance.dismiss();
						};
					}
				]
			});
		}

		/*
		 * Checks for failover eligibility on the list of machines Even if one of the machines is eligible to
		 * failover, allow failover on the entire app
		 */
		function isFailoverAllowed(machines) {
			machines.forEach(function(machine) {
				if (isFailoverAllowedOnMachine(machine)) {
					factory.failoverAllowed = true;
				}
			});
			return factory.failoverAllowed;
		}

		/*
		 * Checks for failback eligibility on the list of machines Even if one of the machines is eligible to
		 * failback, allow failback on the entire app
		 */
		function isFailbackAllowed(machines) {
			machines.forEach(function(machine) {
				if (isFailbackAllowedOnMachine(machine)) {
					factory.failbackAllowed = true;
				}
			});

			return factory.failbackAllowed;
		}

		/*
		 * Checks for failover eligibility on the machines
		 */
		function isFailoverAllowedOnMachine(machine) {
			/*
			 * VSAReplicationStatus.VSAREP_COMPLETE = 1 VSAReplicationStatus.VSAREP_PENDING = 2
			 */
			if (machine.syncStatus != '1' && machine.syncStatus != '2') {
				return false;
			}

			/*
			 * Allow failover only if it not already failed over or failover/ failback is not in
			 * progress/paused
			 *
			 * VSAFailoverStatus.VSAREP_NONE = 0; VSAFailoverStatus.VSAREP_FAILOVER_FAILED = 4
			 * VSAFailoverStatus.VSAREP_FAILBACK_COMPLETE = 5
			 *
			 */
			if (machine.failoverStatus != '0' && machine.failoverStatus != '4' && machine.failoverStatus != '5') {
				return false;
			}

			return true;
		}

		/*
		 * Checks for failback eligibility on the machine
		 */
		function isFailbackAllowedOnMachine(machine) {
			/*
			 * Allow fail back only if it not already failed back
			 *
			 * VSAFailoverStatus.VSAREP_FAILOVER_COMPLETE = 1 VSAFailoverStatus.VSAREP_FAILBACK_FAILED = 8
			 */
			if (machine.failoverStatus != '1' && machine.failoverStatus != '8') {
				return false;
			}

			/*
			 * TO-DO Add an additional check for Snap backup live-sync. Waiting for backend change
			 */

			return true;
		}

		function isTestbootAllowedOnMachine(machine) {
			return isFailoverAllowedOnMachine(machine) && isTestbootSupportedVendor(machine.destVendor);
		}

		function isTestbootSupportedVendor(vendor) {
			return vendor && testbootSupportedVendors.indexOf(vendor) >= 0;
		}

		function areMachinesFromSameSchedule(machines) {
			return machines.every(function(obj, idx, arr) {
				return obj.scheduleName.toLowerCase() === arr[0].scheduleName.toLowerCase();
			});
		}

		/* ===== Helper Functions ==== */

		/**
		 * Shows the cvToaster success message
		 */
		function showSuccessMessage(content) {
			cvToaster.showSuccessMessage({
				ttl: toasterTTL,
				message: content
			});
		}

		/**
		 * Shows the cvToaster error message
		 */
		function showErrorMessage(content) {
			cvToaster.showErrorMessage({
				ttl: toasterTTL,
				message: content ? content : cvLoc('generic_error')
			});
		}
	}
]);

export default failoverMod;
