import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';
import 'modules/disasterRecovery/js/controllers/basicDetails.ctrl.js';
import 'modules/disasterRecovery/js/controllers/machineDetails.ctrl.js';

import { drAppFailoverModule } from 'common/js/modules';

/**
 * Contains the library of CRUD functions used in the DR Group
 */

var failoverMod = drAppFailoverModule;

failoverMod.factory('drCRUDFactory', [
	'$state',
	'$stateParams',
	'$uibModal',
	'$dialogs',
	'cvToaster',
	'cvLoc',
	'failoverService',
	'FailoverConstants',
	function drCRDUFactory(
		$state,
		$stateParams,
		$uibModal,
		$dialogs,
		cvToaster,
		cvLoc,
		failoverService,
		FailoverConstants
	) {
		var factory = {
			// public functions
			addMonitoringMachines: addMonitoringMachines,
			deleteGroup: deleteGroup,
			getDevtestGroups: getDevtestGroups,
			getFailoverGroupDetails: getFailoverGroupDetails,
			getFailoverGroups: getFailoverGroups,
			modifyBasicDetails: modifyBasicDetails,
			modifyMachineDetails: modifyMachineDetails,
			updateFailoverGroup: updateFailoverGroup,

			// public variables
			failoverGroupDetails: [],
			failoverGroups: [],
			machine: [], // refers the current machine being edited
			vApp: [] // refers the current vApp being edited
		};

		// private variables
		var toasterTTL = 10000;

		return factory;

		function addMonitoringMachines(vApp) {
			$uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/autoFailover.jsp',
				backdrop: 'static',
				controller: [
					'$uibModalInstance',
					'vApp',
					function($uibModalInstance, vApp) {
						factory.vApp = vApp;
						factory.closeModal = closeModal($uibModalInstance);
					}
				],
				resolve: {
					vApp: function() {
						return vApp;
					}
				}
			});
		}

		function closeModal(modal) {
			modal.close();
		}

		function deleteGroup(vAppId, vAppName, transitionTo) {
			var callBackFunctions = {
				noFunction: function() {},
				yesFunction: function() {
					failoverService
						.deleteFailoverGroup(JSON.stringify(vAppId))
						.success(function(data) {
							$state.transitionTo(transitionTo || 'failover', $stateParams, {
								reload: true,
								inherit: false,
								notify: true
							});
							cvToaster.showSuccessMessage({
								ttl: toasterTTL,
								message: cvLoc('msg.groupDeleted', '<b>' + vAppName + '</b>')
							});
						})
						.error(function(e) {
							cvToaster.showErrorMessage({
								ttl: toasterTTL,
								message: e ? e : cvLoc('generic_error')
							});
						});
				}
			};

			$dialogs.confirm(
				cvLoc('label.deleteGroup'),
				cvLoc('label.deleteGroupConfirmation', '<b>' + vAppName + '</b>'),
				callBackFunctions
			);
		}

		function getFailoverGroupDetails(vAppId) {
			return failoverService
				.getFailoverGroupDetails(vAppId)
				.success(function(data) {
					if (data && data.length > 0) {
						factory.failoverGroupDetails = data[0];
					}
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: toasterTTL,
						message: e ? e : cvLoc('generic_error')
					});
				});
		}

		/**
		 * Fetches the list of all failover groups and adds the count of the machines available in the group
		 */
		function getFailoverGroups() {
			return failoverService
				.getFailoverGroups()
				.success(function(data) {
					for (var i = 0; i < data.length; i++) {
						data[i].machinesCount = data[i].clientList && data[i].clientList.length ? data[i].clientList.length : 0;
						data[i].isFailoverGroup = true;
						data[i].isSourceReplication = data[i].source.toUpperCase() === FailoverConstants.VAppSource['REPLICATION'];
						data[i].isSourceBackup = data[i].source.toUpperCase() === FailoverConstants.VAppSource['BACKUP'];
						data[i].isSourceArray =
							data[i].replicationType === FailoverConstants.DRReplicationType.SnapArrayReplication;
						data[i].isSourceArrayFailover =
							data[i].isSourceReplication &&
							data[i].isSourceArray &&
							data[i].operationType.toUpperCase() === FailoverConstants.VAppOperation.FAILOVER;
						data[i].isSourceArrayTestFailover =
							data[i].isSourceReplication &&
							data[i].isSourceArray &&
							data[i].operationType.toUpperCase() === FailoverConstants.VAppOperation.TESTFAILOVER;
						if (data[i].isSourceBackup) {
							data[i].sourceString = cvLoc('label.source.backup');
						}
						if (data[i].isSourceReplication) {
							data[i].sourceString = cvLoc('label.source.replication');
						}
						if (data[i].isSourceArrayFailover) {
							data[i].sourceString = `${cvLoc('label.source.arrayreplication')} - ${cvLoc('label.failover')}`;
						}
						if (data[i].isSourceArrayTestFailover) {
							data[i].sourceString = `${cvLoc('label.source.arrayreplication')} - ${cvLoc('label.testFailover')}`;
						}
					}
					factory.failoverGroups = data;
					return data || [];
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: toasterTTL,
						message: e ? e : cvLoc('generic_error')
					});
				});
		}

		function getDevtestGroups() {
			return failoverService
				.getDevtestGroups()
				.success(function(data) {
					for (var i = 0; i < data.length; i++) {
						data[i].machinesCount = data[i].clientList && data[i].clientList.length ? data[i].clientList.length : 0;
						data[i].isDevtestGroup = true;
						data[i].isSourceReplication = data[i].source.toUpperCase() === FailoverConstants.VAppSource['REPLICATION'];
						data[i].isSourceBackup = data[i].source.toUpperCase() === FailoverConstants.VAppSource['BACKUP'];
					}
					factory.devtestGroups = data;
					return data || [];
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: toasterTTL,
						message: e ? e : cvLoc('generic_error')
					});
				});
		}

		function modifyBasicDetails(vApp) {
			$uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/basicDetailsDialog.jsp',
				backdrop: 'static',
				controllerAs: 'basicDetailsController',
				controller: 'basicDetailsController',
				resolve: {
					vApp: function() {
						return vApp;
					}
				}
			});
		}

		function modifyMachineDetails(machine, vApp) {
			$uibModal.open({
				templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/machineDetailsDialog.jsp',
				backdrop: 'static',
				controllerAs: 'machineDetailsController',
				controller: 'machineDetailsController',
				resolve: {
					machine: function() {
						return machine;
					},
					vApp: function() {
						return vApp;
					}
				}
			});
		}

		function updateFailoverGroup(updatedvApp) {
			return failoverService
				.updateFailoverGroup({
					virtualApp: JSON.stringify(updatedvApp)
				})
				.success(function(data) {
					factory.failoverGroupDetails = updatedvApp;
					cvToaster.showSuccessMessage({
						ttl: toasterTTL,
						message: cvLoc('msg.updateSuccessful')
					});
					$state.reload();
				})
				.error(function(e) {
					cvToaster.showErrorMessage({
						ttl: toasterTTL,
						message: e ? e : cvLoc('generic_error')
					});
				});
		}
	}
]);

export default failoverMod;
