import 'modules/servers/js/services/servers.svc.js';

import { drAppFailoverModule } from 'common/js/modules';

const AnyNetworkLabel = 'Any Network';
const AnyNetwork = '*';

class DRNetwork {
	constructor() {
		this.restrict = 'E';
		this.templateUrl = appUtil.appRoot + 'modules/disasterRecovery/partials/drNetwork.jsp';
		this.scope = {
			server: '=',
			sourceClientId: '=',
			destinationClientId: '=',
			ctrl: '='
		};
		this.controllerAs = 'drNetworkCtrl';
		this.controller = DRNetworkController;
		this.bindToController = true;
	}
}

class DRNetworkController {
	constructor($modal, cvLoc, serverService) {
		this.$modal = $modal;
		this.cvLoc = cvLoc;
		this.serverService = serverService;
	}

	addNetwork() {
		this.errors = {};
		if (!this._isValid()) {
			return;
		}
		let nic = {
			sourceNetworkLabel: this.sourceNetwork[0].networkName,
			sourceNetwork: this._isAnyNetworkLabel(this.sourceNetwork[0].networkName)
				? AnyNetwork
				: this.sourceNetwork[0].networkName,
			sourceNetworkId: this._isAnyNetworkLabel(this.sourceNetwork[0].networkName)
				? AnyNetwork
				: this.sourceNetwork[0].networkId,
			destinationNetwork: this.destinationNetwork[0].networkName,
			destinationNetworkLabel: this.destinationNetwork[0].networkName,
			id: _.size(this.server.nics) + 1
		};
		this.server.nics.push(nic);
	}

	deleteNetworkRow(_rowValue) {
		var index = this.server.nics.findIndex(tuple => tuple.id === _rowValue.id);
		this.server.nics.splice(index, 1);
	}

	$onInit() {
		this._initVariables();
		this._initCtrl();
		this._fetchSourceNetworks();
		this._fetchDestinationNetworks();
	}

	_initVariables() {
		let self = this;
		this.errors = {};
		this.sourceNetworks = [this._getAnyNetwork()]; // contains the list of all the source networks
		this.sourceNetwork = []; // contains the currently selected source network
		this.destinationNetworks = []; // contains the list of all the destination networks
		this.destinationNetwork = []; // contains the currently selected destination network
		this.server = this.server || {}; // contains the currently selected ESX server and it's NICs, if any
		let id = 1;
		this.server.nics.forEach((nic, index) => {
			nic.sourceNetworkLabel = self._isAnyNetwork(nic.sourceNetwork) ? AnyNetworkLabel : nic.sourceNetwork;
			nic.destinationNetworkLabel = nic.destinationNetwork;
			nic.id = index;
		});
	}

	_initCtrl() {
		this.ctrl = this.ctrl || {};
		this.ctrl.fetchSourceNetworks = this._fetchSourceNetworks.bind(this);
		this.ctrl.fetchDestinationNetworks = this._fetchDestinationNetworks.bind(this);
	}

	_getAnyNetwork() {
		return {
			networkName: AnyNetworkLabel
		};
	}

	_isAnyNetworkLabel(networkLabel) {
		return networkLabel === AnyNetworkLabel;
	}

	_isAnyNetwork(network) {
		return network === AnyNetwork;
	}

	_fetchSourceNetworks() {
		let self = this;
		let sourceESX = _.get(this, 'server.sourceServer');
		if (!sourceESX) {
			// TO-DO show error
			return Promise.resolve();
		}

		this._resetSourceNetworks();
		this.errors.sourceNetwork = '';
		this.sourceNetworks = [this._loadingISteven('networkName')];
		return this._fetchNetwork(this.sourceClientId, sourceESX)
			.then(function(data) {
				self.sourceNetworks = [self._getAnyNetwork()];
				self.sourceNetworks = self.sourceNetworks.concat(_.get(data, 'data', []));
				self._setDefaultSourceNetwork();
			})
			.catch(function(e) {
				self.errors.sourceNetwork = self.cvLoc('error.networks');
				self._resetSourceNetworks();
			});
	}

	_fetchDestinationNetworks() {
		let self = this;
		let destinationESX = _.get(this, 'server.destinationServer');
		if (!destinationESX) {
			// TO-DO show error
			return Promise.resolve();
		}

		this.errors.destinationNetwork = '';
		this._resetDestinationNetworks();
		this.destinationNetworks = [this._loadingISteven('networkName')];
		return this._fetchNetwork(this.destinationClientId, destinationESX)
			.then(function(data) {
				self.destinationNetworks = _.get(data, 'data', []);
				self._setDefaultDestinationNetwork();
			})
			.catch(function(e) {
				self.errors.destinationNetwork = self.cvLoc('error.networks');
				self._resetDestinationNetworks();
			});
	}

	_fetchNetwork(clientId, esx) {
		let vendor = 'VMW';
		let serverDetails = {
			clientId: clientId
		};
		return this.serverService.getNetworkInfo(serverDetails, esx, vendor);
	}

	_setDefaultSourceNetwork() {
		this._setDefault(this.sourceNetworks, this.sourceNetwork);
	}

	_setDefaultDestinationNetwork() {
		this._setDefault(this.destinationNetworks, this.destinationNetwork);
	}

	_setDefault(inputModel, outputModel) {
		if (inputModel.length > 0) {
			inputModel[0].selected = true;
			outputModel[0] = inputModel[0];
		}
	}

	_isValid() {
		let isValid = true;
		if (_.isEmpty(this.sourceNetwork) || _.isEmpty(this.destinationNetwork)) {
			this.errors.destinationNetwork = this.cvLoc('error.networkSelection');
			isValid = false;
		} else if (this._isSourceDuplicate()) {
			this.errors.sourceNetwork = this.cvLoc('error.network.source.duplicate');
			isValid = false;
		} else if (this._isNetworkLoading()) {
			isValid = false;
		}

		return isValid;
	}

	/*
	 * One to many mapping of the source to destination is not allowed
	 */
	_isSourceDuplicate() {
		let self = this;
		let duplicateEntryIndex = this.server.nics.findIndex(nic => {
			return nic.sourceNetworkLabel === _.get(self, 'sourceNetwork[0].networkName');
		});

		return duplicateEntryIndex >= 0;
	}

	_isDuplicatePair() {
		let self = this;
		let duplicateEntryIndex = this.server.nics.findIndex(nic => {
			return (
				nic.sourceNetworkLabel === _.get(self, 'sourceNetwork[0].networkName') &&
				nic.destinationNetworkLabel === _.get(self, 'destinationNetwork[0].networkName')
			);
		});

		return duplicateEntryIndex >= 0;
	}

	_isNetworkLoading() {
		return _.get(this, 'sourceNetworks[0].id') === -1 || _.get(this, 'destinationNetworks[0].id') === -1;
	}

	_resetSourceNetworks() {
		this.sourceNetworks = [];
		this.sourceNetwork = [];
	}

	_resetDestinationNetworks() {
		this.destinationNetworks = [];
		this.destinationNetwork = [];
	}

	_loadingISteven(name) {
		let loading = {
			id: -1,
			selected: true,
			disableSelection: true // disable from using this as a value to save
		};
		loading[name] = this.cvLoc('Loading');

		return loading;
	}
}

DRNetworkController.$inject = ['$uibModal', 'cvLoc', 'serverService'];
drAppFailoverModule.directive('drNetwork', () => new DRNetwork());

export default drAppFailoverModule;
