import 'modules/disasterRecovery/js/controllers/drArrayMachinesSelect.ctrl.js';
import 'modules/disasterRecovery/js/controllers/drConfigureArrayMachines.ctrl.js';
import 'modules/disasterRecovery/js/controllers/drPriority.ctrl.js';
import 'modules/disasterRecovery/js/directives/dr-array-esx.directive.js';
import 'modules/disasterRecovery/js/directives/dr-hypervisors.directive.js';
import 'modules/disasterRecovery/js/directives/dr-recovery-target.directive.js';
import 'modules/disasterRecovery/js/dr.constants.js';
import 'modules/disasterRecovery/js/factory/dr.factory.js';
import 'modules/disasterRecovery/js/factory/drArrayReplication.factory.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';
import 'modules/vmManagement/js/factory/virtualMachine.factory.js';
import 'vsa/js/constants/servers.constants.js';
import 'vsa/js/factories/replication.factory.js';

import { drAppFailoverModule } from 'common/js/modules';

class DRArrayReplication {
	constructor() {
		this.restrict = 'E';
		this.templateUrl = appUtil.appRoot + 'modules/disasterRecovery/partials/drArrayReplication.jsp';
		this.scope = {
			ctrl: '=',
			vApp: '=',
			vAppOperation: '=' // FailoverConstants.VAppOperation
		};
		this.require = ['^^cvSubmit', '^^form'];
		this.controllerAs = 'drArrayCtrl';
		this.controller = DRArrayReplicationController;
		this.bindToController = true;
	}

	link($scope, elem, attrs, ctrl) {
		$scope.submitController = ctrl[0];
		$scope.formController = ctrl[1];
	}
}

class DRArrayReplicationController {
	constructor(
		$modal,
		cvLoc,
		cvToaster,
		failoverService,
		arrayReplicationFactory,
		drMachinesFactory,
		drOrchestrationFactory,
		serverService,
		VSA_VENDOR,
		VirtualMachineConstants,
		FailoverConstants,
		FailoverEnum
	) {
		this.$modal = $modal;
		this.cvLoc = cvLoc;
		this.cvToaster = cvToaster;
		this.failoverService = failoverService;
		this.arrayReplicationFactory = arrayReplicationFactory;
		this.drMachinesFactory = drMachinesFactory;
		this.drOrchestrationFactory = drOrchestrationFactory;
		this.serverService = serverService;
		this.VSA_VENDOR = VSA_VENDOR;
		this.VMAllocationPolicyType = VirtualMachineConstants.VMAllocationPolicyType;
		this.FailoverConstants = FailoverConstants;
		this.FailoverEnum = FailoverEnum;
	}

	onHypervisorSelect(serversSelected) {
		let self = this;
		this.failoverService.cancelAllPendingRequests();

		if (_.get(this, 'vApp.arrayReplication.source.clientId') !== serversSelected[0].clientId) {
			this.showESXServers && this._reset('esx');
		}
		this.vApp.arrayReplication.source = {
			clientId: serversSelected[0].clientId,
			clientName: serversSelected[0].clientName,
			hostName: serversSelected[0].hostName,
			displayName: serversSelected[0].name
		};

		if (this.isFailoverGroup || this.isDevtestGroup) {
			this.onFailoverSelect();
		} else if (this.isTestFailoverGroup) {
			this.onTestFailoverSelect();
		}
	}

	onFailoverChange() {
		this.failoverService.cancelAllPendingRequests();
		this.showESXServers && this._reset('esx');
		this.onFailoverSelect();
	}

	onFailoverSelect() {
		let self = this;
		let serverId = this._getHypervisorId();
		this._initShowVariables();
		this._initVAppMachines(this.vApp);
		this._getVMGroups(serverId).then(function() {
			if (!_.isEmpty(self.vmGroups)) {
				self._setDefaultVMGroup();
				self.onVMGroupsSelect();
			}
		});
	}

	onTestFailoverChange() {
		this.failoverService.cancelAllPendingRequests();
		this.showESXServers && this._reset('esx');
		this.onTestFailoverSelect();
	}

	onTestFailoverSelect() {
		let self = this;
		this._initShowVariables();
		this._initVAppMachines(this.vApp);
		this._reset('copy');
		this._getCopy().then(function() {
			if (!_.isEmpty(self.copy)) {
				self.copy.forEach(c => (c.copyLabel = `${c.storagePolicyName} / ${c.copyName}`));
				self._setDefaultCopy();
				self.onCopySelect();
			}
		});
	}

	onVMGroupsSelect() {
		let self = this;
		this.vApp.arrayReplication.subclient = {
			subclientId: this.vmGroupsSelected[0].subClientEntity.subclientId,
			subclientName: this.vmGroupsSelected[0].subClientEntity.subclientName,
			subclientGUID: this.vmGroupsSelected[0].subClientEntity.subclientGUID
		};
		this._getCopy().then(function() {
			if (!_.isEmpty(self.copy)) {
				self.copy.forEach(c => (c.copyLabel = `${c.storagePolicyName} - ${c.copyName}`));
				self._setDefaultCopy();
				self.onCopySelect();
			}
		});
	}

	onCopySelect() {
		const self = this;
		this.vApp.arrayReplication.copyId = this.copySelected[0].copyId;
		this.vApp.arrayReplication.copyName = this.copySelected[0].copyName;
		this.vApp.arrayReplication.copy = this.vApp.arrayReplication.copy || {};
		// copy the storage policy id for displaying the storage policy name in the details page.
		this.vApp.arrayReplication.copy.storagePolicyId = this.copySelected[0].storagePolicyId;
		this.vApp.arrayReplication.copy.storagePolicyName = this.copySelected[0].storagePolicyName;
		if (this.isFailoverGroup) {
			this._getMachinesFailover();
		} else if (this.showDatastores) {
			this._getMachines().then(function(datastores) {
				self.onDatastoreSelect(datastores);
			});
		} else if (this.isTestFailoverGroup) {
			this._getMachinesTestFailover();
		} else {
			this._getMachines();
		}
	}

	/*
	 * For Failover groups, select all the datastores and all the VM's under it
	 */
	onDatastoreSelect(datastores) {
		const self = this;
		this.vApp.arrayReplication.datastore = [];
		datastores.forEach(datastore => {
			self.vApp.arrayReplication.datastore.push({
				source: datastore.datastore
			});
		});
	}

	onDestinationHyperVSelect() {
		let destinationHyperV = _.get(this, 'destinationHyperVSelected[0]', {});
		if (_.get(this, 'vApp.arrayReplication.destination.clientId') !== destinationHyperV.clientId) {
			this.showESXServers && this._reset('esx');
		}
		this.vApp.arrayReplication = this.vApp.arrayReplication || [];
		this.vApp.arrayReplication.destination = {
			clientId: destinationHyperV.clientId,
			clientName: destinationHyperV.clientName,
			hostName: destinationHyperV.hostName
		};
	}

	onTargetSelect(recoveryTargetSelected) {
		if (
			_.get(this, 'vApp.arrayReplication.destination.clientId') !==
			_.get(recoveryTargetSelected, 'destinationHyperV.clientId')
		) {
			this.showESXServers && this._reset('esx');
		}
		this.vApp.arrayReplication = this.vApp.arrayReplication || [];
		this.vApp.arrayReplication.destination = _.get(recoveryTargetSelected, 'destinationHyperV', {});
	}

	disableCopy() {
		if (this.isFailoverGroup || this.isDevtestGroup) {
			return !this.vmGroupsSelected || this.vmGroupsSelected.length === 0;
		} else if (this.isTestFailoverGroup) {
			return false;
		}
	}

	$onInit() {
		this.editMode = _.has(this, 'vApp.operationType');
		this.vApp = this._initVApp(this.vApp);
		this.errors = {};
		this.ctrl = this._initCtrl();
		this._initShowVariables();

		this.supportedPolicyTypes = [
			this.VMAllocationPolicyType.VMWARE,
			this.VMAllocationPolicyType.VMW_LIVEMOUNT,
			this.VMAllocationPolicyType.VMW_LABTEMPLATE,
			this.VMAllocationPolicyType.VMW_BACKUP_LABTEMPLATE
		];
		this.supportedHypervisorVendors = [this.VSA_VENDOR.VMWARE];
		if (this.showDestinationHyperV) {
			this._initDestinationHyperV();
		}
	}

	_initVApp(vApp) {
		vApp = vApp || {};
		vApp.operationType = this._getVAppOperationType(vApp.operationType);
		vApp.arrayReplication = vApp.arrayReplication || {};
		vApp.arrayReplication.server = vApp.arrayReplication.server || [];
		vApp.clientList = vApp.clientList || [];
		vApp.config = vApp.config || {};
		this._initVAppMachines(vApp);

		return vApp;
	}

	_initVAppMachines(vApp) {
		vApp.machines = vApp.machines || {};
		let gridOptions = {};
		if (this._isFailoverGroup()) {
			gridOptions.tableName = 'drArrayFailoverMachines';
			gridOptions.columnsTemplate = this.arrayReplicationFactory.getFailoverColumns();
			gridOptions.actionMenu = [this._getAssignPriorityAction(), this._getConfigureIPAction()];
			gridOptions.gridToolbarMenu = [this._getSelectAllMenu(), this._getSelectNoneMenu()];
			gridOptions.enableCheckBoxColumn = true;
		} else if (this._isTestFailoverGroup()) {
			gridOptions.tableName = 'drArrayTestFailoverMachines';
			gridOptions.columnsTemplate = this.arrayReplicationFactory.getFailoverColumns();
			gridOptions.actionMenu = [
				this._getAssignPriorityAction(),
				this._getConfigureIPAction(),
				this._getRemoveMachinesAction()
			];
			gridOptions.gridToolbarMenu = [this._getAddMachinesMenu(), this._getSelectAllMenu(), this._getSelectNoneMenu()];
			gridOptions.enableCheckBoxColumn = true;
			gridOptions.gridEmptyMessage = this.cvLoc('msg.machines.addButton');
		} else if (this._isDevtestGroup()) {
			gridOptions.tableName = 'drArrayDevtestMachines';
			gridOptions.columnsTemplate = this.arrayReplicationFactory.getDevtestColumnsTemplate();
			gridOptions.enableCheckBoxColumn = true;
		}
		vApp.machines.gridOptions = gridOptions;
	}

	/*
	 * Initializes all the variables that handle displaying the fields on the UI
	 */
	_initShowVariables() {
		this.isFailoverGroup = this._isFailoverGroup();
		this.isTestFailoverGroup = this._isTestFailoverGroup();
		this.isDevtestGroup = this._isDevtestGroup();
		this.showOperationTypeSource = this.isFailoverGroup || this.isTestFailoverGroup;
		this.showVMGroups = this.isFailoverGroup || this.isDevtestGroup;
		this.showDatastores = this.isFailoverGroup;
		this.showESXServers = this.isFailoverGroup || this.isTestFailoverGroup;
		this.showDestinationHyperV = this.isFailoverGroup || this.isTestFailoverGroup;
		this.showRecoveryTargets = this.isDevtestGroup;
	}

	_initCtrl() {
		return {
			isArrayReplicationValid: this._isValid,
			isVMGroupsValid: this._isVMGroupsValid,
			isCopyValid: this._isCopyValid,
			isFailoverGroup: this._isFailoverGroup,
			isTestFailoverGroup: this._isTestFailoverGroup,
			isDevtestGroup: this._isDevtestGroup,
			FailoverEnum: this.FailoverEnum,
			onSubmitArrayReplication: this._onSubmit.bind(this)
		};
	}

	_initDestinationHyperV() {
		let self = this;
		this._getDestinationHypervisors().then(function(data) {
			self.destinationHyperV = data || [];
			self._setDefaultDestinationHypervisor();
			self.onDestinationHyperVSelect();
		});
	}

	_getVMGroups(serverId) {
		const self = this;
		const options = {
			fl: 'subClientProperties.subClientEntity,subClientProperties.vsaSubclientProp,subClientProperties.planEntity'
		};

		this._reset('vmGroups');
		this.vmGroups.push(this._loadingISteven('subclientName'));

		return this.serverService
			.getServerCachedSubclientList(serverId, options)
			.success(function(data) {
				self.vmGroups = _.flatMap(data.subclients, sc => {
					let v = sc.sc;
					v.subclientName = v.subClientEntity.subclientName;
					return [v];
				});
			})
			.error(function(e) {
				self.vmGroups = [];
				self.errors.fetchingVMGroups = true;
			});
	}

	_getCopy() {
		this._reset('copy');
		this.copy.push(this._loadingISteven('copyLabel'));

		if (this.isTestFailoverGroup) {
			let serverId = this._getHypervisorId();
			return this._getCopyForTestFailoverGroup(serverId);
		} else if (this.isFailoverGroup) {
			const subclientId = _.get(this, 'vmGroupsSelected[0].subClientEntity.subclientId', null);
			return this._getCopyForFailoverGroup(subclientId);
		}
	}

	_getCopyForFailoverGroup(subclientId) {
		const self = this;
		return this.arrayReplicationFactory
			.getFailoverCopies(subclientId)
			.then(function(data) {
				self.copyMap = data || new Map();
				self.copy = [];
				self.copyMap.forEach((value, key, map) => {
					let copy = {};
					copy = _.get(value, '[0].copy', {});
					self.copy.push(copy);
				});
				if (_.isEmpty(self.copy)) {
					throw new Error(self.cvLoc('error.copyNA'));
				}
			})
			.catch(function(e) {
				self.copy = [];
				self.errors.copyMsg = _.get(e, 'message', self.cvLoc('error.fetchingCopy'));
			});
	}

	_getCopyForTestFailoverGroup(serverId) {
		const self = this;
		return this.arrayReplicationFactory
			.getTestFailoverCopies(serverId)
			.then(function(data) {
				self.copyMap = data || new Map();
				self.copy = [];
				self.copyMap.forEach((value, key, map) => {
					let copy = {};
					copy = _.get(value, '[0].copy', {});
					self.copy.push(copy);
				});
				if (_.isEmpty(self.copy)) {
					throw new Error(self.cvLoc('error.copyNA'));
				}
			})
			.catch(function(e) {
				self.copy = [];
				self.errors.copyMsg = _.get(e, 'message', self.cvLoc('error.fetchingCopy'));
			});
	}

	_getDatastores() {
		const self = this;
		const subclientId = _.get(this, 'vmGroupsSelected[0].subClientEntity.subclientId', null);
		const copyId = _.get(this, 'copySelected[0].copyId', null);

		this.vApp.machines.isLoading = false;
		return this.arrayReplicationFactory.getArrayDatastores(subclientId, copyId).then(function(data) {
			self.vApp.machines.isLoading = false;
			data = data || [];
			self.arrayReplicationFactory.addValuesForEdit(data, self.vApp);
			self.vApp.machines.machinesList = data;
			self._refreshMachinesGrid();
			return data;
		});
	}

	_getDestinationHypervisors() {
		let self = this;
		if (_.get(this, 'drOrchestrationFactory.hypervisors', []).length > 0) {
			let hypervisors = filterHypervisors(this.drOrchestrationFactory.hypervisors, this.supportedHypervisorVendors);
			return Promise.resolve(hypervisors);
		} else {
			return this.drOrchestrationFactory.getHyperVisors().then(function() {
				let hypervisors = filterHypervisors(self.drOrchestrationFactory.hypervisors, self.supportedHypervisorVendors);
				return hypervisors;
			});
		}

		/*
		 * Copies the hypervisors locally and filters out to the supported vendors
		 */
		function filterHypervisors(hypervisors, supportedHypervisorVendors) {
			let filteredHypervisors = angular.copy(hypervisors);
			filteredHypervisors = filteredHypervisors.filter(
				hypervisor => supportedHypervisorVendors.indexOf(hypervisor.type) >= 0
			);
			return filteredHypervisors;
		}
	}

	_getMachines() {
		const self = this;
		const subclientId = _.get(this, 'vmGroupsSelected[0].subClientEntity.subclientId', null);
		const copyId = _.get(this, 'copySelected[0].copyId', null);

		this.vApp.machines.isLoading = true;
		return this.arrayReplicationFactory.getArrayMachines(subclientId, copyId, 0).then(function(data) {
			data = data || [];
			self.arrayReplicationFactory.addValuesForEdit(data, self.vApp);
			self.vApp.machines.isLoading = false;
			self.vApp.machines.machinesList = data;
			self._refreshMachinesGrid();

			return self.vApp.machines.machinesList;
		});
	}

	/*
	 * The machines are already fetched along with copy information.
	 */
	_getMachinesFailover() {
		this._reset('machines');
		let copyId = _.get(this, 'copySelected[0].copyId', 0);
		let machinesList = this.copyMap.get(copyId);
		this.arrayReplicationFactory.addValuesForEdit(machinesList, this.vApp);
		this.vApp.machines.isLoading = false;
		if (this.editMode) {
			let existingMachines = this.vApp.clientList || [];
			let allMachines = machinesList || [];
			this.vApp.machines.machinesList = _.intersectionWith(allMachines, existingMachines, this._isEqualMachines);
		} else {
			this.vApp.machines.machinesList = machinesList;
		}
		this._refreshMachinesGrid();
	}

	/*
	 * In case of test failover, the machines are already fetched along with copy information.
	 */
	_getMachinesTestFailover() {
		this._reset('machines');
		let copyId = _.get(this, 'copySelected[0].copyId', 0);
		let machinesList = this.copyMap.get(copyId);
		this.arrayReplicationFactory.addValuesForEdit(machinesList, this.vApp);
		this.vApp.machines.isLoading = false;
		this.vApp.machines.allMachines = machinesList;
		if (this.editMode) {
			let existingMachines = this.vApp.clientList || [];
			let allMachines = machinesList || [];
			this.vApp.machines.machinesList = _.intersectionWith(allMachines, existingMachines, this._isEqualMachines);
			this._refreshMachinesGrid();
		}
	}

	_setDefaultVMGroup() {
		let indexToSelect = 0;
		let subclientId = _.get(this, 'vApp.arrayReplication.subclient.subclientId');
		if (subclientId) {
			indexToSelect = this.vmGroups.findIndex(subclient => subclient.subClientEntity.subclientId === subclientId);
			indexToSelect = Math.max(indexToSelect, 0);
		}
		this.vmGroups[indexToSelect].selected = true;
		this.vmGroupsSelected[0] = this.vmGroups[indexToSelect];
	}

	_setDefaultCopy() {
		let indexToSelect = 0;
		let copyId = _.get(this, 'vApp.arrayReplication.copyId');
		if (copyId) {
			indexToSelect = this.copy.findIndex(copy => copy.copyId === copyId);
			indexToSelect = Math.max(indexToSelect, 0);
		}
		this.copy[indexToSelect].selected = true;
		this.copySelected[0] = this.copy[indexToSelect];
	}

	_setDefaultDestinationHypervisor() {
		let indexToSelect = 0;
		let clientId = _.get(this, 'vApp.arrayReplication.destination.clientId');
		if (clientId) {
			indexToSelect = this.destinationHyperV.findIndex(server => server.clientId === clientId);
			indexToSelect = Math.max(indexToSelect, 0);
		}
		this.destinationHyperV[indexToSelect].selected = true;
		this.destinationHyperVSelected = [this.destinationHyperV[indexToSelect]];
	}

	/* Checks if the all the inputs are valid */
	_isValid(vApp) {
		let valid = true;
		valid = valid && this.isHypervisorValid(vApp.selectedEntities);
		valid = valid && this.isCopyValid(vApp.arrayReplication.copyId);
		valid = valid && this.areMachinesValid(vApp.machines);
		if (this.isFailoverGroup(vApp.operationType)) {
			valid = valid && this.isVMGroupsValid(vApp.arrayReplication.subclient);
			valid = valid && this.isESXValid(vApp.arrayReplication.server);
		} else if (this.isTestFailoverGroup(vApp.operationType)) {
			valid = valid && this.isESXValid(vApp.arrayReplication.server);
		} else if (this.isDevtestGroup(vApp.operationType)) {
			valid = valid && this.isRecoveryTargetValid(vApp.policy);
		}

		return valid;
	}

	_isVMGroupsValid(subclient) {
		return Boolean(subclient);
	}

	_isCopyValid(copy) {
		return copy && copy >= 0;
	}

	_onSubmit(vApp) {
		let self = this;
		vApp.clientList = [];
		let selectedMachines = [];
		if (this.isDevtestGroup) {
			selectedMachines = vApp.machines.grid.getSelectedRows().values;
		} else if (this.isFailoverGroup) {
			selectedMachines = vApp.machines.machinesList;
		} else if (this.isTestFailoverGroup) {
			selectedMachines = vApp.machines.machinesList;
		}
		vApp.config.vmGroups = [];
		selectedMachines.forEach(function(selectedMachine) {
			if (selectedMachine) {
				self._addMachinetoConfig(vApp, selectedMachine);
			}
		});
	}

	/* Adds the selectedMachine to the correct location in vApp */
	_addMachinetoConfig(vApp, selectedMachine) {
		selectedMachine.groupId = selectedMachine.priority;
		var machine = {};
		machine.clientId = selectedMachine.clientId;
		machine.clientName = selectedMachine.clientName;
		machine.clientGUID = selectedMachine.clientGUID;
		machine.entityId = selectedMachine.replicationId;
		vApp.clientList.push(machine);

		var vmSequence = {};
		var vmInfo = {};
		vmInfo.vmGUID = selectedMachine.clientGUID;
		vmInfo.vmName = selectedMachine.clientName;
		vmSequence.vmInfo = vmInfo;
		vmSequence.delay = 1;
		vmSequence.replicationId = selectedMachine.replicationId;
		vmSequence.vmStaticIPAddressOptions = _.isEmpty(_.get(selectedMachine, 'ip[0].sourceIP', ''))
			? null
			: selectedMachine.ip;
		vmSequence.destComputerName = selectedMachine.destComputerName;

		vApp.config.vmGroups = vApp.config.vmGroups || [];
		var added = false;
		vApp.config.vmGroups.forEach(function(group) {
			if (group.groupId == selectedMachine.groupId) {
				group.vmSequence.push(vmSequence);
				added = true;
			}
		});
		if (!added) {
			let vmGroup = {
				delay: vApp.delay,
				continueOnFailure: vApp.continueOnFailure,
				groupId: selectedMachine.groupId || 1,
				vmSequence: [vmSequence]
			};
			vApp.config.vmGroups.push(vmGroup);
		}
	}

	_getAddMachinesMenu() {
		return {
			id: 'addMachines',
			label: this.cvLoc('label.add'),
			onSelect: this._openAddMachinesModal.bind(this)
		};
	}

	_getSelectAllMenu() {
		return {
			id: 'selectAll',
			label: this.cvLoc('label.selectAll'),
			onSelect: this._selectAllMachines.bind(this)
		};
	}

	_getSelectNoneMenu() {
		return {
			id: 'selectNone',
			label: this.cvLoc('label.selectNone'),
			onSelect: this._clearGridSelection.bind(this)
		};
	}

	_getRemoveMachinesAction() {
		return {
			id: 'removeMachines',
			label: this.cvLoc('label.remove'),
			onSelect: this._removeMachinesFromGrid.bind(this)
		};
	}

	_getConfigureIPAction() {
		return {
			id: 'configureIP',
			label: `${this.cvLoc('label.configure')} ${this.cvLoc('label.ip')}`,
			onSelect: this._openConfigureIPModal.bind(this)
		};
	}

	_getAssignPriorityAction() {
		return {
			id: 'assignPriority',
			label: this.cvLoc('label.assignPriority'),
			onSelect: this._assignPriority.bind(this)
		};
	}

	_openAddMachinesModal() {
		let self = this;
		let selectModal = this.$modal.open({
			templateUrl: appUtil.appRoot + 'modules/disasterRecovery/partials/drArrayMachinesSelect.jsp',
			backdrop: 'static',
			controller: 'drArrayMachinesSelectController',
			controllerAs: 'drMachinesSelectCtrl',
			resolve: {
				machines: self._getSelectableMachines()
			}
		});
		selectModal.result.then(function(result) {
			self.vApp.machines.machinesList = self.vApp.machines.machinesList.concat(result || []);
			self._refreshMachinesGrid();
			self._clearGridSelection(); // this will avoid re-selecting the machines that have been removed and are being added again.
		});
	}

	_selectAllMachines() {
		this.drMachinesFactory.selectAllMachines(this.vApp.machines.grid.grid);
	}

	/*
	 * Returns the list of machines that can be added to the group.
	 * It removes the machines that are part of the group already.
	 */
	_getSelectableMachines() {
		let machines = _.cloneDeep(this.vApp.machines);
		let allMachines = machines.allMachines;
		let selectedMachines = machines.machinesList;
		machines.allMachines = _.differenceWith(allMachines, selectedMachines, this._isEqualMachines);

		return machines;
	}

	_removeMachinesFromGrid(rows) {
		let machinesToRemove = this._getSelectedRows(rows);
		if (machinesToRemove.length == 0) {
			return;
		}

		let selectedMachines = this.vApp.machines.machinesList;
		this.vApp.machines.machinesList = _.differenceWith(selectedMachines, machinesToRemove, this._isEqualMachines);
		this._refreshMachinesGrid();
	}

	_openConfigureIPModal(rows) {
		let selectedRow = {};
		let selectedRows = _.get(rows, 'selectedRowValues');
		selectedRows = selectedRows.filter(row => !_.isUndefined(row));
		if (selectedRows.length > 1) {
			/* Indicates the action was launched from the grid toolbar and multiple rows were selected */
			this._showErrorMessage(this.cvLoc('error.batchActionNotEligible'));
			return;
		} else if (selectedRows.length === 1) {
			/* Indicates the action was launched from the grid toolbar and only one row was selected */
			selectedRow = selectedRows[0];
		} else {
			/* Indicates the action was launched from row actions. */
			selectedRow = _.get(rows, 'rowValue', {});
		}

		let self = this;

		this.arrayReplicationFactory.configureIPAddress(selectedRow).then(function(result) {
			let machine = self.vApp.machines.machinesList.find(machine => machine.clientId === result.clientId);
			machine.ip = result.ip;
			machine.ipText =
				_.get(machine, 'ip', []).length > 0 ? self.cvLoc('label.configured') : self.cvLoc('label.notConfigured');
			machine.destComputerName = result.destComputerName;
			self._refreshMachinesGrid();
		});
	}

	_assignPriority(rows) {
		let selectedRows = this._getSelectedRows(rows);
		if (selectedRows.length == 0) {
			return;
		}

		let self = this;
		this.arrayReplicationFactory
			.assignPriority(
				_.get(
					_.minBy(selectedRows, row => row.priority),
					'priority',
					1
				)
			)
			.then(function(priority) {
				selectedRows.forEach(selectedMachine => {
					selectedMachine.priority = priority || 1;
					let machine = self.vApp.machines.machinesList.find(machine => machine.clientId === selectedMachine.clientId);
					machine.priority = selectedMachine.priority;
				});
				self._refreshMachinesGrid();
			});
	}

	_loadingISteven(name) {
		let loading = {
			id: 0,
			selected: true,
			disableSelection: true // disable from using this as a value to save
		};
		loading[name] = this.cvLoc('Loading');

		return loading;
	}

	_getHypervisorId() {
		return _.get(this, 'vApp.arrayReplication.source.clientId', 0);
	}

	/*
	 * In the edit page, the operation type is sent in the corresponding string value of the enum.
	 * In such cases, convert it to corresponding enum value.
	 */
	_getVAppOperationType(operationType) {
		if (_.isString(operationType)) {
			return this.FailoverEnum.VAppOperation[operationType.toUpperCase()];
		} else if (_.isNumber(operationType)) {
			return operationType;
		} else {
			return this.FailoverEnum.VAppOperation.TESTFAILOVER;
		}
	}

	_isFailoverGroup(operationType) {
		operationType = operationType || this.vApp.operationType;
		return this.FailoverEnum.VAppOperation.FAILOVER === operationType;
	}

	_isTestFailoverGroup(operationType) {
		operationType = operationType || this.vApp.operationType;
		return this.FailoverEnum.VAppOperation.TESTFAILOVER === operationType;
	}

	_isDevtestGroup(operationType) {
		operationType = operationType || this.vApp.operationType;
		return this.FailoverEnum.VAppOperation.LIVEMOUNT === operationType;
	}

	_refreshMachinesGrid() {
		this.vApp.machines.grid.refreshData();
		this._handleGridToolbarMenuVisibility();
	}

	_clearGridSelection() {
		this.drMachinesFactory.clearGridSelection(this.vApp.machines.grid.grid);
	}

	_handleGridToolbarMenuVisibility() {
		if (_.get(this, 'vApp.machines.machinesList', []).length > 0) {
			$('#toolbar-menu_selectAll').show();
			$('#toolbar-menu_selectNone').show();
		} else {
			$('#toolbar-menu_selectAll').hide();
			$('#toolbar-menu_selectNone').hide();
		}
	}

	_isEqualMachines(m1, m2) {
		return m1 && m2 && m1.clientId == m2.clientId;
	}

	_getSelectedRows(rows) {
		let selectedRows = _.get(rows, 'selectedRowValues');
		selectedRows = selectedRows.filter(row => !_.isUndefined(row));
		if (selectedRows.length == 0) {
			selectedRows = _.get(rows, 'rowValue', []);
		}

		return selectedRows;
	}

	_showErrorMessage(errorText) {
		this.cvToaster.showErrorMessage({
			ttl: 10000,
			message: errorText ? errorText : cvLoc('generic_error')
		});
	}

	/* resets the data of the inputs and the selected inputs to empty based on the level */
	_reset(level) {
		this.errors = {};
		if (level === 'vmGroups') {
			this.vmGroups = [];
			this.vmGroupsSelected = [];
			this._reset('copy');
		} else if (level === 'copy') {
			this.copy = [];
			this.copySelected = [];
			this.showDatastores ? this._reset('datastores') : this._reset('machines');
		} else if (level === 'datastores') {
			this.vApp.arrayReplication.datastore = [];
			this._refreshMachinesGrid();
			this._reset('machines');
		} else if (level === 'machines') {
			this.vApp.machines.allMachines = [];
			this.vApp.machines.machinesList = [];
			this._refreshMachinesGrid();
		} else if (level === 'esx') {
			this.vApp.arrayReplication.server = [];
		}
	}
}

DRArrayReplicationController.$inject = [
	'$uibModal',
	'cvLoc',
	'cvToaster',
	'failoverService',
	'arrayReplicationFactory',
	'drMachinesFactory',
	'drOrchestrationFactory',
	'serverService',
	'VSA_VENDOR',
	'VirtualMachineConstants',
	'FailoverConstants',
	'FailoverEnum'
];
drAppFailoverModule.directive('drArrayReplication', () => new DRArrayReplication());

export default drAppFailoverModule;
