import 'jobs/js/filters/elapsedTime.filter.js';
import 'modules/disasterRecovery/js/services/failoverGroup/failoverGroup.svc.js';
import * as columnsTemplate from 'modules/disasterRecovery/js/columns/drJobs.column.template.js';
import { drAppFailoverModule } from 'common/js/modules';

const supportedDRJobTypes = ['DR Orchestration', 'Virtual Machine Management'];

export class DRJobsController {
	constructor($stateParams, $scope, $filter, $document, cvToaster, cvLoc, cvUtil, failoverService) {
		this.$stateParams = $stateParams;
		this.$scope = $scope;
		this.$filter = $filter;
		this.$document = $document;
		this.cvToaster = cvToaster;
		this.cvLoc = cvLoc;
		this.cvUtil = cvUtil;
		this.failoverService = failoverService;
		this._init();
		this._watchJobDetails();
	}

	_init() {
		let self = this;
		this.drJobs = [];
		this.filterList = {
			clientNames: [],
			phases: [],
			phaseStatus: []
		};
		this.jobId = this.$stateParams['jobId'];
		this._getDRJobDetails().then(function() {
			self._setupGridOptions();
			self.showGrid = true;
		});
	}

	_watchJobDetails() {
		const self = this;
		this.$scope.$on('jobDetailsChanged', function(evt, job) {
			if (self._isSupportedDRJob(job)) {
				self._getDRJobDetails();
			}
		});
	}

	_isSupportedDRJob(job) {
		let operationType = _.get(job, 'jobDetail.generalInfo.operationType', '');
		return supportedDRJobTypes.includes(operationType);
	}

	_setupGridOptions() {
		let self = this;
		this._sortAndUniq(this.filterList);
		this.gridOptions = {
			columns: this._initColumnDefs(),
			gridTitle: this.cvLoc('label.phaseDetails'),
			tableName: 'drJobsTable',
			url: this._getData.bind(this),
			usePageToolbar: false,
			gridToolbarMenu: [],
			actionMenu: [],
			pageSize: 10,
			pageSizes: [5, 10, 15, 25, 50, 100, 250, 500, 1000],
			sortDirection: {
				field: 'endTimeString',
				dir: 'desc'
			},
			idField: 'startTime.time',
			beforeGridInitialize: ({ grid }) => {
				self.grid = grid;
			},
			detailGridOptions: this._initDetailGridOptions()
		};
	}

	_getData(options) {
		options.success(this.drJobs);
	}

	_initColumnDefs() {
		return columnsTemplate.getColumns(this.cvLoc, this.filterList);
	}

	_initDetailGridOptions() {
		return {
			columns: this._initDetailsColumnDefs(),
			gridTitle: this.cvLoc('label.machineDetails'),
			tableName: 'drJobsDetailsTable',
			enableCheckBoxColumn: false,
			url: this._getDetailGridData.bind(this),
			gridEmptyMessage: this.cvLoc('msg.noVMsAvailable'),
			enableColumnMenu: false,
			enableFiltering: false,
			hasViews: false,
			gridToolbarMenu: [],
			afterGridInitialize: this._afterGridInitialize.bind(this)
		};
	}

	_getDetailGridData(parentEntity, options) {
		options.success(parentEntity.phases);
	}

	_initDetailsColumnDefs() {
		return columnsTemplate.getDetailsColumns(this.cvLoc);
	}

	_afterGridInitialize(grid) {
		this.$document.find('#drJobsDetailsTable.k-grid .k-grid-toolbar').remove();
	}

	/*
	 * drJobsMap contains a map of the latest phase of each machine
	 * and machine.phases contains the list of all the phases for the machine.
	 */
	_getDRJobDetails() {
		let self = this;
		return this.failoverService
			.getDRJobDetails(this.jobId)
			.success(function(data) {
				self.drJobs = [];
				let drJobsMap = new Map();
				let jobs = _.get(data, 'job', []);
				jobs.forEach(function(job) {
					job.phase.forEach(function(p) {
						/* Construct the columns for each phase */
						p.clientName = _.get(p, 'entity.clientName', '');
						p.phaseText = self.cvUtil.lookupEnumConstant(p.phase);
						p.statusText = self.cvUtil.lookupEnumConstant(p.status);
						p.startTimeString = self._getTimeString(p.startTime);
						if (self._isTimeValid(p)) {
							p.elapsedTimeString = self.$filter('elapsedTime')(p.endTime.time - p.startTime.time);
						} else {
							p.elapsedTimeString = '';
						}
						p.endTimeString = self._getTimeString(p.endTime);
						if (p.phaseInfo && p.phaseInfo.job) {
							p.phaseInfo.job.forEach(function(phaseJob) {
								if (phaseJob.jobid && phaseJob.jobid != self.jobId) {
									p.phaseJobId = phaseJob.jobid;
								} else {
									p.phaseJobId = 0;
								}
								p.failureReason = _.get(phaseJob, 'failure.errorMessage', '');
							});
						}

						/* Construct the machine wise phases */
						let machinePhasesList = [];
						if (drJobsMap.has(p.clientName)) {
							machinePhasesList = drJobsMap.get(p.clientName).phases || [];
						}
						machinePhasesList.push(p);
						let latestPhase = _.cloneDeep(machinePhasesList[0]);
						machinePhasesList.forEach(phase => {
							if (phase.startTime.time > latestPhase.startTime.time) {
								latestPhase = phase;
							}
						});
						latestPhase.phases = _.cloneDeep(machinePhasesList);
						drJobsMap.set(p.clientName, latestPhase);
					});
				});
				self.drJobs.push(...drJobsMap.values());
				self._addToFilterList();
				if (self.grid) {
					self.grid.refreshData();
				}
			})
			.error(function(e) {
				cvToaster.showErrorMessage({
					ttl: '10000',
					message: e ? e : cvLoc('generic_error')
				});
			});
	}

	_isTimeValid(phase) {
		return _.get(phase, 'startTime.time', 0) > 0 && _.get(phase, 'endTime.time', 0) > 0;
	}

	_getTimeString(timeObj) {
		return _.get(timeObj, 'time', 0) > 0 ? kendo.toString(new Date(timeObj.time * 1000), 'MMM d, h:mm:ss tt') : '';
	}

	/*
	 * Adds the phase to the filter list that is provide checkbox for the filterType of each column.
	 */
	_addToFilterList(phase) {
		let self = this;
		this.drJobs.forEach(phase => {
			self.filterList.clientNames.push({
				value: phase.clientName,
				label: phase.clientName
			});
			self.filterList.phases.push({
				value: phase.phaseText,
				label: phase.phaseText
			});
			self.filterList.phaseStatus.push({
				value: phase.statusText,
				label: phase.statusText
			});
		});
	}

	/*
	 * Removes the duplicates and sorts the filterList
	 */
	_sortAndUniq(filterList) {
		filterList = filterList || {};

		for (let list in filterList) {
			filterList[list] = _.uniqWith(filterList[list], uniqWith);
			filterList[list] = _.sortBy(filterList[list], sortBy);
		}

		function uniqWith(obj1, obj2) {
			return obj1.value === obj2.value;
		}

		function sortBy(obj) {
			return obj.value;
		}
	}
}

DRJobsController.$inject = [
	'$stateParams',
	'$scope',
	'$filter',
	'$document',
	'cvToaster',
	'cvLoc',
	'cvUtil',
	'failoverService'
];
drAppFailoverModule.controller('drJobsController', DRJobsController);

export default drAppFailoverModule;
