import 'modules/anomaly/fileAnomaly/js/constants/anomalyFilter.constants.js';
import 'modules/anomaly/fileAnomaly/js/constants/anomalyAction.constants.js';
import 'modules/vmManagement/js/constants/vmLifeCyclePolicy.constants.js';
import 'vsa/js/constants/servers.constants.js';
import 'modules/anomaly/fileAnomaly/js/services/fileAnomaly.svc.js';
import 'adminConsole/js/services/agents.svc.js';
import 'vsa/js/services/restore.svc.js';
import 'adminConsole/js/controllers/restore.ctrl.js';
import 'modules/anomaly/fileAnomaly/js/factories/fileAnomaly.factory.js';
import 'adminConsole/js/factories/acBrowseUtils.factory.js';
import * as fileAnomalyTemplate from 'modules/anomaly/fileAnomaly/js/controllers/columnTemplates/fileAnomaly.column.template';
import { acAppAnomaly } from 'common/js/modules';

var app = acAppAnomaly;

class FileAnomalyDetailsController {
	constructor(
		cvLoc,
		cvBreadcrumbsTabsFactory,
		cvToaster,
		$stateParams,
		$scope,
		$compile,
		$state,
		$modal,
		$dialogs,
		$filter,
		fileAnomalyService,
		agentService,
		restoreService,
		AppTypes,
		acBrowseUtils,
		ANOMALY_FILTER_ENUM,
		ANOMALY,
		ANOMALY_ACTIONS,
		POLICY_APP_TYPE,
		VSA_VENDOR,
		fileAnomalyFactory
	) {
		this.cvLoc = cvLoc;
		this.cvBreadcrumbsTabsFactory = cvBreadcrumbsTabsFactory;
		this.cvToaster = cvToaster;
		this.$stateParams = $stateParams;
		this.$scope = $scope;
		this.$compile = $compile;
		this.$state = $state;
		this.$modal = $modal;
		this.$dialogs = $dialogs;
		this.$filter = $filter;
		this.fileAnomalyService = fileAnomalyService;
		this.agentService = agentService;
		this.restoreService = restoreService;
		this.AppTypes = AppTypes;
		this.acBrowseUtils = acBrowseUtils;
		this.ANOMALY_FILTER_ENUM = ANOMALY_FILTER_ENUM;
		this.ANOMALY = ANOMALY;
		this.ANOMALY_ACTIONS = ANOMALY_ACTIONS;
		this.POLICY_APP_TYPE = POLICY_APP_TYPE;
		this.VSA_VENDOR = VSA_VENDOR;
		this.fileAnomalyFactory = fileAnomalyFactory;
	}

	$onInit() {
		this.clientId = parseInt(this.$stateParams['clientId']);
		this.clientName = this.$stateParams['clientName'];
		this.displayName = this.clientName;
		this.anomalyType = '';
		this.defaultBackupSet = {
			id: 0,
			name: ''
		};
		this.backupSetList = [];
		this.anomalyTime = 0;
		this.recoveryTargets = [];
		this.osInfo = {};
		this.chartInitialized = false;
		this.showChart = true;
		this.noStatsFoundForChart = false;
		this.noFolderRecordsForGrid = false;
		this.noAnomalyDataAvailable = false;
		this.initialAnomalyDataLoaded = false;
		this.isVMeSupported = false;
		this.isClearAnomalySupported = capabilities.hasAgentManagementCapability;
		this.isBackupAnomaly = false;
		this.browseGridData = [];
		this.oneWeekCreated = [];
		this.oneWeekRename = [];
		this.oneWeekModified = [];
		this.oneWeekDeleted = [];
		this.oneWeekTime = [];
		this.oneDayCreated = [];
		this.oneDayRename = [];
		this.oneDayModified = [];
		this.oneDayDeleted = [];
		this.oneDayTime = [];
		this.backupCreated = [];
		this.backupRename = [];
		this.backupModified = [];
		this.backupDeleted = [];
		this.backupJobId = [];
		this.jobIdJobTimeMap = new Map();
		this.appType = this.AppTypes.WINDOWS_FILESYSTEM;
		this.isOneWeekDataLoaded = false;
		this.isOneDayDataLoaded = false;

		// Show 7 days of anomaly records with aggregation of 6 hours to 1 record. Show 1 day of anomaly records with aggregation of 2 hours to 1 record.
		this.oneWeek = 7;
		this.oneDay = 1;
		this.oneWeekAggregateHour = 6;
		this.oneDayAggregateHour = 2;

		// y aixs of the chart is logarithmic and cannot represent 0 value. So holder for subsitute of any 0 values.
		this.logSubForZero = 0.1;

		this._fetchAnomalyInfo(this.ANOMALY_FILTER_ENUM.ALL, this.oneWeekAggregateHour, this.oneWeek);
		this._addBreadCrumbs();
		this._getRecoveryTargets();
		this._initializeLiveBrowseGrid();
		this._initializeBackupBrowseGrid();
	}

	_fetchDefaultBackupSet() {
		let self = this;
		this.agentService
			.getBs(undefined, this.clientId, this.appType, false)
			.success(backupSets => {
				self.backupSetList = backupSets.backupsetProperties;
				let defaultBackupSetEntry = _.filter(
					backupSets.backupsetProperties,
					backupSet => _.get(backupSet, 'commonBackupSet.isDefaultBackupSet') == true
				);

				if (defaultBackupSetEntry && defaultBackupSetEntry.length == 1) {
					self.defaultBackupSet.id = _.get(defaultBackupSetEntry[0], 'backupSetEntity.backupsetId', 0);
					self.defaultBackupSet.name = _.get(defaultBackupSetEntry[0], 'backupSetEntity.backupsetName', '');
				}
			})
			.error(err => {
				self.cvToaster.showErrorMessage({
					message: err
				});
			});
	}

	_fetchAnomalyInfo(filter, aggregateHour, timeFrame) {
		filter = filter ? filter : this.ANOMALY_FILTER_ENUM.ALL;
		aggregateHour = aggregateHour ? aggregateHour : this.oneWeekAggregateHour;
		timeFrame = timeFrame ? timeFrame : this.oneWeek;
		this.anomalyTime = 0;

		let self = this;
		this.fileAnomalyService
			.getFileAnomalyInfo(filter, this.clientId, aggregateHour, timeFrame)
			.success(function(anomalyInfoResp) {
				self.osInfo = _.get(anomalyInfoResp, 'clientInfo[0].osInfo.osInfo', {});
				self.displayName = _.get(
					anomalyInfoResp,
					'clientInfo[0].client.displayName',
					_.get(anomalyInfoResp, 'clientInfo[0].client.clientName', self.clientName)
				);
				let anomalyRecord = _.get(anomalyInfoResp, 'clientInfo[0].anomalyRecordList[0].anomalyRecord', {});
				let recordList = anomalyRecord.anomalyRecords;
				let folderList = anomalyRecord.folderInfoList;
				self.isBackupAnomaly = _.get(anomalyRecord, 'isBackupAnomaly', false);

				if (folderList) {
					if (filter == self.ANOMALY_FILTER_ENUM.ALL || filter == self.ANOMALY_FILTER_ENUM.FOLDER_LEVEL) {
						self._processBrowseData(folderList, self.isBackupAnomaly);
					}
					self.noFolderRecordsForGrid = false;
				} else {
					self.noFolderRecordsForGrid = true;
				}

				if (recordList) {
					if (filter == self.ANOMALY_FILTER_ENUM.ALL || filter == self.ANOMALY_FILTER_ENUM.CLIENT_LEVEL) {
						self._processChartData(recordList, timeFrame);
					}
					self.noStatsFoundForChart = false;
				} else {
					self.noStatsFoundForChart = true;
				}

				self.noAnomalyDataAvailable = self.noFolderRecordsForGrid && self.noStatsFoundForChart;
				if (!self.initialAnomalyDataLoaded) {
					self.initialAnomalyDataLoaded = true;
				}

				self.showChart =
					self.initialAnomalyDataLoaded &&
					!self.noAnomalyDataAvailable &&
					!self.noStatsFoundForChart &&
					self.chartInitialized &&
					self.chart;

				self._fetchDefaultBackupSet();
			})
			.error(function(e) {
				self.noAnomalyDataAvailable = true;
				if (!self.initialAnomalyDataLoaded) {
					self.initialAnomalyDataLoaded = true;
				}
			});
	}

	_reflowChart() {
		this.chart.reflow();
	}

	_processChartData(data, timeFrame) {
		if (!this.isBackupAnomaly) {
			if (timeFrame == this.oneWeek) {
				this.isOneWeekDataLoaded = true;
			} else if (timeFrame == this.oneDay) {
				this.isOneDayDataLoaded = true;
			}
		}

		if (data) {
			if (!this.chartInitialized) {
				this.chartInitialized = true;
				this._initializeChart();
			}

			/*
			 * Backend sends data in reverse time order (latest record as first and oldest record as last). Push them to graph by reversing the order.
			 */
			for (var index = data.length - 1; index >= 0; index--) {
				// First record coming in is anomaly record.
				if (index == 0) {
					/*
					 * Live anomaly only runs on windows physical machine so the client code does not populate appType in the records. It is okay to
					 * assume that appType is 33 in that case. On other hand backup anomaly can be reported for any clients and in this case backend
					 * does populate the appType in the records.
					 */
					this.appType = _.get(data[index], 'appType', 0) > 0 ? data[index].appType : this.AppTypes.WINDOWS_FILESYSTEM;
					this.anomalyType =
						this.cvLoc('header.anomalyType') +
						' - ' +
						this.fileAnomalyFactory.getFileAnomalyType(data[index].anomalyType);
				}
				if (this.isBackupAnomaly) {
					// Anomaly time in case of backup anomaly is second job (non-anomaly job).
					if (index == 1) {
						this.anomalyTime = data[index].refTime;
					}
					this.backupCreated.push(data[index].createCount == 0 ? this.logSubForZero : data[index].createCount);
					this.backupRename.push(data[index].renameCount == 0 ? this.logSubForZero : data[index].renameCount);
					this.backupDeleted.push(data[index].deleteCount == 0 ? this.logSubForZero : data[index].deleteCount);
					this.backupModified.push(data[index].modCount == 0 ? this.logSubForZero : data[index].modCount);
					this.backupJobId.push(data[index].jobId);
					this.jobIdJobTimeMap.set(data[index].jobId, data[index].refTime);
				} else {
					// Anomaly time in case of live anomaly is first record time.
					if (index == 0) {
						this.anomalyTime = data[index].refTime;
					}
					if (timeFrame == this.oneWeek) {
						this.oneWeekCreated.push(data[index].createCount == 0 ? this.logSubForZero : data[index].createCount);
						this.oneWeekRename.push(data[index].renameCount == 0 ? this.logSubForZero : data[index].renameCount);
						this.oneWeekDeleted.push(data[index].deleteCount == 0 ? this.logSubForZero : data[index].deleteCount);
						this.oneWeekModified.push(data[index].modCount == 0 ? this.logSubForZero : data[index].modCount);
						this.oneWeekTime.push(data[index].refTime);
					} else if (timeFrame == this.oneDay) {
						this.oneDayCreated.push(data[index].createCount == 0 ? this.logSubForZero : data[index].createCount);
						this.oneDayRename.push(data[index].renameCount == 0 ? this.logSubForZero : data[index].renameCount);
						this.oneDayDeleted.push(data[index].deleteCount == 0 ? this.logSubForZero : data[index].deleteCount);
						this.oneDayModified.push(data[index].modCount == 0 ? this.logSubForZero : data[index].modCount);
						this.oneDayTime.push(data[index].refTime);
					}
				}
			}

			if (this.chart) {
				this._updateChart(timeFrame);
			}
		}
	}

	_updateChart(timeFrame) {
		if (this.isBackupAnomaly) {
			this.chart.update({
				series: [
					{ data: this.backupCreated },
					{ data: this.backupRename },
					{ data: this.backupDeleted },
					{ data: this.backupModified }
				],
				xAxis: { categories: this.backupJobId }
			});
		} else {
			if (timeFrame == this.oneWeek) {
				this.chart.update({
					series: [
						{ data: this.oneWeekCreated },
						{ data: this.oneWeekRename },
						{ data: this.oneWeekDeleted },
						{ data: this.oneWeekModified }
					],
					xAxis: { categories: this.oneWeekTime }
				});
			} else if (timeFrame == this.oneDay) {
				this.chart.update({
					series: [
						{ data: this.oneDayCreated },
						{ data: this.oneDayRename },
						{ data: this.oneDayDeleted },
						{ data: this.oneDayModified }
					],
					xAxis: { categories: this.oneDayTime }
				});
			}
		}
	}

	_processBrowseData(data, isBackupAnomaly) {
		if (data) {
			for (var index = 0; index < data.length; index++) {
				let folderInfo = {
					id: index + 1,
					folderPath: data[index].folderPath
				};

				if (isBackupAnomaly) {
					folderInfo.jobId = data[index].jobId;
					folderInfo.formattedDetectedTime = this.$filter('customDateTime')(data[index].refTime);
					folderInfo.refTime = data[index].refTime;
				} else {
					folderInfo.createdFiles = data[index].createCount;
					folderInfo.renamedFiles = data[index].renameCount;
					folderInfo.deletedFiles = data[index].deleteCount;
					folderInfo.modifiedFiles = data[index].modCount;
					folderInfo.formattedDetectedTime = this.$filter('customDateTime')(data[index].refTime);
					folderInfo.refTime = data[index].refTime;
				}

				folderInfo.permittedOptions = {
					entityId: index + 1,
					appendToBody: true,
					permittedActionList: this._computePermittedActions(data[index].folderPath, data[index].refTime)
				};

				this.browseGridData.push(folderInfo);
			}
		}
	}

	_computePermittedActions(path, time) {
		let folderInfo = {
			folderPath: path,
			refTime: time
		};
		let permittedActions = [
			{
				show: true,
				value: this.ANOMALY_ACTIONS.RESTORE_PATH,
				label: this.cvLoc('label.restorePath'),
				onClick: () => {
					this._restorePath(folderInfo);
				}
			}
		];
		return permittedActions;
	}

	_initializeChart() {
		let self = this;
		let xAxisChart = {};

		if (this.isBackupAnomaly) {
			xAxisChart = {
				gridLineWidth: 1,
				tickPosition: 'inside',
				tickmarkPlacement: 'on',
				categories: [],
				labels: {
					formatter: function() {
						var xAxisJobId = this.value;
						return self.cvLoc('header.jobId') + ': ' + xAxisJobId;
					}
				}
			};
		} else {
			xAxisChart = {
				type: 'datetime',
				tickWidth: 0,
				gridLineWidth: 1,
				tickLength: 4,
				tickPosition: 'inside',
				tickmarkPlacement: 'on',
				tickPositioner: function(min, max) {
					var tickDist = Math.round((max - min) / 6);
					if (max - min <= 12) {
						tickDist = this.tickInterval;
					}
					var ticks = this.getLinearTickPositions(tickDist, min, max),
						tLen = ticks.length;
					return ticks;
				},
				categories: [],
				labels: {
					formatter: function() {
						var xAxisTS = this.value;
						return self.fileAnomalyFactory.getTimeString(xAxisTS);
					}
				}
			};
		}

		this.chart = new Highcharts.Chart('chartContainer', {
			chart: {
				type: 'spline',
				zoomType: 'x'
			},

			credits: {
				enabled: false
			},

			exporting: {
				enabled: !this.isBackupAnomaly,
				buttons: {
					contextButton: {
						enabled: false
					},
					oneDayButton: {
						text: '1 Day',
						theme: {
							'stroke-width': 1,
							stroke: 'silver',
							r: 3
						},

						onclick: function() {
							if (self.isOneDayDataLoaded) {
								this.update({
									series: [
										{ data: self.oneDayCreated },
										{ data: self.oneDayRename },
										{ data: self.oneDayDeleted },
										{ data: self.oneDayModified }
									],
									xAxis: { categories: self.oneDayTime }
								});
							} else {
								self._fetchAnomalyInfo(self.ANOMALY_FILTER_ENUM.CLIENT_LEVEL, self.oneDayAggregateHour, self.oneDay);
							}
						}
					},
					oneWeekButton: {
						text: '1 Week',
						theme: {
							'stroke-width': 1,
							stroke: 'silver',
							r: 3
						},

						onclick: function() {
							if (self.isOneWeekDataLoaded) {
								this.update({
									series: [
										{ data: self.oneWeekCreated },
										{ data: self.oneWeekRename },
										{ data: self.oneWeekDeleted },
										{ data: self.oneWeekModified }
									],
									xAxis: { categories: self.oneWeekTime }
								});
							} else {
								self._fetchAnomalyInfo(self.ANOMALY_FILTER_ENUM.CLIENT_LEVEL, self.oneWeekAggregateHour, self.oneWeek);
							}
						}
					}
				}
			},

			legend: {
				useHTML: true
			},

			tooltip: {
				useHTML: true,
				shared: false,
				formatter: function() {
					return (
						(self.isBackupAnomaly
							? '<b>' +
							  self.cvLoc('header.jobId') +
							  '</b>: ' +
							  this.x +
							  '<br/> ' +
							  self.fileAnomalyFactory.getTimeString(self.jobIdJobTimeMap.get(this.x))
							: self.fileAnomalyFactory.getTimeString(this.x)) +
						'<br/> ' +
						'<b>' +
						this.series.name +
						'</b>: ' +
						(this.y == self.logSubForZero ? 0 : this.y)
					);
				}
			},

			plotOptions: {
				series: {
					lineWidth: 1,
					marker: {
						enabled: false,
						states: {
							hover: {
								radiusPlus: 2,
								lineWidthPlus: 2
							}
						}
					},
					states: {
						hover: {
							lineWidthPlus: 2
						}
					}
				}
			},

			title: {
				text: this.cvLoc('header.fileAnomaly')
			},

			xAxis: xAxisChart,

			yAxis: {
				title: {
					text: this.cvLoc('label.fileCount')
				},
				type: 'logarithmic',
				labels: {
					formatter: function() {
						var ret;
						if (this.isFirst) {
							ret = 0;
						} else {
							var axis = this.axis,
								numericSymbols = this.chart.options.lang.numericSymbols,
								i = numericSymbols && numericSymbols.length,
								value = this.value,
								multi,
								numSymMagnitude = this.chart.options.lang.numericSymbolMagnitude || 1000,
								numericSymbolDetector = axis.isLog ? Math.abs(value) : axis.tickInterval;

							if (i && numericSymbolDetector >= 1000) {
								/* Decide whether we should add a numeric symbol like k (thousands)
								 * or M (millions). If we are to enable this in tooltip or other
								 * places as well, we can move this logic to the numberFormatter and
								 * enable it by a parameter.
								 */
								while (i-- && typeof ret === 'undefined') {
									multi = Math.pow(numSymMagnitude, i + 1);
									/* Only accept a numeric symbol when the distance is more
									 * than a full unit. So for example if the symbol is k, we
									 * don't accept numbers like 0.5k.
									 * Accept one decimal before the symbol. Accepts 0.5k but
									 * not 0.25k. How does this work with the previous?
									 */
									if (
										numericSymbolDetector >= multi &&
										(value * 10) % multi === 0 &&
										numericSymbols[i] !== null &&
										value !== 0
									) {
										ret = Highcharts.numberFormat(value / multi, -1) + numericSymbols[i];
									}
								}
							}

							if (typeof ret === 'undefined') {
								if (Math.abs(value) >= 10000) {
									// add thousands separators
									ret = Highcharts.numberFormat(value, -1);
								} else {
									// small numbers
									ret = Highcharts.numberFormat(value, -1, void 0, ''); // #2466
								}
							}
						}

						return ret;
					}
				}
			},

			series: [
				{
					name: this.cvLoc('header.createdFiles'),
					data: [],
					color: '#1ebe50',
					type: 'spline'
				},
				{
					name: this.cvLoc('header.renamedFiles'),
					data: [],
					color: '#FF9800',
					type: 'spline'
				},
				{
					name: this.cvLoc('header.deletedFiles'),
					data: [],
					color: '#777777',
					type: 'spline'
				},
				{
					name: this.cvLoc('header.modifiedFiles'),
					data: [],
					color: '#348ab4',
					type: 'spline'
				}
			]
		});
	}

	_initializeLiveBrowseGrid() {
		const self = this;
		this.gridOptionsLive = {};
		this.gridOptionsLive.enableResizing = true;
		this.gridOptionsLive.enableCsvExport = true;
		this.gridOptionsLive.exportAllPages = true;
		this.gridOptionsLive.exportName = 'UnusualFileActivityDetails_' + this.displayName;
		this.gridOptionsLive.columns = this._initColumnDefs(false);
		this.gridOptionsLive.gridTitle = this.cvLoc('header.fileAnomaly');
		this.gridOptionsLive.tableName = 'fileAnomalyLiveBrowseTable';
		this.gridOptionsLive.enableCheckBoxColumn = true;
		this.gridOptionsLive.enableMultiSelect = true;
		this.gridOptionsLive.url = this._fetchData.bind(this);
		this.gridOptionsLive.usePageToolbar = true;
		this.gridOptionsLive.onGridDataBound = this._onGridDataBound.bind(this);
		this.gridOptionsLive.hasViews = true;
		this.gridOptionsLive.enablePaging = true;
		this.gridOptionsLive.pageSize = 20;
		this.gridOptionsLive.pageSizes = [5, 10, 20, 50, 100];
		this.gridOptionsLive.sortDirection = {
			field: 'folderPath',
			dir: 'asc'
		};
		this.gridOptionsLive.onGridSelectionChange = this._onSelectionChange.bind(this);
		this.gridOptionsLive.actionMenu = [
			{
				id: this.ANOMALY_ACTIONS.RESTORE_PATH,
				label: this.cvLoc('label.restorePath'),
				onSelect: this._onBatchActionMenuSelect.bind(this)
			}
		];
		this.gridOptionsLive.gridEmptyMessage = this._emptyGridMessage();
		this.gridOptionsLive.beforeGridInitialize = ({ grid }) => {
			self.gridLive = grid;
		};
		this.gridOptionsLive.afterGridInitialize = ({ grid }) => {
			/*
			 * Launch the grid so that scrollbar is set and then display the chart that will make sure that chart adjusts correctly.
			 * Else the chart right side will hide under scrollbar component.
			 */
			self._reflowChart();
		};
	}

	_initializeBackupBrowseGrid() {
		const self = this;
		this.gridOptionsBackup = {};
		this.gridOptionsBackup.enableResizing = true;
		this.gridOptionsBackup.enableCsvExport = true;
		this.gridOptionsBackup.exportAllPages = true;
		this.gridOptionsBackup.exportName = 'UnusualFileActivityDetails_' + this.displayName;
		this.gridOptionsBackup.columns = this._initColumnDefs(true);
		this.gridOptionsBackup.gridTitle = this.cvLoc('header.fileAnomaly');
		this.gridOptionsBackup.tableName = 'fileAnomalyBackupBrowseTable';
		this.gridOptionsBackup.enableCheckBoxColumn = true;
		this.gridOptionsBackup.enableMultiSelect = true;
		this.gridOptionsBackup.url = this._fetchData.bind(this);
		this.gridOptionsBackup.usePageToolbar = true;
		this.gridOptionsBackup.onGridDataBound = this._onGridDataBound.bind(this);
		this.gridOptionsBackup.hasViews = true;
		this.gridOptionsBackup.enablePaging = true;
		this.gridOptionsBackup.pageSize = 20;
		this.gridOptionsBackup.pageSizes = [5, 10, 20, 50, 100];
		this.gridOptionsBackup.sortDirection = {
			field: 'folderPath',
			dir: 'asc'
		};
		this.gridOptionsBackup.onGridSelectionChange = this._onSelectionChange.bind(this);
		this.gridOptionsBackup.actionMenu = [
			{
				id: this.ANOMALY_ACTIONS.RESTORE_PATH,
				label: this.cvLoc('label.restorePath'),
				onSelect: this._onBatchActionMenuSelect.bind(this)
			}
		];
		this.gridOptionsBackup.gridEmptyMessage = this._emptyGridMessage();
		this.gridOptionsBackup.beforeGridInitialize = ({ grid }) => {
			self.gridBackup = grid;
		};
		this.gridOptionsBackup.afterGridInitialize = ({ grid }) => {
			/*
			 * Launch the grid so that scrollbar is set and then display the chart that will make sure that chart adjusts correctly.
			 * Else the chart right side will hide under scrollbar component.
			 */
			self._reflowChart();
		};
	}

	_onBatchActionMenuSelect({ optionId, selectedRowValues }) {
		if (selectedRowValues.length === 1) {
			this._performOperation(optionId, selectedRowValues[0]);
		} else {
			this._performMultiOperation(optionId, selectedRowValues);
		}
	}

	_performOperation(action, folderInfo) {
		if (action === this.ANOMALY_ACTIONS.RESTORE_PATH) {
			this._restorePath(folderInfo);
		}
	}

	_performMultiOperation(action, folderInfoList) {
		if (action === this.ANOMALY_ACTIONS.RESTORE_PATH) {
			this._restoreMultiplePaths(folderInfoList);
		}
	}

	_initColumnDefs(isBackupAnomaly) {
		return isBackupAnomaly
			? fileAnomalyTemplate.getFileAnomalyBackupFolderBrowseTemplate({ cvLoc: this.cvLoc })
			: fileAnomalyTemplate.getFileAnomalyLiveFolderBrowseTemplate({ cvLoc: this.cvLoc });
	}

	_emptyGridMessage() {
		return '<div class="text-align-center">' + this.cvLoc('label.noDataAvailable') + '</div>';
	}

	browseAndRestore() {
		this.$state.go('fsBrowseForAgents', {
			applicationId: this.appType,
			serverId: this.clientId,
			serverName: this.clientName,
			backupsetId: this.defaultBackupSet.id,
			backupsetName: this.defaultBackupSet.name,
			entityType: 'BACKUPSET_ENTITY',
			restoreType: this.ANOMALY.FILE_ANOMALY,
			path: this.acBrowseUtils.getRootPathForBrowse(this.appType, null),
			showDeletedItems: true,
			skipLatestVersion: false, // TODO: Change to true when indexing changes for beowse are in and remove toTime setting.
			toTime:
				this.anomalyTime > 0 ? this.fileAnomalyFactory.getTimeString(this.anomalyTime, "yyyy-MM-dd'T'HH:mm:ss") : null,
			restoreType: this.ANOMALY.FILE_ANOMALY
		});
	}

	clearAnomaly() {
		let self = this;
		let clientsList = [];
		clientsList.push({
			_type_: 'CLIENT_ENTITY',
			clientId: this.clientId,
			clientName: this.clientName
		});

		this.$dialogs.confirm(this.cvLoc('label.clearAnomaly'), this.cvLoc('label.confirmClearAnomaly', this.clientName), {
			noFunction: () => {},
			yesFunction: () => {
				this.fileAnomalyService
					.clearFileAnomaly({
						clients: clientsList
					})
					.success(data => {
						self.cvToaster.showSuccessMessage({
							ttl: '1000',
							message: self.cvLoc('label.clearAnomalySuccess')
						});
						self.$state.go('fileAnomaly');
					})
					.error(err => {
						self.cvToaster.showErrorMessage({
							message: err
						});
					});
			}
		});
	}

	recoverAsVM() {
		let self = this;
		this.$modal.open({
			templateUrl: appUtil.appRoot + 'vsa/partials/virtualizeFileSystem.jsp',
			backdrop: 'static',
			controller: [
				'$scope',
				'$uibModalInstance',
				'$state',
				'cvLoc',
				'vMeParams',
				function($scope, $modalInstance, $state, cvLoc, vMeParams) {
					$scope.virtualizeToVendor = self.VSA_VENDOR.VMWARE;
					$scope.vendor = $scope.virtualizeToVendor;
					$scope.vMeParams = vMeParams;

					$scope.$modalInstance = $modalInstance; //unfortunately $modalInstance is not being injected in modal's controller, so passing it through scope hierarchy
					$scope.cancel = function() {
						$modalInstance.dismiss();
					};
					$scope.buildVirtualizeModalUrl = function(filePath) {
						return appUtil.appRoot + 'vsa/partials/' + filePath;
					};
				}
			],
			resolve: {
				vMeParams: function() {
					return {
						vMeFor: self.ANOMALY.FILE_ANOMALY,
						serverId: self.clientId,
						serverName: self.clientName,
						serverDisplayName: self.displayName,
						applicationId: self.appType,
						backupSets: self.backupSetList,
						osInfo: self.osInfo,
						recoveryTargets: self.recoveryTargets,
						pointInTime: self.anomalyTime
					};
				}
			}
		});
	}

	_restoreMultiplePaths(folderInfoList) {
		const self = this;
		let folderPathList = [];
		let refTime = 0;
		folderInfoList.forEach(folderInfo => {
			folderPathList.push(folderInfo.folderPath);

			if (refTime == 0) {
				refTime = folderInfo.refTime;
			}
		});

		if (this.defaultBackupSet.id > 0) {
			this.$modal.open({
				templateUrl: appUtil.appRoot + 'adminConsole/partials/fsRestoreOptions.jsp',
				backdrop: 'static',
				controller: [
					'$scope',
					'$uibModalInstance',
					'$state',
					'cvLoc',
					'restoreParams',
					function($scope, $uibModalInstance, $state, cvLoc, restoreParams) {
						$scope.$modalInstance = $uibModalInstance;
						$scope.restoreParams = restoreParams;
						$scope.cancel = function() {
							$uibModalInstance.dismiss();
						};
					}
				],
				resolve: {
					restoreParams: function() {
						return {
							restoreFor: self.ANOMALY.FILE_ANOMALY,
							serverId: self.clientId,
							serverName: self.clientName,
							applicationId: self.appType,
							backupsetId: self.defaultBackupSet.id,
							entityType: 'BACKUPSET_ENTITY',
							showDeletedItems: true,
							skipLatestVersion: _.get(cv, 'additionalSettings.AdminConsole.enableToTimeRestoreFileAnomaly', false)
								? false
								: true,
							noImage: true, // Needed for restoring deleted files.
							toTime: _.get(cv, 'additionalSettings.AdminConsole.enableToTimeRestoreFileAnomaly', false)
								? self.fileAnomalyFactory.getTimeString(refTime, "yyyy-MM-dd'T'HH:mm:ss")
								: null,
							pathList: folderPathList
						};
					}
				}
			});
		} else {
			this.cvToaster.showErrorMessage({
				message: this.cvLoc('label.noDefaultBackupSetAvailable')
			});
		}
	}

	_restorePath(folderInfo) {
		let folderInfoList = [];
		folderInfoList.push(folderInfo);
		return this._restoreMultiplePaths(folderInfoList);
	}

	_fetchData(options) {
		return options.success(this.browseGridData);
	}

	_onGridDataBound(dataItem, row) {
		const permittedOptions = dataItem.permittedOptions;
		const id = permittedOptions.entityId;
		const template = `<cv-permitted-actions cv-permitted-options="permittedOptions${id}"></cv-permitted-actions>`;
		this.$scope[`permittedOptions${id}`] = permittedOptions;
		row.find('.permittedActions').append(this.$compile(template)(this.$scope));
	}

	_onSelectionChange(e) {
		[this.ANOMALY_ACTIONS.RESTORE_PATH].forEach(item => {
			if (e.values.length > 0) {
				this.isBackupAnomaly
					? this.gridBackup.enableActionMenuOption(item)
					: this.gridLive.enableActionMenuOption(item);
			} else {
				this.isBackupAnomaly
					? this.gridBackup.disableActionMenuOption(item)
					: this.gridLive.disableActionMenuOption(item);
			}
		});
	}

	_addBreadCrumbs() {
		var breadCrumbs = [];
		breadCrumbs.push({
			title: this.cvLoc('header.fileAnomaly'),
			link: '#/fileAnomaly'
		});
		this.cvBreadcrumbsTabsFactory.addBreadCrumbs(breadCrumbs);
	}

	_getRecoveryTargets() {
		let self = this;
		let vmPolicyAppType = this.POLICY_APP_TYPE.REGULAR; // Regular life cycle policy app type.

		return this.restoreService.getLifeCyclePolicies(vmPolicyAppType, true, true).then(
			function(data) {
				if (data.data && data.data.length) {
					self.recoveryTargets = data.data;
					self.isVMeSupported = capabilities.hasAgentManagementCapability;
				}
			},
			function(e) {
				// No error needs to be raised, the check is whether virtualization targets are populated or not.
			}
		);
	}
}

FileAnomalyDetailsController.$inject = [
	'cvLoc',
	'cvBreadcrumbsTabsFactory',
	'cvToaster',
	'$stateParams',
	'$scope',
	'$compile',
	'$state',
	'$uibModal',
	'$dialogs',
	'$filter',
	'fileAnomalyService',
	'agentService',
	'restoreService',
	'AppTypes',
	'acBrowseUtils',
	'ANOMALY_FILTER_ENUM',
	'ANOMALY',
	'ANOMALY_ACTIONS',
	'POLICY_APP_TYPE',
	'VSA_VENDOR',
	'fileAnomalyFactory'
];
app.controller('fileAnomalyDetailsController', FileAnomalyDetailsController);

export default app;
