(function() {
	'use strict';

	angular.module('common.userPreference', []).service('userPrefService', [
		'$http',
		'$log',
		function($http, $log) {
			this.updateUserPrefs = function(syncLoginContext) {
				var query = '';
				if (syncLoginContext) {
					query = 'syncLoginContext=true';
				}
				return $http
					.get('getUserPrefs.do?' + query, {
						ignoreLoadingBar: true
					})
					.success(function(data) {
						cv.userPref = data;
					});
			};

			this.saveTableView = function(tableName, view, setAsDefault) {
				return $http.post('saveTableView.do', {
					tableName: tableName,
					unencodedViewName: decodeURIComponent(view.name),
					view: JSON.stringify(view),
					setAsDefault: setAsDefault
				});
			};

			this.updateTableView = function(tableName, view, setAsDefault) {
				return $http.post('updateTableView.do', {
					tableName: tableName,
					unencodedViewName: decodeURIComponent(view.name),
					view: JSON.stringify(view),
					setAsDefault: setAsDefault
				});
			};

			this.deleteTableView = function(tableName, viewName) {
				return $http.post('deleteTableView.do', {
					tableName: tableName,
					viewName: viewName
				});
			};

			this.setViewAsDefault = function(tableName, viewName) {
				return $http.get('setViewAsDefault.do', {
					params: {
						tableName: tableName,
						viewName: viewName
					}
				});
			};

			this.setViewOrder = function(tableName, viewOrder) {
				return $http.post('setViewOrder.do', {
					tableName: tableName,
					viewOrder: JSON.stringify(viewOrder)
				});
			};

			this.putAdditionalPref = function(tableName, key, value) {
				return $http.post('putAdditionalPref.do', {
					tableName: tableName,
					key: key,
					value: value
				});
			};

			this.setAdditionalPrefs = function(tableName, additionalPrefs) {
				return $http.post('setAdditionalPrefs.do', {
					tableName: tableName,
					additionalPrefs: additionalPrefs
				});
			};

			this.toggleColumnVisibility = function(tableName, columnName) {
				return $http.post(
					'toggleColumnVisibility.do',
					{
						tableName: tableName,
						columnName: columnName
					},
					{
						ignoreLoadingBar: true
					}
				);
			};

			/**
			 * @param {string} tableName
			 * @param {Object} columnVisibilities Map of column names to boolean visibilities
			 */
			this.setColumnVisibilities = function(tableName, columnVisibilities) {
				return $http.post(
					'setColumnVisibilities.do',
					{
						tableName: tableName,
						columnsChanged: JSON.stringify(columnVisibilities)
					},
					{
						ignoreLoadingBar: true
					}
				);
			};

			this.setColumnWidth = function(tableName, columnName, columnWidth) {
				return $http.post(
					'setColumnWidth.do',
					{
						tableName: tableName,
						columnName: columnName,
						columnWidth: columnWidth
					},
					{
						ignoreLoadingBar: true
					}
				);
			};

			this.setColumnWidths = function(tableName, columnsChanged, tableWidth) {
				return $http.post(
					'setColumnWidths.do',
					{
						tableName: tableName,
						columnsChanged: JSON.stringify(columnsChanged),
						tableWidth: tableWidth || ''
					},
					{
						ignoreLoadingBar: true
					}
				);
			};

			this.sortColumns = function(tableName, columnsChanged) {
				return $http.post(
					'sortColumns.do',
					{
						tableName: tableName,
						columnsChanged: JSON.stringify(columnsChanged)
					},
					{
						ignoreLoadingBar: true
					}
				);
			};

			/**
			 * Set a User Preference using a boolean variable.
			 *
			 * @param {string}
			 *            key - The key of the User Preference to set.
			 * @param {boolean}
			 *            value - THe value that will be set.
			 * @return {HttpPromise}
			 */
			this.setBooleanUserPref = function(key, value) {
				return $http
					.post('setBooleanUserPref.do', {
						key: String(key),
						value: value ? true : false
					})
					.success(function() {
						cv.userPref[String(key)] = value ? '1' : '0';
					});
			};

			/**
			 * Remove a User Preference.
			 *
			 * @param {string}
			 *            key - The key name of the User Preference to remove.
			 * @return {HttpPromise}
			 */
			this.removeUserPref = function(key) {
				return $http['delete']('removeUserPref.do?key=' + key).success(function() {
					delete cv.userPref[String(key)];
				});
			};

			/**
			 * Retrieve the "markedAsCompleteSetups" user preference.
			 *
			 * @return {Object} The "markedAsCompleteSetups" parse from JSON to a JS object.
			 */
			this.getMarkedAsCompleteSetups = function() {
				if (angular.isDefined(cv.userPref.markedAsCompleteSetups)) {
					try {
						return JSON.parse(cv.userPref.markedAsCompleteSetups);
					} catch (e) {
						$log.error('The "markedAsCompleteSetups" user preference is corrupted!');
						return {};
					}
				}
				return {};
			};

			/**
			 * Checks to see if a setup has been marked as complete.
			 *
			 * @param {string}
			 *            key - THe key of the setup to check if it's been marked as complete.
			 * @return boolean True if the setup given has been marked as complete; False otherwise.
			 */
			this.isMarkedAsComplete = function(setup) {
				return this.getMarkedAsCompleteSetups()[setup] === true;
			};

			/**
			 * Set a User Preference.
			 *
			 * @param {string}
			 *            key - The key of the User Preference to set.
			 * @param {string}
			 *            value - The value that will be set.
			 * @return {HttpPromise}
			 */
			this.setUserPref = function(key, value) {
				return $http
					.post('setUserPref.do', {
						key: String(key),
						value: String(value)
					})
					.success(function() {
						cv.userPref[String(key)] = String(value);
					});
			};

			// Load the user preferences on the starting of the application
			var loadUserPrefsPromise = this.updateUserPrefs();

			/**
			 * Get the user preferences that were loaded when the application first launched.
			 *
			 * @return {HttpPromise} The promise that was initialized on application load to get the user
			 *         preferences.
			 */
			this.getLoadedUserPrefs = function() {
				return loadUserPrefsPromise;
			};
		}
	]);
})();
