import Dialog from '../dialog';

const DEFAULT_CUTOFF_LENGTH = 60;
const HIDE_TIMEOUT = 150; // In milliseconds
const DELAY_BEFORE_SHOW = 200; // In milliseconds

const ATTRS = {
	ALLOW_TOOLTIPS: 'data-cv-k-allow-tooltips', // Usable by context menus to allow tooltips to show on their targets
	TOGGLE: 'cv-toggle', // cv-toggle="tooltip" is used to designate
	TOGGLE_HTML: 'cv-toggle-html-content', // Used to sepcify html tooltip content.
	// If cv-toggle-html-content is used, cv-toggle-content will be ignored and content cutoff will be disabled.
	// As the browser will automatically decode any html entities present in attributes, cv-toggle-html-content is unsafe.
	// If using user-generated content in cv-toggle-html-content, the user generated content should be
	// double-encoded since the browser will decode the content once.
	TOGGLE_CONTENT: 'cv-toggle-content', // Used to specify tooltip content
	TOGGLE_CUTOFF: 'cv-toggle-cutoff', // Used to specify or disable the cutoff length
	TOGGLE_HEADER: 'cv-toggle-header', // Used to specify the header text for the dialog when the tooltip is cut off
	TOGGLE_MORE: 'cv-toggle-more' // Used internally as a selector for the More button
};

function removeZeroWidthChars(string) {
	return string.replace(/[\u200B-\u200F\uFEFF]/g, '');
}

/**
 * Encodes \r and \n into html entities
 * @param {string} string
 * @returns {string} Returns the text with the newline characters encoded
 */
function encodeNewlines(string) {
	return string.replace(/\r/g, '&#13;').replace(/\n/g, '&#10;');
}

/**
 * Determines if an element is visible. Returns false if the element is hidden,
 * opacity is 0, or the element does not consume space
 * @param {HTMLElement} elem
 * @returns {boolean} Whether or not the element is visible
 */
function isElementVisible(elem) {
	if (elem.offsetWidth === 0 && elem.offsetHeight === 0) {
		// Element is not consuming space, it is not visible
		return false;
	}
	const style = getComputedStyle(elem);
	if (style.visibility === 'hidden' || style.opacity == 0) {
		// Element is hidden by css
		return false;
	}
	return true;
}

/**
 * Gets the visible content of an element. Zero width characters and newlines
 * will be removed and white space sequences will be collapsed to a single
 * space.
 *
 * If newlines are needed the cv-toggle-content attribute should be used.
 *
 * This function is needed because Internet Explorer's elem.innerText does not
 * exclude hidden text correctly.
 *
 * @param {Node} node
 * @returns {string} The visible text of the node.
 */
function getVisibleContent(node) {
	let content = '';
	if (node.nodeType === Node.TEXT_NODE) {
		content = removeZeroWidthChars(node.textContent)
			.trim()
			.replace(/\s\s+/g, ' ') // Collapse white space sequences
			.replace(/[\r\n\t\f\v]/g, ''); // Remove newlines
	} else if (node.nodeType === Node.ELEMENT_NODE && isElementVisible(node)) {
		for (let i = 0; i < node.childNodes.length; i++) {
			const childContent = getVisibleContent(node.childNodes[i]);
			if (childContent.length > 0) {
				if (content.length > 0) {
					// Insert a space between child nodes
					content += ' ';
				}
				content += childContent;
			}
		}
	}
	return content;
}

/**
 * When the characters <, >, ', " are encoded and used in element attributes
 * they are automatically decoded by the browser, so we must reencode them.
 * @param {string} attr The text to format
 * @returns {string} The formatted text
 */
function formatContentAttribute(attr) {
	// When the characters <, >, ', " are encoded and used in element attributes
	// they are automatically decoded by the browser, so we must reencode them.
	return removeZeroWidthChars(attr)
		.trim()
		.replace(/<br>/g, "&#10;")
		.replace(/</g, '&lt;')
		.replace(/>/g, '&gt;')
		.replace(/'/g, '&#39;')
		.replace(/"/g, '&quot;');
}

/**
 * Encodes characters into html entities
 * @param {string} text
 * @param {Element} [encoderElem] An element to use for encoding. Useful to
 * prevent the creation of unnecessary elements (for performance).
 * @returns The text with html entities encoded
 */
function encodeText(text, encoderElem = document.createElement('div')) {
	encoderElem.textContent = text;
	return encodeNewlines(encoderElem.innerHTML);
}

/**
 * Decodes html entities
 * @param {string} text
 * @param {Element} [decoderElem] An element to use for decoding. Useful to
 * prevent the creation of unnecessary elements (for performance).
 * @returns The text with html entities decoded
 */
function decodeText(text, decoderElem = document.createElement('div')) {
	decoderElem.innerHTML = text;
	return decoderElem.textContent;
}

export default class Tooltip {
	/**
	 * @param {HTMLElement} element
	 */
	constructor(element) {
		this.element = $(element);
		this.fullContentText = '';
		this.fullContentHeaderText = null;
		this._onMouseLeave = e => {
			this.queueHide();
		};
		this._onMouseEnter = e => {
			this.cancelQueuedHide();
		};
		this._onMoreClicked = e => {
			Dialog.notify(this.fullContentHeaderText, this.fullContentText);
		};
	}

	/**
	 * Shows the tooltip positioned on the target element
	 * @param {(Element|jQuery|string)} target
	 */
	show(target) {
		this.tooltip.show($(target));
	}

	/**
	 * Hides the tooltip
	 */
	hide() {
		this.cancelQueuedHide();
		this.tooltip.hide();
	}

	/**
	 * Cancels the currently queued hide and queues a new hide to execute after
	 * a short time
	 */
	queueHide() {
		this.cancelQueuedHide();
		this.hideTimeoutId = setTimeout(this.hide.bind(this), HIDE_TIMEOUT);
	}

	cancelQueuedHide() {
		clearTimeout(this.hideTimeoutId);
		delete this.hideTimeoutId;
	}

	/**
	 * Returns the parent cell of the target
	 * @returns {jQuery} The parent cell
	 */
	_getCell(target) {
		let td = target.closest('th[cv-col-id]');
		if (!td.length) {
			td = target.closest('td[cv-col-id]');
		}
		if (!td.length) {
			td = target.closest('div');
		}
		return td;
	}

	build() {
		this.tooltip = this.element
			.kendoTooltip({
				filter: `td[cv-col-id]:not(.k-detail-cell):not(.no-tooltip),th[cv-col-id],[${ATTRS.TOGGLE}='tooltip'], span.k-in`,
				position: 'bottom',
				autoHide: false,
				showAfter: DELAY_BEFORE_SHOW,
				content: e => {
					let tooltipTarget = e.target;
					if (tooltipTarget.attr(ATTRS.TOGGLE) !== 'tooltip') {
						// If the tooltip target isn't cv-toggle='tooltip', look for
						// a child that is cv-toggle='tooltip'
						tooltipTarget = e.target.find(`[${ATTRS.TOGGLE}]`);
						if (!tooltipTarget.length) {
							tooltipTarget = e.target;
						}
					}

					let td = this._getCell(e.target);
					// Store the cell's row uid and col id so that we can track it across grid redraws:
					this.targetUid = td.parent().attr('data-uid');
					this.targetColId = td.attr('cv-col-id');

					let hasContextMenu = false;
					if (td.length) {
						// If this tooltip is on a kendo grid cell, check if the
						// tooltip overlaps with some context menu. By default,
						// tootlips will not be shown on cells that contain a
						// context menu
						hasContextMenu = $(`[data-cv-k-context-menu]`)
							.toArray()
							.some(ul => {
								if (ul.hasAttribute(ATTRS.ALLOW_TOOLTIPS)) {
									// Skip the check because the context menu allows tooltips
									return false;
								}
								const contextMenu = $(ul).data('kendoContextMenu');
								let contextMenuTargets = contextMenu.target;
								if (contextMenu.options.filter) {
									contextMenuTargets = contextMenu.target.find(contextMenu.options.filter);
									if (!contextMenuTargets.length) {
										return false;
									}
								}
								if (
									td.is(contextMenuTargets) ||
									contextMenuTargets.has(td).length ||
									_.some(contextMenuTargets.toArray(), target => td.has(target).length)
								) {
									// The cell is a context menu target or the cell contains a context menu target or
									// a context menu target contains the cell
									return true;
								}
							});
					}

					let encoderElem = null;
					let content = '';
					let encodedContent = '';
					let contentIsHtml = false;
					if (!hasContextMenu) {
						encoderElem = document.createElement('div');
						if (tooltipTarget[0].hasAttribute(ATTRS.TOGGLE_HTML)) {
							// If the tooltip has cv-toggle-html-content defined, we always use its value.
							// cv-toggle-content is ignored if cv-toggle-html-content is defined.
							content = tooltipTarget.attr(ATTRS.TOGGLE_HTML);
							contentIsHtml = true;
						} else if (tooltipTarget[0].hasAttribute(ATTRS.TOGGLE_CONTENT)) {
							// If the tooltip has cv-toggle-content defined, we always use its value
							encodedContent = formatContentAttribute(tooltipTarget.attr(ATTRS.TOGGLE_CONTENT));
							content = decodeText(encodedContent, encoderElem);
						} else {
							content = getVisibleContent(td[0]);
							encodedContent = encodeText(content, encoderElem);
						}
					}

					if (hasContextMenu || content.length === 0) {
						// Do not show tooltip if there is no content
						this.tooltip.content.parent().addClass('invisible');
						return '';
					} else {
						this.tooltip.content.parent().removeClass('invisible');
					}

					if (contentIsHtml) {
						// Html content will not be cutoff, as the text length
						// cannot be determined in from the html string
						return content;
					}

					let contentText = encodedContent;
					this.fullContentText = contentText;
					this.fullContentHeaderText = null;

					let cutoffEnabled = true;
					let cutoffLength = DEFAULT_CUTOFF_LENGTH;
					if (tooltipTarget[0].hasAttribute(ATTRS.TOGGLE_CUTOFF)) {
						cutoffLength = parseInt(tooltipTarget.attr(ATTRS.TOGGLE_CUTOFF));
						if (isNaN(cutoffLength)) {
							// If the cutoff length is invalid / not a number, disable the cutoff
							cutoffEnabled = false;
						}
					}
					let maxLength = cutoffLength + 2;
					// Max length is cutoff length + 2 because we don't want to
					// cut off the tooltip if just 2 characters are being removed

					if (cutoffEnabled && content.length > maxLength) {
						// If content length exceeds the max length, replace all text after the
						// cutoff length with an ellipsis, and add the 'More' button:
						content = content.substring(0, cutoffLength).trim();
						encodedContent = encodeText(content, encoderElem); // Re-encode the shortened content
						contentText = `
							<span class="cv-k-tooltip-content"><span>${encodedContent}</span><span>...</span></span>
							<div class="cv-k-tooltip-more">
								<a ${ATTRS.TOGGLE_MORE}>${cvUtil.cvLocalize('label.More')}</a>
							</div>
						`;

						// Read header text used for tooltip dialog:
						let headerText = tooltipTarget.attr(ATTRS.TOGGLE_HEADER);
						if (typeof headerText === 'undefined') {
							if (td.length > 0) {
								// If tooltip is on a grid cell, find the corresponding header and use its
								// inner text as the header
								let node = 	td
								.closest('.cv-kendo-grid')
								.children('.k-grid-header')
								.find(`[cv-col-id=${this.targetColId}]`)[0];
								if(typeof node === 'undefined'){
									node = 	td
									.closest('.cv-kendo-tree')
									.find(`[id=headNode]`)[0];
								}
								headerText = encodeText(getVisibleContent(node),encoderElem);
							}
						} else {
							headerText = formatContentAttribute(headerText);
						}
						this.fullContentHeaderText = headerText;
					} else {
						contentText = `<span class="cv-k-tooltip-content">${encodedContent}</span>`;
					}

					return contentText;
				},
				show: e => {
					this.shown = true;
					this._addTooltipEventHandlers();
					this._addTargetEventHandlers();
				},
				hide: e => {
					this.shown = false;
					this._removeTargetEventHandlers();
				}
			})
			.data('kendoTooltip');
	}

	_addTooltipEventHandlers() {
		if (this.tooltipEventHandlersAdded) {
			// This will only run the first time the tooltip is shown. We must
			// wait for the tooltip to be shown because the popup property is
			// not defined until the tooltip is shown
			return;
		}
		this.tooltip.popup.element.on('mouseleave', this._onMouseLeave);
		this.tooltip.popup.element.on('mouseenter', this._onMouseEnter);
		this.tooltip.popup.element.on('click', `[${ATTRS.TOGGLE_MORE}]`, this._onMoreClicked);
		this.tooltipEventHandlersAdded = true;
	}

	_removeTooltipEventHandlers() {
		if (this.tooltipEventHandlersAdded) {
			this.tooltip.popup.element.off('mouseleave', this._onMouseLeave);
			this.tooltip.popup.element.off('mouseenter', this._onMouseEnter);
			this.tooltip.popup.element.off('click', this._onMoreClicked);
			this.tooltipEventHandlersAdded = false;
		}
	}

	/**
	 * Adds the mouse event handlers to the tooltip's target (or the cell if possible)
	 */
	_addTargetEventHandlers() {
		this.target = this.tooltip.target();
		const targetCell = this._getCell(this.target);
		if (targetCell.length) {
			this.target = targetCell;
		}
		this.target.on('mouseleave', this._onMouseLeave);
		this.target.on('mouseenter', this._onMouseEnter);
	}

	_removeTargetEventHandlers() {
		if (this.target) {
			this.target.off('mouseleave', this._onMouseLeave);
			this.target.off('mouseenter', this._onMouseEnter);
			delete this.target;
		}
	}

	/**
	 * Use the stored row uid and col id to find the cell again and position the tooltip over it
	 */
	repositionInGrid() {
		if (this.targetUid && this.shown) {
			const newTarget = `[data-uid=${this.targetUid}] [cv-col-id=${this.targetColId}]`;
			this.hide();
			requestAnimationFrame(() => {
				this.show(newTarget);
			});
		}
	}

	destroy() {
		this._removeTooltipEventHandlers();
		this._removeTargetEventHandlers();
		this.tooltip.destroy();
	}
}
Tooltip.ATTRS = ATTRS;
