/**
 * @author ssubash.
 *
 * This represents GridColumns
 */

import GridColumn from './gridColumn';
import * as GridConstants from './grid.constants';

export default class GridColumns {
	constructor(gridElement, gridEventEmitter, options, angularLibs) {
		this.gridEventEmitter = gridEventEmitter;
		this.columns = this._buildGridColumns(gridElement, gridEventEmitter, options, angularLibs);
		this.idField = options.idField;
		this.enableMultiSelect = options.enableMultiSelect;
		if (options.enableCheckBoxColumn) {
			this._addColumnForSelectAllCheckboxView(gridElement, gridEventEmitter, options);
		}
		if (options.enableHorizontalScroll) {
			// Add blank column to fill in remaining space for the background
			this._addBlankColumn(gridElement, gridEventEmitter);
		}
	}

	build() {
		_.forEach(this.columns, column => {
			column.build();
		});
	}

	_addColumnForSelectAllCheckboxView(gridElement, gridEventEmitter, options) {
		this.columns.unshift(
			new GridColumn({
				gridElement,
				gridEventEmitter,
				option: {
					width: '5%',
					enableMultiSelect: this.enableMultiSelect,
					customSelectAll: options.customSelectAll
				},
				key: null,
				columnType: 'checkbox'
			})
		);
	}

	_addBlankColumn(gridElement, gridEventEmitter) {
		this.columns.push(
			new GridColumn({
				gridElement,
				gridEventEmitter,
				option: { width: '0px' },
				key: null,
				columnType: 'blank'
			})
		);
	}
	_getCompanyColumn(gridElement, gridEventEmitter, options, angularLibs) {
	
		const companyColumn = {
			title: angularLibs.cvLoc('label.company'),
			type: 'string',
			width: '15%',
			template: `#if(data.entityInfo && data.entityInfo.companyName && data.entityInfo.companyId > 0) {
							#<a class="crop k-link-column" cv-toggle-content="#:entityInfo.companyName#" cv-toggle="tooltip" 
								href="\\#/subscriptions/#:entityInfo.companyId#">#:entityInfo.companyName#</a>#
						} else if (data.entityInfo && data.entityInfo.companyName) {
							#<span>#:entityInfo.companyName#</span>#
						}#`,
			searchable: !options.enableServerLoading,
			sortable: !options.enableServerLoading,
			hidden: true,
			disableColumnFilter: true,
			disableViewFiltering: true,
		};
		
		const key = 'entityInfo.companyName';

		return new GridColumn({
			gridElement,
			gridEventEmitter,
			enableServerLoading: options.enableServerLoading,
			option: companyColumn,
			key,
			enableCheckBoxColumn: false,
			angularLibs
		}); 

	}

	
	
	_getTagsColumn(gridElement, gridEventEmitter, options, angularLibs) {
		const tagsColumn = {
			title: angularLibs.cvLoc('label.tags'),
			type: 'string',
			width: '15%',
			template: `#if(data.tags !== null && data.tags !== undefined && data.tags.length == 1) {
				if(data.tags[0].value !== undefined && data.tags[0].value !== null && data.tags[0].value !== ''){
					#<span>#:tags[0].name#` + `: #:tags[0].value#`+`</span>#
				}else{
					#<span>#:tags[0].name#</span>#
				}
			} else if(data.tags !== null && data.tags !== undefined && data.tags.length > 1){
				if(data.tags[0].value !== undefined && data.tags[0].value !== null && data.tags[0].value !== ''){
					#<span class="tags" cv-toggle="callout" data-content="#:tagsCalloutTemplate#" data-trigger="click"><cv-help-text><span class="help-label">#:tags[0].name#` + 
				`<span>: #:tags[0].value#</span>`+ 
				` + #:tags.length-1 #` + `</span></cv-help-text></span>#
				}else{
					#<span class="tags" cv-toggle="callout" data-content="#:tagsCalloutTemplate#" data-trigger="click"><cv-help-text><span class="help-label">#:tags[0].name#` + 
					` + #:tags.length-1 #` + `</span></cv-help-text></span>#
				}
			}else {
				#<span>`+ angularLibs.cvLoc('label.noTags') + `</span>#
			}#`,
			searchable: !options.enableServerLoading,
			sortable: !options.enableServerLoading,
			hidden: true,
			disableColumnFilter: true,
			disableViewFiltering: true,
		};
		
		const key = 'entityInfo.tags';

		return new GridColumn({
			gridElement,
			gridEventEmitter,
			enableServerLoading: options.enableServerLoading,
			option: tagsColumn,
			key,
			enableCheckBoxColumn: false,
			angularLibs
		}); 

	}



	_buildGridColumns(gridElement, gridEventEmitter, options, angularLibs) {
		let cols = _.values(
			_.mapValues(options.columns, (value, key) => {
				return new GridColumn({
					gridElement,
					gridEventEmitter,
					enableServerLoading: options.enableServerLoading,
					option: value,
					key,
					enableCheckBoxColumn: false,
					angularLibs
				});
			})
		);
		
		if (options.showCompanyColumnAndFilter && !options.showOnlyCompanyFilter) {
			const companyColumn = this._getCompanyColumn(gridElement, gridEventEmitter, options, angularLibs);
			cols.splice(-1, 0, companyColumn);
			options.columns[companyColumn.field] = companyColumn.option;
		}

		if(options.showTags){
			const tagsColumn = this._getTagsColumn(gridElement,gridEventEmitter,options,angularLibs);
			cols.splice(-1,0, tagsColumn);
			options.columns[tagsColumn.field] = tagsColumn.option;
		}
		
		this._firstCol = cols[0];
		return cols;
	}

	getColumns() {
		return this.columns;
	}

	getFirstColumn() {
		return this._firstCol;
	}

	getColumn(columnName) {
		return this.columns.find(column => column.field === columnName);
	}

	getColumnsWithCheckBox() {
		return this.getColumns().filter(e => e.displayName);
	}

	getColumnNames() {
		return this.getColumns()
			.map(gridColumn => {
				return gridColumn.title;
			})
			.filter(columnTitle => columnTitle);
	}

	getFilterValues() {
		const values = [];
		this.getColumns().forEach(column => {
			const value = column.getFilterValue();
			const filterCondition = column.getFilterCondition();
			if (!_.isEmpty(value) && !_.isUndefined(value) && !_.isNull(value)) {
				values.push({
					columnName: column.field,
					filterValue: value,
					filterCondition,
				});
			}
		});
		return values;
	}

	onColumnMenuInit(e) {
		this.getColumns()
			.filter(column => column.field && !column.allowHiding)
			.forEach(column => {
				e.container
					.find(`.k-columns-item input[data-field='${column.field}']`)
					.parent('.k-menu-link')
					.parent('.k-item')
					.addClass('hidden');
			});
		e.container
			.find(`.k-columns-item input[data-field='${this.idField}']`)
			.parent('.k-menu-link')
			.parent('.k-item')
			.addClass('hidden');
		const column = this.getColumn(e.field);
		if (column) {
			column.onColumnMenuInit(e);
		}
	}

	onColumnMenuOpen(e) {
		const column = this.getColumn(e.field);
		if (column) {
			column.onColumnMenuOpen(e);
		}
		return column;
	}

	onFilterMenuOpen(e) {
		const column = this.getColumn(e.field);
		if (column) {
			column.onFilterMenuOpen(e);
		}
		return column;
	}

	clearColumnFilters(refreshGridFilter) {
		this.gridEventEmitter.emit(GridConstants.GRID_CLEAR_ALL_COLUMN_FILTERS, {});
		_.forEach(this.columns, column => {
			column.clearColumnFilter(refreshGridFilter);
		});
	}

	clearColumnFilter(columnName, refreshGridFilter) {
		const column = this.getColumn(columnName);
		if (column) {
			column.clearColumnFilter(refreshGridFilter);
		}
	}

	initializeFilterData(dataSrc) {
		this.getColumns().forEach(column => {
			column.initializeFilterData(dataSrc);
		});
	}
}
