import ColumnFilter from './columnFilter';
import AutoComplete from '../autocomplete';
import Dropdown from '../dropdown';
import Multiselect from '../multiselect';
import { CheckBoxMultiSelect } from '../multiselect';
import DatePicker from '../datetimepicker';
import DateTimeDropdown from '../datetimedropdown';
import NumericTextBox from '../textbox';
import * as GridConstants from './grid.constants';

export default class GridColumn {
	constructor({ gridElement, gridEventEmitter, enableServerLoading, option, key, columnType, angularLibs }) {
		option.width = option.width || '10%';
		this.width = option.width;
		this.gridEventEmitter = gridEventEmitter;
		this.angularLibs = angularLibs;
		this.option = option;
		this.gridElement = gridElement;
		this.enableServerLoading = enableServerLoading;
		this.id = kendo.guid();
		if (columnType === 'checkbox') {
			this.selectable = true;
			this.excludeFromExport = true;
			this.headerAttributes = {
				'data-resizable': 'false',
				'data-cv-multi-select': option.enableMultiSelect ? 'true' : 'false',
				'cv-col-id': this.id,
				id: 'cv-k-grid-th-:checkbox'
			};
			this.attributes = {
				'cv-col-id': this.id,
				id: 'cv-k-grid-td-:checkbox'
			};
			if(option.customSelectAll){
				this.headerTemplate = `<span id="customSelectAllElemId" style="width:56px; height:25px;"></span>`;
				this.width = "65px"
			}			
		} else if (columnType === 'blank') {
			this.excludeFromExport = true;
			this.template = '&nbsp;';
			this.headerAttributes = {
				'data-resizable': 'false',
				'cv-col-id': this.id,
				id: 'cv-k-grid-th-:blank',
				class: 'background-column'
			};
			this.attributes = {
				'cv-col-id': this.id,
				id: 'cv-k-grid-td-:blank',
				class: 'background-column'
			};
		} else {
			this.field = key;
			// this.data = option.data;
			this.filterIcon = $(`<span class="k-icon k-i-filter-sm hidden"></span>`);
			this.filterable = this._buildGridColumnFilter(option, key);
			if (
				this.columnFilter &&
				((_.isArray(this.columnFilter.value) && this.columnFilter.value.length > 0) ||
					(!_.isArray(this.columnFilter.value) && this.columnFilter.value))
			) {
				this.filterIcon.removeClass('hidden');
			}
			this.format = option.format;
			this.title = option.title;
			const sortConfig = { allowUnsort: false };
			let sortField = key;
			if ('sortField' in option) {
				sortField = option.sortField;
			} else if ('searchField' in option) {
				// If sortField is not defined, use searchField if it is defined
				sortField = option.searchField;
			}
			let sortType = option.type;
			if ('sortType' in option) {
				sortType = option.sortType;
			}
			if (sortType === 'number') {
				// Kendo sort does string comparison by default. Use numeric
				// sort function instead:
				sortConfig.compare = (first, second) => {
					const firstValue = _.get(first, sortField);
					const secondValue = _.get(second, sortField);
					if (isNaN(firstValue)) {
						if (isNaN(secondValue)) {
							return 0;
						}
						return -1;
					} else if (isNaN(secondValue)) {
						return 1;
					}
					return firstValue - secondValue;
				};
			} else if (sortField !== key) {
				// Kendo sort will be based on the column field instead of the
				// sort field, so we must replace it:
				sortConfig.compare = function(first, second) {
					return String(_.get(first, sortField)).localeCompare(_.get(second, sortField));
				};
			}
			if ('sortable' in option) {
				this.sortable = option.sortable ? sortConfig : false;
			} else {
				// By default, column is sortable
				this.sortable = sortConfig;
			}
			this.displayName = option.title;
			this.locked = option.locked;
			this.template = option.template;
			if (typeof this.template === 'string') {
				this.template = kendo.template(this.template);
			}
			this.locked = option.locked || false;
			this.minResizableWidth = option.minResizableWidth;
			this.hidden = option.hidden;
			this.allowHiding = 'allowHiding' in option ? option.allowHiding : true; // Default allowHiding to true
			this.viewKey = option.viewKey;
			this.command = option.command;
			this.groupHeaderTemplate = option.groupHeaderTemplate;
			this.groupable = option.groupable;
			this.headerTemplate = `<span cv-toggle="tooltip" cv-toggle-content="${this.title}">${this.title}</span>`;
			this.headerAttributes = {
				'cv-col-id': this.id,
				id: `cv-k-grid-th-${encodeURIComponent(this.field)}`
			};
			this.attributes = {
				...option.attributes,
				'cv-col-id': this.id,
				id: `cv-k-grid-td-${encodeURIComponent(this.field)}`
			};
			this.exportWhileHidden = option.exportWhileHidden;
			this.excludeFromExport = option.excludeFromExport;
			this.toExportValue = option.toExportValue;
			requestAnimationFrame(() => {
				// Once grid is build, append the filterIcon to the column header:
				if (this.sortable) {
					gridElement
						.children('.k-grid-header')
						.find(`thead [data-field="${this.field}"] .k-link`)
						.append(this.filterIcon);
				} else {
					gridElement
						.children('.k-grid-header')
						.find(`thead [data-field="${this.field}"]`)
						.append(this.filterIcon);
				}
			});
		}
	}

	build() {
		if (this.field) {
			requestAnimationFrame(() => {
				// Once grid is build, append the filterIcon to the column header:
				if (this.sortable) {
					this.gridElement
						.children('.k-grid-header')
						.find(`thead [data-field="${this.field}"] .k-link`)
						.append(this.filterIcon);
				} else {
					this.gridElement
						.children('.k-grid-header')
						.find(`thead [data-field="${this.field}"]`)
						.append(this.filterIcon);
				}
			});
		}
	}

	_buildGridColumnFilter(option, key) {
		if (option.disableColumnFilter) {
			return false;
		}
		if (_.isBoolean(option.filterType)) {
			return option.filterType;
		}
		if (_.isObject(option.filterType)) {
			return option.filterType;
		}
		return this._getColumnFilter(option, key);
	}

	_getColumnFilter(option, key) {
		const self = this;
		const columnOption = {
			...option
		};
		if (!columnOption.filterUrl && !columnOption.data) {
			columnOption.data = [];
			self.filterDataUninitialized = true;
		} else {
			self.filterDataUninitialized = false;
		}
		columnOption.key = key;
		this.columnFilter = GridColumn.createColumnFilter({
			options: columnOption,
			enableServerLoading: this.enableServerLoading,
			angularLibs: this.angularLibs
		});
		return {
			ui: function(element) {
				element.attr('id', key);
				self.columnFilter.build(element, self.initialFilterData);
			}
		};
	}

	initializeFilterData(dataSrc) {
		if (!this.filterDataUninitialized) {
			return;
		}
		this.initialFilterData = _.map(
			_.filter(_.uniq(_.map(dataSrc.getDataSource().view(), item => _.get(item, this.field)))),
			item => ({
				label: item,
				value: item
			})
		);
	}

	onColumnMenuInit(e) {
		// Apply alternating colors to menu items:
		e.container.find('ul.k-menu').each((index, ul) => {
			let odd = true;
			let visibleCount = 0;
			let visibleItem = null;
			$(ul)
				.children('li.k-item')
				.each((index, li) => {
					if (!li.classList.contains('k-separator')) {
						visibleItem = li;
						visibleCount += 1;
						if (odd) {
							li.classList.add('context-menu-alt');
						} else {
							li.classList.remove('context-menu-alt');
						}
						odd = !odd;
					}
				});
			if (visibleCount === 1) {
				visibleItem.classList.remove('context-menu-alt');
			}
		});

		if (this.columnFilter) {
			this.columnFilter.setGridColumn(this);
		}
		this.kendoPopup = e.container.getKendoPopup();
		this.resetButton = e.container.find("[type='reset']");
		this.resetButton.on('click', e => {
			this._clearColumnFilter(true);
			this.closeColumnMenu();
			e.preventDefault();
		});
		this.submitButton = e.container.find("[type='submit']");
		this.submitButton.on('click', e => {
			this._submitColumnFilter();
			this.closeColumnMenu();
			e.preventDefault();
		});
		this.filterForm = e.container.find('form.k-filter-menu');
		this.filterForm.off('submit');
		this.filterForm.on('submit', e => {
			this._submitColumnFilter();
			this.closeColumnMenu();
			e.preventDefault();
			e.stopImmediatePropagation();
		});
	}

	onColumnMenuOpen(e) {
		if (!this.columnFilter) {
			return;
		}
		this.columnFilter.onColumnMenuOpen();
	}

	onFilterMenuOpen(e) {
		if (this.columnFilter) {
			this.columnFilter.onFilterMenuOpen();
		}
	}

	closeColumnMenu() {
		if (this.kendoPopup) {
			this.kendoPopup.close();
		}
	}

	getFilterValue() {
		if (this.columnFilter) {
			return this.columnFilter.getValue();
		}
		return null;
	}

	getFilterCondition() {
		if (this.columnFilter) {
			return this.columnFilter.getFilterCondition();
		}
		return null;
	}

	clearColumnFilter(refreshGridFilter) {
		if (this.resetButton) {
			this.resetButton.trigger('click');
		} else {
			this._clearColumnFilter(refreshGridFilter);
		}
	}

	_clearColumnFilter(refreshGridFilter) {
		if (!this.columnFilter) {
			return;
		}
		this.columnFilter.clearFilter();
		this.columnFilter.onFilterSubmit(this.columnFilter.getValue(), this.columnFilter.getFilterCondition());
		this.gridEventEmitter.emit(GridConstants.GRID_COLUMN_FILTER_CLEAR, {
			columnName: this.field
		});
		if (refreshGridFilter) {
			this.gridEventEmitter.emit(GridConstants.GRID_FILTER, [{ columnName: this.field }]);
		}
		this.filterIcon.addClass('hidden');
	}

	submitColumnFilter() {
		this._submitColumnFilter();
		this.closeColumnMenu();
	}

	_submitColumnFilter() {
		const previousValue = this.columnFilter.getValue();
		const previousFilterCondition = this.columnFilter.getFilterCondition();
		const value = this.columnFilter.applyInputValue();
		const filterCondition = this.columnFilter.applyInputFilterCondition();
		this.columnFilter.onFilterSubmit(value, filterCondition);
		if (_.isUndefined(value) || _.isNull(value) || (_.isArrayLike(value) && _.isEmpty(value))) {
			this.clearColumnFilter(true);
		} else {
			this.gridEventEmitter.emit(GridConstants.GRID_COLUMN_FILTER_SET, {
				columnName: this.field,
				previousValue,
				previousFilterCondition,
				value,
				filterCondition,
			});
			this.gridEventEmitter.emit(GridConstants.GRID_FILTER, [
				{
					columnName: this.field,
					filterValue: value,
					filterCondition,
				}
			]);
			this.filterIcon.removeClass('hidden');
		}
	}

	static createColumnFilter({options, enableServerLoading, angularLibs}) {
		let columnFilterClass;
		switch (options.filterType) {
			case 'dropdown':
				columnFilterClass = Dropdown;
				break;
			case 'multiselect':
				columnFilterClass = Multiselect;
				break;
			case 'checkboxmultiselect':
				columnFilterClass = CheckBoxMultiSelect;
				break;
			case 'autocomplete':
				columnFilterClass = AutoComplete;
				break;
			case 'datepicker':
				columnFilterClass = DatePicker;
				break;
			case 'datetimedropdown':
				columnFilterClass = DateTimeDropdown;
				break;
			case 'numeric':
				columnFilterClass = NumericTextBox;
				break;
			default:
				columnFilterClass = ColumnFilter.GenericColumnFilter;
				break;
		}
		let filterConditions = options.filterConditions || columnFilterClass.FILTER_CONDITION_OPTIONS;
		// let viewFilterConditions = enableServerLoading ? filterConditions.slice(0, 1) : filterConditions;
		if (enableServerLoading && !options.filterConditions) {
			filterConditions = filterConditions.slice(0, 1);
		}
		const columnFilter = new columnFilterClass({
			...options,
			filterConditions,
		}, angularLibs);
		return columnFilter;
	}
}
