// grid.export.js

// const CSV_RESERVED_CHARS_REGEXP = /[,\n\r]|\r\n/m
const MAX_INT = 2147483647; // Java Integer.MAX
const UTF8_BOM = '\ufeff';
export default class CsvKendoGridExporter {
	_saveToFile(fileName, text) {
		const data = new Blob([UTF8_BOM + text], { type: 'text/plain;charset=utf-8' });
		if (navigator && navigator.msSaveOrOpenBlob) {
			navigator.msSaveOrOpenBlob(data, fileName);
		} else {
			const url = window.URL.createObjectURL(data);
			
			const elem = document.createElement('a');
			elem.href = url;
			elem.download = fileName;
			elem.style.display = 'none';
			
			document.body.appendChild(elem);
			elem.click();
			document.body.removeChild(elem);
			
			window.URL.revokeObjectURL(url);
		}
	}

	_escapeCsvValue(text) {
		// Replace double quotes with 2 double quotes and then wrap in double quotes:
		return `"${text.replace(/"/gm, '""')}"`;
	}

	async export(grid) {
		const exportedData = []; // Holds 2D array of the csv data

		let fetchDataPromise;
		if (grid.exportAllPages && (grid.enablePaging || grid.enableVirtualization)) {
			if (grid.enableServerLoading) {
				// Copy the dataSource but with max page size and no event listeners:
				const dataSourceConfig = {
					...grid.getDataSrc().dataSrcConfig,
					sort: grid
						.getDataSrc()
						.getDataSource()
						.sort(),
					filter: grid.getDataSrc().buildFilters(),
					pageSize: MAX_INT,
					page: 1,
					change: null
				};
				// Wrap the read function so we can add the export property to
				// the options object:
				const readDataFunction = _.get(dataSourceConfig, 'transport.read');
				if (_.isFunction(readDataFunction)) {
					dataSourceConfig.transport.read = function(options) {
						options.export = {
							type: 'csv',
							allPages: true
						};
						readDataFunction.call(this, options);
					};
				}
				const dataSource = new kendo.data.DataSource(dataSourceConfig);
				fetchDataPromise = dataSource.fetch().then(() => ({
					rows: [],
					values: dataSource.view()
				}));
			} else {
				const dataSource = grid.getDataSrc().getDataSource();
				fetchDataPromise = dataSource.fetch().then(() => {
					// Temporarily set the pageSize to max to get data from all pages and
					// restore page + pageSize afterwards:
					const pageSize = dataSource.pageSize();
					const page = dataSource.page();

					dataSource.pageSize(Number.MAX_SAFE_INTEGER);
					dataSource.page(1);

					const displayedRows = grid.getDisplayedRows();

					dataSource.pageSize(pageSize);
					dataSource.page(page);

					return displayedRows;
				});
			}
		} else {
			fetchDataPromise = Promise.resolve(grid.getDisplayedRows());
		}

		const columnsToExport = [];
		_.forEach(grid.getColumns(), col => {
			if (!col.excludeFromExport && (!col.hidden || col.exportWhileHidden)) {
				columnsToExport.push(col);
			}
		});

		const exportedHeaderRow = [];
		_.forEach(columnsToExport, col => {
			exportedHeaderRow.push(col.displayName || '');
		});
		exportedData.push(exportedHeaderRow);

		const data = await fetchDataPromise;
		_.forEach(data.values, (rowValue, rowIndex) => {
			const exportedRow = [];
			_.forEach(columnsToExport, col => {
				if (_.isFunction(col.toExportValue)) {
					const value = _.get(rowValue, col.field);
					exportedRow.push(col.toExportValue(value, rowValue));
				} else {
					const tr = data.rows[rowIndex];
					if (tr) {
						// If row element exists, use the inner text of the cell
						const td = $(tr).children(`td[cv-col-id="${col.id}"]`);
						exportedRow.push(td.text().trim());
					} else if (col.template) {
						// Else if template is defined, build the cell using the template
						// and use the innerText of that cell
						const td = $(`<td></td>`);
						td.html(col.template(rowValue));
						exportedRow.push(td.text().trim());
					} else {
						// Otherwise, use the value of the cell
						const value = _.get(rowValue, col.field);
						exportedRow.push(String(value));
					}
				}
			});
			exportedData.push(exportedRow);
		});

		const csvText = _.map(exportedData, exportedRow => {
			return _.map(exportedRow, exportedCell => this._escapeCsvValue(exportedCell)).join(',');
		}).join('\n');

		this._saveToFile(`${grid.exportName || 'table'}.csv`, csvText);
	}
}
