import * as GridConstants from '../newGrid/grid.constants';
import ColumnFilter from '../newGrid/columnFilter';

const DEFAULT_FILTER_CONDITION_OPTIONS = [
	GridConstants.FILTER_CONDITIONS.EQUALS,
	GridConstants.FILTER_CONDITIONS.DOES_NOT_EQUAL
];

/**
 * @author ssubash
 * Multiselect
 */
export default class Multiselect extends ColumnFilter {
	static get FILTER_CONDITION_OPTIONS() {
		return DEFAULT_FILTER_CONDITION_OPTIONS;
	}

	get textInput() {
		return this.isBuilt() ? this.multiSelect.input : null;
	}

	get wrapper() {
		return this.isBuilt() ? this.multiSelect.wrapper : null;
	}

	initialize(options) {
		super.initialize(options);
		if ('popup' in options) {
			if (options.popup === null) {
				// null popup not supported by multiselect, use undefined instead
				this.popup = undefined;
			} else {
				this.popup = options.popup;
			}
		} else {
			// By default, position the multiselect dropdown to the right
			this.popup = {
				origin: 'top right',
				position: 'top left',
			};
		}
		if ('animation' in options) {
			if (options.animation === null) {
				// null animation not supported by multiselect, use undefined instead
				this.animation = undefined;
			} else {
				this.animation = options.animation;
			}
		} else {
			// By default, animate the multiselect opening to the right
			this.animation = {
				open: {
					effects: 'slideIn:right',
				},
			};
		}
		if (options.filterUrl) {
			this.dataSrc = new MultiselectDataSource(options, this);
		}
		this.labels = [];
		this._parseViewValue = options.parseViewValue || (val => (val ? val.split(',') : []));
		this._toViewValue = options.toViewValue || (val => val.join(','));
		if (!_.isArray(options.filterValue)) {
			if (_.isString(options.filterValue)) {
				this.value = options.filterValue.split(',');
			} else if (_.isUndefined(options.filterValue) || _.isNull(options.filterValue)) {
				this.value = [];
			} else {
				this.value = [options.filterValue];
			}
		}
		// this.valuePrimitive = true;
	}

	_getKendoOptions() {
		return {
			itemTemplate: `<label class='crop' title="#:${this.dataTextField}#">#:${this.dataTextField}#</label>`,
			dataSource: this.dataSrc,
			dataTextField: this.dataTextField,
			dataValueField: this.dataValueField,
			filter: 'contains',
			value: _.clone(this.value),
			change: this.onFilterInputChange.bind(this),
			meta: this.meta,
			autoClose: false,
			popup: this.popup,
			animation: this.animation,
		};
	}

	build(element, initialFilterData) {
		super.build(element);
		element.kendoMultiSelect(this._getKendoOptions());
		this.multiSelect = element.getKendoMultiSelect();
		this._addValueToDataSource();
		this.saveLabels();
	}

	_addValueToDataSource() {
		if (!this.isBuilt()) {
			return;
		}
		const labels = this.getLabels();
		const currentValue = this.getValue();
		const valueMap = {};
		_.forEach(currentValue, (val, index) => {
			valueMap[val] = {
				label: labels[index],
				isInDataSource: false
			};
		});
		const currentData = this.multiSelect.dataSource.data();
		_.forEach(currentData, data => {
			// TODO
			const value = data[this.dataValueField];
			if (value in valueMap) {
				valueMap[value].isInDataSource = true;
			}
		});
		_.forEach(valueMap, (obj, value) => {
			if (!obj.isInDataSource) {
				const newValue = {};
				newValue[this.dataTextField] = obj.label || value;
				newValue[this.dataValueField] = value;
				this.multiSelect.dataSource.add(newValue);
			}
		});
	}

	onColumnMenuOpen() {
		this._addValueToDataSource();
		super.onColumnMenuOpen();
	}

	onFilterMenuOpen() {
		super.onFilterMenuOpen();
		if (this.isBuilt()) {
			this.multiSelect.close();
		}
	}

	loadUrlParam() {
		if (this.filterLabelQueryParam && this.angularLibs && this.angularLibs.$location) {
			const searchParams = this.angularLibs.$location.search();
			if (this.filterLabelQueryParam in searchParams) {
				// TODO allow dev to parse the label
				this.labels = searchParams[this.filterLabelQueryParam].split(',');
			}
		}
		super.loadUrlParam();
	}

	updateUrlParam() {
		if (this.filterLabelQueryParam && this.angularLibs && this.angularLibs.$location && this.angularLibs.$scope) {
			let labels = this.getLabels();
			if (_.isEmpty(labels) || _.isUndefined(labels) || _.isNull(labels)) {
				labels = null;
			} else {
				// TODO allow dev to format the label for url param
				labels = labels.join(',');
			}
			const self = this;
			this.angularLibs.$scope.$applyAsync(() => {
				self.angularLibs.$location.search(self.filterLabelQueryParam, labels).replace();
			});
		}
		super.updateUrlParam();
	}

	fromViewValue(val) {
		super.fromViewValue(val);
		this._addValueToDataSource();
		this.applyValue();
	}

	addItem(item) {
		if (this.isBuilt()) {
			this.multiSelect.dataSource.add(item);
		}
	}

	removeItem(item) {
		if (this.isBuilt()) {
			this.multiSelect.dataSource.remove(item);
		}
	}

	addSelection(value) {
		const currentValue = this.getInputValue();
		if (_.isArray(value)) {
			currentValue.push(...value);
		} else {
			currentValue.push(value);
		}
		this.setInputValue(currentValue);
		this.setValue(currentValue);
		return currentValue;
	}

	clearFilter() {
		this.setValue([]);
		this.setInputValue([]);
	}

	setValue(val) {
		if (!_.isArray(val)) {
			val = [val];
		}
		this.value = val;
		this.saveLabels();
		super.setValue(val);
	}

	saveLabels() {
		if (!this.isBuilt()) {
			return;
		}
		const val = this.getValue();
		this.labels = [];
		const labelMap = {};
		_.forEach(val, key => {
			labelMap[key] = undefined;
		});
		const dataItems = this.multiSelect.dataItems();
		_.forEach(dataItems, dataItem => {
			const value = dataItem[this.dataValueField];
			const label = dataItem[this.dataTextField];
			if (value in labelMap) {
				labelMap[value] = label;
			}
		});
		_.forEach(val, key => {
			this.labels.push(labelMap[key]);
		});
	}

	getLabels() {
		return this.labels;
	}

	setInputValue(val) {
		if (this.isBuilt()) {
			this.multiSelect.value(val);
			this.refreshInputTooltip();
		}
	}

	getInputValue() {
		if (this.isBuilt()) {
			return this.multiSelect.value();
		}
		return null;
	}

	getInputTooltipText() {
		if (this.isBuilt()) {
			const dataItems = this.multiSelect.dataItems();
			return _.map(dataItems, dataItem => dataItem[this.dataTextField]).join(', ');
		}
		return null;
	}

	restoreInputValue(val) {
		this.setInputValue(val.multiSelectValue);
		if (this.isBuilt()) {
			this.multiSelect.input.val(val.textInputValue);
			if (val.textInputValue) {
				// Search is open, redo the search
				this.multiSelect.search(val.textInputValue);
			}
		}
	}

	getInputValueForRestore() {
		return {
			multiSelectValue: this.getInputValue(),
			textInputValue: this.isBuilt() ? this.multiSelect.input.val() : '',
		};
	}

	destroy() {
		super.destroy();
		if (this.isBuilt()) {
			this.multiSelect.destroy();
		}
	}
}

class MultiselectDataSource extends ColumnFilter.ColumnFilterDataSource {
	constructor(options, columnFilter) {
		super(options, columnFilter);
		// this.sort = {
		// 	field: columnFilter.dataTextField,
		// 	dir: "asc"
		// };
		if (_.isFunction(options.filterUrl)) {
			const self = this;
			const readFunction = options.filterUrl;
			this.transport.read = callbacks => {
				readFunction.call(
					columnFilter,
					{
						...callbacks,
						success: data => {
							callbacks.success(data);
							if (!self.firstReadFinished) {
								this.columnFilter._addValueToDataSource();
								this.columnFilter.applyValue();
								self.firstReadFinished = true;
							}
						},
						error: e => {
							callbacks.error(e);
						}
					},
					columnFilter
				);
			};
		}
	}
}

/**
 * @author ssubash
 * This is more specific multiselect library which has checkboxes
 */
export class CheckBoxMultiSelect extends Multiselect {
	_getKendoOptions() {
		const options = super._getKendoOptions();
		options.itemTemplate = `<input class='multi-select' type='checkbox'/>${options.itemTemplate}`;
		options.dataBound = () => {
			requestAnimationFrame(() => {
				this.refreshCheckboxes();
			});
		};
		options.open = e => {
			this.refreshCheckboxes();
		};
		return options;
	}

	onFilterInputChange() {
		this.refreshCheckboxes();
		return super.onFilterInputChange();
	}

	refreshCheckboxes() {
		if (this.isBuilt()) {
			const items = this.multiSelect.ul.children('li');
			items.each((index, elem) => {
				const element = $(elem);
				const input = element.children('input');
				input.prop('checked', element.hasClass('k-state-selected'));
			});
		}
	}

	onColumnMenuOpen() {
		super.onColumnMenuOpen();
		requestAnimationFrame(() => {
			this.refreshCheckboxes();
		});
	}

	clearFilter() {
		super.clearFilter();
		requestAnimationFrame(() => {
			this.refreshCheckboxes();
		});
	}
}
