import { cvCommonModule } from "common/js/modules";

cvCommonModule.factory('multiCommcell', ['cvToaster', 'cvLoc', 'statusToText', '$transitions', function(cvToaster, cvLoc, statusToText, $transitions) {
    let history = new Set();
    const commcellStatus = {};

    const statusToState = {
        500: 'failed',
        400: 'warning',
        401: 'warning',
        default: 'info',
    }

    // Not saving this in external constant, use functions to identify roles as per requirement
    const commcellRoles = {
        NOTCONFIGURED: 0,
        ROUTERCOMMCELL: 1,
        ROUTERCOMMCELL_SERVICECOMMCELL: 2,
        IDPCOMMCELL: 4,
        IDPCOMMCELL_SERVICECOMMCELL: 8,
        CLOUDSERVICE: 16,
        ONPRIM_SUBSCRIBER_OF_CLOUDSERVICE: 32,
    }

    function updateIfNotShown(failedCommcells, isSubscription) {
        if (isSubscription) {
			return false;
		}
        const commcellNames = new Set(_.map(failedCommcells, 'commcell.commCellName'));
        const isAlreadyShown = _.isEqual(commcellNames, history);
        if (!isAlreadyShown){history = commcellNames;}
        return isAlreadyShown;
    }

    function getStatusString(failedCommcell) {
        let statusString = failedCommcell.statusCode;
        let state = statusToState['default'];
        if (failedCommcell.errorString) {
            statusString += " - " + failedCommcell.errorString;
        } else if (statusToText[failedCommcell.statusCode]) {
            statusString += " - " + cvLoc(statusToText[failedCommcell.statusCode]);
            state = statusToState[failedCommcell.statusCode]
        } else {
            statusString += " - " + cvLoc(statusToText['default']);
        }
        // Save latest statusString for later use
        commcellStatus[_.get(failedCommcell, 'commcell.commCellName')] = {
            statusString: statusString,
            statusCode: failedCommcell.statusCode,
            state: state
        };
        return statusString;
    }

    function createErrorMessage(failedCommcells, isSubscription) {
        let message = '';
        if (isSubscription) {
            message  = cvLoc('error.multiCommcellSubscription');
            _.each(failedCommcells, function (failedCommcell) {
                message += "\n " + _.get(failedCommcell, 'commcell.commCellName');
            });
        } else {
            message = cvLoc('error.multiCommcellFailed');
            _.each(failedCommcells, function (failedCommcell) {
                message += "\n " + _.get(failedCommcell, 'commcell.commCellName') + ": " + getStatusString(failedCommcell);
            });
        }
        return message;
    }

    function clearHistory() {
        history.clear();
    }

    $transitions.onStart({}, clearHistory);

    function showError(cometResponse, isSubscription) {
        if (_.has(cometResponse, 'commcellInfo.length')) {
            const failedCommcells = _.filter(cometResponse.commcellInfo, 'statusCode');
            if (!updateIfNotShown(failedCommcells, isSubscription)) {
                cvToaster.showErrorMessage({
                    message: createErrorMessage(failedCommcells, isSubscription),
                    ttl: '6000'
                });
            }
        }
    }

    return {
        // Show error information returned via header of apis
        processHeader: function(response, isSubscription) {
            let headers = response && response.headers();
            if(_.isUndefined(headers) || _.isNull(headers)){
                headers = {};
            }
            if (!_.has(headers, 'comet-response') && _.has(response,  'data.metaInfo.length')) {
                const allHeaders = _.find(response.data.metaInfo, {
                    name: 'allHeaders'
                });
                if (allHeaders && allHeaders.value) {
                    headers = JSON.parse(allHeaders.value);
                    if (!headers['comet-response'] && headers['Comet-Response']){headers['comet-response'] = headers['Comet-Response'];}
                }
            }
            if (headers['comet-response']) {
                try {
                    const cometResponse = JSON.parse(_.get(headers, 'comet-response'));
                    showError(cometResponse, isSubscription);
                } catch(error) {
                    console.error('Failed to parse header-value');
                }
            }
        },
        samlRedirectRequired: function(commcellName) {
            const commcell = cv.commcellUrls[commcellName];
            if (commcell && commcell.commcellRole) {
                const isRouterCommcell = (commcellRoles.ROUTERCOMMCELL_SERVICECOMMCELL & commcell.commcellRole) > 0;
                const isCloudServiceCommcell = (commcellRoles.CLOUDSERVICE & commcell.commcellRole) > 0;
                return isRouterCommcell || isCloudServiceCommcell;
            }
            return false;
        },
        getOfflineCommcells: function() {
            if (history.size) {
                return _.map(Array.from(history), function(commcellName) {
                    return {
                        title: commcellName,
                        subtitle: commcellStatus[commcellName].statusString,
                        state: commcellStatus[commcellName].state
                    }
                });
            }
            return null;
        },
        isGlobalOffline: function() {
            return history.size !== 0;
        },
        noOfOfflineCells: function() {
            return history.size;
        }
    }
}]);

angular.module('cvCommon').factory('multiCommcellFactory', ['multiCommcell', 'settingsService', 'cvUtil', '$window', 'cvLoc', 'commcellRole', '$transitions', '$state', 'cometFirstLoad', function(multiCommcell, settingsService, cvUtil, $window, cvLoc, commcellRole, $transitions, $state, cometFirstLoad) {
    let factory = {};

    factory.checkRoleExist = function (combinedRole, roleToCheck) {
        return (combinedRole & roleToCheck) === roleToCheck;
    };

    factory.getRoleLabels = function (commcellRoleInt) {
        switch (commcellRoleInt) {
            case commcellRole.NOTCONFIGURED:
                return cvLoc('commcellType.notConfigured');
            case commcellRole.ROUTERCOMMCELL:
                return cvLoc('commcellType.routerCommcell');
            case commcellRole.ROUTERCOMMCELL_SERVICECOMMCELL:
                return cvLoc('commcellType.routerServiceCommcell');
            case commcellRole.IDPCOMMCELL:
                return cvLoc('commcellType.IdPCommcell');
            case commcellRole.IDPCOMMCELL_SERVICECOMMCELL:
                return cvLoc('commcellType.serviceCommcell');
            case commcellRole.CLOUDSERVICE:
                return cvLoc('commcellType.cloudservice');
            case commcellRole.ONPRIM_SUBSCRIBER_OF_CLOUDSERVICE:
                return cvLoc('commcellType.onPrimCS');
            default:
                return '';
        }
    };

    factory.getCommcellRole = function (commcellRoleInt) {
        if (_.isUndefined(commcellRoleInt) || commcellRoleInt === commcellRole.NOTCONFIGURED) {
            return '';
        }

        let roles = [];

        _.forEach(commcellRole, function (roleValue) {
            if (factory.checkRoleExist(commcellRoleInt, roleValue) && roleValue !== commcellRole.NOTCONFIGURED) {
                // Push applicable roles into the array
                roles.push(factory.getRoleLabels(roleValue));
            }
        });

        // Return string roles separated by comma
        return roles.join(', ');
    };

    // kept name existing dashboard function calls
    factory.redirectToCommcell = function(event, commcell) {
        if (event && event.target && event.target.tagName == 'A') {
            event && event.preventDefault && event.preventDefault();
            // TODO: When clicking link in callout in dashboard, remove callout
            _.defaults(commcell, {
                useIframe: true
            });
            if (event.target.href) {
                factory.openUrl(event.target.href, commcell);
            }
        }
    }

    factory.setGridClickFunctionWrapper = function(row, columnNames, commcell) {
        _.forEach(columnNames, columnName => {
            if (row && row.find && row.find(columnName)) {
                factory.setGridClickFunction(row.find(columnName), commcell);
            }
        })
    }

    factory.setGridClickFunction = function(column, commcell) {
        if (!cv.isCometApp) {
            return;
        }
        // Add polymorphism to support opening link with just commcellName if required
        if (_.isString(commcell)) {
            commcell = { 
                commcellName: commcell,
                // Grid actions do not support iframe view yet. (Css doesn't look good)
                useIframe: false,
            }
        }
        column.click(event => {
            factory.redirectToCommcell(event, commcell);
        });
    }

    function isValidUrl(string) {
        try {
          new URL(string);
        } catch (_) {
          return false;  
        }  
        return true;
    }
    
    factory.correctHostName = function(url, commcellName, urlKey) {
        if (cv.isCometApp) {
            if (commcellName) {
                // return idp url
                let commcell = {}
                if (commcellName == 'IDP') {
                    commcell = _.find(cv.commcellUrls, 'isIdp');
                } else {
                    commcell = cv.commcellUrls[commcellName]
                }
                let prefix = _.get(commcell, urlKey || 'webUrl', '');
                if (isValidUrl(url)) {
                    let localUrl = new URL(url);
                    try {
                        let hostName = new URL(_.get(commcell, 'baseUrl', ''));
                        localUrl.hostname = hostName.hostname;
                        url = localUrl.toString().replace('/global/', '/adminconsole/');
                    } catch (_) {}
                } else {
                    url = prefix + url;
                }
            } else {
                url = 'undefinedCometURL';
            }
        }
        return url;
    }

    factory.getCorrectedUrl = function(link = '', commcell) {
        link = link.replace('http://', 'https://')
        if (!commcell || !commcell.commcellName) {
            return Promise.resolve(link);
        }

        let dummy = document.createElement('a');
        dummy.href = link;
        dummy.port = '';
        const callbackUrl = factory.correctHostName(dummy.href, commcell.commcellName);

        if (commcell.skipSAMLRedirect || !multiCommcell.samlRedirectRequired(commcell.commcellName)) {
            return Promise.resolve(callbackUrl);
        }

        const webconsoleUrl = cv.commcellUrls[commcell.commcellName].webconsoleUrl;

        return settingsService.getSAMLToken(true).then(function(result) {
            const token = (result.data && result.data.token) || '';
            /*
			 * For single session logout, a logoutSessionIndex is generated and passed onto the service
			 * commcell On service commcell logout, the logoutSessionIndices connected to it are also logged
			 * out After the first redirect, any subsequent logoutSessionIndices sent are ignored
			 */
            const logoutSessionIndex = cvUtil.generateUUID().replace(/-/g, '');

            const handler = function() {
                if(token){
                    if (commcell.useIframe && cv.isCometApp && cvConfig.enableGlobalIframe) {
                        const linkToRedirect = webconsoleUrl +
                        '/idpSSOLoginCallBackHandler.do?samlToken=' + encodeURIComponent(token) +
                        '&next=' + encodeURIComponent(callbackUrl) +
                        '&logoutSessionIndex=' + logoutSessionIndex +
                        '&logoutSessionCallbackUrl=' + encodeURIComponent(location.origin + location.pathname);
                        return linkToRedirect;
                    }
                    else {
                        const linkToRedirect = webconsoleUrl +'/idpSSOLoginCallBackHandler.do';
                        const samlToken = token;
                        const next = callbackUrl;
                        const logoutSessionCallbackUrl = location.origin + location.pathname;

                        return {
                            linkToRedirect,
                            samlToken,
                            next,
                            logoutSessionIndex,
                            logoutSessionCallbackUrl
                        };

                    }
            	}else{
                    return callbackUrl;
            	}
            }

            return settingsService.saveSessionIndexAndCloudServiceUrl({
                cloudServiceUrl: webconsoleUrl + '/',
                sessionIndex: logoutSessionIndex
            }).then(handler).catch(handler);
        }).catch(function(error) {
            console.error(error);
            return callbackUrl;
        });
    }

    function setupTransitionsForFrame() {
    	var eventMethod = window.addEventListener ? "addEventListener" : "attachEvent";
    	var eventer = window[eventMethod];
    	var messageEvent = eventMethod == "attachEvent" ? "onmessage" : "message";
    	eventer(messageEvent,function(event) {
    	    if (event.data && cv.isCometApp) {
    	    	if (event.data.eventType == 'stateTransitionEvent') {
    	    		$state.go(event.data.to,{}, { reload: true});
    	    	}
    	    }
    	},false);

    	$transitions.onStart({}, function(transition) {
    		if (cv.isCometApp && !inIframe()) {
    			factory.removeContentFrame();
    		}
    		if (inIframe()) {
    			// Inform parent about the transition
    			// Todo add and confirm origin for security
    			parent.postMessage({
    				eventType: 'stateTransitionEvent',
                    to: transition.to().name,
                    param: transition.params()
    			}, '*');
    		}
		});
    }

    factory.addContentFrame = function(content) {
        $('#cv-content-page-iframe').html(content);
        $('#cv-content-page-iframe').removeClass('hide');
        $('#cv-content-page').addClass('hide');
    }

    factory.removeContentFrame = function(content) {
        $('#cv-content-page-iframe').empty();
        $('#cv-content-page-iframe').addClass('hide');
        $('#cv-content-page').removeClass('hide');
    }

    function isWebConsoleUrl(link) {
        return _.includes(link, '/webconsole/') && !_.includes(link, '://webconsole/');
    }

    /**
     * options {
     *  commcellName: Name of commcell to redirect to
     *  useIframe: open details in iframe instead of new tab
     * }
     */
    factory.openUrl = function(link, options = {}) {
        _.defaults(options, {
            useIframe: true
        });
        // If link is webconsole url, do not open it in iframe
        if (isWebConsoleUrl(link)) {
            options.useIframe = false;
        }
        const handler = function(correctedUrl) {
            if (options.useIframe && cvConfig.enableGlobalIframe) {
                let iframeDOM = document.createElement('iframe');
                iframeDOM.setAttribute('class', 'full-height full-width inline-block z-index-10');
                iframeDOM.id = 'iframe' + Date.now();
                factory.addContentFrame(iframeDOM);
                iframeDOM.src = correctedUrl;
            } else if (_.isObject(correctedUrl)) {
                const linkToRedirect = correctedUrl.linkToRedirect.replace('http://', 'https://').replace(":80", "");

                delete correctedUrl.linkToRedirect;

                const params = correctedUrl;

                window.cvUtil.doFormPost(linkToRedirect, null, params, cv.cometTarget);
            } else {
                $window.open(correctedUrl, cv.cometTarget);
            }
        }
        factory.getCorrectedUrl(link, options).then(handler).catch(handler);
    }

    function inIframe () {
        try {
            // TODO: MR 304463: In gitapps if add modal is opened, then header and left nav is hidden even though its not an iframe. Need to fix that regression before re-enabling enableGlobalIframe.
            return window.self !== window.top && cvConfig.enableGlobalIframe;
        } catch (e) {
            return true;
        }
    }

    function isDetailsInIframe() {
        // TODO: check if details page
        return inIframe();
    }

    factory.hideLeftNav = function() {
        return isDetailsInIframe();
    }

    factory.hideHeader = function() {
        return isDetailsInIframe();
    }

    factory.redirectToCCIfNotGlobal = function() {
        setupTransitionsForFrame();
        if (cv.isCometApp) {
            if (!cvConfig.enableGlobalConsole) {
                // If globalconsole is not enabled, redirect to adminconsole
                var adminconsoleUrl = _.replace(_.get($window, 'location.href'), '/global/', '/adminconsole/');
                $window.open(adminconsoleUrl, '_self');
            }            
            if (cvApp.commcellInfo && cvApp.commcellInfo.commcellRole) {
                const globalAccessible = factory.checkRoleExist(cvApp.commcellInfo.commcellRole, commcellRole.IDPCOMMCELL) || factory.checkRoleExist(cvApp.commcellInfo.commcellRole, commcellRole.ROUTERCOMMCELL) || factory.checkRoleExist(cvApp.commcellInfo.commcellRole, commcellRole.ONPRIM_SUBSCRIBER_OF_CLOUDSERVICE);
                if ((factory.checkRoleExist(cvApp.commcellInfo.commcellRole, commcellRole.IDPCOMMCELL_SERVICECOMMCELL)
                ||  factory.checkRoleExist(cvApp.commcellInfo.commcellRole, commcellRole.ROUTERCOMMCELL_SERVICECOMMCELL))
                && !globalAccessible) {
                    // if its a service commcell, redirect to actual IdP global url, Since window.open has cross site issue, using location.href
                    console.warn(`commcellRole ${cvApp.commcellInfo.commcellRole} not matched to global cs, redirecting to global of idp`);
                    var globalUrl = _.replace(cvApp.commcellInfo.globalIdpCommcellWebconsoleUrl, '/webconsole', '/global');
                    if (globalUrl)
                        $window.location.href = globalUrl;
                } else if (!globalAccessible) {
                    console.warn(`commcellRole ${cvApp.commcellInfo.commcellRole} not matched to global cs, redirecting to adminconsole`);
                    // If its a machine where global is not setup, redirect to adminconsole of same machine
                    var adminconsoleUrl = _.replace(_.get($window, 'location.href'), '/global/', '/adminconsole/');
                    $window.open(adminconsoleUrl, '_self');
                }
            }
        }
    }

    factory.handleSubscriptionResponse = function(response) {
        if (!cv.isCometApp) {
			return;
		}
        if (response && response.metaData == 'comet') {
            if (response && response.headers) {
                try {
                    const headers = JSON.parse(response.headers);
                    const dummyResponse = {
                        headers: () => headers
                    };
                    this.multiCommcell.processHeader(dummyResponse, true);
                } catch (error) {
                    console.log(error);
                    console.log('Subscriptions may not work properly');
                }
            }
        }
    }

    // Create a subscription for cometfirstload
    // Print any fanout errors from the subscription
    // Get subscription id from the subscription
    // Call actual api with subscription id
    // Handle subsequent messages in subscription
    factory.multiCommcellLoad = function(subscriptionType, apiCall, onPushSuccess, enablePushMessages) {
        if (!cv.isCometApp) {
			return;
		}
        let unsubscribeMultiCommcellLoad = undefined;
        let serviceCommcellList = [];
        const subscriptionInfo = {
			msg: 'subscribe',
            type: subscriptionType,
            metaData: 'comet',
        };
        // If no response comes with 20 seconds, ignore this method and do direct api call
        const fallBackHandler = function() {
            if (unsubscribeMultiCommcellLoad){unsubscribeMultiCommcellLoad();}
            apiCall && apiCall();
            enablePushMessages && enablePushMessages();
        }
        const fallBackTimeout = setTimeout(fallBackHandler.bind(this), 20 * 1000);

        const onMessageReceived = function(response) {
            if (_.isInteger(response)) {
                // Subscription Id
                const subscriptionId = response;
                apiCall && apiCall(subscriptionId);
            } else if (_.isObject(response)) {
                factory.handleSubscriptionResponse(response);
                if (_.has(response, 'notificationType') && _.has(response, 'serviceCommcells')) {
                    switch (response.notificationType) {
                        case cometFirstLoad.notificationType.serviceCommcellResponse:
                            _.forEach(response.serviceCommcells, serviceCommcell => {
                                if (_.get(serviceCommcell, 'isSuccess', false)) {
                                    serviceCommcell.response = JSON.parse(_.get(serviceCommcell, 'response', '{}'));
                                    onPushSuccess && onPushSuccess(serviceCommcell.response);
                                }
                            });
                            break;
                        case cometFirstLoad.notificationType.serviceCommcellList:
                            clearInterval(fallBackTimeout);
                            enablePushMessages();
                            // stub to use later
                            serviceCommcellList = _.get(response, 'serviceCommcells');
                            break;
                        case cometFirstLoad.notificationType.consolidatedResponse:
                            // Stub for future enhancement - consolidated status of each commcell.
                            serviceCommcellList = _.get(response, 'serviceCommcells');
                            // No more messages will be given after consolidated response.
                            if (unsubscribeMultiCommcellLoad) {
                                unsubscribeMultiCommcellLoad();
                            }
                            break;
                    }
                }
            }

        }
        const onError = function(error) {
            console.log(error)
        }
        unsubscribeMultiCommcellLoad = pushService.subscribe(subscriptionInfo, onMessageReceived.bind(factory), onError.bind(factory));
    }

    return factory;

}]);

export default cvCommonModule;