/**
 * @author: suman Global error handling
 */
import * as constants from "./constants";
import * as utils from "./util";

export class GlobalErrorHandler {
	constructor() {
		this._initializeErrorHandlerConfig();
		this._setupEventErrorEventHandler();
		this._setupPromiseErrorHandlers();
		this.subscribers = [];
	}

	_initializeErrorHandlerConfig() {
		this._lastReportedError = {};
		this._lastReportedErrorTime = null;
	}

	_isImmediateDuplicate(newErrStack)
	{
		newErrStack = newErrStack || {};
		const previousErrKeys = Object.keys(this._lastReportedError);
		const newErrKeys = Object.keys(newErrStack);
		let identicalErrors = false;
		if(previousErrKeys.length && previousErrKeys.length === newErrKeys.length)
		{
			newErrKeys.every((errKey) => {
				if (this._lastReportedError[errKey] === newErrStack[errKey])
				{
					identicalErrors = true;
					return;
				}
			});
		}

		//timestamp comparision
		if(identicalErrors && this._lastReportedErrorTime)
		{
			const newErrDTime = new Date();
			const timeDiff = (newErrDTime.getTime() - this._lastReportedErrorTime.getTime()) / 1000;
			identicalErrors = timeDiff < 6; // skip similar errors with 5sec
		}
		return identicalErrors;
	}

	subscribe(init) {
		this.subscribers.push(init);
	}

	notifyHandlers(stack) {
		this._lastReportedError = stack;
		this._lastReportedErrorTime = new Date();
		this.subscribers.forEach(sub => sub(stack));
	}

	unsubscribe() {}

	_setupEventErrorEventHandler() {
		window.onerror = this._onErrorHandler.bind(this);
	}
	_setupPromiseErrorHandlers() {
		window.unhandledrejection = this._onePromiseErrorHandler.bind(this);
	}

	_onePromiseErrorHandler(event) {
		this._onErrorHandler(null, null, null, null, event.reason);
	}

	captureException(exception) {
		this._onErrorHandler(null, null, null, null, exception);
	}

	_onErrorHandler(message, source, lineno, colno, error) {
		let lines = [];
		if(_.isObject(error) && !_.isUndefined(error.stack)) {
			lines = error.stack.split("\n");
		}
		if(_.isString(error)) {
			lines[0] = error
		}
		const stack = [];
		let parts;
		let element;

		for (var i = 0, j = lines.length; i < j; ++i) {
			if ((parts = constants.chrome.exec(lines[i]))) {
				element = this._chromeErrorHandler(parts);
			} else if ((parts = constants.ie.exec(lines[i]))) {
				element = this._ieErrorHandler(parts);
			} else if ((parts = constants.firefox.exec(lines[i]))) {
				element = this._firefoxErrorHandler(parts, i);
			}
			stack.push(element);
		}

		const stackError = {
			name: error.name ? error.name : error,
			message: error.message? error.message: error,
			info: stack,
			url: window.location.href
		};

		if(!this._isImmediateDuplicate(stackError))
			this.notifyHandlers(stackError);
	}

	_chromeErrorHandler(parts) {
		let submatch;
		const isNative = utils.isNative(parts[2]);
		const isEval = utils.isEval(parts[2]);
		if (isEval && (submatch = constants.chromeEval.exec(parts[2]))) {
			parts[2] = submatch[1];
			parts[3] = submatch[2];
			parts[4] = submatch[3];
		}
		return {
			url: !isNative ? parts[2] : null,
			functionName: parts[1] || constants.UNKNOWN_FUNCTION,
			args: isNative ? [parts[2]] : [],
			line: parts[3] ? +parts[3] : null,
			columnNumber: parts[4] ? +parts[4] : null
		};
	}

	_ieErrorHandler(parts) {
		return {
			url: parts[2],
			functionName: parts[1] || constants.UNKNOWN_FUNCTION,
			args: [],
			line: +parts[3],
			columnNumber: parts[4] ? +parts[4] : null
		};
	}

	_firefoxErrorHandler(parts, i) {
		const isEval = utils.isFireFoxEval(parts[3]);
		if (isEval && (submatch = constants.firefoxEval.exec(parts[3]))) {
			// throw out eval line/column and use top-most line number
			parts[3] = submatch[1];
			parts[4] = submatch[2];
			parts[5] = null; // no column when eval
		} else if (i === 0 && !parts[5] && !_.isUndefined(ex.columnNumber)) {
			stack[0].column = ex.columnNumber + 1;
		}
		return {
			url: parts[3],
			functionName: parts[1] || constants.UNKNOWN_FUNCTION,
			args: parts[2] ? parts[2].split(",") : [],
			lineNumber: parts[4] ? +parts[4] : null,
			columnNumber: parts[5] ? +parts[5] : null
		};
	}
}

const globalErrorHandler = new GlobalErrorHandler();

globalErrorHandler.subscribe(stack => {
	utils.http.post(cvUtil.CONTEXT_PATH + "/consoleError.do", stack, {
		headers: {
			csrf: cvUtil.readCookie('csrf')
		}
	})}
);

export default globalErrorHandler;
