import * as GridConstants from '../newGrid/grid.constants';
import ColumnFilter from "../newGrid/columnFilter";

const DEFAULT_FILTER_CONDITION_OPTIONS = [
	GridConstants.FILTER_CONDITIONS.EQUALS,
	GridConstants.FILTER_CONDITIONS.DOES_NOT_EQUAL
];

export default class Dropdown extends ColumnFilter {
	static get FILTER_CONDITION_OPTIONS() {
		return DEFAULT_FILTER_CONDITION_OPTIONS;
	}

	get wrapper() {
		return this.isBuilt() ? this.dropDownList.wrapper : null;
	}

	initialize(options) {
		super.initialize(options);
		if (options.filterUrl) {
			this.dataSrc = new DropDownDataSource(options, this);
		}
		this.itemsToAddAfterBuild = [];
	}

	build(element, initialFilterData) {
		super.build(element);
		element.kendoDropDownList({
			dataTextField: this.dataTextField,
			dataValueField: this.dataValueField,
			dataSource: this.dataSrc,
			select: this.onDropdownFilterSelect.bind(this),
			change: this.onFilterInputChange.bind(this),
			value: this.value,
			index: null
		});
		this.dropDownList = element.getKendoDropDownList();
		this.addItems(this.itemsToAddAfterBuild);
		this._addEmptyValueToDataSource();
		this._addValueToDataSource();
		this.saveLabel();
	}

	_addValueToDataSource() {
		if (!this.isBuilt()) {
			return;
		}
		const label = this.getLabel();
		const currentValue = this.getValue();
		const currentData = this.dropDownList.dataSource.data();
		let valueInDataSource = false;
		_.forEach(currentData, data => {
			if (data[this.dataValueField] === currentValue) {
				valueInDataSource = true;
			}
		});
		if (!valueInDataSource) {
			const newValue = {};
			newValue[this.dataTextField] = label || currentValue;
			newValue[this.dataValueField] = currentValue;
			this.dropDownList.dataSource.add(newValue);
		}
	}

	_addEmptyValueToDataSource() {
		if (!this.isBuilt()) {
			return;
		}
		const currentData = this.dropDownList.dataSource.data();
		let valueInDataSource = false;
		_.forEach(currentData, data => {
			if (data[this.dataValueField] === "") {
				valueInDataSource = true;
			}
		});
		if (!valueInDataSource) {
			const newValue = {};
			newValue[this.dataTextField] = "";
			newValue[this.dataValueField] = "";
			this.dropDownList.dataSource.insert(0, newValue);
		}
	}

	onColumnMenuOpen() {
		this._addEmptyValueToDataSource();
		this._addValueToDataSource();
		super.onColumnMenuOpen();
	}

	onDropdownFilterSelect(e) {
		return super.onDropdownFilterSelect(e, e.dataItem.value);
	}

	loadUrlParam() {
		if (this.filterLabelQueryParam && this.angularLibs && this.angularLibs.$location) {
			const searchParams = this.angularLibs.$location.search();
			if (this.filterLabelQueryParam in searchParams) {
				// TODO allow dev to parse the label
				this.label = searchParams[this.filterLabelQueryParam];
			}
		}
		super.loadUrlParam();
	}

	updateUrlParam() {
		if (this.filterLabelQueryParam && this.angularLibs && this.angularLibs.$location && this.angularLibs.$scope) {
			let label = this.getLabel();
			if (_.isEmpty(label) || _.isUndefined(label) || _.isNull(label)) {
				label = null;
			}
			// TODO allow dev to format the label for url param
			const self = this;
			this.angularLibs.$scope.$applyAsync(() => {
				self.angularLibs.$location.search(self.filterLabelQueryParam, label).replace();
			});
		}
		super.updateUrlParam();
	}

	fromViewValue(val) {
		super.fromViewValue(val);
		this._addValueToDataSource();
		this.applyValue();
	}

	addItem(item) {
		if (this.isBuilt()) {
			return this.dropDownList.dataSource.add(item);
		}
		this.itemsToAddAfterBuild.push(item);
		return null;
	}

	addItems(items) {
		if (this.isBuilt()) {
			return _.map(items, item => this.addItem(item));
		}
		this.itemsToAddAfterBuild.push(...items);
		return null;
	}

	getItems() {
		if (this.isBuilt()) {
			return this.dropDownList.dataSource.data();
		}
		return null;
	}

	getItemByValue(val) {
		if (this.isBuilt()) {
			const data = this.dropDownList.dataSource.data();
			for (let i = 0; i < data.length; i++) {
				if (data[i][this.dataValueField] === val) {
					return data[i];
				}
			}
		}
		return null;
	}

	removeItem(item) {
		if (this.isBuilt()) {
			return this.dropDownList.dataSource.remove(item);
		}
		this.itemsToAddAfterBuild = this.itemsToAddAfterBuild.filter(
			savedItem => savedItem !== item
		);
		return null;
	}

	removeItemByUid(uid) {
		if (this.isBuilt()) {
			const data = this.dropDownList.dataSource.data();
			for (let i = 0; i < data.length; i++) {
				if (data[i].uid === uid) {
					return this.dropDownList.dataSource.remove(data[i]);
				}
			}
		}
		return null;
	}

	removeItemByValue(val) {
		if (this.isBuilt()) {
			const data = this.dropDownList.dataSource.data();
			for (let i = 0; i < data.length; i++) {
				if (data[i][this.dataValueField] === val) {
					return this.dropDownList.dataSource.remove(data[i]);
				}
			}
		}
		this.itemsToAddAfterBuild = this.itemsToAddAfterBuild.filter(
			savedItem => savedItem[this.dataValueField] !== val
		);
		return null;
	}

	setValue(val) {
		this.value = val;
		this.saveLabel();
		super.setValue(val);
	}

	saveLabel() {
		if (this.isBuilt()) {
			const dataItem = this.dropDownList.dataItem();
			if (dataItem) {
				this.label = dataItem[this.dataTextField];
			} else {
				this.label = undefined;
			}
		}
	}

	getLabel() {
		return this.label;
	}

	setInputValue(val) {
		if (this.isBuilt()) {
			this.dropDownList.value(val);
			this.refreshInputTooltip();
		}
	}

	getInputValue() {
		if (this.isBuilt()) {
			return this.dropDownList.value();
		}
		return null;
	}

	getInputTooltipText() {
		if (this.isBuilt()) {
			return this.dropDownList.text();
		}
		return null;
	}

	destroy() {
		super.destroy();
		if (this.isBuilt()) {
			this.dropDownList.destroy();
		}
	}
}

class DropDownDataSource extends ColumnFilter.ColumnFilterDataSource {
	constructor(options, columnFilter) {
		super(options, columnFilter);
		// this.sort = {
		// 	field: columnFilter.dataTextField,
		// 	dir: "asc"
		// };
		if (_.isFunction(options.filterUrl)) {
			const self = this;
			const readFunction = options.filterUrl;
			this.transport.read = callbacks => {
				readFunction.call(
					columnFilter,
					{
						...callbacks,
						success: data => {
							callbacks.success(data);
							if (!self.firstReadFinished) {
								this.columnFilter._addValueToDataSource();
								this.columnFilter.applyValue();
								self.firstReadFinished = true;
							}
						},
						error: e => {
							callbacks.error(e);
						}
					},
					columnFilter
				);
			};
		}
	}

	// _appendSelectionToData(data) {
	// 	if (!this.columnFilter) {
	// 		return data;
	// 	}
	// 	const currentSelection = this.columnFilter.getValue();
	// 	if (!currentSelection) {
	// 		return data;
	// 	}
	// 	if (!data) {
	// 		const result = {};
	// 		result[this.columnFilter.dataTextField] = this.columnFilter.getLabel();
	// 		result[this.columnFilter.dataValueField] = currentSelection;
	// 		return [result];
	// 	}
	// 	if (!_.isArray(data)) {
	// 		data = [data];
	// 	}
	// 	let addSelection = true;
	// 	_.forEach(data, obj => {
	// 		if (obj[this.columnFilter.dataValueField] === currentSelection) {
	// 			addSelection = false;
	// 		}
	// 	});
	// 	if (addSelection) {
	// 		const result = {};
	// 		result[this.columnFilter.dataTextField] = this.columnFilter.getLabel();
	// 		result[this.columnFilter.dataValueField] = currentSelection;
	// 		data.push(result);
	// 	}
	// 	return data;
	// }
}
