import { cvCommonModule } from "common/js/modules";
import "common/js/directives/monitor-item.directive.js";
import NotifQueue from "common/js/directives/NotifQueue";
import 'jobs/js/services/jobs.svc';
import "modules/alerts/js/services/alerts.svc.js"
import "events/js/services/events.svc.js";
import CvDOMPurify from 'common/js/DOMPurify';

const MonitoringNotif = [
    function() {
        return {
            restrict: 'E',
            templateUrl: appUtil.appRoot + 'common/partials/monitoringNotf.jsp',
            replace:true,
            bindToController: true,
            controllerAs: 'mctrl',
            controller: ['$uibModal','$filter', '$location','jobService', 'triggeredAlertsService', 'userPrefService', 'eventService', 'cvToaster', '$q', '$timeout', '$scope', 'cvUtil', 'cvLoc', '$state', 'multiCommcell', function($modal, $filter, $location, jobService, triggeredAlertsService, userPrefService, eventService, cvToaster, $q, $timeout, $scope, cvUtil, cvLoc, $state, multiCommcell) {
                const ctrl = this;
                this.$onInit = () => {
                    ctrl.type = 'job';
                    $scope.isLoading = true;
                    if(cvConfig.pushEnabled && _.isUndefined(cv.userPref.alertNotifSubscription)) {
                        userPrefService.setBooleanUserPref('alertNotifSubscription', true).success(() => {
                            userPrefService.updateUserPrefs();
                        });
                    };
                    ctrl.linkinfo = {
                        'job': {text: cvLoc('label.viewActiveJobs'), onClick: () => redirectJobPage('activeJobs')},
                        'alert':{text: cvLoc('label.viewTriggeredAlerts'), onClick: () => redirectJobPage('triggeredAlerts')},
                        'event':{text: cvLoc('label.viewEvents'), onClick: () => redirectJobPage('events')}
                    };
                    ctrl.isCometApp = cv.isCometApp;
                    ctrl.isGlobalOffline = multiCommcell.isGlobalOffline;
                    ctrl.noOfOfflineCells = multiCommcell.noOfOfflineCells;

                    const eventsParams = {
                            pagingInfo: JSON.stringify({
                                    'limit' : 5
                            })
                        };

                    $scope.newNotification = function(ind) {
                        // fetchData();
                        switch(ind) {
                            case 0 : fetchJobData(); break;
                            case 1 : fetchAlertData(); break;
                            case 2 : fetchEventsData(); break;
                            case 3 : fetchCommcellData(); break;
                        }
                    }

                    const getAlertSeverityClass = (severity) => {
                            switch (severity) {
                                case 1:
                                    return "critical";
                                case 2:
                                    return "major";
                                case 3:
                                    return "info";
                            }
                    };
                    
                    const getEventSeverityClass = (severity) => {
                            switch (severity) {
                            case 9:
                                return "critical";
                            case 6:
                                return "major";
                            case 3:
                                return "minor";
                            case 0:
                                return 'info';
                        }
                    };

                    function fetchCommcellData() {
                        ctrl.commcellData = multiCommcell.getOfflineCommcells();
                    }

                    function timeSince(timeSource) {

                        const eventTime = new Date(timeSource * 1000);
                        const timeAgo = new Date() - eventTime;

                        // time ago in seconds 
                        const timeAgoSeconds = Math.round(timeAgo/1000);
                        const timeAgoMinutes = Math.round(timeAgoSeconds/60);

                        if(timeAgoMinutes < 60) {
                            if (timeAgoMinutes < 1) {
                                return timeAgoSeconds + ' ' + cvLoc('label.secondsAgo');
                            } else {
                                return timeAgoMinutes + ' ' + (timeAgoMinutes === 1 ? cvLoc('label.minuteAgo') : cvLoc('label.minutesAgo'));
                            }
                        }

                        return $filter('date')(timeSource * 1000, 'MMM d, y h:mm:ss a');;
                    }

                    function loadAlertDetails (liveFeedId, client) {
                        triggeredAlertsService.getAlertDescription({
                            'alertId': liveFeedId,
                            'commcellName': _.get(client, 'commCellName')
                        }).success(function (data) {
                            let purifier = new CvDOMPurify();
                            const sanitizedContent = purifier.sanitize(data.description);
                            const parsedAlert = triggeredAlertsService.parseAlertContent(sanitizedContent);
                            const alertContent =
                              (parsedAlert && triggeredAlertsService.getTransformedAlertContent(angular.copy(parsedAlert), data.severity)) ||
                              sanitizedContent;
                            let alertTitle = parsedAlert ? cvLoc('label.alertDetails') : "";

                            $modal.open({
                                windowClass: 'descriptionClass',
                                backdrop: 'static',
                                template:
                                  '<div class="setup-title"><h1>' +
                                  alertTitle +
                                  '</h1></div>' +
                                  '<div class="bootstrap-wrapper modal-margin">' +
                                  alertContent +
                                  '<div class="button-container"><button class="btn btn-default" data-ng-click="close()">' +
                                  cvLoc('label.close') +
                                  '</button></div></div>',
                                controller: [
                                    '$scope',
                                    '$uibModalInstance',
                                    function($scope, $modalInstance) {
                                        $scope.close = function() {
                                            $modalInstance.dismiss();
                                        };
                                    }
                                ]
                            });
                            }).error(function (e) {
                                cvToaster.showErrorMessage({
                                    'ttl' : '5000',
                                    'message': cvLoc('error.triggeredAlertsLoadError'),
                                });
                            });
                        };
                
                    function redirectJobPage(state, param){
                            $state.transitionTo(state, param || {}, {
                                reload: true
                            })
                    }

                    function fetchJobData() {
                        ctrl.jobLoading = true;
                        if(!ctrl.jobDropdown) return;
                        const deferred = $q.defer();
                        jobService.getJobSummary().then(({data})=>{
                            const jobList = [
                                {state: 'failed', status: 'Failed', title: cvLoc('Failed'), info: data.failedJobs},
                                {state: 'warning', status: 'Suspended', title: cvLoc('Suspended'), info: data.suspendedJobs},
                                {state: 'complete', status: 'Completed', title: cvLoc('Completed'), info: data.completedJobs},
                                {state: 'pending', status: 'Pending', title: cvLoc('Pending'), info: data.pendingJobs},
                                {state: 'running', status: 'Running', title: cvLoc('Running'), info: data.runningJobs},
                                {state: 'running', status: 'Waiting', title: cvLoc('Waiting'), info: data.waitingJobs},
                                {state: 'running', status: 'Queued', title: cvLoc('Queued'), info: data.queuedJobs},
                                {state: 'failed', status: 'Killed', title: cvLoc('Killed'), info: data.killedJobs},
                                {state: 'warning', view: 'anomalousJobs', title: cvLoc('Anomalous'), info: data.anomalousJobs}
                            ].filter(({info})=>info>0);
                            jobList.forEach(job=>job.onClick = () => redirectJobPage('activeJobs',{status: job.status, view: job.view}));
                            ctrl.jobs = new NotifQueue(jobList,10).data;   
                            ctrl.jobLoading = false;
                            $q.resolve();
                        },(e)=>{
                            $q.reject(e);
                        })
                    }

                    function fetchAlertData() {
                        ctrl.alertLoading = true;
                        if(!ctrl.alertDropdown) return;
                        const deferred = $q.defer();
                        triggeredAlertsService.getAlerts({'pageSize': 5,'pageNumber': 1}, true).then(({data})=>{
                            const alertQueue = (data.feedsList || []).map(({severity,liveFeedId,alertName,alertcriteria,client})=>({
                                state: getAlertSeverityClass(severity),
                                onClick: ()=> loadAlertDetails(liveFeedId, client),
                                title: alertName, 
                                subtitle: alertcriteria, 
                                id: liveFeedId
                            }));
                            ctrl.alerts = new NotifQueue(alertQueue).data;
                            ctrl.alertLoading = false;
                            $q.resolve();
                        },(e)=>{
                            $q.reject();
                        })
                    } 

                    if (cvConfig.pushEnabled && cv.userPref.alertNotifSubscription) {
                        enablePush();
                    }

                    function enablePush() {
                        ctrl.unsubscribeAlerts =  subscribeAlerts(onPushSuccess.bind(this), onPushError.bind(this));
                    }

                    function subscribeAlerts(onMessage, onError) {
                        // This gets LIVEFEED (Console Alerts)
                        const subscribeInfo = {
                            msg: 'subscribe',
                            type: 'ALERT'
                        };
                        pushService.subscribe(subscribeInfo, onMessage, onError);
                    }

                    function onPushSuccess(data) {
                        if (data.feedsList && data.feedsList[0].alertName) {
                            let pushAlert = (data.feedsList || []).map(({severity,liveFeedId,alertName,alertcriteria, alertType})=>({
                                state: getAlertSeverityClass(severity),
                                onClick: ()=> loadAlertDetails(liveFeedId),
                                title: alertName, 
                                subtitle: alertcriteria, 
                                alertType: alertType,
                                id: liveFeedId
                            }));
                            if(ctrl.alertDropdown && ctrl.alerts) {
                                // If Alert notification dropdown is open then append latest console alert into it
                                ctrl.alerts.unshift(pushAlert[0]);
                                ctrl.alerts.pop();
                            }
                            if ($state.current.name !== 'triggeredAlerts') {
                                // Don't show toasted notifications if triggered Alerts page is open
                                let alertMessage = `<h4>${pushAlert[0].title}</h4><div>${pushAlert[0].alertType}</div><div>${pushAlert[0].subtitle}</div>`;
                                $scope.$apply(()=> {
                                    cvToaster.showWarnMessage ({
                                        ttl: '5000', // 5 sec
                                        message: alertMessage
                                    })
                                });
                            }
                        }
                    }
                
                    function onPushError(error) {
                        cvToaster.showErrorMessage ({
                            ttl: '5000', // 5 sec
                            message: error
                        });
                    }

                    function fetchEventsData() {
                        ctrl.eventLoading = true;
                        if(!ctrl.eventDropdown) return;
                        const deferred = $q.defer();
                        eventService.getEvents(eventsParams, true).then(({data})=>{
                            const eventsQueue = data.map(({severity,id,description,timeSource,clientEntity})=>({
                                state: getEventSeverityClass(severity), 
                                link: cvUtil.modifyUrlForComet(`#/event/${id}`, _.get(clientEntity, 'commCellName', 'IDP')),
                                target: cv.cometTarget,
                                title: description, 
                                subtitle: timeSince(timeSource), 
                                id: id
                            }));

                            ctrl.events = new NotifQueue(eventsQueue).data;
                            ctrl.eventLoading = false;
                            $q.resolve();
                        },(e)=>{
                            $q.reject(e);
                        })
                    }
                    const fetchData = function () {
                        $scope.isLoading = true;
                        $q.all([
                            jobService.getJobSummary(),
                            triggeredAlertsService.getAlerts({
                                'pageSize': 5,
                                'pageNumber': 1
                            }, true),
                            eventService.getEvents(eventsParams, true)
                        ]).then(res => {
                            // getJobSummary sets the information for first tab
                                let jobList = [
                                    {state: 'failed', status: 'Failed', title: cvLoc('Failed'), info: res[0].data.failedJobs},
                                    {state: 'warning', status: 'Suspended', title: cvLoc('Suspended'), info: res[0].data.suspendedJobs},
                                    {state: 'info', status: 'Completed', title: cvLoc('Completed'), info: res[0].data.completedJobs},
                                    {state: 'pending', status: 'Pending', title: cvLoc('Pending'), info: res[0].data.pendingJobs},
                                    {state: 'running', status: 'Running', title: cvLoc('Running'), info: res[0].data.runningJobs},
                                    {state: 'running', status: 'Waiting', title: cvLoc('Waiting'), info: res[0].data.waitingJobs},
                                    {state: 'running', status: 'Queued', title: cvLoc('Queued'), info: res[0].data.queuedJobs},
                                    {state: 'failed', status: 'Killed', title: cvLoc('Killed'), info: res[0].data.killedJobs},
                                    {state: 'warning', view: 'anomalousJobs', title: cvLoc('Anomalous'), info: res[0].data.anomalousJobs}
                                ];
                                jobList.forEach(job => {
                                    job.onClick = () => redirectJobPage('activeJobs', {
                                        status: job.status,
                                        view: job.view
                                    })
                                })
                            const jobsQueue = new NotifQueue(jobList, 10);

                            // filter the jobs that currently have > 0
                            const jobsQueueFiltered = jobsQueue.data.filter(queue => _.get(queue, 'info') > 0);
                        
                            // filter the data for job alerts and sort the request by date to show the last 5 alerts
                            const filteredAlerts = res[1].data.feedsList || [];
                            const alertQueue = filteredAlerts.map(alert => {
                                return {state: getAlertSeverityClass(alert.severity), onClick: () => loadAlertDetails(alert.liveFeedId), title: alert.alertName, subtitle: alert.alertcriteria, id: alert.liveFeedId}
                            });

                            //filter the data for events and sort the request by date to show the last 5 events
                            const eventsQueue = res[2].data.map(event => {
                                return {state: getEventSeverityClass(event.severity), link: '#/event/' + event.id, title: event.description, subtitle: timeSince(event.timeSource), id: event.id}
                            });
                            
                            ctrl.events = new NotifQueue(eventsQueue).data;
                            ctrl.alerts = new NotifQueue(alertQueue).data;
                            ctrl.jobs = jobsQueueFiltered;
                            $scope.isLoading = false;
                        });
                    };
                };
                this.$onDestroy = () => {
                    if (cvConfig.pushEnabled && cvConfig.enableAlertSubscription) {
                        $scope.$on('$destroy', function() {
                            ctrl.unsubscribeAlerts();
                        });
                    }
                }
            }]
        }
    }
]

cvCommonModule.directive('monitorNotification', MonitoringNotif);
export default cvCommonModule;
