import CvWizard from 'common/js/directives/cv-wizard/cvWizard.js';
import CvWizardStep from 'common/js/directives/cv-wizard/cvWizardStep.js';
import EventEmitter from 'eventemitter3';

/**
 * This service provides an api to programmatically navigate a wizard, listen to
 * events affecting the specified step, and read the current state of the wizard
 * or step.
 */
export default class CvWizardStepInstanceService extends EventEmitter {
	constructor(wizard, step) {
		super();
		this._wizard = wizard;
		this._step = step;
		this._wizardEventListeners = {};
		this._registerWizardListeners();
	}
	
	/**
	 * Registers an event listener to the specified emitter which will reemit
	 * events it hears if the service's step is active.
	 */
	_registerWizardListener(emitter, eventType) {
		const listener = event => {
			if (this._step.isActive()) {
				// If this step is active, reemit the event
				this.emit(eventType, event);
			}
		};
		emitter.on(eventType, listener);
		this._wizardEventListeners[eventType] = {
			emitter,
			listener,
			eventType,
		};
	}
	
	/**
	 * Registers event listeners for the wizard and wizard step event emitters.
	 * If the service's step is active, the event will be reemitted through the
	 * service.
	 */
	_registerWizardListeners() {
		this._registerWizardListener(this._step, CvWizardStep.EVENT.STEP_ENTER);
		this._registerWizardListener(this._step, CvWizardStep.EVENT.STEP_LEAVE);
		this._registerWizardListener(this._wizard, CvWizard.EVENT.NEXT);
		this._registerWizardListener(this._wizard, CvWizard.EVENT.PREVIOUS);
		this._registerWizardListener(this._wizard, CvWizard.EVENT.GOTO);
		this._registerWizardListener(this._wizard, CvWizard.EVENT.FINISH);
	}
	
	/**
	 * Remove the event listeners from the wizard and step.
	 */
	_deregisterWizardListeners() {
		_.forEach(this._wizardEventListeners, el => {
			el.emitter.removeListener(el.eventType, el.listener);
		});
		this._wizardEventListeners = {};
	}
	
	/**
	 * @param {boolean} completed 
	 */
	setCompleted(completed) {
		this._step.setCompleted(completed);
	}
	
	/**
	 * @returns {boolean} Whether or not the step has been marked as complete.
	 */
	isCompleted() {
		return this._step.isCompleted();
	}
	
	/**
	 * @returns {*} The model assigned to the wizard.
	 */
	getModel() {
		return this._wizard.getModel();
	}
	
	setModel(model) {
		this._wizard.setModel(model);
	}
	
	/**
	 * @returns {boolean} Whether or not the step is marked as skippable.
	 */
	isSkippingEnabled() {
		return this._wizard.isSkippingEnabled();
	}
	
	/**
	 * @returns {number} The index of the step.
	 */
	getIndex() {
		return this._wizard.indexOf(this._step);
	}
	
	/**
	 * Moves the wizard to the step at the specified index.
	 * @param {number} index The index of the step to go to.
	 * @returns {boolean} Whether or not the step was changed successfully.
	 */
	goToStep(index) {
		return this._wizard.goToStep(index);
	}
	
	/**
	 * Move the wizard to the next step. The current step will be marked as complete.
	 * @returns {boolean} Whether or not the step was changed successfully.
	 */
	next() {
		return this._wizard.next();
	}
	
	/**
	 * Move the wizard to the next step. The current step will not be marked as complete.
	 * @returns {boolean} Whether or not the step was changed successfully.
	 */
	skip() {
		return this._wizard.skip();
	}
	
	/**
	 * Move the wizard to the previous step. The current step will be marked as incomplete.
	 * @returns {boolean} Whether or not the step was changed successfully.
	 */
	previous() {
		return this._wizard.previous();
	}
	
	/**
	 * Calls the finish listener of the wizard.
	 */
	finish() {
		this._wizard.finish();
	}

	isLastStep() {
		return this._wizard.isLastStep();
	}
}