import 'common/js/directives/cv-wizard/cv-wizard-step.directive.js';
import { cvCommonModule } from "common/js/modules";
import CvWizard from 'common/js/directives/cv-wizard/cvWizard.js';
import { CvWizardButtonTypes } from 'common/js/directives/cv-wizard/cv-wizard-button.directive.js';
import CvWizardStep from 'common/js/directives/cv-wizard/cvWizardStep.js';
import CvWizardStepInstanceService from 'common/js/directives/cv-wizard/cvWizardStepInstanceService.js';
import EventEmitter from 'eventemitter3';

export class CvWizardDirective {
    constructor() {
        this.restrict = 'E';
        this.templateUrl = appUtil.appRoot + 'adminConsole/partials/cvWizard.jsp';
        this.transclude = true;
        this.bindToController = true;
        this.controllerAs = 'wizCtrl';
        this.scope = {
            model: '=?cvModel',
            title: '=cvTitle',
            enableSkipping: '=cvEnableSkipping',
            hideNav: '=?cvHideNav',
            skipLabel: '@?cvSkipLabel',
            previousLabel: '@?cvPreviousLabel',
            nextLabel: '@?cvNextLabel',
            finishLabel: '@?cvFinishLabel',
            cancelLabel: '@?cvCancelLabel',
            onCancel: '=?cvOnCancel',
            onFinish: '=?cvOnFinish',
            useClassicHeader: '=?cvUseClassicHeader',
            initialStepId: '@?cvInitialStepId',
            // TODO allow active step binding
        };
        this.controller = CvWizardController;
    }
    
    link(scope, element) {
        scope.wizCtrl.element = element;
        scope.wizCtrl.directiveLinked = true;
    }
};

export class CvWizardController {
    static $inject = ['$scope', 'cvLoc', 'cvNavigationFactory'];
    constructor($scope, cvLoc, cvNavigationFactory) {
        this.$scope = $scope;
        this.cvLoc = cvLoc;
        this.cvNavigationFactory = cvNavigationFactory;
        
        // initialStepLoaded is used to prevent showing the wizard until the
        // initial step has been loaded. If the initial step has not been specified
        // it will be used to keep the wizard on step-index 0 as steps are loaded,
        // up until the user navigates the wizard
        this.initialStepLoaded = false;
    }
    
    $onInit() {
        // Default useClassicHeader to true
        this.useClassicHeader = this.useClassicHeader !== false;
        
        this.cvNavigationFactory.setNavHidden(!!this.hideNav);
        this.skipLabel = this.skipLabel || this.cvLoc('label.skip');
        this.previousLabel = this.previousLabel || this.cvLoc('label.previous');
        this.nextLabel = this.nextLabel || this.cvLoc('label.next');
        this.finishLabel = this.finishLabel || this.cvLoc('label.finish');
        this.cancelLabel = this.cancelLabel || this.cvLoc('label.cancel');
        
        if (!this.useClassicHeader) {
            // Set the background
            angular.element('#wrapper').find('.wrapper-view').first().addClass('cv-wizard-wrapper-view');
        }
        
        const self = this;
        this.wizard = new CvWizard({
            // Use getter/setters here to allow for two-way binding between the
            // CvWizard data object and the controller
            get model() { return self.model; },
            set model(value) { self.model = value; },
            get enableSkipping() { return self.enableSkipping; },
            set enableSkipping(value) { self.enableSkipping = value; },
        });
        this.registerWizardListeners();
        
        this.$scope.$on(CvWizardStep.EVENT.STEP_DESTROY, (event, details) => {
            if (details.wizardId === this.wizard.getId()) {
                event.stopPropagation();
                if (this.directiveLinked && !this.destroyed) {
                    this.wizard.removeStepById(details.stepId, !this.initialStepLoaded);
                    this.refreshStepOrder();
                }
            }
        });
        this.$scope.$on('$destroy', () => {
            this.destroyed = true;
            this.cvNavigationFactory.setNavHidden(false);
            if (!this.useClassicHeader) {
                angular.element('#wrapper').find('.wrapper-view').first().removeClass('cv-wizard-wrapper-view');
            }
        });
    }
    
    registerWizardListeners() {
        [CvWizard.EVENT.NEXT, CvWizard.EVENT.PREVIOUS, CvWizard.EVENT.GOTO].forEach(eventType => {
            this.wizard.once(eventType, () => {
                // If the user performs navigation, we assume the initial step
                // has already been loaded.
                this.initialStepLoaded = true;
            });
        });
        this.wizard.on(CvWizard.EVENT.LATE_FINISH, this._onFinish.bind(this));
        this.wizard.on(CvWizard.EVENT.CANCEL, this._onCancel.bind(this));
    }
    
    _onCancel() {
        if (angular.isFunction(this.onCancel)) {
            this.onCancel();
        }
    }
    
    _onFinish() {
        if (angular.isFunction(this.onFinish)) {
            this.onFinish();
        }
    }
    
    goToStep(index) {
        this.wizard.goToStep(index);
    }
    
    getLabels() {
        return {
            [CvWizardButtonTypes.SKIP]: this.skipLabel,
            [CvWizardButtonTypes.NEXT]: this.nextLabel,
            [CvWizardButtonTypes.PREVIOUS]: this.previousLabel,
            [CvWizardButtonTypes.FINISH]: this.finishLabel,
            [CvWizardButtonTypes.CANCEL]: this.cancelLabel,
        };
    }
    
    /**
     * Requests a step order refresh to run asynchronously
     */
    requestStepOrderRefresh() {
        if (this.stepOrderRefreshRequested) {
            return;
        }
        this.stepOrderRefreshRequested = true;
        return new Promise((resolve, reject) => {
            this.$scope.$applyAsync(() => {
                this.stepOrderRefreshRequested = false;
                this.refreshStepOrder();
                resolve();
            });
        });
    }
    
    /**
     * Traverses the DOM to determine the step order. The active step will be
     * preserved according to id
     */
    refreshStepOrder() {
        if (!this.directiveLinked || this.destroyed) {
            return;
        }
        let activeStepId = this.wizard.getActiveStepId();
        const stepOrder = [];
        this.element
            .find(`[cv-wizard-step-parent=${this.wizard.getId()}]`)
            .each((index, elem) => {
                const stepId = $(elem).attr('id');
                stepOrder.push(stepId);
            });
        this.wizard.setStepOrder(stepOrder);
        
        if (!this.initialStepLoaded) {
            if (this.initialStepId) {
                // Initial id has been specified. Check if the initial step has
                // been added. If it has, activate that step
                const index = stepOrder.indexOf(this.initialStepId);
                if (index >= 0) {
                    this.initialStepLoaded = true;
                    this.wizard.setActiveStepIndexById(this.initialStepId);
                    this.wizard.getActiveStep()._active = true;
                }
            } else {
                // No initial step id specified. If the user has not navigated,
                // we keep the user on the first step as the steps are updated.
                // This is a workaround as we have no way of knowning how many
                // steps will be added
                const firstStep = this.wizard.getFirstStep();
                if (!firstStep.isActive()) {
                    firstStep._active = true;
                    for (let i = 1; i < this.wizard.getStepCount(); i++) {
                        // Ensure the remaining steps are inactive
                        this.wizard.getStep(i)._active = false;
                    }
                    this.wizard.setActiveStepIndex(0);
                }
            }
        } else if (activeStepId) {
            // Preserve the active step
            this.wizard.setActiveStepIndexById(activeStepId);
        }
    }
    
    /**
     * Creates a child scope and a stepServiceInstance for use by a step. The
     * child scope's parent will be this directive's parent.
     */
    createStepControllerLocals(step) {
        return {
            scope: this.$scope.$parent.$new(false, this.$scope.$parent),
            stepServiceInstance: new CvWizardStepInstanceService(this.wizard, step),
        };
    }
    
    getActiveStepTitle() {
        const step = this.wizard.getActiveStep();
        return step ? step.getTitle() : "";
    }
    
    getStepIds() {
        return this.wizard.getStepOrder();
    }
    
    isSkippingEnabled() {
        return this.wizard.isSkippingEnabled();
    }
    
    getStepTitle(id) {
        const step = this.wizard.getStepById(id);
        return step ? step.getTitle() : '';
    }
    
    isStepComplete(id) {
        const step = this.wizard.getStepById(id);
        return step ? step.isCompleted() : false;
    }
    
    isStepActive(id) {
        return this.wizard.getActiveStepId() === id;
    }
};

cvCommonModule.directive('cvWizard', () => new CvWizardDirective());
