(function() {
	'use strict';

	var cvCommon = angular.module('cvCommon');

	/*-
	 * usage: <cv-advanced-tree cvparams="cvparams" class="advanced-tree"></cv-advanced-tree>
	 * cvparams: {
	 *
	 *		mode: <string> (optional, defaults to edit) 'edit': checkboxes will display for the user to make selections and exclusions;
	 * 													'view': checkboxes will display based on data sent in cvparams.selected and cvparams.excluded. User cannot edit.
	 * 													'summary': only displays the selected and excluded nodes sent by cvparams.selected and cvparams.excluded. requestChildren and processChilren will not be needed in this mode
	 *		clientSidePagingSize: <integer> (optional) if this parameter is used (and nonzero), enables client-side-paging so each list in the tree will have a "Load more" button and load the specified number of items at a time.
	 *		useExternalPaging: <boolean> (optional) enables external paging. Cannot be used with client-side paging. External paging takes priority if external and client-side are both defined.
	 *
	 * 		// the following are required for edit mode
	 *  	requestChildren: function(node) returns a promise which contains data with children of the given node. Data should be an empty array if the node with the given id has no child items. If using external paging, the node will have a property node.page which starts at zero and keeps track of the current page.
     *		processChildren: function(data) processes the data in the promise of requestChildren() and returns an array of children in the format below.
	 * 		topNodes: <array> array of top-level nodes. [
	 * 			externalPaging : <boolean> (optional) enables external paging on top node. Node Level property.If this is enabled searching will be treated as server side searching.
	 * 			externalPageSize : <integer> (optional) if externalPaging option is used , this allows to indicate page sie. Default value is 20.
	 * 			clientSidePaging : <boolean> (optional) enables clientSide paging on top node. Node Level property.
	 * 			clientSidePageSize : <integer> (optional) if clientSidePaging option is used , this allows to indicate page sie. Default value is 20.
	 * 			returnAllSelectedChildren: <boolean> (optional) If a node is expanded and selected such that all its children are also selected, return that node alongwith its children nodes as 'selected'. This is useful when a node is selcted and then one child node is deselected.
	 *
	 * 		multiSelect: <boolean> if true, more than one item can be selected at a time.
	 * 		editParams: {
	 *			inputSelected: <array> optional, provide a list of items that are selected. Must follow the format listed below...
	 *			[
	 *				{
	 *					ItemId: <unique String>,
	 *					ItemName: <name>,
	 *					parentNode: {
	 *						ItemId:<unique String>,
	 *						ItemName: <name>,
	 *						parentNode: {
	 *							ItemId: <unique String>,
	 *							ItemName: <name>,
	 *							parentNode: {
	 *								ItemId: <unique String> // this must correspond to a topNode
	 *								ItemName: <name>
	 *							}
	 *						}
	 *					}
	 *				},
	 *				...
	 *			]
	 *			...which is the same format that can be obtained from editParams.result.selected and editParams.result.excluded
	 *			inputExcluded: <array> optional, provide a list of items that are excluded. Must follow the same format as inputSelected
	 *			result: <empty object>, collects the lists of selected and excluded items into result.selected and result.excluded
	 *			noExclude: <boolean> optional. If true, the user cannot exclude items.
	 *			allowToggle: <boolean> optional. If true, exposes the function cvparams.toggleMode() which, when called from the controller, toggles the display between edit mode and summary mode
	 * 		}
	 *
	 * 		// the following are required if using view mode.
	 *
	 * 		requestChildren: same as above
	 * 		processChildren: same as above
	 * 		topNodes: same as above
	 * 		multiSelect: same as above, usually should be set to true
	 * 		viewParams: {
	 * 			selects: [] array of items that are selected.
	 *  		excludes: [] array of items that are excluded.
	 * 		}
	 *
	 *
	 *  	// the following are required if using summary mode.
	 *  	summaryParams: {
	 *			selected: <array> array of selected items. Same format as editParams.inputSelected, except you also need to include ItemName
	 *  		excluded: <array> similar to selected, a list of excluded items
	 *  	}
	 *
	 *  expandAll : boolean. default value is false. optional property of a node for edit/view mode.
	 *  			If set to true for a node, on expanding that node, entire hierarchy of that node will be expanded.
	 *  			It should be used only when tree size is known to be small.
	 *
	 *	onTreeEmpty : object. optional.
	 *				set custom message if tree is empty
	 *				eg {message : 'custom message goes here'}
	 *
	 * }
	 *
	 * Every array returned by processChildren must have the following format:
	 * [
	 * 		{
	 * 			ItemId: 'abc123',
	 * 			ItemName: 'Item0',
	 * 			icon: <string>, // (optional) css class that has background-image property as the image url
	 * 			forceDisable: <boolean>, // (optional) If true, this node cannot be selected by the user
	 * 			noChild: <boolean>, // (optional) If true, the item is a leaf and cannot be expanded
	 * 			// any other node properties needed for requestChildren or processChildren
	 * 		},
	 * 		{
	 * 			ItemId: 'def456',
	 * 			ItemName: 'Item1',
	 * 			icon: <string> // (optional) the url of an icon to use
	 * 		},
	 * 		...
	 * 		totalItems: <integer> // (optional) If using external paging, add this property to your array so the directive knows when there are no more items to load.
	 * ]
	 *
	 * topNodes has the same format.
	 *
	 * Every ItemId must be unique.
	 *
	 */

	cvCommon.factory('cvAdvancedTreeFactory', [
		'$uibModal',
		function($modal) {
			var factory = {};

			factory.openAdvancedCvTreeWrapperModal = function(params) {
				// open modal
				return $modal.open({
					templateUrl : appUtil.appRoot + 'common/partials/cvAdvancedTreeWrapperModal.jsp',
					backdrop : 'static',
					controller : [
							'$scope',
							'$uibModalInstance',
							'cvLoc',
							function($scope, $modalInstance, cvLoc) {
								$scope.cancel = function() {
									$modalInstance.dismiss('cancel');
								};

								$scope.submitSelections = function() {
									var result;
									if ($scope.cvParams.editParams) result = $scope.cvParams.editParams.result;
									$modalInstance.close(result);
								};
								$scope.cvParams = params;
								// set dialog title
								$scope.treeTitle = params.title;
							} ]
				});
			}

			return factory;

		}
	]);

	cvCommon.directive('cvAdvancedTree', ['cvUtil', '$compile', '$log', '$window', 'cvLoc', '$timeout', function(cvUtil, $compile, $log, $window, cvLoc, $timeout) {
		return {
			restrict : 'E',
			templateUrl: appUtil.appRoot + 'common/partials/cvAdvancedTree.jsp',
			scope : {
				cvParams : '=cvparams',
				onShowSelected : '=onshowselected'
			},
			link : function (scope, tElement, tAttrs, transclude) {
				scope.treeScope = scope;
				var mode = scope.cvParams.mode || 'edit'; // defaults to 'edit'
				var editMode = mode === 'edit' ? true : false;

				// see if external paging or client-size paging is used
				if (scope.cvParams.useExternalPaging) {
					scope.externalPaging = true;
				}
				else if (scope.cvParams.clientSidePagingSize && !isNaN(scope.cvParams.clientSidePagingSize) && scope.cvParams.clientSidePagingSize > 0) {
					scope.clientSidePaging = true;
				}

				if (!scope.cvParams.editParams) scope.cvParams.editParams = {};
				if (!scope.cvParams.editParams.result) scope.cvParams.editParams.result = {};
				scope.cvParams.editParams.result.selected = [];
				scope.cvParams.editParams.result.excluded = [];

				var selectList = scope.cvParams.editParams.result.selected;
				var excludeList = scope.cvParams.editParams.result.excluded;
				if (scope.cvParams.editParams) var noExcludes = scope.cvParams.editParams.noExcludes;

				scope.loadMoreExternal = function(n) { // uses external paging to load more items
					 var newChildrenPromise = scope.cvParams.requestChildren(getNodeWithSearchKeyWord(n));
					 var lastChildIndex = n.ItemDetails.ChildItems.length - 1;
					 n.pagingLoading = true;
					 newChildrenPromise.then(function(data){
						 var newChildren = scope.cvParams.processChildren(data, n);
						 n.page++;
						 addChildren(newChildren, n);
						 $timeout(function() {n.pagingLoading = undefined;}, 150, false); // delay on the loading bars collapsing
					 });
				}

				scope.showMore = function(node) {
					if (node.remainingItems > 0 && node.clientSidePaging) {
						showMore(node);
					}
				}


				/*scope.searchTree = function(){
					if (typeof scope.cvParams.searchTree === 'function') {
						scope.cvParams.searchTree();
					}
				}*/

				function noPagination(node){
					return !node.externalPaging && !node.clientSidePaging;
				}

				function getClientSidePagingSize(node){
					return node.clientSidePagingSize?node.clientSidePagingSize:20;
				}

				function showMore(parent, noAnimate) { // client-side paging
					var childItems = parent.ItemDetails.ChildItems;
					var i = 0;
					var target = i + getClientSidePagingSize(childItems[i]);

					parent.pageNo = !parent.pageNo?1 :  (parent.pageNo + 1);

/*					while (childItems[i] && childItems[i].appear) i++;
					if (noAnimate) { // uib animates the first set, so animating them here would be redundant
						while (childItems[i] && (i < target)) {
							childItems[i].appear = true;
							i++;
						}
					}
					else {
						showOne(childItems[i]); // adds the new items one at a time to match each item's individual animation
					}*/

					//parent.remainingItems = childItems.length - parent.limit   < 0 ? 0 : childItems.length - parent.limit;

					//parent.showMoreMessage = cvLoc('label.showMore', parent.remainingItems);

					function showOne(elem) {
						if (elem && (i < target)) {
							elem.appear = true;
							i++;
							$timeout(showOne, 30, false, childItems[i]) // match the transition duration in scss
						}
					}
				}

				if (mode !== 'summary') {
					// initial list of top level nodes
					scope.initTopLevelNodes = function () {
						scope.localNodes = [];
						for (let i = 0; i < scope.cvParams.topNodes.length; i++) {
							// copy each item so the reference is not modified
							scope.localNodes.push(Object.assign({
								ItemDetails: {} // can be overwritten if the user wants to send childItems data. (not supported yet)
							}, scope.cvParams.topNodes[i]))
							// add certain initial properties for edit/view mode.
							var enforceProperties = {
								notLoading: true,
								disabled: !editMode,
								enableSearch : true
							};
							var newLocalNode = _.last(scope.localNodes);

							if(newLocalNode.clientSidePaging){
								newLocalNode.clientSidePagingSize = getClientSidePagingSize(newLocalNode);
							}
							else
							{
							if(scope.clientSidePaging){
								if(noPagination(newLocalNode)){
									newLocalNode.clientSidePaging = true;
									newLocalNode.clientSidePagingSize = scope.cvParams.clientSidePagingSize;
								}
							}
							else if(scope.externalPaging){
								if(noPagination(newLocalNode)){
									newLocalNode.externalPaging = true;
								}
							}
							}

							for (let prop in enforceProperties) {
								newLocalNode[prop] = enforceProperties[prop];
							}
						}
					}
					scope.initTopLevelNodes();
				}

				scope.cvParams.resetTree = function() {
					scope.cvParams.editParams.result.selected = selectList = [];
					scope.cvParams.editParams.result.excluded = excludeList = [];
					scope.cvParams.searchKeyWord = '';
					scope.cvParams.showSelectedOnly = false;
					scope.initTopLevelNodes();
					renderDirective();
				}

				// add nodes for the summary view
				function summaryAdd(item, property, localNodes,editMode) {
					if (!item.ItemId) return; //error, skip this item
					item[property] = true;
					// add node
					var ancestry = [item]; // list of the hierarchy
					var current = item;
					while (current.parentNode) {
						ancestry.unshift(current.parentNode)
						current = current.parentNode;
					}
					findOrCreate(ancestry, 0, localNodes,editMode);

					function findOrCreate(ancestors, index, list,editMode) {
					// find ancestors[index] in list or create it in list if it's not there
						//check valid index
						if (index >= ancestors.length) return;

						for (let i = 0; i < list.length; i++) {
							if (list[i].ItemId === ancestors[index].ItemId) {
								ancestors[index] = list[i]; // insert the existing node back into the ancestry list
								// item already exists
								if (!list[i].ItemDetails) list[i].ItemDetails={};
								if (!list[i].ItemDetails.ChildItems) list[i].ItemDetails.ChildItems=[];
								findOrCreate(ancestors, index + 1, list[i].ItemDetails.ChildItems,editMode);
								return;
							}
						}
						// item does not already exist
						// generate properties
						ancestors[index].ItemDetails = {
							ChildItems: [
							]
						};
						if (ancestors[index - 1]) ancestors[index].parentNode = ancestors[index - 1];
						ancestors[index].notLoading = true;
						ancestors[index].searchedForChildren = true;
						ancestors[index].disabled = !editMode?true:false;
						ancestors[index].show = true;
						ancestors[index].appear = true;
						// determine if it should be checked, because of a checked parent
						if (ancestors[index].parentNode) {

							ancestors[index].ItemDetails.Parent = {
								ItemId : ancestors[index].parentNode.ItemId
							};

							if(ancestors[index].parentNode.checked && !ancestors[index].exclude){
								ancestors[index].checked = true;
							}
						}

						list.push(ancestors[index]);
						updateAllAncestors(list[list.length - 1]); // determines if any ancestors should be partial
						findOrCreate(ancestors, index + 1, list[list.length - 1].ItemDetails.ChildItems,editMode);
						return;
					}
				}

				var preSetPartials = [];
				// process inputSelected and inputExcluded
				function processInputSelected(input) {
					for (let i = 0; i < input.length; i++) {
						addToSelectList(input[i])
					}
					setAllPartials(input)
				}
				function processInputExcluded(input){
					for (let i = 0; i < input.length; i++) {
						addToExcludeList(input[i])
					}
					setAllPartials(input)
				}
				function setAllPartials(inputs){
					for (let i = 0; i < inputs.length; i++) {
						if (inputs[i].parentNode) setPartials(inputs[i].parentNode);
					}
				}

				function setPartials(node) {
					if (preSetPartials.indexOf(node.ItemId) === -1) preSetPartials.push(node.ItemId);
					if (node.parentNode) setPartials(node.parentNode);
				}

				// find partially selected ancestor nodes for preselected nodes, make list of input ids
				if (mode === "edit" && scope.cvParams.editParams) {
					if (scope.cvParams.editParams.inputSelected) {
						processInputSelected(scope.cvParams.editParams.inputSelected);
						scope.cvParams.inputSelectedIds = [];
						for (let i = 0; i < scope.cvParams.editParams.inputSelected.length; i++) {
							scope.cvParams.inputSelectedIds.push(scope.cvParams.editParams.inputSelected[i].ItemId)
						}
					}
					if (scope.cvParams.editParams.inputExcluded) {
						processInputExcluded(scope.cvParams.editParams.inputExcluded);
						scope.cvParams.inputExcludedIds = [];
						for (let i = 0; i < scope.cvParams.editParams.inputExcluded.length; i++) {
							scope.cvParams.inputExcludedIds.push(scope.cvParams.editParams.inputExcluded[i].ItemId)
						}
					}
				}
				else if (mode === "view" && scope.cvParams.viewParams) {
					if (scope.cvParams.viewParams.selects) {
						processInputSelected(scope.cvParams.viewParams.selects);
						scope.cvParams.inputSelectedIds = [];
						for (let i = 0; i < scope.cvParams.viewParams.selects.length; i++) {
							scope.cvParams.inputSelectedIds.push(scope.cvParams.viewParams.selects[i].ItemId)
						}
					}
					if (scope.cvParams.viewParams.excludes) {
						processInputExcluded(scope.cvParams.viewParams.excludes);
						scope.cvParams.inputExcludedIds = [];
						for (let i = 0; i < scope.cvParams.viewParams.excludes.length; i++) {
							scope.cvParams.inputExcludedIds.push(scope.cvParams.viewParams.excludes[i].ItemId)
						}
					}
				}
				else if (mode === "summary" && scope.cvParams.summaryParams) {
					var selected = angular.copy(scope.cvParams.summaryParams.selected); // copy the lists so they don't modify the original nodes
					var excluded = angular.copy(scope.cvParams.summaryParams.excluded);

					var selectedItems = selected || [];
					var excludedItems = excluded || [];
					scope.localNodes = [];
					var localNodes = [];

					for (let i = 0; i < selectedItems.length; i++){
						summaryAdd(selectedItems[i], 'checked', localNodes);
					}
					for (let i = 0; i < excludedItems.length; i++){
						summaryAdd(excludedItems[i], 'exclude', localNodes);
					}
				}
				else {
					$log.error("No mode selected for cv advanced tree");
				}

				function isPartial(node) { // check if a node is partially selected due to a preselected descendant
					if (preSetPartials) {
						return preSetPartials.indexOf(node.ItemId) > -1;
					}
					return false;
				}

				// see if any top nodes are partially selected or fully selected
				if (mode !== "summary") {
					for (let i = 0; i < scope.localNodes.length; i++) {
						if (isPartial(scope.localNodes[i])) {
							scope.localNodes[i].partial = true;
						}
						// determine if top node is checked
						if (isSelected(scope.localNodes[i])) {
							scope.localNodes[i].checked = true;
						}
					}
				}

				scope.showHide = function(node) {
					node.show = !node.show;
				}

				function getNodeWithSearchKeyWord(node){
					let currNode = _.cloneDeep(node);
					currNode.searchKeyWord = scope.cvParams.searchKeyWord;
					return currNode;
				}

				const tryExpandNode = function(node, event) {
					var id = node.ItemId;
					if (node.searchedForChildren) {
						if (node.ItemDetails.ChildItems !== undefined)
							scope.showHide(node);
					}
					else {
						if (node.externalPaging && !node.page) node.page = 0;
						// send request for new items

						if (angular.isFunction(scope.cvParams.requestChildren)) var newChildrenPromise = scope.cvParams.requestChildren(getNodeWithSearchKeyWord(node));
						else {
							$log.error("cvparams.requestChildren is not a function");
							return;
						}

						if (!angular.isUndefined(node.page)) node.page++; // keep track of paging
						var newChildren;
						node.notLoading = undefined; // displays the loading bars
						newChildrenPromise.then(function(data){
							if (angular.isFunction(scope.cvParams.processChildren)) newChildren = scope.cvParams.processChildren(data, node);
							else return $log.error("cvparams.processChildren is not a function");

							node.searchedForChildren = true;
							if (!node.ItemDetails) node.ItemDetails = {};
							if (newChildren.length > 0) {
								if (node.ItemId === 'ROOT' && scope.cvParams.returnAllSelectedChildren) {
									node.forceDisable = false;
								}
								addChildren(newChildren, node, true); // true signifies the first list of children. there could be more with external paging
								if (node.clientSidePaging) {
									showMore(node, true); // load the first set of items
								}
							}
							else {
								if(scope.cvParams.searchKeyWord && scope.cvParams.searchKeyWord!="")
								{
									node.ItemDetails = {};
								}
								node.noChild = true; // hides the dropdown arrow
								node.remainingItems = 0;
							}
							if(node.expandAll){
								angular.forEach(node.ItemDetails.ChildItems, function(value, key){
									if(!(value.noChild  === false))//if it is a leaf node then no need to attempt expanding it. if noChild is not set, it is assumed that the node has children.
										tryExpandNode(value);
								})
							}
							$timeout(function() {node.notLoading = true;}, 100, false); // delay hiding the loading bars
						}, function error(res) {
							node.notLoading = true;
							$log.error("Error:", res.status);
						});
					}
				}

				scope.tryExpand = function(node, event) { // expands the node's children if there are any, calls requestChildren() to find them if they aren't already found
					tryExpandNode(node, event);

				}




				// determine if an item was pre-selected
				function isSelected(node) {
					let isSelected = false;
					if (scope.cvParams.inputSelectedIds) {
						isSelected = scope.cvParams.inputSelectedIds.indexOf(node.ItemId) > -1;
					}

					if (scope.cvParams.editParams.result.selected) {
						var selectedNodes = _.filter(scope.cvParams.editParams.result.selected,function(item){
							return item.ItemId === node.ItemId;
						});
						isSelected = selectedNodes && selectedNodes.length?true:false;
					}

					return isSelected;
				}

				function isExcluded(node) {
					let isExcluded = false;
					if (scope.cvParams.inputExcludedIds) {
						isExcluded = scope.cvParams.inputExcludedIds.indexOf(node.ItemId) > -1;
					}

					if (scope.cvParams.editParams.result.excluded) {
						var selectedNodes = _.filter(scope.cvParams.editParams.result.excluded,function(item){
							return item.ItemId === node.ItemId;
						});
						isExcluded = selectedNodes && selectedNodes.length?true:false;
					}

					return isExcluded;
				}

				// add child items to a node
				function addChildren(newChildren, node, firstLoad) {

					for (let i = 0; i < newChildren.length; i++) {
						if (!newChildren[i].ItemDetails) newChildren[i].ItemDetails = {};
						// add reference to parent
						newChildren[i].ItemDetails.Parent = {
							ItemId: node.ItemId
						}
						newChildren[i].parentNode = node;
						newChildren[i].notLoading = true;
						if (node.externalPaging && firstLoad) {
							newChildren[i].appear = true;
						}
					}

					if (!firstLoad && node.externalPaging) { // if external paging loads another set of items, animate the new items
						showOne(newChildren[0], 0);
					}

					function showOne(elem, i) {
						if (elem) {
							elem.appear = true;
							i++;
							$timeout(showOne, 30, false, newChildren[i], i);
						}
					}

					// at this point the new child elements will automatically generate

					// Process elements sent with select/exclude data

					var numberChildren = newChildren.length;
					var numberExclude = 0; // number of excluded children
					var numberSelect = 0; // number of selected children
					var anyPartialChildren = false; // becomes true if isPartial finds a partially selected item
					for (var i = 0; i < numberChildren; i++) {


						if(node.clientSidePaging){
							if(noPagination(newChildren[i])){
								newChildren[i].clientSidePaging = node.clientSidePaging;
								newChildren[i].clientSidePagingSize = node.clientSidePagingSize;
							}
						}
						else if(node.externalPaging){
							if(noPagination(newChildren[i])){
								newChildren[i].externalPaging = node.externalPaging;
							}
						}

						if(!scope.cvParams.multiSelect){
							//On first load, selectedNode needs to be set so that we have a hold on the node. On selecting another node, selected node needs to be unchecked.
							//It can be then marked unchecked.
							if (angular.isUndefined(scope.cvParams.selectedNode) ) { // first load
								if(isSelected(newChildren[i])){
									newChildren[i].checked = true;
									numberSelect++;
									addToSelectList(newChildren[i]);
									scope.cvParams.selectedNode = newChildren[i];
								}

							}
						}else{
							// check if new children are pre-selected or excluded or partially selected
							if (isSelected(newChildren[i])) {
								newChildren[i].checked = true;
								numberSelect++;
								addToSelectList(newChildren[i]);
								//updateAllAncestors(node.ItemDetails.ChildItems[i]);
							}
							else if (isExcluded(newChildren[i])) {
								newChildren[i].exclude = true;
								numberExclude++;
								addToExcludeList(newChildren[i]);
								//updateAllAncestors(node.ItemDetails.ChildItems[i]);
							}
						}

						if (isPartial(newChildren[i])) {
							newChildren[i].partial = true;
							anyPartialChildren = true;
						}
					}
					if (numberSelect > 0 && !node.checked && scope.cvParams.multiSelect) {
						node.partial = true;
					}
					else if (numberExclude > 0 && node.checked) {
						node.partial = true;
						// make all children checked unless they are excluded
						for (let i = 0; i < newChildren.length; i++) {
							if (!newChildren[i].exclude) {
								newChildren[i].checked = true;
							}
						}
					}
					else if (anyPartialChildren) {
						// cannot run parentCheckChange because it will remove the partial state of any items.
						for (let i = 0; i < newChildren.length; i++) {
							newChildren[i].checked = node.checked;
						}
					}
					else { // no new child is selected or excluded. run parentCheckChange.
						parentCheckChange(node, newChildren); // apply updates to the new child elements
					}
					if (firstLoad) {
						node.ItemDetails.ChildItems = newChildren;
					}
					else {
						node.ItemDetails.ChildItems = node.ItemDetails.ChildItems.concat(newChildren);
					}

					if (node.externalPaging) { // check if external paging has reached the last page
						if (node.ItemDetails.ChildItems) {
							if (node.ItemDetails.ChildItems.length >= newChildren.totalItems) node.remainingItems = 0; // the load more button will disappear if the number of children reaches totalItems
							else if (newChildren.length === 0) node.remainingItems = 0; // the load more button will disappear if no new items are sent, if totalItems was undefined
							else node.remainingItems = newChildren.totalItems - node.ItemDetails.ChildItems.length;
							node.showMoreMessage = cvLoc("label.loadMore", node.remainingItems);
						}
						else {
							node.remainingItems = 0;
						}
					}
					else if (node.clientSidePaging) {
						// determine the number of remaining items after the node is first expanded
						if (node.ItemDetails.ChildItems) {
							node.remainingItems = node.ItemDetails.ChildItems.length - getClientSidePagingSize(node);
							node.showMoreMessage = cvLoc("label.showMoreItems", node.remainingItems);
						}
						else node.remainingItems = 0;
					}

					if (!editMode) { // view mode, display disabled checkboxes
						for (let i = 0; i < newChildren.length; i++) {
							newChildren[i].disabled = true;
						}
					}

					node.show = true;
				}

				scope.checkIfChildren = function(node) {
					if (node.searchedForChildren && node.ItemDetails.ChildItems !== undefined) return true;
				}

				// functions for maintaining the selectList and excludeList, which are the output lists of items
				function addToSelectList(node) {
					addToList(selectList, node);
					/*checkIfDescendantInList(selectList, node);
					checkIfDescendantInList(excludeList, node);*/
				}
				function emptySelectList() {
					selectList.length = 0;
				}

				function removeExcludedNode(node, removeChildren) {
					if (_.isEmpty(node)) {
						return;
					}
					removeFromList(selectList, node);
					if (!_.isEmpty(node.parentNode)) {
						removeExcludedNode(node.parentNode, false);
					}
					if (removeChildren && !_.isUndefined(node.ItemDetails) && _.isArray(node.ItemDetails.ChildItems)) {
						node.ItemDetails.ChildItems.forEach(child => {
							removeExcludedNode(child, true);
						});
					}
				}

				function addToExcludeList(node) {
					if (!noExcludes) addToList(excludeList, node);
					if (scope.cvParams.returnAllSelectedChildren) {
						removeExcludedNode(node, true);
					}
					/*checkIfDescendantInList(selectList, node);
					checkIfDescendantInList(excludeList, node);*/
				}

				function addToList(list, node) {
					for (let i = 0; i < list.length; i++) {
						// check if it's already on the list
						if (list[i].ItemId === node.ItemId) {
							for (var prop in node) {
								// only add essential properties to the object in the list, such as name and id, and any additional properties sent to the directive.
								if (!list[i][prop] && propertySafe(prop)) list[i][prop] = node[prop];
							}
							return;
						}
					}

					function propertySafe(prop) {
						// returns false if any of the following properties are detected, true otherwise
						var ignoreProperties = [
							"ItemDetails",
							"notLoading",
							"disabled",
							"show",
							"searchedForChildren",
							"partial",
							"checked",
							"exclude",
							"appear",
							"remainingItems",
							"pagingLoading",
							"userModified"
						];
						return ignoreProperties.indexOf(prop) === -1;
					}
					function pushSafeProperties(list, node) { // push item to list, but without properties that are internal to the directive and should not be saved
						var newNode = {};
						for (var prop in node) {
							if (propertySafe(prop)) {
								newNode[prop] = node[prop];
							}
						}
						list.push(newNode);
					}

					// if not on list, add to list
					pushSafeProperties(list, node);
				}

				function removeFromSelectList(node) {
					removeFromList(selectList, node);
					if (scope.cvParams.returnAllSelectedChildren) {
						removeExcludedNode(node, true);
				}
				}

				function removeFromExcludeList(node) {
					if (!noExcludes) removeFromList(excludeList, node);
				}

				// removes a node from a list of nodes, searching by node.ItemId
				function removeFromList(list, node) {
					if(!scope.doNotUpdateList){
						_.remove(list, function(item) {
							return item.ItemId === node.ItemId;
						});
					}
				}

				// removes an id (string) from a list of ids
				function removeFromIdList(list, id) {
					if(!scope.doNotUpdateList){
						_.remove(list, function(e) {
							return e === id;
						});
					}
				}

				// checks the selectList and excludeList to see if any descendants of node are in those lists, in which case the descendant should be removed
				function checkIfDescendantInLists(node) {
					checkIfDescendantInList(selectList, scope.cvParams.inputSelectedIds, node);
					checkIfDescendantInList(excludeList, scope.cvParams.inputExcludedIds, node);
				}

				function checkIfDescendantInList(list, idList, parent) { // checks if a child item is in the list and removes it
					for (var i = list.length - 1; i >= 0; i--) {
						if (list[i].ItemId !== parent.ItemId) { // don't double check the parent node
							if (checkAncestry(list[i], parent.ItemId)) { // if a match is found in the hierarchy
								var removeNode = list.splice(i, 1)[0]; // remove the node from list
								// remove all ancestors from preSetPartials
								var item = removeNode.parentNode;
								while (item) {
									removeFromIdList(preSetPartials, item.ItemId);
									item = item.parentNode;
								}
								// remove from inputSelectedIds or inputExcludedIds
								removeFromIdList(idList, removeNode.ItemId);
							}
						}
					}
					// check the hierarchy to see if a node has ancestor with the given id
					function checkAncestry(node, ancestorId) {
						var nextParent = node.parentNode;
						if (nextParent) {
							if (nextParent.ItemId === ancestorId) {
								return true; // found, delete the duplicate
							}
							else {
								return checkAncestry(nextParent, ancestorId)
							}
						}
						else {
							return false;
						}
					}
				}

				// update children when a parent changes
				function parentCheckChange(item, childItems) {
					if (!childItems) childItems = item.ItemDetails.ChildItems;
					if (!scope.cvParams.multiSelect) return;
					for (var i = 0; i < childItems.length; i++) {

						childItems[i].userModified = true;

						childItems[i].checked = item.checked;
						childItems[i].exclude = item.exclude;
						childItems[i].disabled = item.exclude || noExcludes && item.checked; // disable only if the parent is excluded

						if (childItems[i].exclude) {
								childItems[i].partial = false;
						}
						if (item.checked) {
							childItems[i].partial = false;
						}

						if (scope.cvParams.returnAllSelectedChildren && item.checked) {
							addToSelectList(childItems[i]);
						}

						if (childItems[i].ItemDetails.ChildItems) {
							parentCheckChange(childItems[i]);
						}
					}
				}

				function updateAllAncestors(node) {
					if (!node.parentNode) return;
					var parent = node.parentNode;
					var oldParentPartialState = parent.partial;
					parent.partial = false;
					for (var i = 0; i < parent.ItemDetails.ChildItems.length; i++) {
						var siblingChecked = parent.ItemDetails.ChildItems[i].checked;
						var siblingPartial = parent.ItemDetails.ChildItems[i].partial;
						var siblingState = siblingChecked || siblingPartial;
						if ((siblingPartial || !siblingChecked) && parent.checked || (siblingState && !parent.checked)) {
							parent.partial = true;
							break;
						}
					}

					if (parent.exclude) parent.partial = false;

					if (oldParentPartialState !== parent.partial) {
						updateAllAncestors(parent);
					}
				}

				scope.checkChange = function(node,updateOriginalTree) {
					node.userModified = true;

					if (!scope.cvParams.multiSelect) { // no multiselect. only one item can be checked
						if (node.checked) {
							scope.cvParams.selectedNode = null;
							node.checked = false;
							removeFromSelectList(node);
						}
						else {
							if (scope.cvParams.selectedNode) {
								scope.cvParams.selectedNode.checked = false;
								updateAllAncestors(scope.cvParams.selectedNode);
							}
							scope.cvParams.selectedNode = node;
							node.checked = true;
							emptySelectList();
							addToSelectList(node);

						}
						updateAllAncestors(node);
						return;
					}
					if (!(node.checked && node.partial)) { // if partially excluded, go to full check
						node.checked = !node.checked // stay checked if it was partially excluded. otherwise, flip the checked status
					}

					if (node.checked) node.exclude = false;

					if (node.ItemDetails.Parent !== undefined && node.ItemDetails.Parent.ItemId) {
						// examine the parent node, give it the correct check, and give the child node an x instead of blank
						var parent = node.parentNode;
						if (parent.checked) { // parent was checked
							//removeFromSelectList(node);
							if (!node.checked && noExcludes) removeFromSelectList(node);
							if (!node.checked && !noExcludes) {
								node.exclude = true;
							}
							else if (node.partial) {
								checkIfDescendantInLists(node); // clear out any excluded descendants
							}
							else {
								removeFromExcludeList(node);
							}
							if (node.exclude) {
								addToExcludeList(node);
								node.partial = false;
							}
						}
						else { // parent was not checked
							if (node.checked) {
								addToSelectList(node);
								checkIfDescendantInLists(node);
							}
							else {
								removeFromSelectList(node);
							}
						}

						updateAllAncestors(node);

					}
					else {
						if (node.checked) {
							addToSelectList(node);
							checkIfDescendantInLists(node);
						}
						else removeFromSelectList(node);
					}
					node.partial = false;
					if (node.ItemDetails.ChildItems) {
						parentCheckChange(node);
					}

					//when in show selected mode, any modifications doen to the node , update the corresponding node in the tree when show selected mode is not true.
					if(scope.cvParams.showSelectedOnly && updateOriginalTree){
						let filteredNode = _findNode(scope.originalTree,node);
						if(filteredNode){
							scope.doNotUpdateList = true;
							scope.checkChange(filteredNode);
							scope.doNotUpdateList = false;
						}
					}
				}

				scope.submitSelections = function() {
					scope.cvParams.submit(selectList, excludeList);
				}

				scope.cvParams.searchTree = function() {

					if(!scope.cvParams.showSelectedOnly)
					{
						for (let i = 0; i < scope.localNodes.length; i++) {
							scope.localNodes[i].noChild = false;
							if(scope.localNodes[i].externalPaging || !scope.localNodes[i].show){

								if(scope.localNodes[i].show === undefined || scope.localNodes[i].show)
									scope.localNodes[i].searchedForChildren = false;
								scope.localNodes[i].page = 0;
								scope.tryExpand(scope.localNodes[i]);
							}
							else
							{
								scope.localNodes[i].pageNo = 1;
							}
						}
					}
				}

				if (editMode && scope.cvParams.editParams.allowToggle) {
					scope.toggleToSummary = false;
					// switch between summary and view mode
					scope.cvParams.toggleMode = function() {
						if(scope.cvParams.showSelectedOnly){
							let originalNodes = _.cloneDeep(scope.localNodes);
							scope.originalTree = {
									ItemDetails : {
										ChildItems : originalNodes
									}
								};
							scope.originalEditParams = _.cloneDeep(scope.cvParams.editParams.result);
							//scope.originalSearchKeyWord = _.cloneDeep(scope.cvParams.searchKeyWord);

							var selected = angular.copy(selectList); // copy the lists so they don't modify the original nodes
							var excluded = angular.copy(excludeList);

							var selectedItems = selected || [];
							var excludedItems = excluded || [];

							var localNodes = [];

							_.forEach(selectedItems, function(item) {
								summaryAdd(item, 'checked', localNodes,true);
								});

							_.forEach(excludedItems, function(item) {
								summaryAdd(item, 'exclude', localNodes,true);
								});

							if (localNodes)
							{
								scope.localNodes = localNodes;
								renderDirective();
							}
						}
						else
						{
							scope.localNodes = scope.originalTree.ItemDetails.ChildItems;
							renderDirective();
						}
					}
					if(scope.cvParams.showSelectedOnly && scope.cvParams.isShowSelectedOnlyByDefault) {
						scope.cvParams.toggleMode()
					}
				}

					function _findNode(node,item){ //find node in child items
						let filteredNode = undefined;
						if(!node.ItemDetails.ChildItems) //reached to last level
						    return filteredNode;

						filteredNode = _.find(node.ItemDetails.ChildItems,{ItemId : item.ItemId});

						if(!filteredNode){ //search in children
							_.forEach(node.ItemDetails.ChildItems,function(currentNode){
								filteredNode = _findNode(currentNode,item);
								if(filteredNode){
									return false;
								}
							});
						}

						return filteredNode;
					}

				// compiles the directive for the first time, displaying the top level nodes or all summary nodes
				function renderDirective() {
					var text = "";
					if (scope.localNodes.length === 0) {
						if(scope.cvParams.onTreeEmpty && scope.cvParams.onTreeEmpty.message)
							text = scope.cvParams.onTreeEmpty.message;
						else
							text = cvLoc('label.noDataAvailable'); // error here
					}
					else {
						text = '<ul class="tree-view-wrapper"';
						if (editMode && scope.cvParams.editParams.allowToggle) {
							// add properties to hide the directive and toggle to summary mode
							text += ' data-ng-show="!toggleToSummary || cvParams.mode==\'summary\'"';
						}
						text += '>';
						text += '<li ng-repeat="n in ::localNodes" class="cv-tree-ele">';
						text += '<cv-advanced-node tree-scope="treeScope" n="::n" edit-mode="' + editMode + '"></cv-advanced-node>';
						text += '</li>';
						text += '</ul>';
						if (editMode && scope.cvParams.editParams.allowToggle) {
							text += '<cv-advanced-tree data-ng-if="toggleToSummary" cvparams="toggleSummaryParams"></cv-advanced-node>';
						}
					}
					tElement.find(".treeContents").html(text);
					$compile(tElement.contents())(scope);
				}

				try {
					if (localNodes) scope.localNodes = localNodes;
					renderDirective();
				}
				catch(error) {
					scope.serverMessage = cvUtil.errMsg(error); // error here
					$compile(tElement.contents())(scope);
				}
			}
		};

	}]);

	cvCommon.directive('cvAdvancedNode', ['$compile', 'RecursionHelper', function($compile, RecursionHelper) {
		return {
			restrict : "E",
			scope : {
				n : '=',
				treeScope : '=',
				editMode: '='
			},
			templateUrl : 'common/partials/cvAdvancedNode.jsp',
			compile : function(element) {
				// Use the compile function from the RecursionHelper, defined in cvtree.js
				return RecursionHelper.compile(element);
			}
		};
	}]);
})();
