import * as DateTimePickerModal from '../newGrid/grid.datetime.modal';
import Dropdown from '../dropdown';

const DEFAULT_FILTER_CONDITION_OPTIONS = [
	{
		value: 'equals',
		label: 'label.equals'
	}
];

// The default dropdown options for DateTimeDropdown:
const UNLOCALIZED_DEFAULT_DATA = [{
	value: 'last1hour',
	label: 'label.dateRangeOptions.lastHour',
	buildFilterRange: () => {
		const filterDate = new Date();
		filterDate.setHours(filterDate.getHours() - 1);
		return {
			from: filterDate
		};
	},
}, {
	value: 'last2hour',
	label: 'label.dateRangeOptions.last2Hours',
	buildFilterRange: () => {
		const filterDate = new Date();
		filterDate.setHours(filterDate.getHours() - 2);
		return {
			from: filterDate
		};
	},
}, {
	value: 'last24hour',
	label: 'label.dateRangeOptions.last24Hours',
	buildFilterRange: () => {
		const filterDate = new Date();
		filterDate.setDate(filterDate.getDate() - 1);
		return {
			from: filterDate
		};
	},
}, {
	value: 'last48hour',
	label: 'label.dateRangeOptions.last48Hours',
	buildFilterRange: () => {
		const filterDate = new Date();
		filterDate.setDate(filterDate.getDate() - 2);
		return {
			from: filterDate
		};
	},
}, {
	value: 'last1week',
	label: 'label.dateRangeOptions.lastWeek',
	buildFilterRange: () => {
		const filterDate = new Date();
		filterDate.setDate(filterDate.getDate() - 7);
		return {
			from: filterDate
		};
	},
}, {
	value: 'custom',
	label: 'label.dateRangeOptions.customRange',
}];

export default class DateTimeDropdown extends Dropdown {
	static get FILTER_CONDITION_OPTIONS() {
		return DEFAULT_FILTER_CONDITION_OPTIONS;
	}

	/**
	 * @param {Function} cvLoc 
	 * @returns {Object[]} The default dropdown data. If cvLoc is provided, the labels will be localized
	 */
	static getDefaultData(cvLoc) {
		if (!cvLoc) {
			// No localization function provided, return copy of default data
			return UNLOCALIZED_DEFAULT_DATA.map(dataObj => ({...dataObj}));
		}
		return UNLOCALIZED_DEFAULT_DATA.map(dataObj => ({
			...dataObj,
			label: cvLoc(dataObj.label)
		}));
	}

	constructor(options, angularLibs) {
		super(
			{
				...options,
				parseViewValue: function(val) {
					if (val && !_.some(this.dataSrc.data, dataObj => val === dataObj.value)) {
						// If the view value is not in the dropdown data, it is assumed to be a custom range:
						this.currentCustomRange = val;
						const range = JSON.parse(val);
						let label;
						label = angularLibs.$filter('date')(range.from * 1000, 'MMM d, h:mm a'); // TODO angular lib
						if (range.to) {
							label = angularLibs.cvLoc(
								'label.dateRangeOptions.fromTo',
								label,
								angularLibs.$filter('date')(range.to * 1000, 'MMM d, h:mm a')
							);
						} else {
							label = angularLibs.cvLoc('label.dateRangeOptions.from', label);
						}
						this.addItem({
							label: label,
							value: val
						});
					}
					return val;
				},
				onFilterSubmit: function(value) {
					if (this.currentCustomRange && this.currentCustomRange !== value) {
						this.removeItemByValue(this.currentCustomRange);
						delete this.currentCustomRange;
					}
				},
				onDropdownFilterSelect: function(event, selectedFilter) {
					if (selectedFilter === 'custom') {
						const self = this;
						// TODO load current filter
						const previousInputValue = this.getInputValue();
						const gridColumn = this.getGridColumn();
						DateTimePickerModal.customDateTime(angularLibs.$modal).then(
							range => {
								let label;
								label = angularLibs.$filter('date')(range.from.getTime(), 'MMM d, h:mm a');
								if (range.to) {
									label = angularLibs.cvLoc(
										'label.dateRangeOptions.fromTo',
										label,
										angularLibs.$filter('date')(range.to.getTime(), 'MMM d, h:mm a')
									);
								} else {
									label = angularLibs.cvLoc('label.dateRangeOptions.from', label);
								}

								// convert range to seconds:
								range.from = range.from.getTime() / 1000;
								if (range.to) {
									range.to = range.to.getTime() / 1000;
								}
								if (self.currentCustomRange) {
									self.removeItemByValue(self.currentCustomRange);
								}
								self.currentCustomRange = JSON.stringify(range);
								self.addItem({
									label: label,
									value: self.currentCustomRange
								});
								self.setValue(self.currentCustomRange);
								self.applyValue();
								if (gridColumn) {
									gridColumn.submitColumnFilter();
								}
							},
							() => {
								// If user dismisses modal
								self.setInputValue(previousInputValue);
							}
						);
						if (gridColumn) {
							gridColumn.closeColumnMenu();
						}
					}
				},
				data: options.data && options.data.length ? options.data : DateTimeDropdown.getDefaultData(angularLibs.cvLoc),
				filterUrl: null, // Not supported by DateTimeDropdown
			},
			angularLibs
		);
	}

	/**
	 * Builds the kendo filter
	 * @param {string|Object} filterValue Either a value in the dropdown or an object with from and to properties in seconds
	 * @param {String} columnName The name of the column for the kdno filter
	 * @param {[Object[]]} data The dropdown data list
	 * @returns {Object} The kendo filter object
	 */
	static buildFilter(filterValue, columnName, data) {
		let filterRange = {
			from: new Date(),
		};
		if (_.isObject(filterValue)) {
			filterRange.from = filterValue.from;
			filterRange.to = filterValue.to;
		} else {
			if (!data) {
				// Labels are not needed here, so default to unlocalized data
				data = UNLOCALIZED_DEFAULT_DATA;
			}
			const filterDataObj = _.find(data, dataObj => dataObj.value === filterValue);
			if (filterDataObj) {
				filterRange = filterDataObj.buildFilterRange();
				// convert range dates to seconds:
				filterRange.from = filterRange.from.getTime() / 1000;
				if (filterRange.to) {
					filterRange.to = filterRange.to.getTime() / 1000;
				}
			} else {
				try {
					filterRange = JSON.parse(filterValue);
				} catch (e) {}
			}
		}

		let operator;
		if (filterRange.to) {
			if (filterRange.to === filterRange.from) {
				operator = date => date == filterRange.to;
			} else {
				operator = date => filterRange.from <= date && date <= filterRange.to;
			}
		} else {
			operator = date => date >= filterRange.from;
		}
	
		return {
			field: columnName,
			// value: value,
			operator,
		};
	}
}
