(function() {
	'use strict';
	String.prototype.endsWith = function(suffix) {
		if (!suffix || (this.length < suffix.length)) {
			return false;
		}
		return this.indexOf(suffix, this.length - suffix.length) !== -1;
	};
	var cvCommon = angular.module('cvCommon');
	/**
	 * Use this service cache your http calls and return cached data until page is refreshed. This doesn't use
	 * angular's $http.cache but adds custom caching
	 */
	cvCommon.factory('cvHttp', [ '$http', '$q', function($http, $q) {
		var cache = {};
		var cvCache = {
			contains : function(key) {
				return key in cache;
			},
			get : function(key) {
				return cache[key];
			},
			put : function(key, val) {
				cache[key] = val;
			},
			clear : function(key) {
				delete cache[key];
			},
			clearAll : function() {
				cache = {};
			}
		};
		return {
			get : function(url) {
				if (cvCache.contains(url)) {
					var cachedData = cvCache.get(url);
					var promise = $q.when(cachedData);
					//return $q.when(cachedData) is enough. Attaching success to the promise so the callers don't have to change their code
					promise.success = function(cb) {
						cb(cachedData)
					};
					return promise;
				}
				return $http.get(url).success(function(data) {
					cvCache.put(url, data);
					return data;
				});
			}
		};
	} ]);
	cvCommon.config([ 'growlProvider', function(growlProvider) {
		growlProvider.globalEnableHtml(true);
	} ]);
	cvCommon.readCookie = function(name) {
		var nameEQ = name + "=";
		var ca = document.cookie.split(';');
		for (var i = 0; i < ca.length; i++) {
			var c = ca[i];
			while (c.charAt(0) == ' ') {
				c = c.substring(1, c.length);
			}
			if (c.indexOf(nameEQ) == 0) {
				return c.substring(nameEQ.length, c.length);
			}
		}
		return null;
	};
	/**
	 * Set cookies
	 * 
	 * name : key for the cookie, If the name exist it will overwrite the existing value value : value of the
	 * cookie to be set expires : date-in-GMTString-format, If not specified it will expire at the end of
	 * session. maxAge : max-age-in-seconds (e.g., 60*60*24*365 or 31536e3 for a year)
	 * 
	 */
	cvCommon.writeCookie = function(name, value, expires, maxAge, path) {
		document.cookie = name + "=" + value + ";" + (window.location.protocol == "https:" ? "secure;" : "")
				+ (expires ? " expires=" + expires : "") + (path ? " path=" + path : "")
				+ (maxAge ? " max-age=" + maxAge : "");
	};
	cvCommon.config(function() {
		$("form").on("submit", function() {
			var csrfParam = cvCommon.readCookie('csrf');
			$("<input type='hidden' id='csrf' name='csrf' value='" + csrfParam + "'>").appendTo($(this));
			if ($(this).attr("enctype") == "multipart/form-data") {
				var action = $(this).attr("action") + "?csrf=" + csrfParam;
				$(this).attr("action", action);
			}
			return true;
		});
	});
	/**
	 * A directive for appending the link to the user's dashhboard based on the app's OEM ID.
	 */
	cvCommon.directive('dashboardLink', [ 'cvUtil', function(cvUtil) {
		return {
			restrict : 'A',
			link : function(scope, element,attrs) {
				// We only want this directive to work for anchor tags
				if (element.prop('tagName') === 'A') {
					var supportCustomLink = attrs.supportCustomLink;
					if(supportCustomLink !== null && supportCustomLink !== undefined && supportCustomLink === "true" ){
						if (cvConfig != null && cvConfig.customHomeUrl != null) {
							element.attr('href', cvConfig.customHomeUrl);
						} else {
							element.attr('href', '#' + cvUtil.getAppDashboard());
						}
					}else{
						element.attr('href', '#' + cvUtil.getAppDashboard());
					}
				}
			}
		};
	} ]);
	cvCommon
			.directive(
					'cvDefaultLogo',
					[
							'$location',
							'cvLoc',
							function($location, cvLoc) {
								return {
									restrict : 'E',
									template : '<img class="hidden-xs" src={{defaultLogo}} alt="{{companyLogoAltText}}" /><img class="hidden-md hidden-lg" src={{defaultLogoSymbol}} alt="{{companyLogoAltText}}" />',
									controller : [
											'$scope',
											'$location',
											'cvLoc',
											function($scope, $location, cvLoc) {
												$scope.companyLogoAltText = cvLoc('companyLogo');
												$scope.defaultLogo = $location.protocol() + "://" + $location.host()
														+ ":" + $location.port() + "/webconsole/OEM/" + cvApp.oemId
														+ "/images/Company_logo.png";
												$scope.defaultLogoSymbol = $location.protocol() + "://"
														+ $location.host() + ":" + $location.port()
														+ "/webconsole/OEM/" + cvApp.oemId
														+ "/images/Company_logo_symbol.png";
											} ]
								};
							} ]);
	cvCommon.directive('uibAccordionGroup', [ function() {
		return {
			priority : 2,
			require : '^uibAccordion',
			restrict : 'A',
			link : function(scope, element, attrs) {
				if (angular.isDefined(attrs.scrollToWhenOpen)) {
					scope.$watch(attrs.isOpen, function(isOpen) {
						if (isOpen) {
							angular.element('html,body').animate({
								scrollTop : element.offset().top
							});
						}
					});
				}
			}
		}
	} ]);
	/**
	 * A directive to only be applied to input tags that automatically selects an input's value when the input
	 * is generated.
	 */
	cvCommon.directive('autoselect', [ function() {
		return {
			restrict : 'A',
			link : function(scope, element, attrs) {
				if (element.prop('nodeName') === 'INPUT') {
					var dereg = scope.$watch(function() {
						return element.val();
					}, function(newVal, oldVal) {
						if (!angular.isDefined(attrs.autoselect) || (attrs.autoselect.length <= 0)
								|| attrs.autoselect === 'true') {
							if (angular.isDefined(newVal) && (newVal.length > 0)) {
								dereg();
								element.select();
							}
						}
					});
				}
			}
		}
	} ]);
	/*
	 * The cvUtil service contains utility functions to be used throughout the AdminConsole application.
	 * 
	 * DO NOT ADD FUNCTIONS TO THIS SERVICE UNLESS THEY ARE GENERIC AND USEFUL TO MULTIPLE MODULES.
	 * 
	 * DO NOT ADD FUNCTIONS TO THIS SERVICE WITHOUT DOCUMENTING THEM.
	 */
	cvCommon
			.service(
					'cvUtil',
					[
							'$injector',
							'$filter',
							'$rootScope',
							'DASHBOARD_LINKS',
							function($injector, $filter, $rootScope, DASHBOARD_LINKS) {

								//validation patterns for window,linux and network paths
								this.localPathPattern = /^([a-zA-Z]:){1}(\\[^<>:"/\\|?*]+)*\\?$/;
								this.linuxPathPattern = /^(\/[^<>:"/\\|?*]+)+\/?$/;
								this.networkPathPattern = /^\\(\\[^<>:"/\\|?*]+)+\\?$/;

								this.readCookie = function(name) {
									return cvCommon.readCookie(name);
								};
								this.writeCookie = function(name, value, expires, maxAge, path) {
									return cvCommon.writeCookie(name, value, expires, maxAge, path);
								};

								/**
								 * Returns the path to the dashboard of the current app based on the OEM ID.
								 * 
								 * @return string The path to the app's dashboard.
								 */
								this.getAppDashboard = function() {
									if (typeof cvApp === "undefined") {
										return;
									}
									if (cvApp && cvApp.globalParams && cvApp.globalParams.StateReportOverride) {
										var stateToReportMap = JSON.parse(cvApp.globalParams.StateReportOverride);
										if (angular.isDefined(stateToReportMap["dashboard"])) {
											return '/dashboard';
										}
									}
									switch (cvApp.oemId) {
									case 106: // VSA
									case 114: // VSA
										if (cv.isMspUser || cv.userRole == 'Role_Restricted_User'
												|| cv.userRole == 'Role_Tenant_User') {
											return DASHBOARD_LINKS.TENANT;
										}
										return DASHBOARD_LINKS.VIRTUALIZATION;
									case 118: //zeal
										if (cv.userRole == 'Role_Msp_Admin') {
											if (cv.csEnvironmentInfo
												&& (cv.csEnvironmentInfo.isApplianceCS
													|| cv.csEnvironmentInfo.isRefArchitecture
													|| cv.csEnvironmentInfo.isCSWithApplianceNode)) {
													return DASHBOARD_LINKS.APPLIANCE;
											}
											return DASHBOARD_LINKS.DISASTER_RECOVERY;
										} else if(cv.userRole == 'Role_Tenant_Admin') {
											return DASHBOARD_LINKS.DISASTER_RECOVERY;
										}
										return DASHBOARD_LINKS.TENANT;
									case 113: // Endpoint
										//return '/endpointDashboard';
									case 1: // Admin Console
									case 16: // Admin Console
									default:
										if (cv.isTenantAdmin || cv.isMspUser || cv.userRole == 'Role_Restricted_User'
												|| cv.userRole == 'Role_Tenant_User') {
											return DASHBOARD_LINKS.TENANT;
										} else if (cv.userRole == 'Role_Msp_Admin') {
											if (cv.csEnvironmentInfo) {
												if (cv.csEnvironmentInfo.isApplianceCS
														|| cv.csEnvironmentInfo.isRefArchitecture
														|| cv.csEnvironmentInfo.isCSWithApplianceNode) {
													return DASHBOARD_LINKS.APPLIANCE;
												}
											}
											return DASHBOARD_LINKS.OVERVIEW;
										}
										return DASHBOARD_LINKS.TENANT;
									}
								};
								/*
								 * method to form url for server caching enabled entities url: url to be
								 * called (this may either be the api or the java .do call options : has the
								 * following structure: additionalQuery: Additional hardcoded query (for
								 * eg:clientId=0) pagingOptions.limit:Number of rows to be fetched
								 * pagingOptions.page: the page # to start from options.sortOptions: specifies
								 * the column we need to sort the records by and the direction of the sort
								 * (asc/desc) fl(optional): list of fields to be fetched , if not specified
								 * the api will return all of them hardRefresh: Indicates if the cache needs
								 * to be refreshed globalFilter.searchText : Specifies the text that needs to
								 * matched across columns (this is the page level search)
								 * globalFilter.supportedColumns: Specifies the CSV columns which the
								 * globalFilter.searchText should be applied options.columnFilters : Specifies
								 * the column filters (the filters are in the format
								 * ColName:FilterQualifier:ColValue (for eg: name:contains:x)
								 */
								this.getCachedEntityUrl = function(options) {
									let sort = "";
									let url = "";
									if (options.sortOptions && options.sortOptions.fields) {
										sort = "Sort=" + options.sortOptions.fields;
										sort = (!options.sortOptions.directions || options.sortOptions.directions
												.toLowerCase() == "asc") ? sort + ":1&" : sort + ":-1&"
									}
									let pageAndLimit = "";
									if (options.pagingOptions && options.pagingOptions.limit
											&& options.pagingOptions.page) {
										pageAndLimit = "Limit=" + options.pagingOptions.limit;
										pageAndLimit += "&Start="
												+ (options.pagingOptions.limit * (options.pagingOptions.page - 1))
												+ "&";
									}
									let fl = "";
									if (options.fl) {
										fl = "Fl=" + options.fl + "&";
									}
									let sep = '&';
									if (options.additionalQuery) {
										url = options.additionalQuery + sep;
									}
									if (options && options.hardRefresh) {
										url = url + 'hardRefresh=true' + sep;
									}

									let search = "";
									if (options && options.globalFilter && options.globalFilter.searchText) {
										search = "Search="
												+ options.globalFilter.supportedColumns
												+ encodeURIComponent(":contains:"
														+ options.globalFilter.searchText.replace(/\\/g, "\\\\")) + "&";
									}
									let fq = options.fq ? options.fq : "";
									if (options && options.columnFilters && options.columnFilters.length > 0) {
										for (var i = 0; i < options.columnFilters.length; i++) {
											fq = fq
													+ "fq="
													+ encodeURIComponent(options.columnFilters[i]
															.replace(/\\/g, "\\\\")) + "&";
										}
									}
									url = url + fl + sort + pageAndLimit + search + fq;
									return url;
								}
								/*
								 * Utility method to create a blank info message object for display in a
								 * serverMessage.message field in a jsp.
								 */
								this.emptyMsg = function() {
									return {
										message : '',
										type : 'ok'
									};
								};
								/*
								 * Utility method to create a info message object for display in a
								 * serverMessage.message field in a jsp. All strings passed into this function
								 * should be localized!
								 */
								this.infoMsg = function(message) {
									return {
										message : message.trim(),
										type : 'ok'
									};
								};

								this.getDaysforLicenseExpiry = function(expiryDate) {
									//check if production license expires within 60 days
									var licenseExpiryDate = new Date(expiryDate * 1000);
									var today = new Date();

									var timeDiff = Math.abs(today.getTime() - licenseExpiryDate.getTime());
									var dateDiff = Math.ceil(timeDiff / (1000 * 3600 * 24));

									return dateDiff;

								};

								this.getDateforLicenseExpiry = function(expiryDate) {
									//check if production license expires within 60 days

									var expiresOn = moment.unix(expiryDate).format('dddd, MMMM Do YYYY');
									return expiresOn;

								};

								this.isLicenseExpired = function(expiryDate) {

									var licenseExpiryDate = new Date(expiryDate * 1000);
									var today = new Date();
									if (licenseExpiryDate < today) {
										return true;
									}
									return false;
								};

								/*
								 * Utility method to create an error message object for display in a
								 * serverMessage.message field in a jsp. All strings passed into this function
								 * should be localized!
								 */
								this.errMsg = function(message) {
									return {
										message : message.trim(),
										type : 'error'
									};
								};
								/*
								 * Utility method to that functions as a wrapper around errMsg function
								 * handling different message types
								 */
								this.errMsgWrapper = function(errorMessage) {
									if (errorMessage) {
										if (angular.isString(errorMessage)) {
											return this.errMsg(errorMessage);
										} else if (angular.isString(errorMessage.data)) {
											return this.errMsg(errorMessage.data);
										} else {
											return this.errMsgLoc('generic_error');
										}
									} else {
										return this.emptyMsg();
									}
								}
								/*
								 * Utility method to create a localized info message object for display in a
								 * serverMessage.message field in a jsp. This method accepts a key for the
								 * cvLoc factory.
								 */
								this.infoMsgLoc = function(key) {
									var cvLoc = $injector.get("cvLoc");
									return {
										message : cvLoc.apply(this, arguments),
										type : 'ok'
									};
								};
								/*
								 * Utility method to create a localized error message object for display in a
								 * serverMessage.message field in a jsp. This method accepts a key for the
								 * cvLoc factory.
								 */
								this.errMsgLoc = function(key) {
									var cvLoc = $injector.get("cvLoc");
									return {
										message : cvLoc(key),
										type : 'error'
									};
								};
								/*
								 * Utility method to return the locale specific labels for iSteven multiselect
								 */
								this.getIStevenLocLabels = function() {
									var cvLoc = $injector.get("cvLoc");
									return {
										selectAll : cvLoc('label.selectAll'),
										selectNone : cvLoc('label.selectNone'),
										reset : cvLoc('label.reset'),
										search : cvLoc('label.search'),
										nothingSelected : cvLoc('label.nothingSelected'),
										showSelected : cvLoc('label.showSelected')
									};
								};
								this.getDateTimeValue = function(datePickerValue, timePickerValue) {
									var toDateTime = datePickerValue;
									toDateTime.setHours(timePickerValue.getHours());
									toDateTime.setMinutes(timePickerValue.getMinutes());
									return toDateTime;
								};
								this.getUTCDateValue = function(toDateTime) {
									return new Date(toDateTime.getUTCFullYear(), toDateTime.getUTCMonth(), toDateTime
											.getUTCDate(), toDateTime.getUTCHours(), toDateTime.getUTCMinutes());
								};
								this.getUTCDate = function(toDateTime) {
									return new Date(
											toDateTime.getUTCFullYear(),
											toDateTime.getUTCMonth(),
											toDateTime.getUTCDate(),
											toDateTime.getUTCHours(),
											toDateTime.getUTCMinutes(),
											toDateTime.getUTCSeconds());
								};
								this.getDateAndTime = function(datePickerValue, timePickerValue) {
									var toDateTime = datePickerValue;
									toDateTime.setHours(timePickerValue.getHours());
									toDateTime.setMinutes(timePickerValue.getMinutes());
									toDateTime.setSeconds(timePickerValue.getSeconds());
									return toDateTime;
								};
								/*
								 * Creates a string for a single parameter for use in a URL. The parameter's
								 * value, if any, is escaped. Example: given "name" and "my value", returns
								 * "name=my%20val".
								 */
								this.createSingleParamString = function(name, value) {
									if (typeof value === "undefined") {
										return name;
									} else {
										return name + "=" + encodeURIComponent(value);
									}
								};
								/*
								 * Takes a target path and a parameter object and converts it to a
								 * properly-encoded URL string. For example, the input urlPath='something.do',
								 * paramObj={param1: 'value 1', param2: 'value 2'} would be converted to this
								 * output: something.do?param1=value%201&param2=value%202.
								 */
								this.createUrlParamString = function(urlPath, paramObj) {
									var that = this;
									var str = urlPath;
									if (paramObj) {
										str += '?';
										var first = true;
										_.forOwn(paramObj, function(val, key) {
											if (first) {
												first = false;
											} else {
												str += "&";
											}
											str += that.createSingleParamString(key, val);
										});
									}
									return str;
								};
								/*
								 * Takes a filename and check if the filename is a valid windows file name
								 */
								this.isValidWindowFileName = function(fileName) {
									var fileNameRegEx = /^(?!^(PRN|AUX|CLOCK\$|NUL|CON|COM\d|LPT\d|\..*)(\..+)?$)[^\x00-\x1f\\?*:<>\"";|/]+$/i;
									if ((fileName != "undefined") && (fileName != null) && fileName.trim().length !== 0) {
										return fileNameRegEx.test(fileName);
									} else {
										return false;
									}
								};
								/*
								 * Takes a CV time and converts it to date object, that datepicker can
								 * understand
								 */
								this.convertCvTimeToDate = function(time) {
									var hours = time / 3600;
									var mins = (time % 3600) / 60;
									var date = new Date();
									date.setHours(hours);
									date.setMinutes(mins);
									return date;
								};
								/*
								 * Takes two date object and returns a condensed string form of the range Eg:
								 * 5th Jan 2016 - 7th Feb 2016 -> Jan 5 - Feb 7
								 */
								this.dateRangeToString = function(start, end) {
									var str = "";
									var showYear = start.getFullYear() !== end.getFullYear();
									var showMonth = start.getMonth() !== end.getMonth();
									var showDate = start.getDate() !== end.getDate();
									var showTime = start.getHours() !== end.getHours();
									if (showYear) {
										// if year is different, no need to show time
										str = $filter('date')(start, 'shortDate') + ' - '
												+ $filter('date')(end, 'shortDate');
									} else {
										// if year is same
										if (showMonth) {
											// if month is different, no need to show time
											str = $filter('date')(start, 'MMM d') + ' - '
													+ $filter('date')(end, 'MMM d');
										} else {
											// if month is also same
											if (showDate) {
												// if date is different, show day and date
												str = $filter('date')(start, 'MMM d') + ' - '
														+ $filter('date')(end, 'MMM d');
											} else {
												// if date is also same
												str = $filter('date')(start, 'shortTime') + ' - '
														+ $filter('date')(end, 'shortTime');
											}
										}
									}
									return str;
								};
								/**
								 * Utility method to convert the UTC date to local time zone date
								 */
								this.convertUTCDateToLocalDate = function(utcDate) {
									var convertdLocalTime = new Date(utcDate.getTime()
											- (utcDate.getTimezoneOffset() * 60000));
									return convertdLocalTime;
								}
								/**
								 * Utility method to format the date based on the parameters When getting a
								 * date, without specifying the time zone, the result is converted to the
								 * browser's time zone. Parameters : dateValue : date object it may be
								 * date,UTC integer value or string of date format : format for the date to
								 * alter timeZone : in case format needs in UTC pass it as 'UTC' String,
								 * otherwise it will be formated for the browser timezone
								 */
								this.formatDate = function(dateValue, format, timeZone) {
									var tempDate = null;
									var typeOfDate = typeof dateValue;
									if (typeOfDate === 'number') {
										tempDate = new Date((dateValue * 1000));
									} else if (typeOfDate === 'string') {
										tempDate = new Date(dateValue);
									} else {
										tempDate = dateValue;
									}
									if (!tempDate && (tempDate == null)) {
										return null;
									}
									if (timeZone && timeZone === 'UTC') {
										tempDate = this.getUTCDate(tempDate); // get UTC date value
									} else {
										tempDate = new Date(tempDate + 'Z'); // Format with the browse timezone value
									}
									return $filter('date')(tempDate, format);
								}
								/*
								 * sets a cookie to skip setup until next login
								 * 
								 */
								this.skipSetup = function() {
									this.writeCookie('skipSetup', 'true');
								}
								this.sortAscending = function(data, property) {
									var temProperty = property.split('.');
									var nameA, nameB;
									data.sort(function(a, b) {
										if (temProperty.length === 1) {
											nameA = (typeof a[property] === 'string') ? a[property].toLowerCase()
													: a[property];
											nameB = (typeof b[property] === 'string') ? b[property].toLowerCase()
													: b[property];
										} else {
											for (var i = 0; i < temProperty.length; i++) {
												if (a[temProperty[i]] !== undefined) {
													a = a[temProperty[i]];
												} else {
													break;
												}
												if (b[temProperty[i]] !== undefined) {
													b = b[temProperty[i]];
												} else {
													break;
												}
											}
											nameA = (typeof a === 'string') ? a.toLowerCase() : a;
											nameB = (typeof b === 'string') ? b.toLowerCase() : b;
										}
										if (nameA < nameB) {
											return -1;
										}
										if (nameA > nameB) {
											return 1;
										}
										return 0; //default return value (no sorting)
									})
									return data;
								}
								this.sortDescending = function(data, property) {
									var temProperty = property.split('.');
									var nameA, nameB;
									data.sort(function(a, b) {
										if (temProperty.length === 1) {
											nameA = a[property].toLowerCase();
											nameB = b[property].toLowerCase();
										} else {
											for (var i = 0; i < temProperty.length; i++) {
												if (a[temProperty[i]] !== undefined) {
													a = a[temProperty[i]];
												} else {
													break;
												}
												if (b[temProperty[i]] !== undefined) {
													b = b[temProperty[i]];
												} else {
													break;
												}
											}
											nameA = (typeof a === 'string') ? a.toLowerCase() : a;
											nameB = (typeof b === 'string') ? b.toLowerCase() : b;
										}
										if (nameA < nameB) {
											return 1;
										}
										if (nameA > nameB) {
											return -1;
										}
										return 0; //default return value (no sorting)
									})
									return data;
								}
								/*
								 * This method returns the text value of the enum key
								 */
								this.lookupEnumConstant = function(key, field) {
									var result = "???" + key + (field ? "." + field : "") + "???";
									if (enumMap[key]) {
										return field ? (enumMap[key][field] ? enumMap[key][field] : result)
												: enumMap[key]['text'];
									}
									return result;
								}
								/*
								 * This method encodes the given string into an array of bytes, returning the
								 * result as a new byte array.
								 */
								this.getBytes = function(inputString) {
									if (!inputString) {
										return null;
									}
									var bytes = [];
									for (var i = 0; i < inputString.length; ++i) {
										bytes.push(inputString.charCodeAt(i));
									}
									return bytes;
								}
								this.controllerExists = function(name) {
									var $controller = $injector.get("$controller");
									var $log = $injector.get("$log");
									try {
										// inject '$scope' as a dummy local variable
										// and flag the $controller with 'later' to delay instantiation
										$controller(name, {
											"$scope" : {}
										}, true);
										return true;
									} catch (ex) {
										$log.error("unable to find the controller : " + name);
										return false;
									}
								}
								this.getParameterByName = function(name, url, defaultValue) {
									name = name.replace(/[\[]/, "\\\[").replace(/[\]]/, "\\\]");
									var regexS = "[\\?&]" + name + "=([^&#]*)";
									var regex = new RegExp(regexS);
									if (typeof url === "undefined" || (url == null)) {
										url = window.location.href;
									}
									// if (logging.trace)
									//  console.debug("getting parameter [" + name + "] from url [" + url + "]"); // !!!
									var results = regex.exec(url);
									if (results == null) {
										if (typeof (defaultValue) !== 'undefined') {
											return defaultValue;
										} else {
											return "";
										}
									} else {
										return decodeURIComponent(results[1].replace(/\+/g, " "));
									}
								};
								// Convenience method that delegates to getParameterByName using only name and (optionally) defaultValue params
								this.getParameter = function(name, defaultValue) {
									return this.getParameterByName(name, null, defaultValue);
								};
								this.makeSynchronousCall = function(method, url) {
									var response = {};
									var xhttp = new XMLHttpRequest();
									xhttp.onreadystatechange = function() {
										if (this.readyState == 4) {
											// Typical action to be performed when the document is ready:
											response = {
												'status' : this.status,
												'response' : JSON.parse(this.response)
											}
										}
									};
									xhttp.open(method, url, false);
									xhttp.send();
									return response;
								}
								this.showJobLoader = function(action, loadTime) {
									$rootScope.$broadcast('jobLoader', action, loadTime);
								}
								this.getTimeZone = function() {
									return /\((.*)\)/.exec(new Date().toString())[1];
								};
								this.calculateTimeInSecs = function(unit, value) {
									switch (unit) {
									case 'Days':
										return value * 24 * 60 * 60;
									case 'Hours':
										return value * 60 * 60;
									case 'Minutes':
										return value * 60;
									case 'Seconds':
										return value;
									default:
										return 0;
									}
								}
								this.generateUUID = function() {
									var currentTime = new Date().getTime();
									var uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(chr) {
										var randomNum = (currentTime + Math.random() * 16) % 16 | 0;
										currentTime = Math.floor(currentTime / 16);
										return (chr == 'x' ? randomNum : (randomNum & 0x7 | 0x8)).toString(16);
									});
									return uuid;
								};
								this.convertTime = function (input) {
									var time = input / 60;
									var minutes;
									if (time < 720) {
										time = time / 60;
										if (time < 1) {
											time = time + 12;
										}
										minutes = (time - Math.floor(time)) * 60;
										if (minutes < 10) {
											time = Math.floor(time) + ":0" + minutes.toFixed(0) + " am";
										} else {
											time = Math.floor(time) + ":" + minutes.toFixed(0) + " am";
										}
									} else {
										time = time / 60 - 12;
										if (time < 1) {
											time = time + 12;
										}
										minutes = (time - Math.floor(time)) * 60;
										if (minutes < 10) {
											time = Math.floor(time) + ":0" + minutes.toFixed(0) + " pm";
										} else {
											time = Math.floor(time) + ":" + minutes.toFixed(0) + " pm";
										}
									}
									return time;
								};
								this.convertDate = function (input) {
									var cvLoc = $injector.get("cvLoc");
									if (input === 0) {
										return cvLoc('label.notSet');
									}
									var date = new Date(input * 1000);
									return date.getMonth() + 1 + "/" + date.getDate() + "/" + date.getFullYear();
								};
								this.xmlToJson = function(xml) {
									try {
										var obj = {};
										if (xml.children.length > 0) {
											for (var i = 0; i < xml.children.length; i++) {
												var item = xml.children.item(i);
												var nodeName = item.nodeName;
												if (typeof (obj[nodeName]) == "undefined") {
													obj[nodeName] = this.xmlToJson(item);
												} else {
													if (typeof (obj[nodeName].push) == "undefined") {
														var old = obj[nodeName];
														obj[nodeName] = [];
														obj[nodeName].push(old);
													}
													obj[nodeName].push(this.xmlToJson(item));
												}
											}
										} else {
											obj = xml.textContent;
										}
										return obj;
									} catch (e) {
										console.log(e.message);
									}
								}
								/**
								 * @param{string} input contains string which is a nested property like
								 *                a.b.c.d returns if the nested property value is not null or
								 *                not empty object or not empty string.
								 */
								this.objectContainsNestedProperty = function(nestedProperty, rootObject) {
									const objectHierarchy = nestedProperty.split(".");
									const hierarchy = objectHierarchy.slice(1, objectHierarchy.length).join(".");
									return !!hierarchy.split(".").reduce(function(obj, prop) {
										return obj && obj[prop] ? obj[prop] : undefined;
									}, rootObject);
								}
								this.getPermissionsForEntityDetailsResponse = function(metaInfo) {
									if (metaInfo && angular.isArray(metaInfo)) {
										for (var i = 0; i < metaInfo.length; i++) {
											var nameValue = metaInfo[i];
											if (nameValue && angular.isDefined(nameValue) && nameValue.name
													&& angular.isDefined(nameValue.name)
													&& nameValue.name === "permissions") {
												if (nameValue.value && angular.isDefined(nameValue.value)
														&& nameValue.value !== "") {
													return JSON.parse(nameValue.value);
												}
											}
										}
									}
									return {
										permissionIds : []
									};
								}
								/**
								 * takes in string and cvLoc factory and returns localized value, If it is
								 * already localized , it returns the value without localizing takes either
								 * the label key of localized string or actual localized string
								 */
								this.getLocalizedString = function(st) {
									var cvLoc = $injector.get("cvLoc");
									if (st && angular.isDefined(st)) {
										let localizedString = cvLoc(st);
										if (localizedString.substring(0, 3) === "???") {
											return st;
										}
										return localizedString;
									}
									return "";
								}
								this.removeParamsFromURLString = function(key, sourceURL) {
									var rtn = sourceURL.split("?")[0], param, params_arr = [], queryString = (sourceURL
											.indexOf("?") !== -1) ? sourceURL.split("?")[1] : "";
									if (queryString !== "") {
										params_arr = queryString.split("&");
										for (var i = params_arr.length - 1; i >= 0; i -= 1) {
											param = params_arr[i].split("=")[0];
											if (param === key) {
												params_arr.splice(i, 1);
											}
										}
										if (params_arr.length >= 1) {
											rtn = rtn + "?" + params_arr.join("&");
										}
									}
									return rtn;
								}
								this.removeRequiredAndOptionalParamsFromURLString = function(url) {
									if (url.includes(":")) {
										url = url.substring(0, url.indexOf(':'));
									}
									//If url has optional parameters, we remove the optional parameters
									if (url.includes("?")) {
										url = url.substring(0, url.indexOf('?'));
									}
									return url;
								}
								/*
								 * Takes a list of String , returns a sentence string with comma separated
								 * values and "and" appended to the last word Example : input [a,b,c] =>
								 * output "a, b and c" input [] => output "" input null,undefined => output ""
								 * input [a] => output "a" input [a,b] => output "a and b" . check for type
								 * array is pending
								 */
								this.getSentenceFromList = function(list) {
									let sentenceString = '';
									if (!list || list.length === 0) {
										return sentenceString;
									} else if (list.length === 1) {
										sentenceString = sentenceString + list[0];
									} else {
										for (let i = 0; i < list.length; ++i) {
											let appendToFront = ', ';
											if (i === 0) {
												appendToFront = '';
											} else if (i === list.length - 1) {
												appendToFront = ' and ';
											}
											sentenceString = sentenceString + appendToFront + list[i];
										}
									}
									return sentenceString;
								};

								this.toByteArray = function(str) {
									if (!str) {
										return null;
									}
									var bytes = [];
									for (var i = 0; i < str.length; ++i) {
										bytes.push(str.charCodeAt(i));
									}
									return bytes;
								};

								/*
								 * Returns additional setting value for an additional setting key set by user
								 * at the Commcell.
								 * 
								 * keyString - additional setting key additionalSettingType - Example
								 * :AdminConsole, Console defaultValue - If key is not present or set at the
								 * Commserver
								 * 
								 */
								this.getAdditionalSettingValue = function(keyString, additionalSettingType,
										defaultValue) {
									let property = 'additionalSettings.' + additionalSettingType + '.' + keyString;
									return _.get(cv, property, defaultValue);
								};

								this.setAdditionalSettingValue = function(keyString, additionalSettingType, value) {
									let property = 'additionalSettings.' + additionalSettingType + '.' + keyString;
									_.set(cv, property, value);
								}

								this.setTitle = function(titleName) {
									if (typeof titleName !== 'string') {
										return;
									}

									document.title = titleName;
								};

								this.modifyUrlForComet = function(url, commcellName, useBaseUrl) {
									if (cvUtil) {
										return cvUtil.modifyUrlForComet(url, commcellName, useBaseUrl);
									} else {
										return url;
									}
								}

								this.redirectHttpConfig = function(commCell, ignoreComet) {
									if (cvUtil) {
										return cvUtil.redirectHttpConfig(commCell, ignoreComet);
									} else {
										return {};
									}
								}

								this.checkCustomParam = function(key, val) {
									var actionAllowed = cv.customParams.filter(function(item) {
										if(!val) {
											return item[0]===key;
										}
										return item[0]===key && item[1]===val;
									});
									if(actionAllowed.length===1) {
										return true;
									}
									return false;
								}

								this.getCustomParamVal = function(key) {
									for(let i=0;i<cv.customParams.length;i++) {
										if(cv.customParams[i][0]===key) {
											return cv.customParams[i][1];
										}
									}
									return null;
								}

								this.checkCustomParamRedirect = function() {
									if(this.checkCustomParam('redirectURL')) {
										let $http = $injector.get("$http");
										let actionVal = this.getCustomParamVal('mAction');
										let url = decodeURIComponent(this.getCustomParamVal('redirectURL'));
										return new Promise(function(res, rej) {
											let call = $http.get('checkValidWhiteListDomain.do?domain='+url)
											call.then(function(resp) {
												if(resp.data=='false') {
													res(false);
												} else if(resp.data=='true') {
													//redirect here
													window.location = url + "?action="+actionVal;
												}
											}).catch(function(resp) {
												res(false);
											});
										});
									} else {
										return false;
									}
								}

								this.isValidIpAddress = function(hostName){
									return /^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/.test(
											hostName
										);
								}


							} ]);

	//global variable to hide toaster in case of save as script
		cvCommon.value('hideToaster',{
			hide : false
		});
	/**
	 * 
	 */
	cvCommon
			.factory(
					'cvToaster',
					[
							'growl',
							'cvLoc',
							'hideToaster',
							function(growl, cvLoc, hideToaster) {
								var toaster = {};
								var showMessage = function(message, msgType, ttl) {
									var fn = null;
									switch (msgType) {
									case 'warn':
										fn = growl.addWarnMessage;
										break;
									case 'error':
										fn = growl.addErrorMessage;
										break;
									case 'info':
										fn = growl.addInfoMessage;
										break;
									case 'success':
									default:
										fn = growl.addSuccessMessage;
									}
									if (ttl && (ttl > 0)) {
										fn(message, {
											ttl : ttl
										});
									} else {
										fn(message);
									}
								};
								var errorIcon = '<span class="toaster-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 104 104"><path d="M52 0a52 52 0 1 0 0 104A52 52 0 0 0 52 0zm0 94a42 42 0 1 1 0-84 42 42 0 0 1 0 84z"/><path d="M52.1 64.5c-2 0-3.8.8-5.2 2.2A7 7 0 0 0 52 78.5c1.9 0 3.6-.7 5-2 1.4-1.4 2-3 2-5a7 7 0 0 0-6.9-7zM48.4 58.6a4.7 4.7 0 0 0 3.6 1.8c1.4 0 2.6-.5 3.5-1.7l.1-.1c.7-1 1.1-2.1 1.3-3.4l1.4-27.1c.1-1.3-.3-2.6-1.2-3.8l-.1-.2a4.6 4.6 0 0 0-3.6-1.8h-2.9c-1 0-2.4.3-3.6 1.8-.9 1-1.3 2.3-1.3 3.8L47 55a5 5 0 0 0 1.4 3.6z"/></svg></span>';
								var successIcon = '<span class="toaster-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 104 104"><path d="M52 10a42 42 0 1 1-42 42 42 42 0 0 1 42-42m0-10a52 52 0 1 0 52 52A52 52 0 0 0 52 0z"/><path d="M46 71.8a5 5 0 0 1-3.5-1.5l-17-17a5 5 0 0 1 7-7L46 59.6l24-24a5 5 0 0 1 7 7L49.5 70.4a5 5 0 0 1-3.5 1.5z"/></svg></span>';
								toaster.showWarnMessage = function(ops) {
									// no need to show empty popup message
									if (!ops.message) {
										return;
									}
									showMessage(errorIcon + '<div>' + ops.message + '</div>', 'warn', ops.ttl || '3000');
								};
								toaster.showErrorMessage = function(ops) {
									//if state changed and request got cancelled, do not show toaster
									if (hideToaster.hide || System.hideToaster || !ops.message) {
										return;
									}
									showMessage(errorIcon + '<div>' + ops.message + '</div>', 'error', ops.ttl
											|| '3000');
								};
								toaster.showInfoMessage = function(ops) {
									// no need to show empty popup message
									if (hideToaster.hide ||!ops.message) {
										return;
									}
									showMessage(successIcon + '<div>' + ops.message + '</div>', 'info', ops.ttl
											|| '3000');
								};
								toaster.showSuccessMessage = function(ops) {
									// no need to show empty popup message
									if (!ops.message) {
										return;
									}
									showMessage(successIcon + '<div>' + ops.message + '</div>', 'success', ops.ttl
											|| '3000');
								};
								toaster.showErrorMessageWrapper = function(error, ttl) {
									var ops = {
										'message' : '',
										'ttl' : ttl
									}
									if (error) {
										if (angular.isString(error) && !_.startsWith(error, "<")) {
											ops.message = error;
										} else if (angular.isString(error.data) && !_.isEmpty(error.data)) {
											ops.message = error.data;
										} else if (angular.isString(error.errorMessage)
												&& !_.isEmpty(error.errorMessage)) {
											ops.message = error.errorMessage;
										} else {
											ops.message = cvLoc('generic_error');
										}
									}
									toaster.showErrorMessage(ops)
								};
								return toaster;
							} ]);
	/**
	 * 
	 */
	cvCommon.factory('cvConfirmDialog', [
			'$uibModal',
			function($modal) {
				var factory = {};
				factory.openDialog = function(opts) {
					return $modal.open({
						templateUrl : appUtil.appRoot + 'common/partials/cvConfirmDialog.jsp',
						backdrop : 'static',
						windowClass : 'cv-confirm-dialog confirm-dialog',
						controller : [
								'$rootScope',
								'$scope',
								'$uibModalInstance',
								'cvLoc',
								'$log',
								function($rootScope, $scope, $modalInstance, cvLoc, $log) {
									$scope.confirmationText = opts.confirmationText;
									$scope.okText = opts.okText;
									$scope.cancelText = opts.cancelText;
									$scope.okCallback = function() {
										if (typeof opts.okCallback === 'function') {
											opts.okCallback();
										}
										$modalInstance.close();
									};
									$scope.cancelCallback = function() {
										if (typeof opts.cancelCallback === 'function') {
											opts.cancelCallback();
										}
										$modalInstance.dismiss();
									};
								} ]
					});
				};
				return factory;
			} ]);
	/**
	 * 
	 */
	cvCommon.factory('cvGridSearchFactory', [ '$rootScope', '$parse', function($rootScope, $parse) {
		var factory = {};
		factory.singleFilter = function(renderableRows, searchText, fieldNameArray) {
			// check if there fields eligible for search and there is a search text
			if (fieldNameArray.length && searchText) {
				// create a regex for search text
				var matcher = new RegExp(searchText, "i");
				renderableRows.forEach(function(row) {
					var match = false;
					fieldNameArray.forEach(function(field) {
						// get the entity qualified field for the row
						var fieldVarString = row.getEntityQualifiedColField({
							field : field
						});
						// create angular scope resolver
						var getter = $parse(fieldVarString);
						// resolve scope
						var searchField = getter(row);
						// if evaluated field has a valid value and its a match
						if (searchField && searchField.toString().match(matcher)) {
							match = true;
						}
					});
					if (!match) {
						row.visible = false;
					}
				});
			}
			return renderableRows;
		}
		return factory;
	} ]);
	cvCommon.factory('cvMultiCommcell', [ '$rootScope', '$location', '$transitions', function($rootScope, $location, $transitions) {
		var mc = {};
		var multicomcellCapability = false;
		var ccName = cv.commcellName;
		var commCellNameforLinks;
		var selectedCClength = cv.sessionContext.dataSourceList.length;
		var multiselection = selectedCClength < 2 ? false : true;
		var selectedCommcells;
		mc.setcommCellNameforLinks = function(commcellName) {
			commCellNameforLinks = commcellName;
		}
		mc.resetcommCellNameforLinks = function() {
			commCellNameforLinks = undefined;
		}
		mc.getcommCellNameforLinks = function() {
			return commCellNameforLinks;
		}
		mc.setCapability = function() {
			multicomcellCapability = true;
		}
		mc.resetCapability = function() {
			multicomcellCapability = false;
		}
		mc.getmultiselection = function() {
			return multiselection;
		}
		mc.setMultiSelection = function(isMultiselected) {
			multiselection = isMultiselected;
		}
		mc.getccName = function() {
			return ccName;
		}
		mc.getCapability = function() {
			return multicomcellCapability;
		}
		mc.setselectedCClength = function(len) {
			selectedCClength = len;
		}
		mc.getselectedCClength = function() {
			return selectedCClength;
		}
		mc.setSelectedCommcell = function(cn) {
			selectedCommcells = cn;
		}
		mc.getSelectedCommcell = function() {
			return selectedCommcells;
		}
		if (cv.sessionContext.isMultiComcellAware) {
			$transitions.onStart({}, function() {
				mc.resetCapability();
				mc.resetcommCellNameforLinks();
			});
		}
		return mc;
	} ]);
	cvCommon.factory('cvBreadcrumbsTabsFactory', [ '$rootScope', '$transitions', function($rootScope, $transitions) {
		var bc = {};
		var bcData = [];
		var tabData = [];
		var bcCallbacks = [];
		var tabCallbacks = [];
		bc.getBreadCrumbs = function() {
			return bcData;
		};
		bc.getTabData = function() {
			return tabData;
		};
		function validateAndAddBreadcrumb(item) {
			var repeated = false;
			// Since total breadcrumb length is small, can just iterate over it every time breadcrumb is added.
			_.forEach(bcData, function(breadcrumb) {
				if (_.get(breadcrumb, 'link') === _.get(item, 'link') &&
				_.get(breadcrumb, 'title') === _.get(item,'title') &&
				_.get(breadcrumb, 'current') === _.get(item, 'current')) {
					repeated = true;
					// Break out of loop
					return false;
				}
			});
			if (!repeated){bcData.push(item);}
		}
		bc.addBreadCrumbs = function(newItem) {
			if (newItem instanceof Array) {
				angular.forEach(newItem, function(item) {
					validateAndAddBreadcrumb(item);
				});
			} else {
				validateAndAddBreadcrumb(newItem);
			}
			angular.forEach(bcCallbacks, function(fn) {
				fn(bcData);
			});
		};
		bc.addTabs = function(newItem) {
			if (newItem instanceof Array) {
				angular.forEach(newItem, function(item) {
					tabData.push(item);
				});
			} else {
				tabData.push(newItem);
			}
			angular.forEach(tabCallbacks, function(fn) {
				fn(tabData);
			});
		};
		bc.registerBcDataChangeObserver = function(callback) {
			bcCallbacks.push(callback);
		};
		bc.registerTabDataChangeObserver = function(callback) {
			tabCallbacks.push(callback);
		};
		bc.clearBc = function() {
			bcData = [];
		};
		bc.clearTabs = function() {
			tabData = [];
		};
		$transitions.onStart({}, function(transition) {
			/**
			 * Fix for avoiding reset of global breadcrumbs if only subroute is reloaded with different params
			 * but state is same
			 */
			var toState = transition.to();
			var fromState = transition.from();
			if (!(toState.name === fromState.name)) {
				bc.clearBc();
				bc.clearTabs();
				angular.forEach(bcCallbacks, function(fn) {
					fn(bcData);
				});
				angular.forEach(tabCallbacks, function(fn) {
					fn(tabData);
				});
			}
		});
		return bc;
	} ]);
	cvCommon.factory('cvEscapeRegExp', [ function() {
		var escapeRegExp = function(str) {
			if (!str) {
				return '';
			}
			return str.replace(/[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, "\\$&");
		};
		return escapeRegExp;
	} ]);
	/*
	 * Adding a close button to all modals This is achieved by extending the modalWindow directive part of the
	 * bootstrap modal module
	 */
	var uiBt = angular.module('ui.bootstrap.modal');
	uiBt.config([
			'$provide',
			'$injector',
			function($provide, $injector) {
				$provide.decorator('uibModalWindowDirective', [
						'$delegate',
						'$controller',
						'$injector',
						function($delegate, $controller, $injector) {
							var directive = $delegate[0];
							var compile = directive.compile;
							var modal = $injector.get('$uibModalStack');
							directive.compile = function(tElement, tAttrs) {
								var link = compile.apply(this, arguments);
								return function(scope, elem, attrs) {
									link.apply(this, arguments);
									var closeAnchor = $('<a class="modal__close-btn" href=""></a>');
									if (elem.find('.modal-content').find('.modal__close-btn').length == 0) {
										elem.find('.modal-content').append(closeAnchor);
										closeAnchor.bind("click", function(e) {
											e.preventDefault();
											modal.dismiss(modal.getTop().key, 'close');
										});
									}
								};
							};
							return $delegate;
						} ]);
			} ]);
	cvCommon.filter('capacity', [ 'cvLoc', function(cvLoc) {
		var format = function(number, unit) {
			if (number.toFixed(2) == Math.floor(number.toFixed(2))) {
				return Math.floor(number.toFixed(2)) + unit; //in case of 3.999 .toFixed(2) makes it 4.00 - so important to retain .toFixed
			}
			return number.toFixed(2) + unit;
		};
		return function(x) {
			var value = Number(x);
			if (isNaN(value)) {
				return x;
			}
			if (value <= 1024) {
				return format(value, ' ' + cvLoc('labelBytes'));
			}
			value = value / 1024;
			if (value.toFixed(2) < 1024) {
				return format(value, ' ' + cvLoc('labelKB'));
			}
			value = value / 1024;
			if (value.toFixed(2) < 1024) {
				return format(value, ' ' + cvLoc('labelMB'));
			}
			value = value / 1024;
			if (value.toFixed(2) < 1024) {
				return format(value, ' ' + cvLoc('labelGB'));
			}
			value = value / 1024;
			if (value.toFixed(2) < 1024) {
				return format(value, ' ' + cvLoc('labelTB'));
			}
			value = value / 1024;
			if (value.toFixed(2) < 1024) {
				return format(value, ' ' + cvLoc('labelPB'));
			}
			value = value / 1024;
			return format(value, ' ' + cvLoc('labelEB'));
		};
	} ]);
	/**
	 * filter that encodes parameter
	 */
	cvCommon.filter('encodeParam', function() {
		return function(input) {
			return encodeURIComponent(input);
		};
	});
	/**
	 * filter to show localized string if it exists
	 */
	cvCommon.filter('localizedString', [ 'cvUtil', function(cvUtil) {
		return function(input) {
			return cvUtil.getLocalizedString(input);
		};
	} ]);
	/**
	 * 
	 * 
	 * /** An angular filter that utilizes the cvLoc factory.
	 * 
	 * @param {string}
	 *            input - The label to pass to cvLoc.
	 */
	cvCommon.filter('cvLoc', [ 'cvLoc', function(cvLoc) {
		return cvLoc;
	} ]);
	/**
	 * An angular filter that preprends the "appUtil.appRoot" variable before the given path.
	 * 
	 * @param {string}
	 *            path - The path to prepend "appUtil.appRoot"
	 */
	cvCommon.filter('appRoot', [ function() {
		return function(path) {
			return appUtil.appRoot + path;
		};
	} ]);
	/**
	 * A filter that will transform all text inside brackets [] into hyperlinks. Since this filter outputs
	 * HTML, you will have to use the data-ng-bind-html directive to ensure there is proper output with this
	 * filter.
	 * 
	 * @param {string}
	 *            text - The text to transform all text inside brackets with hyperlinks.
	 * @param {...string}
	 *            var_args - An indefinite amount of URLs to be used when transorming the text inside brackets
	 *            into hyperlinks. The order in which the argument was given will be match the order of the
	 *            text found inbetween brackets. As an example, the first URL given will be used for the first
	 *            instance of text inside brackets.
	 */
	cvCommon.filter('replaceBracketsWithLinks', [ function() {
		return function(text) {
			var i = 0;
			var filterArgs = arguments;
			return text.replace(/\[(.+?)\]/g, function($0, $1) {
				if (angular.isDefined(filterArgs[i + 1])) {
					var str = '<a href="' + filterArgs[i + 1] + '">' + $1 + '</a>';
					i++;
					return str;
				}
				return $1;
			});
		};
	} ]);
	cvCommon.filter('spaceless', function() {
		return function(input) {
			if (input) {
				return input.replace(/\s+/g, '-');
			}
		}
	});
	cvCommon.filter('daysOfWeek', [ 'cvLoc', function(cvLoc) {
		return function(input) {
			var DaysEnum = {
				"MONDAY" : cvLoc('label.monday'),
				"TUESDAY" : cvLoc('label.tuesday'),
				"WEDNESDAY" : cvLoc('label.wednesday'),
				"THURSDAY" : cvLoc('label.thursday'),
				"FRIDAY" : cvLoc('label.friday'),
				"SATURDAY" : cvLoc('label.saturday'),
				"SUNDAY" : cvLoc('label.sunday')
			};
			var stringToDisplay = '';
			if (input) {
				for (var i = 0; i < input.length; i++) {
					if (DaysEnum[input[i]]) {
						if (i === input.length - 1) {
							stringToDisplay += DaysEnum[input[i]] + " ";
						} else {
							stringToDisplay += DaysEnum[input[i]] + ", ";
						}
					}
				}
			}
			return stringToDisplay;
		}
	} ]);
	cvCommon.filter('getDay', [ 'cvLoc', function(cvLoc) {
		return function(input) {
			if (input === 0) {
				return cvLoc('label.notSet');
			}
			var date = new Date(input * 1000)
			return date.getMonth() + 1 + "/" + date.getDate() + "/" + date.getFullYear();
		}
	} ]);
	/** Filter to display date in different date formats * */
	cvCommon.filter('getDateFormat', [ 'cvLoc', function(cvLoc) {
		return function(input, pattern) {
			if (input === 0) {
				return cvLoc('label.notSet');
			}
			var defaultPattern = 'DD/MM/YYYY';
			if (!pattern) {
				pattern = defaultPattern;
			}
			var date = new Date(input * 1000);
			return moment(date).format(pattern);
		}
	} ]);
	cvCommon.filter('getTime', function() {
		return function(input) {
			var time = input / 60;
			var minutes;
			if (time < 720) {
				time = time / 60;
				if (time < 1) {
					time = time + 12;
				}
				minutes = (time - Math.floor(time)) * 60;
				if (minutes < 10) {
					time = Math.floor(time) + ":0" + minutes.toFixed(0) + " am"
				} else {
					time = Math.floor(time) + ":" + minutes.toFixed(0) + " am"
				}
			} else {
				time = time / 60 - 12;
				if (time < 1) {
					time = time + 12;
				}
				minutes = (time - Math.floor(time)) * 60;
				if (minutes < 10) {
					time = Math.floor(time) + ":0" + minutes.toFixed(0) + " pm"
				} else {
					time = Math.floor(time) + ":" + minutes.toFixed(0) + " pm"
				}
			}
			return time;
		}
	});
	cvCommon.filter('dateTime', [ 'cvLoc', '$filter', function(cvLoc, $filter) {
		return function(text) {
			if (-1 === text) {
				return cvLoc('schedule.automatic');
			}
			var date = new Date(text * 1000);
			return $filter('date')(date, 'MMM d, h:mm:ss a');
		};
	} ]);
	/**
	 * An angular filter to parse Html and return the text content
	 * 
	 * @param {string}
	 *            text - input text.
	 */
	cvCommon.filter('extractTextFromHtml', [ '$sce', function($sce) {
		var div = document.createElement('div');
		return function(text) {
			div.innerHTML = text;
			//replace br tags with space ,
			//to fix no space issue for job status text
			text.replace(/<br>|<br\/>/g,' ');
			return $sce.trustAsHtml(div.textContent.trim());
		};
	} ]);

	/*
	 * Return text by making hyperlink clickable
	 */

	cvCommon.filter('parseUrl', ['$sce', 'REGEX', function($sce, REGEX) {

		return function(text, asTrusted) {
			if(text.match(REGEX.URL)) {
				text = text.replace(REGEX.URL, "<a href=\"$1\" target=\"_blank\">$1</a>")
			}

			if(asTrusted) {
				return $sce.trustAsHtml(text);
			}
			return text;
		}
	}]);

	cvCommon.controller('paginationController', [
			'$scope',
			'$log',
			'serverService',
			'jobService',
			'cvLoc',
			'cvUtil',
			function($scope, $log, serverService, jobService, cvLoc, cvUtil) {
				$scope.pageNext = function() {
					var maxPages = Math.ceil($scope.totalServerItems / $scope.pagingOptions.pageSize);
					var t = parseInt($scope.pagingOptions.currentPage) + 1;
					$scope.pagingOptions.currentPage = Math.min(t, maxPages);
				};
			} ]);
	/**
	 * Page level Actions
	 */
	cvCommon.factory('cvPageActionsFactory', [ '$rootScope', '$transitions', function($rootScope, $transitions) {
		var actions = [];
		var updateUiMethods = [];
		this.getActions = function() {
			return actions;
		}
		this.add = function(newActions) {
			if (newActions instanceof Array) {
				angular.forEach(newActions, function(newAction) {
					actions.push(newAction);
				});
			} else {
				actions.push(newActions);
			}
			angular.forEach(updateUiMethods, function(fn) {
				fn(actions);
			});
		};
		this.registerChangeObserver = function(updateUiMethod) {
			updateUiMethods.push(updateUiMethod);
		}
		$transitions.onStart({}, function() {
			actions = [];
			angular.forEach(updateUiMethods, function(fn) {
				fn(actions);
			});
		});
		return this;
	} ]);
	cvCommon
			.constant(
					'REGEX',
					{
						EMAIL : /^['_a-zA-Z0-9-]+(\.['_a-zA-Z0-9-]+)*@[a-zA-Z0-9-]+(\.[a-zA-Z0-9-]+)*(\.[a-zA-Z]{2,64})$/,
						URL: /(\b(https?|ftp):\/\/[A-Z0-9+&@#\/%?=~_|!:,.;-]*[-A-Z0-9+&@#\/%=~_|])/gim,
						DOMAIN : /^([a-z0-9]+(-[a-z0-9]+)*\.)+[a-z]{2,}$/,
						DOMAIN_WITH_WILDCARDS : /(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?\.)+[a-z0-9][a-z0-9-]{0,61}[a-z0-9](?!:)/,
						WEBSITES:/^((https?|http):\/\/)?(www.)?[a-z0-9]+\.[a-z]+(\/[a-zA-Z0-9#]+\/?)*$/,
						WINDOWS_LOCAL_PATH : /^(([a-zA-Z]:{1})|\*|\?)((\\|[*?]?)[^<>:"/\\|]+)*\\?$/,
						WINDOWS_GLOBAL_FILTERS_PATH : /^((<P:e=.>){0,1}([a-zA-Z*?]:{1})|\*|\?)((\\|[*?]?)[^<>:"/\\|]+)*\\?$/,
						WINDOWS_NETWORK_PATH : /^[\\*?]((\\|[*?]?)[^<>:"/\\|]+)+\\?$/,
						UNIX_PATH : /^[/*?]([^\0]+\/?)*$/,
						UNIX_GLOBAL_FILTERS_PATH : /^[\/*?\[]([^\0:]+)*$|^[a-zA-Z0-9.]+[\*?\[]([^\0:]+)*$/,
						SMTP : /^[A-Za-z0-9]+(?:-[a-zA-Z0-9]+)*(?:\.[A-Za-z0-9]+(?:-[a-zA-Z0-9]+)*)*(?=.{3,64}$)(\.[a-zA-Z0-9](?:-?[a-zA-Z0-9])*)$/,
						WEBCONSOLE_URL_WITH_MANDATORY_PORT_NUMBER : /^(http|https):\/\/[a-zA-Z0-9-_]+(\.[a-zA-Z0-9-_]+)*(\.[a-zA-Z]{2,4})?:\d{1,5}\/(webconsole)$/,
						EXTERNAL_USERNAME_WITH_DOMAIN : /^[a-zA-Z0-9-._]{1,100}\\[a-zA-Z0-9-._]{1,100}$/,
						AZURE_VM_NAME : /^[a-zA-Z0-9](([a-zA-Z0-9_[-])){1,64}[a-zA-Z0-9]$/,
						OCI_VM_NAME : /^[a-zA-Z0-9]*[A-Za-z0-9_-]*[a-zA-Z0-9]$/,
						GCP_VM_NAME : /^[a-z]([-a-z0-9]*[a-z0-9])?$/,
						SMTP_COMMA_SEPARATED : /^[A-Za-z0-9]+(-[a-zA-Z0-9]+)*(\.[A-Za-z0-9]+(-[a-zA-Z0-9]+)*)*(?=.{3,64}$)(\.[a-zA-Z0-9](?:-?[a-zA-Z0-9])*)\s*(,\s*[A-Za-z0-9]+(-[a-zA-Z0-9]+)*(\.[A-Za-z0-9]+(-[a-zA-Z0-9]+)*)*(?=.{3,64}$)(\.[a-zA-Z0-9](?:-?[a-zA-Z0-9])*)\s*)*$/,
						OPENVMS_PATH : /^@([A-Za-z0-9_$~][-A-Za-z0-9_$~]*):\[(([A-Za-z0-9_$~][-A-Za-z0-9_$~]*((\^[.+])?[-A-Za-z0-9_$~]?)*)\.)*([A-Za-z0-9_$~][-A-Za-z0-9_$~]*((\^[.+])?[-A-Za-z0-9_$~]?)*)\][A-Za-z0-9_$][-A-Za-z0-9_$~]*((\^[.+])?[-A-Za-z0-9_$~]?)*\.[-A-Za-z0-9_$]{1,38}(;\d+)?(\ [^\ ]*)*$/,
						PASSWORD_COMPLEXITY : {
							1 : /(.*)/,
							2 : /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#\$%\^&\*])(?=.{8,})/,
							3 : /^(?=(.*[a-z]){2})(?=(.*[A-Z]){2})(?=(.*\d){2})(?=(.*[!@#\$%\^&\*]){2})(?=.{8,})/
						},
						IP_ADDRESS_COMMA_SEPARATED : /^(([a-zA-Z0-9]+[.])*([a-zA-Z0-9]+)([,]([a-zA-Z0-9]+[.])*([a-zA-Z0-9]+))*)*$/,
						IP_ADDRESS : '^([01]?\\d\\d?|2[0-4]\\d|25[0-5]|\\*)\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5]|\\*)\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5]|\\*)\\.([01]?\\d\\d?|2[0-4]\\d|25[0-5]|\\*)$',
						//allowed_netmasks =
						//		["128.0.0.0", "192.0.0.0", "224.0.0.0", "240.0.0.0", "248.0.0.0", "252.0.0.0", "254.0.0.0", "255.0.0.0", "255.128.0.0", "255.192.0.0",
						//		"255.224.0.0", "255.240.0.0", "255.248.0.0", "255.252.0.0", "255.254.0.0", "255.255.0.0", "255.255.128.0", "255.255.192.0", "255.255.224.0", "255.255.240.0", "255.255.248.0",
						//		"255.255.252.0", "255.255.254.0", "255.255.255.0", "255.255.255.128", "255.255.255.192", "255.255.255.224", "255.255.255.240", "255.255.255.248", "255.255.255.252",
						//		"255.255.255.254", "255.255.255.255"];
						NETMASK : '^(((255\.){3}(255|254|252|248|240|224|192|128+))|((255\.){2}(255|254|252|248|240|224|192|128|0+)\.0)|((255\.)(255|254|252|248|240|224|192|128|0+)(\.0+){2})|((255|254|252|248|240|224|192|128|0+)(\.0+){3}))$',
						NUMBER_ONLY: /^[0-9]+$/,
						ALPHANUMERIC: /^[a-zA-Z0-9]+$/,
						K8s_URL_DOMAIN: /^(http|https):\/\/([a-zA-Z0-9][a-zA-Z0-9-_]*\.)*[a-zA-Z0-9]*[a-zA-Z0-9-_]*[[a-zA-Z0-9](:[0-9]+)?$/im,
						LOOPBACK_RGX: /localhost|127(?:\.[0-9]+){0,2}\.[0-9]+|^(?:0*\:)*?:?0*1$/i,
						K8s_ACCESSNODE_PATTERN:/^[a-z0-9]([-a-z0-9]*[a-z0-9])?(\.[a-z0-9]([-a-z0-9]*[a-z0-9])?)*$/,
						SALEFORCE_LOGIN_URL: /^(http|https):\/\/[a-zA-Z0-9-_]+(\.[a-zA-Z0-9-_\/]+)*$/
					});
	cvCommon.constant('PASSWORD_COMPLEXITY_LEVEL', {
		NONE : "1",
		MEDIUM : "2",
		STRONG : "3"
	})
	cvCommon.constant('FEATURE_TYPE', {
		LAPTOP : 145,
		DLP : 148,
		EDGE_DRIVE : 146,
		ARCHIVING : 215
	});
	cvCommon.constant('SCHEDULE_PROPERTY', {
		SLA_FLAG : 65536,
		SLA_REPEAT_HOURS_4 : 4,
		SLA_REPEAT_HOURS_12 : 12,
		SLA_INTERVALS : [ 1, 2, 3, 5, 7, 14, 21, 30, 90 ],
		FULL_BACKUP_RPO : 4194304
	});
	cvCommon.constant('ADVANCE_JOB_DETAIL_TYPE', {
		RETENTION_INFO : 1,
		REFERENCE_COPY_INFO : 2,
		DASH_COPY_INFO : 4,
		ADMIN_DATA_INFO : 8,
		BKUP_INFO : 16
	});
	cvCommon.constant('SPECIAL_CLIENT_GROUPS', {
		ALL_CLIENTS : {
			'clientGroupId' : 0
		}
	})
	//Below constants are related to CloudApps agent. These are added in common module because CAPPS_CONSTANTS needs to be referred in few controllers eg idaBrowse.
	//Other agents eg sql do not refer cappsMod so these are added in cvCommon
	cvCommon.constant("CAPPS_CONSTANTS", {
						"PLAN_TYPE_MSP" : "MSP",
						"PLAN_SUBTYPE_SERVER" : "Server",
						"MSG_LOADING" : "Loading",
						"REGEX_EMAIL" : /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/,
						"CLOUD_APP_URL_TEXT" : { // to avoid loop search for text
							"1" : "gsuite/clientDetails",
							"2" : "gsuite/clientDetails",
							"3" : "salesForce",
							"7" : "oneDrive",
							"8" : "amazonSQL",
							"9" : "azureSQL",
							"22" : "dynamoDB"
						},
						CLOUD_APP_INSTANCE_TYPE : {
							GOOGLE_MAIL : {
								key : "GOOGLE_MAIL",
								value : 1,
								localizeTypeKey : "label.clientType.gmail",
								urlText : "gsuite/clientDetails",
								svg : 'common/img/gmail.svg'
							},
							GOOGLE_DRIVE : {
								key : "GOOGLE_DRIVE",
								value : 2,
								localizeTypeKey : "label.clientType.gdrive",
								urlText : "gsuite/clientDetails",
								svg : 'common/img/gdrive.svg'
							},
							SALESFORCE : {
								key : "SALESFORCE",
								value : 3,
								localizeTypeKey : "label.clientType.salesforce",
								urlText : "salesForce",
								svg : 'common/img/salesforce.svg'
							},
							ONEDRIVE : {
								key : "ONEDRIVE",
								value : 7,
								localizeTypeKey : "label.clientType.onedrive",
								urlText : "oneDrive",
								svg : 'common/img/onedrive.svg'
							},
							AMAZON_S3 : {
								key : "AMAZON_S3",
								value : 5,
								localizeTypeKey : "label.clientType.amazonS3",
							},
							AZURE_BLOB : {
								key : "AZURE_BLOB",
								value : 6,
								localizeTypeKey : "label.clientType.azureBlob",
							},
							AMAZON_SQL : {
								key : "AMAZON_SQL",
								value : 8,
								localizeTypeKey : "label.clientType.amazonSql",
								urlText : "amazonSQL",
								svg : 'common/img/amazonsql.svg'
							},
							AZURE_SQL : {
								key : "AZURE_SQL",
								value : 9,
								localizeTypeKey : "label.clientType.azureSql",
								urlText : "azureSQL",
								svg : 'common/img/azuresql.svg'
							},
							DISTRIBUTED_SYSTEM : {
								key : "DISTRIBUTED_SYSTEM",
								value : 10,
								localizeTypeKey : "label.clientType.distributedSystem",
								urlText : "distributedSystem",
								svg : 'common/img/distributedsystem.svg'
							},
							//AMAZON_RDS is not integrated with cloud apps solution, add value and key when it's ready to integrate
							AMAZON_RDS : {
								key : "AMAZON_RDS",
								value : 4
							},
							GOOGLE_CLOUD : {
								key : "GOOGLE_CLOUD",
								value : 20,
								localizeTypeKey : "label.clientType.googleCloud",
							},
							AZURE_DATA_LAKE : {
								key : "AZURE_DATA_LAKE",
								value : 21,
								localizeTypeKey : "label.clientType.azureDataLake",
							},
							AMAZON_DYNAMODB : {
								key : "AMAZON_DYNAMODB",
								value : 22,
								localizeTypeKey : "label.clientType.amazonDynamoDb"
							},
							AMAZON_REDSHIFT : {
								key : "AMAZON_REDSHIFT",
								value : 26
							},
							AMAZON_DOCUMENTDB : {
								key : "AMAZON_DOCUMENTDB",
								value : 27
							},
							IBM_COS : {
								key : "IBM_COS",
								value : 24,
								localizeTypeKey : "label.clientType.ibmCOS",
							},
							ALIBABA_OSS : {
								key : "ALIBABA_OSS",
								value : 25,
								localizeTypeKey : "label.clientType.alibabaOSS",
							},
							AZURE_TABLE_STORAGE : {
								key : "AZURE_TABLE_STORAGE",
								value : 32,
								localizeTypeKey : "label.clientType.azureTableStorage",
							},
							GIT_HUB : {
								key : "GIT_HUB",
								value : 33,
								localizeTypeKey : "label.clientType.github",
								svg: "common/img/github.svg"
							},
							AZURE_DEVOPS : {
								key : "AZURE_DEVOPS",
								value :34,
								localizeTypeKey : "label.clientType.azuredevops",
								svg: "common/img/Azure-DevOps.svg"
							},
							AZURE_COSMOS_DB_SQL_API : {
								key : "AZURE_COSMOS_DB_SQL_API",
								value : 23,
								localizeTypeKey : "label.clientType.azureCosmosDBSQL",
							},
							GCP_SPANNER: {
								key: 'GCP_SPANNER',
								value: 37,
								localizeTypeKey: "label.clientType.GCPSpanner"
							}
						},
						AMAZON_DYNAMODB_CONSTANTS : {
							MAX_TEMP_READ_CAPACITY : 40000,
							MAX_TEMP_WRITE_CAPACITY : 40000
						},
						AZURE_COSMOS_DB_CONSTANTS: {
							MAX_THROUGHPUT: 1000000
						},
						//referred from GUIMessageConstants.java
						CLOUD_APP_CONTENT_TYPE : {
							GOOGLE_CONTENT_TYPE_USER_ACC : {
								key : "GOOGLE_CONTENT_TYPE_USER_ACC",
								value : 134,
								toLocalizeTypeKey : "content.type.google.user.acc",
								iconClass : "capps-icon capps-email-type"
							},
							GOOGLE_CONTENT_TYPE_USER_GRP : {
								key : "GOOGLE_CONTENT_TYPE_USER_GRP",
								value : 135,
								toLocalizeTypeKey : "content.type.google.group.acc",
								iconClass : "capps-icon capps-group-type"
							},
							ONEDRIVE_CONTENT_TYPE_AZURE_GRP : {
								key : "ONEDRIVE_CONTENT_TYPE_AZURE_GRP",
								value : 135,
								toLocalizeTypeKey : "content.type.azure.group.acc",
								iconClass : "capps-icon capps-group-type"
							},
							GOOGLE_CONTENT_TYPE_USER_REGEX : {
								key : "GOOGLE_CONTENT_TYPE_USER_REGEX",
								value : 0,
								toLocalizeTypeKey : "",
								iconClass : "capps-icon capps-regex-type"
							}
						},
						//end
						"NO_OF_GOOGLE_USERS_FOR_PREVIEW" : 2,
						"NO_OF_GOOGLE_GRPS_FOR_PREVIEW" : 3,
						"NO_OF_REGEX_PATTERN_FOR_PREVIEW" : 2,
						"SC_CONTENT_REGEX_LOC_KEY" : "content.type.regex",
						"REGEX_TEXT" : "REGEX",
						"USER_GRP_TEXT" : "USER_GROUP",
						"BROWSE_PATH_PREFIX_GDRIVE" : "\\GDrive",
						"BROWSE_PATH_PREFIX_GMAIL" : "\\Gmail",
						"BROWSE_PATH_PREFIX_ONEDRIVE" : "\\OneDrive",
						"SALESFORCE_OBJECT_PREFIX" : "/Objects/",
						"SALESFORCE_FILE_PREFIX" : "/Files/",
						'SALESFORCE_DEFAULT_TABLE' : "Organization",
						//Start Backup types
						BACKUP_TYPE : {
							INCREMENTAL : "INCREMENTAL",
							FULL : "FULL",
							SYNTH_FULL : "SYNTHETIC_FULL"
						},
						RESTORE_TYPE : {
							DISK : "disk",
							GOOGLE : "google",
							ACCOUNT : "account"
						},
						MAILBOX_NODE : {
							ROOT : 'root',
							MAILBOX : 'mailbox',
							MAILBOX_FOLDER : 'mailbox_folder'
						},
						SALESFORCE_NODE : {
							ROOT : 'sf_root',
							FILE_VIEW : 'sf_file_view',
							FILE_VIEW_FOLDER : 'sf_file_view_folder',
							OBJECT_VIEW : 'sf_object_view',
							METADATA_VIEW : 'sf_metadata_view',
							METADATA_VIEW_PACKAGE : 'sf_metadata_view_package',
							METADATA_VIEW_TYPE : 'sf_metadata_view_type',
							METADATA_VIEW_FOLDER : 'sf_metadata_view_folder'
						},
						SYNC_DATABASE_TYPE : {
							SQLSERVER : "SQLSERVER",
							POSTGRESQL : "POSTGRESQL",
							ORACLE: 'ORACLE',
							H2_EMBEDDED: 'H2_EMBEDDED'
						},
						BROWSE_TYPE : {
							METADATA : "metadata",
							RECORD_RESTORE : "Record Restore",
							SANDBOX_SEEDING : "sandbox seeding"
						},
						SALESFORCE_CONTENT : {
							ALL_OBJECTS : {
								LOC_KEY : "label.allSfObjects"
							},
							ALL_FILES : {
								LOC_KEY : "label.allSfFiles"
							},
							ALL_METADATA : {
								LOC_KEY : "label.allSfMetadata"
							}
						},
						SF_SEEDING_RULES : {
							ALL_IDSS : {
								KEY : "ALL_IDSS",
								LOC_KEY : "label.sfAllIdsText",
							},
							USER_QUERY : {
								KEY : "USER_QUERY",
								LOC_KEY : "label.sfUserQueryText",
								PARAM_FIELD : "query"
							},
							ADDED_IN_LAST_N_DAYS : {
								KEY : "ADDED_IN_LAST_N_DAYS",
								LOC_KEY : "label.sfAddedInLastNDaysText",
								PARAM_FIELD : "nDays",
								DEFAULT : 7
							},
							ROW_COUNT : {
								KEY : "ROW_COUNT",
								LOC_KEY : "label.sfRowCountText",
								PARAM_FIELD : "nRows",
								DEFAULT : 1000
							}
						},
						//gmail sorting
						//"NODE_ROOT" : 'root',
						//"NODE_MAILBOX" : 'mailbox',
						//"NODE_FOLDER" : 'folder',
						"SORT_ASC" : "asc",
						"SORT_DESC" : "desc",
						//end gmail sorting
						"DUMMYPASSWORD" : "DummyPassword",
						POLICY_MODAL_MODE : {
							NEW : "new",
							EDIT : "edit",
							CLONE : "clone"
						},
						INFRASTRUCTURE_POOL_APPTYPE : {
							SALESFORCE: 2
						},
						"POLICY_AVAILABLE" : "PolicyAvailable"

	});
	cvCommon.constant('BROWSE_NODE_TYPE', {
		FOLDER : 'folder'
	});
	cvCommon.constant('TREE_NODE_STATE', {
		FULLY_SELECTED : 0,
		PARTIALLY_SELECTED : 1,
		NOT_SELECTED : 2
	});
	cvCommon.constant('SCG_CONSTANTS', {
		OPERATION : {
			ALL : "SCGP_AND",
			ANY : "SCGP_OR",
			NOTANY : "SCGP_NOT"
		},
		FILTER_TYPES : {
			UNDEFINED : "SCGP_UNDEF",
			BOOL : "SCGP_BOOL",
			STRING : "SCGP_STRING",
			INT : "SCGP_INT",
			ENUM : "SCGP_ENUM",
			DATE : "SCGP_DATE",
			SELECTION : "SCGP_SELECTION",
			SELECTION_V2 : "SCGP_SELECTION_V2",
			INT_2_V2 : "SCGP_INT_2_V2",
			INT_3_V2 : "SCGP_INT_3_V2",
			STRING_V2 : "SCGP_STRING_V2"
		},
		FILTER_LABELS : {
			IS_TRUE : "dropdownOption.isTrue",
			IS_FALSE : "dropdownOption.isFalse",
			CONTAINS : "dropdownOption.contains",
			DOES_NOT_CONTAIN : "dropdownOption.doesNotContain",
			IS : "dropdownOption.is",
			IS_NOT : "dropdownOption.isNot",
			STARTS_WITH : "dropdownOption.startsWith",
			ENDS_WITH : "dropdownOption.endsWith",
			EQUAL_TO : "dropdownOption.equalTo",
			NOT_EQUAL_TO : "dropdownOption.notEqualTo",
			GREATER_THAN : "dropdownOption.greaterThan",
			LESS_THAN : "dropdownOption.lessThan",
			GREATER_THAN_OR_EQUAL_TO : "dropdownOption.greaterThanOrEqualTo",
			LESS_THAN_OR_EQUAL_TO : "dropdownOption.lessThanOrEqualTo",
			BETWEEN : "dropdownOption.between",
			NOT_IN : "dropdownOption.notIn",
			ANY_IN_SELECTION : "dropdownOption.anyInSelection",
			NOT_IN_SELECTION : "dropdownOption.notInSelection",
			IN : "dropdownOption.in",
			NOT_BETWEEN : "dropdownOption.notBetween"
		}
	});
	cvCommon.constant('OVERRIDE_OPTIONS', {
		MUST : 'Must',
		OPTIONAL : 'Optional',
		NOT_ALLOWED : 'NotAllowed'
	});
	cvCommon.constant('LIST_OPERATION_TYPES', { // Corresponds to ListOperationType.java
		NONE : 0,
		OVERWRITE : 1,
		ADD : 2,
		DELETE : 3,
		CLEAR : 4
	});
	cvCommon.constant('SIM_IDS', {
		SIM_ID_NAS_OS : 16,
		SIM_ID_EMC_CELERRA_OS : 17,
		SIM_ID_CHAMELEON_FILER_OS : 18,
		SIM_ID_GENERIC_FILER_OS : 19,
		SIM_ID_BLUEARC_FILER_OS : 29,
		SIM_ID_HITACHI_ENAS_OS : 45,
		SIM_ID_OPENVMS_OS : 49,
		SIM_ID_EMC_CENTERA_OS : 55,
		SIM_ID_IBM_ISERIES_OS : 118
	});
	cvCommon.constant('CLIENT_TYPE', {
		SALESFORCE : 3,
		VIRTUAL_SERVER_CLIENT : 12,
		CLOUD_CONNECTOR_CLIENT : 15,
		EDGE_DRIVE_CLIENT : 19,
		SHAREPOINT_FARM : 27,
		DISTRIBUTED_IDA : 29,
		UNIX_CLUSTER_CLIENT : 11,
		GIT_HUB : 33,
		AZURE_DEVOPS : 34
	});
	cvCommon.constant('SLA_OPTIONS', {
		USE_COMMCELL_LEVEL_SETTINGS : 0,
		USE_SLA_FOR_TIME_INTERVAL : 1,
		EXCLUDE_FROM_SLA : 2
	});
	cvCommon.constant('PLAN_TYPE', {
		ANY : 'Any',
		DLO : 'DLO',
		MSP : 'MSP',
		FS : 'FS',
		SNAP : 'SNAP',
		VSA : 'VSA',
		EXCHANGE : 'EXCHANGE',
		EDISCOVERY : 'EDISCOVERY'
	});
	cvCommon.constant('ATTRIBUTE_MAPPINGS', {
		USER_ATTRIBUTES : [ "user name", "user groups", "Email", "user guid", "SID", "company name" ]
	});
	cvCommon.constant('CLOUD_CONSTANTS', {
		SUBSCRIPTION_TYPE : {
			O365 : {
				key : "Office365",
				value : 2,
				localizeTypeKey : "label.nav.office365",
				urlText : "O365",
				svg : 'common/img/OfficeOrange.svg'
			},
			ENDPOINT : {
				key : "ENDPOINT",
				value : 1,
				localizeTypeKey : "label.nav.endpointprotection",
				urlText : "endpoint",
				svg : 'common/img/Endpoint-devices.svg'
			}
		}
	});
	cvCommon
			.constant(
					'OFFICE_365_CONSTANTS',
					{
						//For Office365
						"PLAN_TYPE_MSP" : "MSP",
						"PLAN_SUBTYPE_SERVER" : "Server",
						"MSG_LOADING" : "Loading",
						"REGEX_EMAIL" : /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/,
						"OFFICE_365_URL_TEXT" : { // to avoid loop search for text
							"1" : "exchange",
							"2" : "sharepoint",
							"7" : "oneDrive"
						},
						OFFICE_365_INSTANCE_TYPE : {
							//svg : 'common/img/sharepoint.svg'
							//svg : 'common/img/exchange-dag.svg'
							EXCHANGE : {
								key : "EXCHANGEONLINE",
								value : 1,
								localizeTypeKey : "label.nav.exchangeOnline",
								urlText : "exchange",
								svg : 'common/img/exchange.svg'
							},
							SHAREPOINT : {
								key : "SHAREPOINTONLINE",
								value : 2,
								localizeTypeKey : "label.nav.sharepointOnline",
								urlText : "sharepoint",
								svg : 'common/img/sharepoint.svg'
							},
							ONEDRIVE : {
								key : "ONEDRIVE",
								value : 7,
								localizeTypeKey : "label.nav.oneDriveForBusiness",
								urlText : "365oneDrive",
								svg : 'common/img/onedrive.svg'
							}
						},
						OFFICE_365_CONTENT_TYPE : {
							GOOGLE_CONTENT_TYPE_USER_ACC : {
								key : "GOOGLE_CONTENT_TYPE_USER_ACC",
								value : 134,
								toLocalizeTypeKey : "content.type.google.user.acc",
								iconClass : "capps-icon capps-email-type"
							},
							GOOGLE_CONTENT_TYPE_USER_GRP : {
								key : "GOOGLE_CONTENT_TYPE_USER_GRP",
								value : 135,
								toLocalizeTypeKey : "content.type.google.group.acc",
								iconClass : "capps-icon capps-group-type"
							},
							ONEDRIVE_CONTENT_TYPE_AZURE_GRP : {
								key : "ONEDRIVE_CONTENT_TYPE_AZURE_GRP",
								value : 135,
								toLocalizeTypeKey : "content.type.azure.group.acc",
								iconClass : "capps-icon capps-group-type"
							},
							GOOGLE_CONTENT_TYPE_USER_REGEX : {
								key : "GOOGLE_CONTENT_TYPE_USER_REGEX",
								value : 0,
								toLocalizeTypeKey : "",
								iconClass : "capps-icon capps-regex-type"
							}
						},
						"NO_OF_GOOGLE_USERS_FOR_PREVIEW" : 2,
						"NO_OF_GOOGLE_GRPS_FOR_PREVIEW" : 3,
						"NO_OF_REGEX_PATTERN_FOR_PREVIEW" : 2,
						"SC_CONTENT_REGEX_LOC_KEY" : "content.type.regex",
						"REGEX_TEXT" : "REGEX",
						"USER_GRP_TEXT" : "USER_GROUP",
						"BROWSE_PATH_PREFIX_GDRIVE" : "\\GDrive",
						"BROWSE_PATH_PREFIX_GMAIL" : "\\Gmail",
						'SALESFORCE_DEFAULT_TABLE' : "Organization",
						//Start Backup types
						BACKUP_TYPE : {
							INCREMENTAL : "INCREMENTAL",
							FULL : "FULL",
							SYNTH_FULL : "SYNTHETIC_FULL"
						},
						RESTORE_TYPE : {
							DISK : "disk",
							GOOGLE : "google",
							ACCOUNT : "account"
						},
						MAILBOX_NODE : {
							ROOT : 'root',
							MAILBOX : 'mailbox',
							MAILBOX_FOLDER : 'mailbox_folder'
						},
						'DUMMYPASSWORD' : 'DummyPassword'
					//end
					});
	cvCommon.constant('CONTENT_TYPES', {
		PATH : 'path',
		EXCLUDEPATH : 'excludePath',
		INCLUDEPATH : 'includePath',
		GLOBALFILTERS : 'globalFilters',
		SUBCLIENTPOLICYPATH : 'subclientPolicyPath',
		SUBCLIENTPOLICYEXCLUDEPATH : 'subclientPolicyExcludePath',
		SUBCLIENTPOLICYINCLUDEPATH : 'subclientPolicyIncludePath',
	});
	cvCommon.constant('GLOBAL_EXCLUSIONS_BIT_MASK', {
		POLICY_BIT : 16,
		CELL_LEVEL_BIT : 8,
		CLIENT_GROUP_BIT : 4,
		COMMCELL_BIT : 2,
		READ_ONLY_BIT : 1,
	});
	cvCommon.constant('LAPTOP_OWNERSHIP_TYPES', {
		CONFIGURE_USER_PROFILES_AS_OWNERS : 'CONFIGURE_USER_PROFILES_AS_OWNERS',
		CONFIGURE_LOGGED_IN_USER_AS_OWNER_ONCE : 'CONFIGURE_LOGGED_IN_USER_AS_OWNER_ONCE',
		CONFIGURE_USER_GROUPS_AS_OWNERS : 'CONFIGURE_USER_GROUPS_AS_OWNERS',
	});
	cvCommon.constant('ADDITIONAL_SETTING_CATEGORY', {
		ADMINCONSOLE : 'AdminConsole',
		CLIENT : 'Client',
		CONSOLE : 'Console',
		GXGLOBALPARAM : 'GxGlobalParam',
		MEDIAAGENT : 'MediaAgent',
		RESOURCEMANAGER : 'ResourceManager',
		WEBCONSOLE : 'WebConsole'
	});
	cvCommon.constant('UI_EVENT', {
		PASSKEY_AUTHORIZED : 'passkey:authorized'
	});


	cvCommon.factory('cvGridFactory', [ function() {
		var factory = {};
		//convert last1 hour,last 24 hours, last 48 hours and last week to actual time
		factory.convertTimeDuration = function(value) {
			var start;
			var end = new Date();
			switch (value) {
			case 'last1hour':
				start = new Date(end - 3600000)
				break;
			case 'last2hour':
				start = new Date(end - 7200000)
				break;
			case 'last24hour':
				start = new Date(end - 86400000)
				break;
			case 'last48hour':
				start = new Date(end - 172800000)
				break;
			case 'last1week':
				start = new Date(end - 604800000)
				break;
			default:
				start = 'custom';
			}
			return start;
		}
		return factory;
	} ]);

	cvCommon.factory('customAPIHeaderFactory', function() {
		var pairs = {};
		return {
			add : function(key, val) {
				pairs[key] = val;
			},
			set : function(allPairs) {
				pairs = allPairs;
			},
			remove : function(key) {
				delete pairs[key];
			},
			clear : function() {
				pairs = {};
			},
			get: function(){
				return pairs;
			},
			getAndClear : function() {
				let copy = Object.assign({}, pairs);
				pairs = {};
				return copy;
			}
		};
	});

	cvCommon.factory('httpWrapper', ['$http', '$q', function ($http, $q) {
        let wrapper = function (verb) {
            let promise = verb;
            promise.success = function (fn) {
                promise.then(function (response) {
                    fn(response.data, response.status, response.headers);
                });
                return promise;
            };

            promise.error = function (fn) {
                promise.then(null, function (response) {
                    fn(response.data, response.status, response.headers);
                });
                return promise;
            };
            return promise;
        };
        return {
            get: function (url, data) {
                return wrapper($http.get(url, data));
            },
            post: function (url, data, config) {
                return wrapper($http.post(url, data, config));
            },
            put: function (url, data, config) {
                return wrapper($http.put(url, data, config));
            },
            delete: function (url, data) {
                return wrapper($http.delete(url, data));
            }
        };
    }]);

	cvCommon.constant('DASHBOARD_LINKS', {
		DASHBOARD: '/dashboard',
		OVERVIEW : '/dashboard/commcell',
		TENANT : '/dashboard/tenant',
		VIRTUALIZATION : '/dashboard/virtualization',
		APPLIANCE : '/dashboard/hyperscale',
		APPLIANCE2 : '/dashboard/hyperscalecommcell',
		ENDPOINT : '/dashboard/endpoint',
		SERVER: '/dashboard/server',
		ACTIVATE: '/dashboard/activate',
		DISASTER_RECOVERY: '/dashboard/disasterRecovery'
	});

	cvCommon.constant('CLIENT_CONSTANT', {
		RetireClientPhase: {
			0: 'UNKNOWN',
			1: 'RETIRE_IN_PROGRESS',
			2: 'DECONFIGURED'
		},
		DISALLOWED_CHARS : "\\|`~!@#$%^&*()+=<>/?,[]{}:;'\" "
	});
	cvCommon.constant('USER_SETTINGS',{
		LANGUAGE: 'lang',
		USER_PREF: 'userPreference',
		LICENSE: 'license',
		ACCESSIBILITY_ON: 'accessibilityOn',
		ACCESSIBILITY_OFF: 'accessibilityOff',
		ABOUT: 'about',
		HELP: 'help',
		FEEDBACK: 'feedback',
		SUPPORT: 'support',
		CHANGE_PASS: 'changePassword',
		LOCK: 'lock',
		LOGOUT: 'logout',
		ACCESSIBILITY: 'accessibility'
	})

})();

